function M = factorygpuhelper(M)
% Returns a manifold struct to optimize over unit-norm vectors or matrices.
%
% function M = factorygpuhelper(M)
%
% Helper tool to add GPU support to factories. The input is a factory M
% created by one of Manopt's factories. The output is the same factory,
% with gather() and gpuArray() added in a number of places, following the
% logic that points, tangent vectors and ambient vectors are stored on the
% GPU (but scalars should be 'gathered' to the CPU). The name of the
% factory is also appended with '(GPU)'.
%
% This tool is typically called inside the factory itself, at the very end.
% It is not enough to call this tool: one also needs to create all arrays
% on the GPU directly. See spherefactory for an example.
%
% This tool is still in beta: please let us know about any issues via the
% forum on http://www.manopt.org. Thanks!
%
% See also: spherefactory

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Aug. 3, 2018.
% Contributors: 
% Change log: 

    % Tag the factory name.
    M.name = @() [M.name(), ' (GPU)'];
    
    % Gathering scalar outputs: it's unclear whether this is necessary.
    M.inner = @(x, u, v) gather(M.inner(x, u, v));
    M.norm = @(x, u) gather(M.norm(x, u));
    M.dist = @(x, y) gather(M.dist(x, y));
    
    % TODO: check that this works for manifolds whose points are not
    % matrices (but are structs or cells).
    M.hash = @(x) M.hash(gather(x));
    
    % The vec/mat pair is mostly used in the hessianspectrum tool, where
    % the vector representation of tangent vectors is assumed to be in
    % 'normal' memory (as opposed to GPU). But it's unclear whether we
    % actually need this too.
    M.vec = @(x, u_mat) gather(M.vec(x, u_mat));
    M.mat = @(x, u_vec) M.mat(x, gpuArray(u_vec));

end
