#!/usr/bin/env python
# coding=utf-8
# Copyright 2020 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning the library models for sequence classification on GLUE."""
# You can also adapt this script on your own text classification task. Pointers for this are left as comments.

import logging
import json
import time
import re
import os
import random
import sys
import math
import wandb
from dataclasses import dataclass, field
from typing import Optional, List, Dict
from rational.torch import Rational

import numpy as np

import torch
from torch import nn
from torch.utils.data import Dataset


from datasets import load_dataset, load_metric

import transformers
from transformers import (
    AdapterConfig,
    AutoConfig,
    AutoModelForSequenceClassification,
    AutoTokenizer,
    DataCollatorWithPadding,
    EvalPrediction,
    HfArgumentParser,
    MultiLingAdapterArguments,
    PretrainedConfig,
    Trainer,
    TrainingArguments,
    TrainerCallback,
    TrainerState,
    default_data_collator,
    set_seed,
)
from transformers.trainer_utils import get_last_checkpoint, is_main_process
from transformers.utils import check_min_version
import transformers.adapters.composition as ac

from transformers.integrations import WandbCallback, INTEGRATION_TO_CALLBACK

Rational.use_kde = False

# Will error if the minimal version of Transformers is not installed. Remove at
# your own risks.
check_min_version("4.5.0")

task_to_keys = {
    "cola": ("sentence", None),
    "mnli": ("premise", "hypothesis"),
    "mrpc": ("sentence1", "sentence2"),
    "qnli": ("question", "sentence"),
    "qqp": ("question1", "question2"),
    "rte": ("sentence1", "sentence2"),
    "sst2": ("sentence", None),
    "stsb": ("sentence1", "sentence2"),
    "wnli": ("sentence1", "sentence2"),
}


logger = logging.getLogger(__name__)


SWITCH_INPUT_CHOICES = ["minimal", "pfeiffer"]


def _get_switch_input(name: str):

    if len(tag.split(":")) == 1:
        adapter_name = tag.split(":")[0]


@dataclass
class CustomTrainingArguments(TrainingArguments):

    # Are we using the fused lamb optimizer?
    use_fused_lamb: bool = False

    # Temperature control.
    temp_N: int = 1
    temp_r: float = None
    temp_initial: float = 1.0
    temp_min: float = 0.1

    # Where to put switches.
    switches_at: List[int] = field(default_factory=list)

    # Fixed switch positions.
    fixed_configuration: List[int] = None

    # Fix last and first n layers.
    adapter_last_layers: int = None
    adapter_first_layers: int = None
    adapter_from_layer: int = None
    adapter_to_layer: int = None

    # If switches are used they use the same inputs.
    use_switches: bool = False
    switch_inputs: str = None

    # Another way to define switches.
    switch_at_0: bool = False
    switch_at_1: bool = False
    switch_at_2: bool = False
    switch_at_3: bool = False
    switch_at_4: bool = False
    switch_at_5: bool = False
    switch_at_6: bool = False
    switch_at_7: bool = False
    switch_at_8: bool = False
    switch_at_9: bool = False
    switch_at_10: bool = False
    switch_at_11: bool = False

    # Use a switch regularization.
    switch_regularization: str = None
    switch_regularization_weight: float = None
    switch_regularization_inputs_costs: List[float] = field(default_factory=list)

    # Default adapter.
    default_adapter: str = "rational"

    # Activate the bug in switch.
    switch_bug: bool = False

    # Another way to define switches.
    switch_at_0: bool = False
    switch_at_1: bool = False
    switch_at_2: bool = False
    switch_at_3: bool = False
    switch_at_4: bool = False
    switch_at_5: bool = False
    switch_at_6: bool = False
    switch_at_7: bool = False
    switch_at_8: bool = False
    switch_at_9: bool = False
    switch_at_10: bool = False
    switch_at_11: bool = False

    # Fix the switches
    fix_rational_switch: bool = False

    # Probability for soft fixed
    prob_for_soft_fixed: float = 0.9

    # Learning rate for probabilities.
    lr_for_switches: float = None

    # Probability regularization weight.
    prob_reg_weight: float = 0.0
    prob_reg_power: float = 0.5

    # Learning rate for the rational adapters.
    lr_for_rational_adapters: float = None
    lr_for_rational_activations: float = None

    # Replace rational by identity.
    rational_adapter_non_linearity: str = "rational:sigmoid"
    default_adapter_non_linearity: str = "relu"

    # Limit the total number of input_1 selections
    limit_input_1: bool = False
    limit_input_1_after: int = None
    limit_input_1_after_weight: float = 0.1
    limit_input_1_after_scale: float = 10.0

    # Simple regularization.
    simple_regularization_weight: float = None

    # Shuffle the samples
    shuffle_samples: bool = False

    # Use the validaton splt for testing.
    low_resources: int = None

    # Save the rational plots.
    save_rational_plots: bool = False

    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate "
            "the number of training examples to this value if set."
        },
    )
    max_val_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate "
            "the number of validation examples to this value if set."
        },
    )
    max_test_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate "
            "the number of test examples to this value if set."
        },
    )

    def __post_init__(self):

        if self.use_switches:
            if self.switch_inputs is None:
                raise ValueError("Please provide the inputs to the switches.")

        if self.switch_inputs is not None:
            switch_inputs = self.switch_inputs.split(",")
            for switch_input in switch_inputs:
                if switch_input.split(":")[0] not in SWITCH_INPUT_CHOICES:
                    raise ValueError("Incorrect switch options")

            a = len(self.switch_regularization_inputs_costs)
            b = len(switch_inputs)
            if a != b:
                raise ValueError(
                    "The argument switch_inputs_costs should have {b} values."
                )

        self.output_dir = self.output_dir.replace("%t", str(int(1e7 * time.time())))

        super().__post_init__()

        if self.adapter_first_layers is not None:
            n = self.adapter_first_layers
            self.fixed_configuration = [1] * n + [0] * (12 - n)

        elif self.adapter_last_layers is not None:
            n = self.adapter_last_layers
            self.fixed_configuration = [0] * (12 - n) + [1] * n

        elif self.adapter_from_layer is not None and self.adapter_to_layer is not None:
            a = self.adapter_from_layer
            b = self.adapter_to_layer
            self.fixed_configuration = [0] * a + [1] * (b - a + 1) + [0] * (12 - b - 1)

        if self.fixed_configuration is not None:
            assert len(self.fixed_configuration) == 12, "We need 12 positions."

        if self.lr_for_switches is None:
            self.lr_for_switches = self.learning_rate

        if (
            self.lr_for_rational_activations is not None
            and self.lr_for_rational_adapters is not None
        ):
            raise Exception("Both lr_for_rational_* can not be set.")

        if self.lr_for_rational_activations is None:
            self.lr_for_rational_activations = self.learning_rate

        if self.lr_for_rational_adapters is None:
            self.lr_for_rational_adapters = self.lr_for_rational_activations

        # Default value for temp_r.
        if self.temp_r is None:
            n = self.num_train_epochs / 2
            self.temp_r = -math.log(self.temp_min / self.temp_initial) / n

        for i in range(12):

            # If any flag is on put it in the list.
            if getattr(self, f"switch_at_{i}") and i not in self.switches_at:
                self.switches_at.append(0)

            # Turn on the flags based on the list.
            setattr(self, f"switch_at_{i}", i in self.switches_at)


class CustomTrainer(Trainer):

    args: CustomTrainingArguments

    def set_switch_mode(self, eval_mode: str):
        for name, module in self.model.named_modules():
            if name.endswith(".adapter_switch_layer"):
                for key, adapter_switch in module.items():
                    adapter_switch.set_mode(eval_mode)

    def number_of_required_parameters(self) -> int:

        per_layer = {}

        # Regular explressions to detect the layer.
        re_adapters = re.compile(
            r"^.*layer\.(?P<layer>[0-9]+).*adapters\.(?P<adapter>.*?)\..*$"
        )

        re_switch = re.compile(
            r"^.*layer\.(?P<layer>[0-9]+)\.output.*switch_layer\.(?P<name>.*?)\.switch_logits$"
        )

        # Track the current values of the switches.
        positions = {}

        for n, p in self.model.named_parameters():

            if not p.requires_grad:
                continue

            res = re_adapters.match(n)
            if res:
                layer = int(res["layer"])

                # Add the size of the current parameter.
                if layer not in per_layer:
                    per_layer[layer] = {}
                if res["adapter"] not in per_layer[layer]:
                    per_layer[layer][res["adapter"]] = 0
                per_layer[layer][res["adapter"]] += math.prod(p.size())

            res = re_switch.match(n)
            if res:
                layer = int(res["layer"])
                names = res["name"].split(",")
                idx = torch.argmax(p, dim=-1)
                positions[layer] = names[idx]

        num_params = 0
        for layer in range(12):
            if layer in positions:
                if positions[layer] in per_layer[layer]:
                    num_params += per_layer[layer][positions[layer]]
            elif layer in per_layer:
                num_params += sum(per_layer[layer].values())
        return num_params

    def predict(
        self,
        test_dataset: Dataset,
        ignore_keys: Optional[List[str]] = None,
        metric_key_prefix: str = "test",
    ):
        results = super().predict(
            test_dataset, ignore_keys=ignore_keys, metric_key_prefix=metric_key_prefix
        )

        # Add the required params in the test run.
        tag = f"{metric_key_prefix}_required_params"
        results.metrics[tag] = self.number_of_required_parameters()

        return results

    def evaluate(
        self,
        eval_dataset: Optional[Dataset] = None,
        ignore_keys: Optional[List[str]] = None,
        metric_key_prefix: str = "eval",
    ) -> Dict[str, float]:

        prefix = metric_key_prefix

        # Evaluate the position of the switches.
        metrics = {f"{prefix}_required_params": self.number_of_required_parameters()}

        if eval_dataset is not None:
            metrics[f"{metric_key_prefix}_samples"] = len(eval_dataset)

        # Change the switches.
        self.set_switch_mode("hard")

        # Call the original evaluation loop.
        metrics.update(
            super().evaluate(
                eval_dataset, ignore_keys, metric_key_prefix=metric_key_prefix
            )
        )

        self.log(metrics)
        self.log_metrics("eval", metrics)
        self.save_metrics("eval", metrics)
        return metrics


def rewrite_logs(d):
    new_d = {}

    re_eval = re.compile("^eval_")
    re_test = re.compile("^test_")

    for k, v in d.items():
        if re_eval.match(k):
            new_d[re_eval.sub("eval/", k)] = v
        elif re_test.match(k):
            new_d[re_test.sub("test/", k)] = v
        else:
            new_d["train/" + k] = v
    return new_d


class CustomWandbCallback(WandbCallback):
    _re = re.compile(r".*\.layer\.([0-9]+)\.output.*\.switch_logits")

    def on_step_begin(
        self, args: CustomTrainingArguments, state: TrainerState, control, **kwargs
    ):
        if args.save_rational_plots:
            if state.global_step % args.logging_steps == 0:
                Rational.save_all_inputs(True)

    def on_step_end(
        self, args: CustomTrainingArguments, state: TrainerState, control, **kwargs
    ):
        if args.save_rational_plots:
            if state.global_step % args.logging_steps == 0 and len(Rational.list) > 0:
                Rational.capture_all(f"Global Step {state.global_step}")
                filename = f"{args.output_dir}/ra_{state.global_step}.png"
                Rational.export_graphs(filename)
                self._wandb.log(
                    {
                        "train/rational_activations": wandb.Image(filename),
                        "train/global_step": state.global_step,
                    }
                )
                Rational.save_all_inputs(False)

    def on_log(self, args, state, control, **kwargs):

        if self._wandb is None:
            return

        # Capture the model.
        model: nn.Module = kwargs.pop("model", None)

        if not self._initialized:
            self.setup(args, state, model)

        # Capture the logs.
        logs = kwargs.pop("logs", {}) or {}

        # Get the number of parameters that need training.
        params_t = [p for p in model.parameters() if p.requires_grad]

        # Total number of parameters.
        logs["num_params"] = sum(math.prod(p.size()) for p in params_t)

        # Logs temperature.
        for name, buf in model.named_buffers():
            if name.endswith(".temperature"):
                logs["temperature"] = buf[0].item()
                break

        # Logs probs.
        for name, param in model.named_parameters():
            match = self._re.match(name)
            if match:
                layer_idx = int(match.groups()[0])
                prob = torch.softmax(param, dim=-1)
                for i in range(prob.size()[0]):
                    logs[f"layer.{layer_idx}.prob.{i}"] = prob[i].item()

        if state.is_world_process_zero:
            logs = rewrite_logs(logs)
            self._wandb.log({**logs, "train/global_step": state.global_step})


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for
    training and eval.

    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """

    task_name: Optional[str] = field(
        default=None,
        metadata={
            "help": "The name of the task to train on: "
            + ", ".join(task_to_keys.keys())
        },
    )
    max_seq_length: int = field(
        default=128,
        metadata={
            "help": "The maximum total input sequence length after "
            "tokenization. Sequences longer than this will be truncated, "
            "sequences shorter will be padded."
        },
    )
    overwrite_cache: bool = field(
        default=False,
        metadata={"help": "Overwrite the cached preprocessed datasets or not."},
    )
    pad_to_max_length: bool = field(
        default=True,
        metadata={
            "help": "Whether to pad all samples to `max_seq_length`. "
            "If False, will pad the samples dynamically when batching to the maximum length in the batch."
        },
    )
    train_file: Optional[str] = field(
        default=None,
        metadata={"help": "A csv or a json file containing the training data."},
    )
    validation_file: Optional[str] = field(
        default=None,
        metadata={"help": "A csv or a json file containing the validation data."},
    )
    test_file: Optional[str] = field(
        default=None,
        metadata={"help": "A csv or a json file containing the test data."},
    )

    def __post_init__(self):
        if self.task_name is not None:
            self.task_name = self.task_name.lower()
            if self.task_name not in task_to_keys.keys():
                raise ValueError(
                    "Unknown task, you should pick one in "
                    + ",".join(task_to_keys.keys())
                )
        elif self.train_file is None or self.validation_file is None:
            raise ValueError("Need either a GLUE task or a training/validation file.")
        else:
            train_extension = self.train_file.split(".")[-1]
            assert train_extension in [
                "csv",
                "json",
            ], "`train_file` should be a csv or a json file."

            validation_extension = self.validation_file.split(".")[-1]
            assert validation_extension == train_extension, (
                "`validation_file` should have the same extension "
                "(csv or json) as `train_file`."
            )


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to
    fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={
            "help": "Path to pretrained model or model identifier from "
            "huggingface.co/models"
        }
    )
    config_name: Optional[str] = field(
        default=None,
        metadata={
            "help": "Pretrained config name or path if not the same as model_name"
        },
    )
    tokenizer_name: Optional[str] = field(
        default=None,
        metadata={
            "help": "Pretrained tokenizer name or path if not the same as model_name"
        },
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={
            "help": "Where do you want to store the pretrained models downloaded from huggingface.co"
        },
    )
    use_fast_tokenizer: bool = field(
        default=True,
        metadata={
            "help": "Whether to use one of the fast tokenizer (backed by the tokenizers library) or not."
        },
    )
    model_revision: str = field(
        default="main",
        metadata={
            "help": "The specific model version to use (can be a branch name, tag name or commit id)."
        },
    )
    use_auth_token: bool = field(
        default=False,
        metadata={
            "help": "Will use the token generated when running `transformers-cli login` (necessary to use this script "
            "with private models)."
        },
    )


class TemperatureControl(TrainerCallback):
    """
    This callback controls the temperature according to the rule
    in arxiv:1611.01144v5.
    """

    def on_train_begin(
        self, args: CustomTrainingArguments, state, control, model=None, **kwargs
    ):
        self._temp_buffers = []
        for name, buf in model.named_buffers():
            if name.endswith(".temperature"):
                buf[0] = args.temp_initial
                self._temp_buffers.append(buf)

    def on_step_begin(
        self,
        args: CustomTrainingArguments,
        state: TrainerState,
        control,
        model=None,
        **kwargs,
    ):

        # Use the epoch to control the temperature.
        t = args.temp_N * (state.epoch // args.temp_N)

        # Compute the temperature
        temp = max(args.temp_min, args.temp_initial * math.exp(-args.temp_r * t))

        # Set all te temperature buffers at the same value.
        for buf in self._temp_buffers:
            buf[0] = temp


def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser(
        (
            ModelArguments,
            DataTrainingArguments,
            CustomTrainingArguments,
            MultiLingAdapterArguments,
        )
    )

    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        all_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    elif len(sys.argv) == 1:
        # Arguments passed by stdin as json.
        data = sys.stdin.read()
        print("Loading from STDIN: ", data)
        all_args = parser.parse_dict(json.loads(data))
    else:
        all_args = parser.parse_args_into_dataclasses()

    args: CustomTrainingArguments

    model_args, data_args, args, adapter_args = all_args

    # Adding the current time to the output dir.
    args.output_dir = args.output_dir.replace("%t", str(int(1e7 * time.time())))

    # Detecting last checkpoint.
    last_checkpoint = None

    if (
        os.path.isdir(args.output_dir)
        and args.do_train
        and not args.overwrite_output_dir
    ):
        last_checkpoint = get_last_checkpoint(args.output_dir)
        if last_checkpoint is None and len(os.listdir(args.output_dir)) > 0:
            raise ValueError(
                f"Output directory ({args.output_dir}) already exists and is not empty. "
                "Use --overwrite_output_dir to overcome."
            )
        elif last_checkpoint is not None:
            logger.info(
                f"Checkpoint detected, resuming training at {last_checkpoint}. To avoid this behavior, change "
                "the `--output_dir` or add `--overwrite_output_dir` to train from scratch."
            )

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s -   %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )
    logger.setLevel(logging.INFO if is_main_process(args.local_rank) else logging.WARN)

    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {args.local_rank}, device: {args.device}, n_gpu: {args.n_gpu}"
        + f"distributed training: {bool(args.local_rank != -1)}, 16-bits training: {args.fp16}"
    )
    # Set the verbosity to info of the Transformers logger (on main process only):
    if is_main_process(args.local_rank):
        transformers.utils.logging.set_verbosity_info()
        transformers.utils.logging.enable_default_handler()
        transformers.utils.logging.enable_explicit_format()
    logger.info(f"Training/evaluation parameters {args}")

    # Set seed before initializing model.
    set_seed(args.seed)

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).
    #
    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.
    #
    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)
    #
    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if data_args.task_name is not None:
        # Downloading and loading a dataset from the hub.
        datasets = load_dataset("glue", data_args.task_name)
    else:
        # Loading a dataset from your local files.
        # CSV/JSON training and evaluation files are needed.
        data_files = {
            "train": data_args.train_file,
            "validation": data_args.validation_file,
        }

        # Get the test dataset: you can provide your own CSV/JSON test file (see below)
        # when you use `do_predict` without specifying a GLUE benchmark task.
        if args.do_predict:
            if data_args.test_file is not None:
                train_extension = data_args.train_file.split(".")[-1]
                test_extension = data_args.test_file.split(".")[-1]
                assert (
                    test_extension == train_extension
                ), "`test_file` should have the same extension (csv or json) as `train_file`."
                data_files["test"] = data_args.test_file
            else:
                raise ValueError(
                    "Need either a GLUE task or a test file for `do_predict`."
                )

        for key in data_files.keys():
            logger.info(f"load a local file for {key}: {data_files[key]}")

        if data_args.train_file.endswith(".csv"):
            # Loading a dataset from local csv files
            datasets = load_dataset("csv", data_files=data_files)
        else:
            # Loading a dataset from local json files
            datasets = load_dataset("json", data_files=data_files)
    # See more about loading any type of standard or custom dataset at
    # https://huggingface.co/docs/datasets/loading_datasets.html.

    # Labels
    if data_args.task_name is not None:
        is_regression = data_args.task_name == "stsb"
        if not is_regression:
            label_list = datasets["train"].features["label"].names
            num_labels = len(label_list)
        else:
            num_labels = 1
    else:
        # Trying to have good defaults here, don't hesitate to tweak to your needs.
        is_regression = datasets["train"].features["label"].dtype in [
            "float32",
            "float64",
        ]
        if is_regression:
            num_labels = 1
        else:
            # A useful fast method:
            # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
            label_list = datasets["train"].unique("label")
            label_list.sort()  # Let's sort it for determinism
            num_labels = len(label_list)

    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(
        model_args.config_name
        if model_args.config_name
        else model_args.model_name_or_path,
        num_labels=num_labels,
        finetuning_task=data_args.task_name,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name
        if model_args.tokenizer_name
        else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    model = AutoModelForSequenceClassification.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    if adapter_args.train_adapter and args.use_switches:

        # Configurations
        switch_inputs = []

        # First collect the inputs for the switches.
        for s_input in args.switch_inputs.split(","):

            adapter_identifier = s_input.replace(":", "_")
            switch_inputs.append(adapter_identifier)

            adapter_name = s_input.split(":")[0]
            if adapter_name == "minimal":
                adapter_name = "rational"

            adapter_activation = None
            if len(s_input.split(":")) > 1:
                adapter_activation = s_input.split(":", 1)[1]

            # Add the switch adapters.
            model.add_adapter(
                adapter_identifier,
                config=AdapterConfig.load(
                    adapter_name, non_linearity=adapter_activation
                ),
            )

        adapter_switch = ac.Switch(*switch_inputs)
        switch_config = {
            "strategy": "global",
            "regularization": args.switch_regularization,
            "regularization_weight": args.switch_regularization_weight,
            "regularization_inputs_costs": args.switch_regularization_inputs_costs
        }
        model.add_adapter_switch(adapter_switch, config=switch_config)
        model.train_adapter_switch(adapter_switch)

    # Setup adapters
    elif adapter_args.train_adapter:

        leave_out_rational = []
        leave_out_pfeiffer = []

        if args.fixed_configuration is not None:
            for i, pos in enumerate(args.fixed_configuration):
                if pos == 0:
                    leave_out_pfeiffer.append(i)
                else:
                    leave_out_rational.append(i)

        # Fixed switch at these layers.
        switches_at = args.switches_at
        fixed_switches = [k for k in range(12) if k not in switches_at]

        # Load basic adapters.
        config_rational = AdapterConfig.load(
            "rational",
            leave_out=leave_out_rational,
            non_linearity=args.rational_adapter_non_linearity,
        )

        config_pfeiffer = AdapterConfig.load(
            "pfeiffer",
            leave_out=leave_out_pfeiffer,
            non_linearity=args.default_adapter_non_linearity,
        )

        # Add the switch adapters.
        model.add_adapter("rational", config=config_rational)
        model.add_adapter("pfeiffer", config=config_pfeiffer)

        if args.fixed_configuration is not None:
            logger.info("Fixing the configuration without switches")
            stack = ac.Stack("rational", "pfeiffer")
            model.active_adapters = stack
            model.train_adapter(stack)
        else:
            default_input = 0
            if args.default_adapter == "pfeiffer":
                default_input = 1

            sel_reg_weights = [0.0, 0.0]
            if args.limit_input_1:
                sel_reg_weights[1] = 0.1

            adapter_switch = ac.Switch("rational", "pfeiffer")
            switch_config = {
                "strategy": "global",
                "fixed": {k: default_input for k in fixed_switches},
                "fixed_soft": {k: default_input for k in switches_at},
                "soft_fixed_prob": args.prob_for_soft_fixed,
                "bug": args.switch_bug,
                "prob_regularization_power": args.prob_reg_power,
                "prob_regularization_weight": args.prob_reg_weight,
                "selection_regularization_weights": sel_reg_weights,
                "limit_input_1_after": args.limit_input_1_after,
                "limit_input_1_after_scale": args.limit_input_1_after_scale,
                "limit_input_1_after_weight": args.limit_input_1_after_weight,
                "simple_regularization_weight": args.simple_regularization_weight,
            }
            model.add_adapter_switch(adapter_switch, config=switch_config)
            model.train_adapter_switch(adapter_switch)

    else:
        if adapter_args.load_adapter or adapter_args.load_lang_adapter:
            raise ValueError(
                "Adapters can only be loaded in adapters training mode."
                "Use --train_adapter to enable adapter training"
            )

    # Preprocessing the datasets
    if data_args.task_name is not None:
        sentence1_key, sentence2_key = task_to_keys[data_args.task_name]
    else:
        # Again, we try to have some nice defaults but don't hesitate to tweak to your use case.
        non_label_column_names = [
            name for name in datasets["train"].column_names if name != "label"
        ]
        if (
            "sentence1" in non_label_column_names
            and "sentence2" in non_label_column_names
        ):
            sentence1_key, sentence2_key = "sentence1", "sentence2"
        else:
            if len(non_label_column_names) >= 2:
                sentence1_key, sentence2_key = non_label_column_names[:2]
            else:
                sentence1_key, sentence2_key = non_label_column_names[0], None

    # Padding strategy
    if data_args.pad_to_max_length:
        padding = "max_length"
    else:
        # We will pad later, dynamically at batch creation, to the max sequence length in each batch
        padding = False

    # Some models have set the order of the labels to use, so let's make sure we do use it.
    label_to_id = None
    if (
        model.config.label2id != PretrainedConfig(num_labels=num_labels).label2id
        and data_args.task_name is not None
        and not is_regression
    ):
        # Some have all caps in their config, some don't.
        label_name_to_id = {k.lower(): v for k, v in model.config.label2id.items()}
        if list(sorted(label_name_to_id.keys())) == list(sorted(label_list)):
            label_to_id = {
                i: int(label_name_to_id[label_list[i]]) for i in range(num_labels)
            }
        else:
            logger.warn(
                "Your model seems to have been trained with labels, but they don't match the dataset: ",
                f"model labels: {list(sorted(label_name_to_id.keys()))}, dataset labels: {list(sorted(label_list))}."
                "\nIgnoring the model labels as a result.",
            )
    elif data_args.task_name is None and not is_regression:
        label_to_id = {v: i for i, v in enumerate(label_list)}

    if data_args.max_seq_length > tokenizer.model_max_length:
        logger.warn(
            f"The max_seq_length passed ({data_args.max_seq_length}) is larger than the maximum length for the"
            f"model ({tokenizer.model_max_length}). Using max_seq_length={tokenizer.model_max_length}."
        )
    max_seq_length = min(data_args.max_seq_length, tokenizer.model_max_length)

    def preprocess_function(examples):

        # Tokenize the texts
        sentences = (examples[sentence1_key],)
        if sentence2_key is not None:
            sentences += (examples[sentence2_key],)

        result = tokenizer(
            *sentences, padding=padding, max_length=max_seq_length, truncation=True
        )

        # Map labels to IDs (not necessary for GLUE tasks)
        if label_to_id is not None and "label" in examples:
            result["label"] = []
            for l in examples["label"]:
                result["label"].append(-1 if l == -1 else label_to_id[l])
        return result

    datasets = datasets.map(
        preprocess_function,
        batched=True,
        load_from_cache_file=not data_args.overwrite_cache,
    )

    # Default values for the datasets.
    train_dataset = eval_dataset = test_dataset = None

    if args.do_train:
        if "train" not in datasets:
            raise ValueError("--do_train requires a train dataset")
        train_dataset = datasets["train"]
        if args.max_train_samples is not None:
            if args.shuffle_samples:
                train_dataset = train_dataset.shuffle()
            train_dataset = train_dataset.select(range(args.max_train_samples))

    if args.do_eval:
        if "validation" not in datasets and "validation_matched" not in datasets:
            raise ValueError("--do_eval requires a validation dataset")

        if data_args.task_name == "mnli":
            eval_dataset = datasets["validation_matched"]
        else:
            eval_dataset = datasets["validation"]

        if args.max_val_samples is not None:
            eval_dataset = eval_dataset.select(range(args.max_val_samples))

    if (
        args.do_predict
        or data_args.task_name is not None
        or data_args.test_file is not None
    ):
        if "test" not in datasets and "test_matched" not in datasets:
            raise ValueError("--do_predict requires a test dataset")
        test_dataset = datasets[
            "test_matched" if data_args.task_name == "mnli" else "test"
        ]
        if args.max_test_samples is not None:
            test_dataset = test_dataset.select(range(args.max_test_samples))
        test_dataset.remove_columns_("label")

    if args.low_resources is not None:
        split_at = int(args.low_resources * 0.75)
        train_dataset = datasets["train"].shuffle()
        eval_dataset = train_dataset.select(range(split_at, args.low_resources))
        train_dataset = train_dataset.select(range(split_at))
        if data_args.task_name == "mnli":
            test_dataset = datasets["validation_matched"]
        else:
            test_dataset = datasets["validation"]

    # Log a few random samples from the training set:
    if train_dataset is not None:
        for index in random.sample(range(len(train_dataset)), 3):
            logger.info(f"Sample {index} of the training set: {train_dataset[index]}.")

    # Get the metric function
    if data_args.task_name is not None:
        metric = load_metric("glue", data_args.task_name)
    # TODO: When datasets metrics include regular accuracy, make an else here and remove special branch from

    # compute_metrics
    # You can define your custom compute_metrics function. It takes
    # an `EvalPrediction` object (a namedtuple with a predictions and
    # label_ids field) and has to return a dictionary string to float.
    def compute_metrics(p: EvalPrediction):

        preds = p.predictions
        if isinstance(preds, tuple):
            preds = preds[0]

        if is_regression:
            preds = np.squeeze(preds)
        else:
            preds = np.argmax(preds, axis=1)

        result = {}

        if data_args.task_name is not None:
            result = metric.compute(predictions=preds, references=p.label_ids)
            if len(result) > 1:
                result["combined_score"] = np.mean(list(result.values())).item()
        elif is_regression:
            result["mse"] = ((preds - p.label_ids) ** 2).mean().item()
        else:
            result["accuracy"] = (preds == p.label_ids).astype(np.float32).mean().item()

        return result

    # Data collator will default to DataCollatorWithPadding, so we change it
    # if we already did the padding.
    if data_args.pad_to_max_length:
        data_collator = default_data_collator
    elif args.fp16:
        data_collator = DataCollatorWithPadding(tokenizer, pad_to_multiple_of=8)
    else:
        data_collator = None

    # Default optimizer.
    optimizer = None

    # All params
    params_n = [(n, p) for n, p in model.named_parameters() if p.requires_grad]

    # Split the parameters for differential learning rates.
    params_rational = []
    params_switches = []
    params_rest = []

    for n, p in model.named_parameters():
        if not p.requires_grad:
            continue
        if n.endswith(".switch_logits"):
            params_switches.append(p)
        elif n.endswith(".f.numerator"):
            params_rational.append(p)
        elif n.endswith(".f.denominator"):
            params_rational.append(p)
        else:
            params_rest.append(p)

    # Parameters for the optimizers
    params_optim = [
        {"params": params_rest},
        {"params": params_switches, "lr": args.lr_for_switches},
        {"params": params_rational, "lr": args.lr_for_rational_activations},
    ]

    # Check if we should use the fused_lamb optimizer.
    if args.use_fused_lamb:
        logger.info("Using the fused lamb optimizer.")

        from apex.optimizers import FusedLAMB

        # Use LAMB.
        optimizer = FusedLAMB(
            params_optim,
            lr=args.learning_rate,
            bias_correction=False,
            max_grad_norm=1.0,
            weight_decay=args.weight_decay,
        )

    else:
        optimizer = torch.optim.Adam(
            params_optim, lr=args.learning_rate, weight_decay=args.weight_decay
        )

    # Initialize our Trainer
    trainer = CustomTrainer(
        model=model,
        args=args,
        train_dataset=train_dataset,
        eval_dataset=eval_dataset,
        compute_metrics=compute_metrics,
        tokenizer=tokenizer,
        data_collator=data_collator,
        do_save_full_model=not adapter_args.train_adapter,
        do_save_adapters=adapter_args.train_adapter,
        optimizers=(optimizer, None),
        callbacks=[TemperatureControl],
    )

    # Change the switches.
    trainer.set_switch_mode("hard")

    # Training
    if train_dataset is not None:

        checkpoint = None

        if last_checkpoint is not None:
            checkpoint = last_checkpoint
        elif os.path.isdir(model_args.model_name_or_path):
            # Check the config from that potential checkpoint has the right
            # number of labels before using it as a checkpoint.
            model_cand = AutoConfig.from_pretrained(model_args.model_name_or_path)

            if model_cand.num_labels == num_labels:
                checkpoint = model_args.model_name_or_path

        # Get the number of parameters that need training.
        params = model.named_parameters()
        params_t = {n: p for n, p in params if p.requires_grad}

        # Total number of parameters.
        num_params = sum(math.prod(p.size()) for p in params_t.values())

        logger.info(f"Training {len(params_t)} params of total size {num_params}.")
        logger.info(
            "The following parameters will be trained:\n"
            + "\n".join(f" - {n}" for n in params_t)
        )

        # Execute training.
        train_result = trainer.train(resume_from_checkpoint=checkpoint)

        # Get the number of parameters that need training.
        num_params = sum(
            math.prod(p.size()) for p in model.parameters() if p.requires_grad
        )

        metrics = train_result.metrics
        max_train_samples = (
            args.max_train_samples
            if args.max_train_samples is not None
            else len(train_dataset)
        )
        metrics["train_samples"] = min(max_train_samples, len(train_dataset))

        # Save the last model.
        trainer.save_model()

        trainer.log_metrics("train", metrics)
        trainer.save_metrics("train", metrics)
        trainer.save_state()

    # Evaluation
    if args.do_eval:
        logger.info("*** Evaluate ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.task_name]
        eval_datasets = [eval_dataset]
        # if data_args.task_name == "mnli":
        #     tasks.append("mnli-mm")
        #     eval_datasets.append(datasets["validation_mismatched"])

        for eval_dataset, task in zip(eval_datasets, tasks):
            trainer.evaluate(eval_dataset=eval_dataset, metric_key_prefix="eval_final")

    if args.do_predict:
        logger.info("*** Test ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.task_name]
        test_datasets = [test_dataset]

        # if data_args.task_name == "mnli":
        #     tasks.append("mnli-mm")
        #     test_datasets.append(datasets["test_mismatched"])

        for test_dataset, task in zip(test_datasets, tasks):
            # Removing the `label` columns because it contains -1 and Trainer won't like that.
            trainer.set_switch_mode("hard")
            result = trainer.predict(test_dataset=test_dataset)
            metrics = result.metrics.copy()
            max_test_samples = len(test_dataset)
            if args.max_test_samples is not None:
                max_test_samples = args.max_test_samples
            metrics["test_samples"] = min(max_test_samples, len(test_dataset))

            trainer.log(metrics)
            trainer.log_metrics("test", metrics)
            trainer.save_metrics("test", metrics)

            predictions = result.predictions
            if is_regression:
                predictions = np.squeeze(predictions)
            else:
                predictions = np.argmax(predictions, axis=1)

            output_test_file = os.path.join(args.output_dir, f"test_results_{task}.txt")

            if trainer.is_world_process_zero():
                with open(output_test_file, "w") as writer:
                    logger.info(f"***** Test results {task} *****")
                    writer.write("index\tprediction\n")
                    for index, item in enumerate(predictions):
                        if is_regression:
                            writer.write(f"{index}\t{item:3.3f}\n")
                        else:
                            item = label_list[item]
                            writer.write(f"{index}\t{item}\n")


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    INTEGRATION_TO_CALLBACK["wandb"] = CustomWandbCallback
    main()
