import os
import pandas as pd
from tqdm import tqdm
import pickle

import openai
from tenacity import (
    retry,
    stop_after_attempt,
    wait_random_exponential,
)

# Load your API key from an environment variable or secret management service
openai.api_key = os.getenv("OPENAI_API_KEY")

@retry(wait=wait_random_exponential(min=5, max=60), stop=stop_after_attempt(6))
def completion_with_backoff(**kwargs):
    return openai.ChatCompletion.create(**kwargs)

def truncate_text(text):
    if len(text.split()) < 2500:
        return text

    _text_turns = text.split('\n')
    _n_turns = len(_text_turns)
    _discard_idx = int(_n_turns/2)
    if _text_turns[_discard_idx].startswith('C'):
        _discard_idx -= 1
    _discard_indices = [_discard_idx, _discard_idx+1]
    _remaining_turns = [_text_turns[idx] for idx in range(_n_turns) if idx not in _discard_indices]
    
    return truncate_text('\n'.join(_remaining_turns))

def retrieve_data():
    # df_transcripts = pd.read_csv(os.getcwd()+'/data/transcripts.tsv', sep='\t')
    df_transcripts = pd.read_csv(os.getcwd()+'/data/extended_transcripts.tsv', sep='\t')

    # id_to_text = {}
    datadict = {'id':[], 'text':[], 'label':[]}
    for idx, row in df_transcripts.iterrows():
        _text = row['text'].replace('___HelpSeeker___','HelpSeeker: ').replace('___Counselor___','Counselor: ')
        _text = truncate_text(_text)
        datadict['id'].append(row['id'])
        datadict['text'].append(_text)
        datadict['label'].append(row['positive'])

    return pd.DataFrame(data=datadict)

def prompt_clf(text_list):
    answer_texts = []
    for _conv in tqdm(text_list):
        _question = "Do you think the HelpSeeker would have felt more positive after the conversation? Answer '1' if they would feel more positive, and answer '0' if they would feel neither more positive, nor more negative, and answer '-1' if they would feel more negative."
        
        message_list = [
            {"role": "system", "content": "You are a helpful assistant to help me understand the chat conversation between HelpSeeker and Counselor. Briefly answer questions about the conversation."},
            {"role": "user", "content": _conv + "\n\n" + _question}
        ]

        response = completion_with_backoff(model="gpt-3.5-turbo", messages=message_list)
        answer_texts.append(response['choices'][0]['message']['content'])

    return answer_texts

data_df = retrieve_data()
all_texts, all_ids, all_labels = data_df['text'].tolist(), data_df['id'].tolist(), data_df['label'].tolist()
predictions = prompt_clf(all_texts)

df = pd.DataFrame(data={'id':all_ids, 'preds':predictions})
df.to_csv(os.getcwd()+'/data/chatgpt_convClf.tsv', sep='\t', index=False)