function M = spherecomplexfactory(n, m)
% Returns a manifold struct to optimize over unit-norm complex matrices.
%
% function M = spherecomplexfactory(n)
% function M = spherecomplexfactory(n, m)
%
% Manifold of n-by-m complex matrices of unit Frobenius norm.
% By default, m = 1, which corresponds to the unit sphere in C^n. The
% metric is such that the sphere is a Riemannian submanifold of the space
% of 2nx2m real matrices with the usual trace inner product, i.e., the
% usual metric.
% 
% See also: spherefactory

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Dec. 30, 2012.
% Contributors: 
% Change log: 
%
%   Sep. 4, 2014 (NB):
%       Added ehess2rhess.
%
%   April 7, 2015 (NB):
%       Added vec/mat pair (for use with hessianspectrum, for example).
%
%   April 13, 2015 (NB):
%       Added logarithm
%
%   Oct. 8, 2016 (NB)
%       Code for exponential was simplified to only treat the zero vector
%       as a particular case.
%
%   Oct. 22, 2016 (NB)
%       Distance function dist now significantly more accurate for points
%       within 1e-7 and less from each other.

    
    if ~exist('m', 'var')
        m = 1;
    end

    if m == 1
        M.name = @() sprintf('Complex sphere S^%d', n-1);
    else
        M.name = @() sprintf('Unit F-norm %dx%d complex matrices', n, m);
    end
    
    M.dim = @() 2*(n*m)-1;
    
    M.inner = @(x, d1, d2) real(d1(:)'*d2(:));
    
    M.norm = @(x, d) norm(d, 'fro');
    
    M.dist = @(x, y) real(2*asin(.5*norm(x - y, 'fro')));
    
    M.typicaldist = @() pi;
    
    M.proj = @(x, d) reshape(d(:) - x(:)*(real(x(:)'*d(:))), n, m);
    
    % For Riemannian submanifolds, converting a Euclidean gradient into a
    % Riemannian gradient amounts to an orthogonal projection.
	M.egrad2rgrad = M.proj;
    
	M.ehess2rhess = @ehess2rhess;
	function rhess = ehess2rhess(x, egrad, ehess, u)
        rhess = M.proj(x, ehess) - real((x(:)'*egrad(:)))*u;
	end
    
	M.tangent = M.proj;
    
    M.exp = @exponential;
    
    M.retr = @retraction;

    M.log = @logarithm;
    function v = logarithm(x1, x2)
        v = M.proj(x1, x2 - x1);
        di = M.dist(x1, x2);
        % If the two points are "far apart", correct the norm.
        if di > 1e-6
            nv = norm(v, 'fro');
            v = v * (di / nv);
        end
    end
    
    M.hash = @(x) ['z' hashmd5([real(x(:)) ; imag(x(:))])];
    
    M.rand = @() random(n, m);
    
    M.randvec = @(x) randomvec(n, m, x);
    
    M.lincomb = @matrixlincomb;
    
    M.zerovec = @(x) zeros(n, m);
    
    M.transp = @(x1, x2, d) M.proj(x2, d);
    
    M.pairmean = @pairmean;
    function y = pairmean(x1, x2)
        y = x1+x2;
        y = y / norm(y, 'fro');
    end

    mn = m*n;
    M.vec = @(x, u_mat) [real(u_mat(:)) ; imag(u_mat(:))];
    M.mat = @(x, u_vec) reshape(u_vec(1:mn), m, n) + 1i*reshape(u_vec((mn+1):end), m, n);
    M.vecmatareisometries = @() true;

end

% Exponential on the sphere
function y = exponential(x, d, t)

    if nargin == 2
        % t = 1;
        td = d;
    else
        td = t*d;
    end
    
    nrm_td = norm(td, 'fro');
    
    if nrm_td > 0
        y = x*cos(nrm_td) + td*(sin(nrm_td)/nrm_td);
    else
        y = x;
    end

end

% Retraction on the sphere
function y = retraction(x, d, t)

    if nargin == 2
        t = 1;
    end
    
    y = x+t*d;
    y = y/norm(y, 'fro');

end

% Uniform random sampling on the sphere.
function x = random(n, m)

    x = randn(n, m) + 1i*randn(n, m);
    x = x/norm(x, 'fro');

end

% Random normalized tangent vector at x.
function d = randomvec(n, m, x)

    d = randn(n, m) + 1i*randn(n, m);
    d = reshape(d(:) - x(:)*(real(x(:)'*d(:))), n, m);
    d = d / norm(d, 'fro');

end
