from playwright.sync_api import sync_playwright
from .searcher import *
from typing import List, Dict, Tuple, Optional

import json

def get_bing_search_raw_page(question):
    """
    使用Playwright获取Bing搜索结果，返回一个包含标题和链接的列表。
    
    Args:
        question (str): 需要搜索的问题字符串。
    
    Returns:
        list[dict]: 每个元素是一个字典，包含两个键值对：'title'（标题）和'url'（URL）。
    
    Raises:
        None
    
    """
    results = []
    with sync_playwright() as p:
        browser = p.chromium.launch()
        context = browser.new_context()
        page = context.new_page()
        try:
            page.goto("https://www.bing.com/search?q=" + question)
        except:
            page.goto("https://www.bing.com")
            page.fill('input[name="q"]', question)
            page.press('input[name="q"]', 'Enter')
        try:
            page.wait_for_load_state('networkidle', timeout=3000)
        except:
            pass
        # page.wait_for_load_state('networkidle')
        search_results = page.query_selector_all('.b_algo h2')
        for result in search_results:
            title = result.inner_text()
            a_tag = result.query_selector('a')
            if not a_tag: continue
            url = a_tag.get_attribute('href')
            if not url: continue
            # print(title, url)
            results.append({
                'title': title,
                'url': url
            })
        browser.close()
    return results

def query_bing(question, max_tries=3):
    """
    使用Bing搜索引擎查询问题，并返回结果。如果没有获取到结果，则重试max_tries次。如果仍然无法获取到结果，则返回None。
    
    Args:
        question (str): 需要查询的问题字符串。
        max_tries (int, optional): 最大尝试次数，默认为3。默认值：3.
    
    Returns:
        Union[List[Dict], None]: 如果成功获取到结果，则返回一个包含多个字典对象的列表，每个字典对象包含了一个图片链接和一个标题；否则返回None。默认值：None。
    
    Raises:
        None
    
    """
    cnt = 0
    while cnt < max_tries:
        cnt += 1
        results = get_bing_search_raw_page(question)
        if results:
            return results
    print('No Bing Result')
    return None


if __name__ == '__main__':
    
    with open('crawl.json', 'w', encoding='utf-8') as f:
        json.dump(query_bing('how to cook a steak'), f, ensure_ascii=False, indent=4)
        
    exit(0)


class Searcher(SearcherInterface):
    def __init__(self):
        """
        初始化类的属性和方法
        Args:
            None
        Returns:
            None
        Raises:
            None
        """
        pass

    def _parse(self, result):
        """
        解析搜索结果，将每个结果转换为 SearchResult 对象
        Args:
            result (list[dict]): 来自搜索引擎的搜索结果，包含 title（标题）和 url（链接）两个字段，格式如下：
                                   [{'title': 'xxx', 'url': 'yyy'}, ...]
        
        Returns:
            list[SearchResult], optional:
                如果没有任何结果，返回 None；否则，返回一个包含 SearchResult 对象的列表，格式如下：
                [SearchResult('xxx', 'yyy', None), ...]
        """
        if not result:
            return None
        ret = []
        for item in result:
            ret.append(SearchResult(item['title'], item['url'], None))
        return ret

    def search(self, query):
        """
        搜索指定的内容，返回一个列表，包含所有搜索结果。每个元素都是一个字典，包含以下键值对：
            - "title" (str)：标题，可能为空；
            - "url" (str)：URL，不会为空；
            - "snippet" (str)：摘要，可能为空。
            如果没有找到任何结果，则返回一个空列表。
        
        Args:
            query (str): 需要搜索的内容。
        
        Returns:
            List[Dict[str, str]]: 搜索结果列表，每个元素都是一个字典，包含以下键值对：
                    {
                        "title": str,  # 标题，可能为空
                        "url": str,  # URL，不会为空
                        "snippet": str  # 摘要，可能为空
                    }
        """
        return self._parse(query_bing(query))



if __name__ == '__main__':
    
    print(json.dumps(query_bing('how to cook a cake?'), ensure_ascii=False, indent=4))