import random
from collections import namedtuple
from typing import List

import torch
from tqdm import tqdm, trange
from transformers import AdamW, get_linear_schedule_with_warmup

from models.bart_utils import ShardedBART

TextPairData = namedtuple('TextPairData', ['src_text', 'tgt_text'])


class BART:
    def __init__(self, args):
        self.args = args
        assert args.gpu in [0, 1, 2, 3, 4]
        self.device = 'cuda' if args.gpu > 0 else 'cpu'

        self.src_max_len = args.src_max_len
        self.tgt_max_len = args.tgt_max_len

        self.bart = ShardedBART(args)
        self.optimizer = None
        self.lr_scheduler = None
        self.data = []

        # Number of optimization performed
        self.train_steps = 0

    def get_optimizer(self, lr, train_steps, warmup_steps,
                      weight_decay, adam_epsilon):
        no_decay = ["bias", "LayerNorm.weight"]
        optimizer_grouped_parameters = [
            {"params": [p for n, p in self.bart.named_parameters()
                        if not any(nd in n for nd in no_decay)],
             "weight_decay": weight_decay},
            {"params": [p for n, p in self.bart.named_parameters()
                        if any(nd in n for nd in no_decay)],
             "weight_decay": 0.0}]
        self.optimizer = AdamW(
            optimizer_grouped_parameters, lr=lr, eps=adam_epsilon)
        self.lr_scheduler = get_linear_schedule_with_warmup(
            self.optimizer, num_warmup_steps=warmup_steps,
            num_training_steps=train_steps)

    def save_model(self, path):
        torch.save(self.bart.state_dict(), path)
        print(f'Model saved in {path}.')

    def load_model(self, path):
        self.bart.load_state_dict(torch.load(path, map_location=self.device))
        print(f'Model {path} loaded.')

    def load_data(self, src_texts, tgt_texts):
        """ Just go through all the data (maybe once), no more preprocessing """
        for src_text, tgt_text in tqdm(zip(src_texts, tgt_texts),
                                       total=len(src_texts),
                                       desc='Loading data...'):
            self.data.append(TextPairData(
                src_text=src_text,
                tgt_text=tgt_text
            ))
        print(f'Data size: {len(self.data)}')

    def train_epoch(self, batch_size):
        self.bart.shard()
        self.bart.train()
        random.shuffle(self.data)
        for i in trange(0, len(self.data), batch_size,
                        desc='BART Training...'):
            batch = self.data[i: i + batch_size]
            self.optimizer.zero_grad()
            for j in range(0, len(batch)):
                data = batch[j]
                src_encoded = self.bart.encode(data.src_text, self.src_max_len)
                src_tokens = src_encoded['input_ids']
                src_attn_mask = src_encoded['attention_mask']

                tgt_encoded = self.bart.encode(data.tgt_text, self.tgt_max_len)
                tgt_tokens = tgt_encoded['input_ids']
                tgt_attn_mask = tgt_encoded['attention_mask']

                loss = self.bart.forward(
                    input_ids=src_tokens,
                    attention_mask=src_attn_mask,
                    decoder_attention_mask=tgt_attn_mask,
                    labels=tgt_tokens
                )
                loss = loss / batch_size
                loss.backward()

            self.optimizer.step()
            self.train_steps += 1
            self.lr_scheduler.step()

            # Save checkpoint
            if self.train_steps % 50 == 0:
                self.save_model(f'{self.args.save_dir}/bart_{self.train_steps}.pth')

            if self.train_steps == 6000:
                exit(0)

    def generate(self, src_sents: List[str]):
        self.bart.eval()
        input_ids = self.bart.tokenizer(
            src_sents,
            max_length=self.src_max_len,
            padding=True,
            truncation=True,
            return_tensors='pt'
        )['input_ids'].to(self.device)

        output = self.bart.generate(
            input_ids=input_ids,
            max_length=self.args.gen_max_len,
            min_length=self.args.gen_min_len,
            num_beams=self.args.beam,
            length_penalty=self.args.lenpen,
            no_repeat_ngram_size=self.args.no_repeat_ngram_size
        )

        hypos = [self.bart.tokenizer.decode(g, skip_special_tokens=True,
                                            clean_up_tokenization_spaces=False).strip()
                 for g in output]

        return hypos
