import random
from bs4 import BeautifulSoup
import logging
from SurveyUtils import submit_button, detailed_instructions
import omegaconf.dictconfig

logger = logging.getLogger(__name__)


def generate_survey(config: omegaconf.dictconfig.DictConfig) -> str:
    logger.info('Start Generating Survey')

    # read the html template file
    logger.info(f'read template file from {config.survey.template_path}')
    with open(config.survey.template_path) as fp:
        soup = BeautifulSoup(fp.read(), 'html.parser')

    # read and add styles
    logger.info(f'read style file from {config.survey.style_path}')
    with open(config.survey.style_path) as fp:
        style_text = f"""
        <style>
            {fp.read()}
        </style>
        """
    style = BeautifulSoup(style_text, "html.parser")
    soup.insert(position=0, new_child=style)

    # create prompts list
    logger.info('creating prompt list')
    prompts_list = ["${prompt" + str(i) + "}" for i in range(config.survey.prompt_per_hit)]

    # prepare different sections of the survey
    small_instructions = BeautifulSoup("""
<strong>Please read the detailed instructions and examples before you begin.</strong><br>
Complete the following prompts with the first thing that comes to your mind.<br>
Remember: <br>
<span>1. No pronouns for objects</span><br>
<span>2. Use pronouns for humans</span><br>
<span>3. Do not overthink!</span><br>
<span>4. Be specific</span><br>
<span>5. Avoid negatives</span><br>
<span>6. Write actual sentences</span>
""", "html.parser")
    question_template = """
<div class="prompt-box" id="questionbox{myid}">
    <p class="golden">{prompt}</p>
    <p class="validcheckbox">
        <crowd-checkbox id="notvalid{myid}" name="notvalid{myid}">It does not make sense</crowd-checkbox>
    </p>
    <div class="row">
        <div class="col-xs-10 col-xs-offset-1">
            <crowd-input class="crowd-input" name="response{myid}" id="response{myid}" auto-focus="true"
                         placeholder="your response goes here!" required
                         auto-validate="true" allowed-pattern="[a-zA-z\.\!\, ]">
            </crowd-input>
        </div>
    </div>
</div>
<br>
"""
    event_log = BeautifulSoup("""
<crowd-input name="eventlog" id="myevents" placeholder="" style="text-indent: -1000em"></crowd-input>
<br>
""", "html.parser")
    script = BeautifulSoup("""
<script>
    function getTimeStamp() {
        var now = new Date();
        return ((now.getMonth() + 1) + '/' + (now.getDate()) + '/' + now.getFullYear() + " " + now.getHours() + ':'
            + ((now.getMinutes() < 10) ? ("0" + now.getMinutes()) : (now.getMinutes())) + ':' + ((now.getSeconds() < 10) ? ("0" + now
                .getSeconds()) : (now.getSeconds())));
    }

    function logTimeStamps(myid) {
        var time = getTimeStamp()
        var message = document.getElementById("myevents").value;
        var newMessage = message.toString() + ";" + myid.toString() + time.toString()
        // alert(textID + time.toString())
        document.getElementById("myevents").value = newMessage;
    }

    function createlog(myname) {
        var textID = "[" + myname.toString() + "]"

        function wrapper() {
            logTimeStamps(textID)
        }

        return wrapper
    }

    function createInvalidator(i) {
        var textID = "[box" + i.toString() + "]"
        function invalidate() {
            document.getElementById('response'+i.toString()).value = "Invalid"
            logTimeStamps(textID)
        }
        return invalidate
    }
    
    function logkey(myname) {
        var textID = "[" + myname.toString() + "]"
        function wrapper_key(e) {
            let key= `(${e.code.toString().replace('Key', '')})`;
            logTimeStamps(textID+key)
        }

        return wrapper_key
    }
</script>
""", "html.parser")

    # add the small instructions
    logger.info('add small instructions')
    small_instr = soup.find(id='small-instruction-box')
    assert small_instr is not None
    small_instr.append(small_instructions)

    # add the questions
    logger.info('prompt boxes')
    content = soup.find(id='AllContent')
    assert content is not None
    for i, (prompt) in enumerate(prompts_list):
        new_q = BeautifulSoup(
            question_template.format(prompt=prompt, myid=i),
            "html.parser"
        )
        content.append(new_q)

    logger.info('add event log')
    # add event log
    content.append(event_log)

    logger.info('add submit button')
    # add submit button
    content.append(submit_button)

    logger.info('add time and style scripts')
    # Add the script
    soup.append(script)
    captured_events = "\n".join([f"""
        document.getElementById('response{i}').shadowRoot.querySelector('input').onclick = createlog('click{i}');
        document.getElementById('response{i}').shadowRoot.querySelector('input').onfocus = createlog('focus{i}');
        document.getElementById('response{i}').shadowRoot.querySelector('input').onkeydown = logkey('key{i}');
        document.getElementById('notvalid{i}').onclick = createInvalidator({i}); 
    """ for i in range(len(prompts_list))] + ["""
        document.getElementById("mainform").onsubmit = createlog('submit')
    """])
    soup.append(BeautifulSoup("""
<script>
    function setup_events() {
        try {
    """ + captured_events + """
        } catch (err) {
            setTimeout(setup_events, 100)
        }
    }
    setup_events()
    
    function start_log() {
        try {
            logTimeStamps("[start]");
        } catch (err) {
            setTimeout(start_log, 100)
        }
    }
    start_log()
</script>
""", "html.parser"))

    clean_survey = str(soup.prettify())

    # out_name = "MturkHTML/MowgliCQSurvey.html" if not FINAL else "MturkHTML/MowgliCQSurvey_final.html"
    if config.results.survey_path is not None:
        logger.info(f'Writing survey to file {config.results.survey_path}')
        with open(config.results.survey_path, "w") as outf:
            outf.write(clean_survey)
    return clean_survey


if __name__ == '__main__':
    config = omegaconf.dictconfig.DictConfig({
        'results': {
            'survey_path': './MowgliCQSurvey.html'
        },
        'survey': {
            'name': 'some name',
            'template_path': 'nas/home/qaesmi/Mowgli-CoreQuisite/FormGenerators/MTurk/Assets/MTurkDesignTemplate.html',
            'style_path': 'nas/home/qaesmi/Mowgli-CoreQuisite/FormGenerators/MTurk/Assets/MTurkStyles.css',
            'prompt_per_hit': 1,
        },
    })

    generate_survey(config)
