import numpy as np
import argparse
import os
import jsonlines
import gzip

import torch
import torch.nn as nn
import torch.optim as optim
from torch.nn import DataParallel
from torch.utils.data import DataLoader
from torch.autograd import Variable

from utils.config import load_config

from models.Oracle import Oracle

from gw_utils import *

def get_qas(pgame):
    """
    Gets the list of questions and their answers from a played game. These
    played games have to be generated with the Gameplay/inference.py script.

    Parameters
    ----------
    pgame: dict
        data of a played game, made with the inference script

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in pgame
    """
    qas = []
    q = ''
    for wd in pgame.lower().strip().split():
        if wd == '<start>':
            continue
        if wd == '<no>' or wd == '<yes>' or wd == '<n/a>':
            qas.append((q.strip(),wd[1:-1]))
            q = ''
        else:
            q += ' '+wd
    return qas


def get_raw_qas(game):
    """
    Gets the list of questions and their answers from a played game. These
    played games are taken from the human-human GuessWhat!? corpus.

    Parameters
    ----------
    pgame: dict
        data of a played game from the human-human corpus

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in game
    """
    qas = []
    for qa in game['qas']:
        qfixed = qa['question'][:-1]+' ?'
        # Get rid of the '?' of the last word
        qas.append((qfixed, qa['answer'].lower()))
    return qas

class FeatGetter():
    def __init__(self, vf_map_file, vf_file, split='test'):
        partitions = {'train': ('train2id', 'train_img_features'),
                      'val'  : ('val2id', 'val_img_features'),
                      'test' : ('test2id', 'test_img_features')}

        map_split, feat_split = partitions[split]
        with open(vf_map_file, 'rb') as fl:
            self.vf_map = json.load(fl)[map_split]
        self.visual_feat = np.asarray(h5py.File('data/ResNet_avg_image_features.h5',
                                                'r')[feat_split])

    def get_vf(self, key):
        img_feat_idx = self.vf_map[key]
        img_feat = self.visual_feat[img_feat_idx]
        return img_feat

def load_visual_features(split='val'):
    """
    Load visual features from file and a given split.

    Parameters
    ----------
    split: str, optional
        Name of the partition to load. Options are 'train', 'val' or 'test' 

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in pgame
    """
    # TODO make this function take paths as arguments
    partitions = {'train': ('train2id', 'train_img_features'),
                  'val'  : ('val2id', 'val_img_features'),
                  'test' : ('test2id', 'test_img_features')}

    map_split, feat_split = partitions[split]
    print('Using {}'.format(split))
    with open('data/ResNet_avg_image_features2id.json', 'rb') as fl:
        vf_map = json.load(fl)
        vf_map = vf_map[map_split]
    visual_features = np.asarray(h5py.File('data/ResNet_avg_image_features.h5',
        'r')[feat_split])
    return vf_map, visual_features


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument("-breaking", action='store_true', help='Run with just one sample for test purpose')
    parser.add_argument("-dataset", type=str, help='Path to the dataset file')
    parser.add_argument("-split", type=str,default='test', help='Partition of the dataset to  use. Options are "train", "val" or "test". Default: val.')
    parser.add_argument("-from_gameplay", action='store_true', help='If the question_data file was generated by the Gameplay/inference.py script')
    parser.add_argument("-question_data", type=str, help='Path to the file of questions. Not needed if -from_gameplay is not used.')
    parser.add_argument("-out_fname", type=str, default='out.json', help='Name of the generated file.')
    
    args = parser.parse_args()

    # Raw GuessWhat?! dataset
    
    # TODO make this path not so hardcoded
    datapath = 'data/guesswhat.{}.jsonl.gz'.format('valid' if args.split == 'val' else args.split)

    with open(args.dataset, 'r') as fl:
        imgdata = json.load(fl)

    # TODO make this load with parameters
    # Load the visual mapping and visual features
    featgetter = FeatGetter('data/ResNet_avg_image_features2id.json',
                            'data/ResNet_avg_image_features.h5',
                            split=args.split)

    # Load the vocabulary manager and Oracle model
    # TODO make this load with parameters
    vocab, oracle = load_vocab_oracle()

    # Dicts to transfrom from token to word
    ans2tok = {'no': 0, 'yes': 1, 'n/a': 2}
    tok2ans = {0:'no', 1: 'yes', 2: 'n/a'}

    # TODO make the path a parameter
    # If we get our dialogues from the Gameplay/inference.py script
    # load the dialogues
    with open(args.question_data, 'r') as fl:
        played_games = json.loads(fl.read())

    # Data to write to file
    data = []

    # Check effectiveness for each game played in the dataset
    for j, entry in enumerate(played_games): 

        # For test purposes
        if args.breaking and j > 5:
            break

        game = imgdata[entry['image_id']]

        # Get image related information
        im_width = game['image']['width']
        im_height = game['image']['height']

        # Get the visual features corresponding to the image
        # of the current game
        image = featgetter.get_vf(game['image']['file_name'])

        # Get a dict of all objects in the image indexed by their id 
        objects = {obj['id']: obj for obj in game['objects']}

        # Create a sample to make inference with the oracle
        # It has to have the fields:
        # answers: non-important
        # crop_features: non-important
        # img_features: visual features of the image
        # lenght: lenght of the question
        # question: a list of numbers that represent the words in ids
        # obj_cat: the category of the target object for the question
        # spatial: the bounding box of the target object for the question
        sample = {}
        sample['answer']         = torch.tensor([2])
        sample['crop_features']  = torch.tensor([0])
        sample['img_features'] = torch.tensor([image])

        # If we're using questions from the Gameplay/inference.py script,
        # parse question with get_qas. Otherwise parse them with get_raw_qas
        qas = get_qas(entry['gen_dialogue'])

        print("Computing {}".format(j))

        rs = [set(objects.keys())] # List of referece sets
                                   # start with a set of all
                                   # objects in the image

        gdata = [] # Create the entry for logging for this game

        # ------------------------
        #
        # COMPUTE REFERENCE SETS:
        # The definition states:
        #
        #     RS(qj ) = {o in the image | o ∈ RS(q_{j-1})
        #                              ∧
        #                answer(q_j, o) = answer(q_j, target_object)}
        # ------------------------
        # TODO make this efficient
        for q, ans in qas:
            # Reset list of objects that have the same answer than the target
            # object. Let q_j be the j-th question, rs_j corresponds to
            # RS(q_j).
            rs_j = []

            # Variable to store if the answer given by the oracle is the same as
            # given by the human if processing human-human dialogues
            same_as_human = False

            # Transform the question string in a list of word ids
            qseq = vocab.words2seq(q)

            # For each question in the gameplay:
            # set the questions and its lenght
            sample['length'] = torch.tensor([len(q.strip().split(' '))])
            sample['question'] = torch.tensor([qseq])

            # Only iterate over the objects of the last reference set in order
            # with the first condition:
            #
            #     o ∈ RS(q_{j-1}) 
            #
            for o in rs[-1]:
                obj = objects[o]
                # For each object in the image:
                # set that object's caterogy and spatial information
                sample['obj_cat'] = torch.tensor([obj['category_id']])
                sample['spatial'] = torch.tensor([get_spatial_feat(obj['bbox'],
                                                              im_width,
                                                              im_height)])

                # Get prediction from the oracle
                pred = oracle.forward(sample).argmax(dim=1).numpy()

                # If the answer is the same for obj than for the target object
                # then obj goes to RS(q_j), in order with the second
                # condition:
                # 
                #    answer(o) == answer(target_object)
                #
                if pred == ans2tok[ans]:
                    rs_j.append(obj['id'])

            # ----------------------------------------------------
            # COMPUTE EFFECTIVENESS:
            #
            #   Effectiveness => RS(q_j) ⊂ RS(q_{j-1})
            # ㄱEffectiveness <= RS(q_j) ⊇ RS(q_{j-1})
            # 
            # We have the inclusion by definition, so we only have
            # to check for the equality.
            # ----------------------------------------------------
            effective = not (set(rs_j)) == (rs[-1])
            rs.append(set(rs_j))

            # Add the data to the entry
            gdata.append({'question': q, 'ans': ans,
                          'rs': list(rs[-1]), 'effective': effective})
        # Store data for this game
        data.append({'qas':gdata, 'states':entry['state']})

    with open(args.out_fname, 'w') as fl:
        json.dump(data, fl)
