import json
import os
import random

import pytest
from dotenv import load_dotenv
from openai import OpenAI

load_dotenv(override=True)

client = OpenAI(
    api_key=os.environ["OPENAI_API_KEY"],
    organization=os.environ["OPENAI_ORGANIZATION_KEY"],
)


def read_json(file_path):
    with open(file_path, "r") as f:
        return json.load(f)


args = read_json("config.json")
PREV_DEALS = [13000, 13500, 13000, 12000, 13000, 13000, 13500, 13000, 13000, 13250, 12500, 12750, 13400]
RATIONALE_FEEDBACK = "If accompanied by a compelling rationale, a very low opening offer can come across as principled bargaining rather than greed or ignorance. The offer rationales that work best in negotiations are those that make reference to the market,  to one's budget, to norms of fairness, and to the buyer and seller's levels of urgency."
QUESTION_FEEDBACK = "Its wise to plan the questions that you will ask before coming to the table.  One purpose of questions is to gain information, such as how far the seller can be pushed.  Another purpose is to persuade the seller to be reasonable by raising issues such as your relative ages or wealth levels.  If you don't plan out questions beforehand, you may forget to ask them once the action begins."


def chatgpt_chatbot(messages, model):
    completion = client.chat.completions.create(
        model=model,
        messages=messages,
        # max_tokens=100,
        # temperature=0.1,
    )
    answer = completion.choices[0].message.content.strip()
    return answer


def first_third(num1, num2):
    # Calculate the range within the first third
    first_third_range = (num2 - num1) // 3

    # Generate a random integer within that range
    random_num = random.randint(num1, num1 + first_third_range)

    return random_num

def get_first_third_range(min_val, max_val):
    first_third_range = (max_val - min_val) // 3
    return min_val, min_val + first_third_range

class PreNegotiationFeedbackModule:
    def __init__(self):
        self.id = None

    def feedback_reservationPrice(self, reservationPrice, marketMax, factsMax):
        if reservationPrice == factsMax:
            edit = "Your reservation price corresponds accurately to the role information about your $13,500 budget for the car purchase."
        else:
            # prompt = f""" You are an assistant aimed to give advices to help a buyer in a negotiation.
            #             You are adressing directly to the buyer, use the second person (You).
            #             The prices for this item in the market do not exceed ${marketMax}.
            #             The buyer was told that they would hope to get a price below ${factsMax}.
            #             Given this information, a good reservation price would be in between ${factsMax} and ${marketMax}
            #             The buyer set their reservation price to {reservationPrice}.

            #             Give them a feedback saying wether it is a relevant price or not. If not, give more details about what would be a good reservation price.
            #             """
            # edit = chatgpt_chatbot(
            #     messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            # )
            edit = "According to your role, your walk away price should be $13,500."
        return edit

    def feedback_targetPrice(self, targetPrice, marketMin, reservationPrice):
        _, first_third = get_first_third_range(marketMin, reservationPrice)
        if targetPrice >= marketMin and targetPrice <= first_third:
            # edit = "Your target price is relevant to the scenario."
            edit = "Your target price is well chosen given the role information about the market, the range of prices at which this car is sold. A target at the low end of this range helps you learn how far this seller can be pushed."
        elif targetPrice < marketMin:
            edit = "This overly ambitious target is below the market range for the car.  It may cause offense. By overreaching, you may miss out on good deal."
            prompt = f""" You are an assistant aimed to give advice to help a buyer in a negotiation.
                        You are adressing directly to the buyer, use the second person (You).

                        The buyer made an error setting their target price for the negotiation. The buyer set their target price to ${targetPrice}.
                        However a good target price should be above the minimum market value for the car which is ${marketMin}.

                        Give the buyer feedback explaining their error including details about what would be a good target price.
                        Here is an example of good feeback:
                        This overly ambitious target is below the market range for the car.  It may cause offense. By overreaching, you may miss out on good deal.
                        """
            edit = chatgpt_chatbot(
                messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            )
        else:
            edit = "Your target price of x is not ambitious enough to test how far this seller can be pushed. You should aspire to a price at the low end of the market range."
            prompt = f""" You are an assistant aimed to give advice to help a buyer in a negotiation.
                        You are adressing directly to the buyer, use the second person (You).

                        The buyer made an error setting their target price for the negotiation. The buyer set their target price to ${targetPrice}.
                        However a good target price should be below ${first_third} and closer to the minimum market range for the car which is ${marketMin}.

                        Give the buyer feedback explaining their error including details about what would be a good target price.
                        Here is an example of good feeback:
                        Your target price of {targetPrice} is not ambitious enough to test how far this seller can be pushed. You should aspire to a price at the low end of the market range.
                        """
            edit = chatgpt_chatbot(
                messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            )
        # else:
            # prompt = f""" You are an assistant aimed to give advices to help a buyer in a negotiation.
            #             You are adressing directly to the buyer, use the second person (You).
            #             In our case, the prices for this item out in the market were ranging from a minimum of {marketMin},
            #             and the buyer set their reservation price to {reservationPrice}.

            #             Given this information, a good target price would not be like: {marketMin} < target price < {marketMin + (reservationPrice-marketMin)/3}.
            #             The buyer set their target price to {targetPrice}.

            #             Give them a feedback saying wether it is a relevant target price or not. If not, give more details about what would be a good target price.
            #             """
            # edit = chatgpt_chatbot(
            #     messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            # )
        return edit

    def feedback_openingPrice(self, openingPrice, targetPrice):
        if openingPrice <= .9*targetPrice and openingPrice >= .8*targetPrice:
            edit = "Your opening price is well chosen relative to target. It gives you room to make concessions from your opening point and still settle near your target price."
        elif targetPrice > .9*targetPrice:
            edit = f"Your opening price of ${openingPrice:,} is not ambitious enough relative to your target. If you open at this price, you don't have much room to make concessions and still end up at your target price."
            prompt = f""" You are an assistant aimed to give advice to help a buyer in a negotiation.
                        You are adressing directly to the buyer, use the second person (You).

                        The buyer made an error setting their opening price for the negotiation. The buyer set their opening price to ${openingPrice}.
                        However a good opening price should be below 90% of their target price which is ${.9*targetPrice}.

                        Give the buyer feedback explaining their error including details about what would be a good opening price.
                        Here is an example of good feeback:
                        Your opening price of ${openingPrice:,} is not ambitious enough relative to your target. If you open at this price, you don't have much room to make concessions and still end up at your target price.
                        """
            edit = chatgpt_chatbot(
                messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            )
        else:
            edit = "This opening price is too ambitious.  A price far below the market range can make you appear greedy or uninformed. A seller may refuse to negotiate with you."
            prompt = f""" You are an assistant aimed to give advice to help a buyer in a negotiation.
                        You are adressing directly to the buyer, use the second person (You).

                        The buyer made an error setting their opening price for the negotiation. The buyer set their opening price to ${openingPrice}.
                        However a good opening price should be above 80% of their target price which is ${.8*targetPrice}.

                        Give the buyer feedback explaining their error including details about what would be a good opening price.
                        Here is an example of good feeback:
                        This opening price is too ambitious.  A price far below the market range can make you appear greedy or uninformed. A seller may refuse to negotiate with you.
                        """
            edit = chatgpt_chatbot(
                messages=[{"role": "system", "content": prompt}], model="gpt-4o"
            )
        # else:
        #     prompt = f""" You are an assistant aimed to give advices to help a buyer in a negotiation.
        #                 You are adressing directly to the buyer, use the second person (You).
        #                 The buyer set their target price to {targetPrice}.

        #                 Given this information, a good opening price would not be higher than {0.9*targetPrice}.
        #                 The buyer planned to open the negotiation with a price of {openingPrice}.

        #                 Give them a feedback saying wether it is a opening price or not. If not, give more details about what would be a good one.
        #                 """
        #     edit = chatgpt_chatbot(
        #         messages=[{"role": "system", "content": prompt}], model="gpt-4o"
        #     )
        return edit
    
    def get_percentile(self, deal_data):
        deal_price = float(deal_data[0]['answer'])

        n = 0
        for deal in PREV_DEALS:
            if deal_price < deal:
                n += 1
        return int((n/len(PREV_DEALS)) * 100)

    def get_prenegotiation_feedback(self, history, deal_data):
        reservationPrice = int(history[0]["answer"])
        targetPrice = int(history[1]["answer"])
        openingPrice = int(history[2]["answer"])

        response = history.copy()
        response[0]["feedback"] = self.feedback_reservationPrice(
            reservationPrice=reservationPrice,
            marketMax=args["buyer"]["marketMax"],
            factsMax=args["buyer"]["factsMax"],
        )
        response[1]["feedback"] = self.feedback_targetPrice(
            targetPrice=targetPrice,
            marketMin=args["buyer"]["marketMin"],
            reservationPrice=reservationPrice,
        )
        response[2]["feedback"] = self.feedback_openingPrice(
            openingPrice=openingPrice, targetPrice=targetPrice
        )
        response[3]["feedback"] = RATIONALE_FEEDBACK
        response[4]["feedback"] = QUESTION_FEEDBACK

        deal_percentile = min(99, self.get_percentile(deal_data))

        return dict(
            metrics=[
                f"The final deal you reached was in the {deal_percentile}th percentile of agreements of our users with this simulation.",
            ],
            feedback=response,
        )
