///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <getopt.h>

#include "nl-iomacros.h"
#include "nl-cpt.h"


////////////////////////////////////////////////////////////////////////////////
//
//  Given Variables
//
////////////////////////////////////////////////////////////////////////////////

char psX[]   = "";
char psSpc[] = " ";
char psLBrace[]="{";
char psRBrace[]="}";
char psColon[]=":";

//// C: syntactic symbol
DiscreteDomain<int> domC;
typedef DiscreteDomainRV<int,domC> C;
const C C_BOT ("-");
const C C_TOP ("REST");

//// L: relation symbol
DiscreteDomain<int> domL;
typedef DiscreteDomainRV<int,domL> L;
const L L_BOT ("-");
const L L_TOP ("h");

//// E: headword/referent symbol
DiscreteDomain<int> domE;
typedef DiscreteDomainRV<int,domE> E;
const E E_BOT ("-");
const E E_TOP ("-");

//// CE: ref-annotated syntax
typedef DelimitedJoint2DRV<psX,C,psLBrace,E,psRBrace> CE;
const CE CE_TOP (C_TOP,E_TOP);

//// LC: rel-annotated syntax
typedef DelimitedJoint2DRV<psX,L,psColon,C,psX> LC;
const LC LC_TOP (L_TOP,C_TOP);

//// LCLC: child rel-annotated syntax
typedef DelimitedJoint2DRV<psX,LC,psSpc,LC,psX> LCLC;

//// G: grammar rule symbol
typedef DelimitedJoint3DRV<psX,L,psColon,C,psLBrace,E,psRBrace> G;
const G G_BOT (L_BOT,C_BOT,E_BOT);
const G G_TOP (L_TOP,C_TOP,E_TOP);

//// GG: pair of grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,G,psSpc,G,psX> GG;
const GG GG_BOT ( G_BOT, G_BOT );

//// P: pos symbol
DiscreteDomain<int> domP;
typedef DiscreteDomainRV<int,domP> P;
const P P_BOT ("-");

//// W: pos symbol
DiscreteDomain<int> domW;
typedef DiscreteDomainRV<int,domW> W;
const W W_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Given Models
//
////////////////////////////////////////////////////////////////////////////////

//// Model of G (prior)
typedef CPT1DModel<G,Prob> GModel;
//// Model of GG given G (binary branching expansion)
typedef CPT2DModel<GG,G,Prob> GGgModel;
//// Model of E given L and E
typedef CPT3DModel<E,L,E,Prob> LModel;

//// Model of term P given G
typedef CPT2DModel<P,C,Prob> PcModel;
//// Model of term P given W
typedef CPT2DModel<P,W,Prob> PwModel;
//// Model of term P
typedef CPT1DModel<P,Prob> PModel;
//// Model of word W
typedef CPT1DModel<W,Prob> WModel;


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Variables
//
////////////////////////////////////////////////////////////////////////////////

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
typedef DiscreteDomainRV<char,domD> D;

//// K: length (input only, to HHMM models)...
DiscreteDomain<char> domK;
typedef DiscreteDomainRV<char,domK> K;

//// Q: incomplete category state
DiscreteDomain<int> domQ;
typedef DiscreteDomainRV<int,domQ> Q;
const Q Q_BOT ("-/-");

//// B: boolean
DiscreteDomain<int> domB;
typedef DiscreteDomainRV<int,domB> B;
const B B_0 ("0");
const B B_1 ("1");

//// F: final state
DiscreteDomain<int> domF;
typedef DiscreteDomainRV<int,domF> F;
const F F_0 ("0");
const F F_1 ("1");
const F F_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Models
//
////////////////////////////////////////////////////////////////////////////////

/*
//// Placeholder model of B given D and G (from above)
typedef CPT3DModel<B,D,G,Prob> BphModel;
*/
// //// Expansion model of G given D and G (from above)
// typedef CPT3DModel<G,D,G,Prob> GeModel;
// //// Transition model of G given D and G (awaited cat from previous) and G (from reduction)
// typedef CPT4DModel<G,D,G,G,Prob> GtmModel;
// //// Transition model of G (active cat) given D and G (active cat from previous) and G (awaited cat from above)
// typedef CPT4DModel<G,D,G,G,Prob> GtpModel;
// //// Transition model of G (awaited cat) given D and G (active cat from current) and G (active cat from previous)
// typedef CPT4DModel<G,D,G,G,Prob> GtqModel;
// //// Reduction model of F given D and G (active cat from previous) and G (awaited cat from above) (only check if previous awaited = reduction)
// typedef CPT4DModel<F,D,G,G,Prob> FrModel;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
///
///  Main function
///
////////////////////////////////////////////////////////////////////////////////

static struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {0, 0, 0, 0}};


void printUsage(char* progname) {
  //fprintf(stderr,"Usage: cat <RAW_FILE> | %s [OPTIONS]... [MODEL_FILE1] [MODEL_FILE2] ...\n",progname);
  //fprintf(stderr,"Runs the speech recognition system on RAW_FILE using the models in MODEL_FILE1, 2, etc.\n");
  fprintf(stderr," System variable Settings:\n");
  fprintf(stderr,"  -h, --help\t\tPrint this message\n");
}


int main (int nArgs, char* argv[]) {

  // Given models...
  GModel   mGr;
  GGgModel mGGg;
  LModel   mL;
  PcModel  mPc;
  PwModel  mPw;
  PModel   mP;
  WModel   mW;

  // Derived variable constants...
  const int D_MAX=4;
  const int K_MAX=98;
  // Init D values (strings)...
  for ( int d=0; d<=D_MAX+2; d++ ) { String s; s<<d; cerr<<D(s.c_array())<<"\n"; }
  // Init K values (strings)...
  for ( int k=0; k<=K_MAX+1; k++ ) { String s; if(k<10)s<<"0"; s<<k;; cerr<<K(s.c_array())<<"\n"; }


  //////////////////// PART 0: COMMAND-LINE OPTIONS

  // Parse command-line options
  char* progname = strdup(argv[0]);
  int option_index = 0;
  char opt;
  while( (opt = getopt_long(nArgs, argv, "hp:q:Vv", long_options, &option_index)) != -1 ) {
    switch(opt) {
    case 'h': printUsage(progname);
              exit(0);
    //case 'p': UPDATE_PORT_NUM  = atoi(optarg); break;
    //case 'q': QUERY_PORT_NUM = atoi(optarg); break;
    //case 'V': WorldModel::OUTPUT_VERY_VERBOSE = true; break; // do break.
    //case 'v': WorldModel::OUTPUT_VERBOSE = true; break;
    default: break;
    }
  }
  // Complain if too few args...
  if (optind > nArgs ) {
    printUsage(progname);
    exit(0);
  }


  //////////////////// PART I. LOAD GIVEN MODELS (FROM STDIN)

  FILE* pf = stdin;

  int c=' '; int i=0; int line=1; String sBuff(1000);                          // Lookahead/ctrs/buffers
  CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                   // Get to first record
  while ( c!=-1 && c!='\0' && c!='\5' ) {                                      // For each record
    CONSUME_STR ( pf, c, (c!='\n' && c!='\0' && c!='\5'), sBuff, i, line );    //   Consume line
    StringInput si(sBuff.c_array());
    if ( !( sBuff[0]=='#' ||                                                   //   Accept comments/fields
	    (si>>"Gr ">>mGr >>"\0"!=NULL) ||
	    (si>>"GG ">>mGGg>>"\0"!=NULL) ||
	    (si>>"L " >>mL>>"\0"!=NULL) ||
            (si>>"Pc ">>mPc >>"\0"!=NULL) ||
	    (si>>"Pw ">>mPw >>"\0"!=NULL) ||
	    (si>>"P " >>mP  >>"\0"!=NULL) ||
	    (si>>"W " >>mW  >>"\0"!=NULL) ) )
      cerr<<"\nERROR: unable to process line "<<line<<": "<<sBuff<<"\n";
    CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                 //   Consume whitespace
  }

  mGr.dump(cout,"Gr");
  mL.dump(cout,"L");
  mPc.dump(cout,"Pc");
  mPw.dump(cout,"Pw");
  mP.dump(cout,"P");
  mW.dump(cout,"W");

  mPc.clear();
  mPw.clear();
  mP.clear();
  mW.clear();

  //////////////////// PART II. BUILD DERIVED MODELS

  ////////////////////////////////////////////////////////////
  ////////// 1. Derive Helper Model Probs...

  ///// 1a. Add REST -> START REST for all START symbols...

  // For each Ae expanded from zero level...
  GModel::IterVal gAe; for ( bool bAe=mGr.setFirst(gAe); bAe; bAe=mGr.setNext(gAe) ) {
    GG ggAe_TOP ( gAe, G_TOP );
    mGGg.setProb ( ggAe_TOP, G_TOP ) += mGr.getProb(gAe);
  }

  //mGGg.dump(cout,"GGg");

  ///// 1b. Derive depth-bounded subtree probs...

  typedef CPT4DModel<B,K,D,G,Prob> BkdgModel;
  BkdgModel mBboundedLC;
  BkdgModel mBboundedRC;
  // For chains of increasing length k...
  for ( int k=0; k<=K_MAX; k++ ) {
    // For decreasing depths d...
    for ( int d=D_MAX+2; d>=1 && d>=D_MAX+1-k; d-- ) {

      cerr<<"1b,"<<k<<","<<d<<"...\n";

      // Ground bottom values...
      if ( d<=D_MAX+1 ) mBboundedLC.setProb ( B_1, k, d, G_BOT ) = 1.0;
      if ( d<=D_MAX+1 ) mBboundedRC.setProb ( B_1, k, d, G_BOT ) = 1.0;

      // For each given Ah...
      for ( GGgModel::const_iterator iGGg=mGGg.begin(); iGGg!=mGGg.end(); iGGg++ ) {
        G gAh = iGGg->first.getX1();

        // For each Ah0, Ah1 by rule from Ah...
        GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
                                           bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
          G gAh0 = ggAh0_Ah1.first;
          G gAh1 = ggAh0_Ah1.second;

          if (  mBboundedLC.getProb(B_1,k-1,d,gAh0)!=Prob() && mBboundedRC.getProb(B_1,k-1,d,gAh1)!=Prob() ) {
            // Define prob...
            mBboundedLC.setProb ( B_1, k, d, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,k-1,d,gAh0) *          mBboundedRC.getProb(B_1,k-1,d,gAh1);
          }
          if (  mBboundedLC.getProb(B_1,k-1,d+1,gAh0)!=Prob() && mBboundedRC.getProb(B_1,k-1,d,gAh1)!=Prob() ) {
            // Define prob...
            mBboundedRC.setProb ( B_1, k, d, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,k-1,d+1,gAh0) *          mBboundedRC.getProb(B_1,k-1,d,gAh1);
          }
        }
      }
    }
  }
  mBboundedRC.setProb ( B_1, K_MAX, 0, G_TOP ) = 1.0;

  //mBboundedLC.dump(cout,"BbLC");
  //mBboundedRC.dump(cout,"BbRC");


  ///// 1c. Derive depth-bounded PCFG probs...

  //typedef CPT3DModel<LCLC,D,CE,Prob> MdModel;
  typedef CPT5DModel<E,D,LC,LCLC,E,Prob> MdModel;
  MdModel mMdL;
  MdModel mMdR;
  // For decreasing depths d...
  for ( int d=D_MAX+1; d>=0; d-- ) {
    cerr<<"1c,"<<d<<"...\n";
    // Base case...
    if ( 0==d ) {
      // For each Ae expanded from zero level...
      GModel::IterVal gAe; for ( bool bAe=mGr.setFirst(gAe); bAe; bAe=mGr.setNext(gAe) ) {
        LCLC lclcAe_TOP ( LC(gAe.first,gAe.second), LC_TOP );
        mMdR.setProb ( E_TOP, d, LC_TOP, lclcAe_TOP, E_TOP ) += mGr.getProb(gAe) * mBboundedLC.getProb(B_1,K_MAX,d+1,gAe);
      }
    }
    // Induction step...
    else {
      // For each given Ah...
      for ( GGgModel::const_iterator iGGg=mGGg.begin(); iGGg!=mGGg.end(); iGGg++ ) {
        G gAh = iGGg->first.getX1();
        // For each Ah0, Ah1 by rule from Ah...
        GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
                                           bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
          G gAh0 = ggAh0_Ah1.first;
          G gAh1 = ggAh0_Ah1.second;
	  LC lcAh = LC(gAh.first,gAh.second);
	  E eAh = E(gAh.third);
	  LCLC lclcAh0_Ah1 = LCLC( LC(gAh0.first,gAh0.second), LC(gAh1.first,gAh1.second) );
          if (  mBboundedLC.getProb(B_1,K_MAX,d,gAh0)!=Prob() && mBboundedRC.getProb(B_1,K_MAX,d,gAh1)!=Prob() ) {
            // Define prob...
            mMdL.setProb ( eAh, d, lcAh, lclcAh0_Ah1, eAh ) = mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,K_MAX,d,gAh0) *          mBboundedRC.getProb(B_1,K_MAX,d,gAh1);
          }
          if (  mBboundedLC.getProb(B_1,K_MAX,d+1,gAh0)!=Prob() && mBboundedRC.getProb(B_1,K_MAX,d,gAh1)!=Prob() ) {
            // Define prob...
            mMdR.setProb ( eAh, d, lcAh, lclcAh0_Ah1, eAh ) = mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,K_MAX,d+1,gAh0) *          mBboundedRC.getProb(B_1,K_MAX,d,gAh1);
          }
        }
      }
    }
  }
  mMdL.normalize();
  mMdR.normalize();

  mMdL.dump(cout,"MdL");
  mMdR.dump(cout,"MdR");


//   ///// 1d. Derive depth-bounded PCFG left star probs...

//   //typedef CPT4DModel<C,K,D,C,Prob> CdkcModel;
//   class GdkgModel : public CPT4DModel<G,K,D,G,Prob> {
//    public:
//     //typedef CPT4DModel<G,K,D,G,Prob>::IterVal IterVal;
//     bool  setFirst (       CPT4DModel<G,K,D,G,Prob>::IterVal& g, const K& k, const D& d, const G& gP ) { return CPT4DModel<G,K,D,G,Prob>::setFirst(g,k.toInt()%2,d,gP); }
//     bool  setNext  (       CPT4DModel<G,K,D,G,Prob>::IterVal& g, const K& k, const D& d, const G& gP ) { return CPT4DModel<G,K,D,G,Prob>::setNext (g,k.toInt()%2,d,gP); }
//     Prob& setProb  ( const CPT4DModel<G,K,D,G,Prob>::IterVal& g, const K& k, const D& d, const G& gP ) { return CPT4DModel<G,K,D,G,Prob>::setProb (g,k.toInt()%2,d,gP); }
//     Prob  getProb  ( const CPT4DModel<G,K,D,G,Prob>::IterVal& g, const K& k, const D& d, const G& gP ) { return CPT4DModel<G,K,D,G,Prob>::getProb (g,k.toInt()%2,d,gP); }
//   };
//   GdkgModel mGlstar;
//   typedef CPT3DModel<G,D,G,Prob> GdgModel;
//   GdgModel mGlzerosum;
//   GdgModel mGlplussum;
//   GdgModel mGlstarsum;
//   // For chains of increasing length k...
//   for ( int k=0; k<=K_MAX; k++ ) {
//     // For increasing depth limits d...
//     for ( int d=1; d<=D_MAX+1; d++ ) {
//       cerr<<"1d,"<<k<<","<<d<<"...\n";
//       // For each Ah0^0...
//       G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//         // Base case...
//         if ( 0==k ) {
//           mGlstar.setProb ( gAh, k, d, gAh ) = 1.0;
//         }
//         // Induction step...
//         else {
//           // Clear model at length k...
//           GdkgModel::IterVal gAh0k; for ( bool bAk=mGlstar.setFirst(gAh0k,k,d,gAh);
//                                           bAk; bAk=mGlstar.setNext (gAh0k,k,d,gAh) ) {
// 	    mGlstar.setProb(gAh0k,k,d,gAh)=0.0;
// 	  }

//           // For each Ah0^k-1 by left-star from Ah0^0...
// 	  GdkgModel::IterVal gAh0kdec; for ( bool    bAkdec=mGlstar.setFirst(gAh0kdec,k-1,d,gAh);
//                                              bAkdec; bAkdec=mGlstar.setNext (gAh0kdec,k-1,d,gAh) ) {
//             // For each Ah0^k, Ah0^(k-1)1 by rule from simplified Ah0^k-1...
// 	    GGgModel::IterVal ggAh0k_Ah0kdec1; for ( bool bAk=mMdL.setFirst(ggAh0k_Ah0kdec1,d,gAh0kdec);
//                                                      bAk; bAk=mMdL.setNext (ggAh0k_Ah0kdec1,d,gAh0kdec) ) {
//               //if ( ggAh0k_Ah0kdec1!=GG_BOT ) {
//                 G gAh0k = ggAh0k_Ah0kdec1.first;
//                 // Define prob...
//                 mGlstar.setProb ( gAh0k, k, d, gAh ) += mGlstar.getProb(gAh0kdec,k-1,d,gAh) * mMdL.getProb(ggAh0k_Ah0kdec1,d,gAh0kdec);
//               //}
//             }
//             /*
//             // For each Ah0^k, Ah0^(k-1)1 by rule from simplified Ah0^k-1...
// 	    GGgModel::IterVal ggAh0k_Ah0kdec1; for ( bool bAk=mGGg.setFirst(ggAh0k_Ah0kdec1,gAh0kdec); bAk; bAk=mGGg.setNext(ggAh0k_Ah0kdec1,gAh0kdec) ) {
//               G gAh0k     = ggAh0k_Ah0kdec1.first;
//               G gAh0kdec1 = ggAh0k_Ah0kdec1.second;
//               if ( ggAh0k_Ah0kdec1!=GG_BOT && mBboundedRC.getProb(B_1,K_MAX,d,gAh0kdec1)!=Prob() ) {
//                 // Define prob...
//                 mGlstar.setProb ( gAh0k, k, d, gAh ) +=
//                   mGlstar.getProb(gAh0kdec,k-1,d,gAh) * mGGg.getProb(ggAh0k_Ah0kdec1,gAh0kdec) * mBboundedRC.getProb(B_1,K_MAX,d,gAh0kdec1);
//               }
//             }
//             */
//           }
//         }

// 	// Store zero-level...
// 	if ( 0==k ) {
//           // For each output, store zero...
//           GdkgModel::IterVal gAh0k; for ( bool bAk=mGlstar.setFirst(gAh0k,0,d,gAh); bAk; bAk=mGlstar.setNext(gAh0k,0,d,gAh) ) {
// 	    //gerr<<"adding "<<cAk<<" "<<d<<" "<<cAh<<"\n";
//             mGlzerosum.setProb ( gAh0k, d, gAh ) += mGlstar.getProb(gAh0k,0,d,gAh);
//           }
// 	}
//         // If k sufficiently large...
//         if ( k>0 ) {
//           // For each output, calc running total...
//           GdkgModel::IterVal gAh0k; for ( bool bAk=mGlstar.setFirst(gAh0k,k,d,gAh); bAk; bAk=mGlstar.setNext(gAh0k,k,d,gAh) ) {
//             mGlplussum.setProb ( gAh0k, d, gAh ) += mGlstar.getProb(gAh0k,k,d,gAh);
//           }
//         }
//         // If k sufficiently large...
//         if ( k>=0 ) {
//           // For each output, calc running total...
//           GdkgModel::IterVal gAh0k; for ( bool bAk=mGlstar.setFirst(gAh0k,k,d,gAh);
//                                           bAk; bAk=mGlstar.setNext (gAh0k,k,d,gAh) ) {
//             mGlstarsum.setProb ( gAh0k, d, gAh ) += mGlstar.getProb(gAh0k,k,d,gAh);
//           }
//         }
//       }
//     }
//   }
//   ////mGlstarsum.normalize();
//   //mGlzerosum.dump(cout,"Glzerosum");
//   //mGlstarsum.dump(cout,"Glstarsum");
//   //exit(0);

//   //mBboundedLC.clear();
//   //mBboundedRC.clear();
//   mGlstar.clear();


//   ////////////////////////////////////////////////////////////
//   ////////// 2. Derive HHMM Probs...

//   ///// 2a. Derive G-Expand probs (to preterminal)...

//   /*
//   BphModel mBph;
//   */
//   GeModel  mGe;
//   // For increasing depth d...
//   for ( int d=1; d<=D_MAX+1; d++ ) {
//     // For each given Ah...
//     G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//       cerr<<"2a,"<<d<<","<<gAh<<"...\n";
//       // For each Ah0, Ah1 by rule from Ah (RC at d-1)...
//       MdModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mMdR.setFirst(ggAh0_Ah1,d-1,gAh);
//                                           bAh0_Ah1; bAh0_Ah1=mMdR.setNext (ggAh0_Ah1,d-1,gAh) ) if ( GG_BOT != ggAh0_Ah1 ) {
//       /*
//       // For each Ah0, Ah1 by rule from Ah...
//       GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
//                                          bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
//       */
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         /*
//         // Record placeholder probability...
//         mBph.setProb ( B_1, d, gAh ) +=
//           mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedLC.getProb(B_1,K_MAX,d,gAh0) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) *
//           mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedLC.getProb(B_1,K_MAX,d,gAh0) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) ;
//         */
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) if ( G_BOT != gAh0m ) {
//           if ( mGGg.getProb(GG_BOT,gAh0m)!=Prob() ) {
//             mGe.setProb ( gAh0m, d, gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlstarsum.getProb(gAh0m,d,gAh0);
//               ////* mGlzerosum.getProb(gAh0,d,gAh0) / mGlstarsum.getProb(gAh0,d,gAh0);  // * zero/star (this last part is relocated from end reduction)
//           }
//         }
//         /*
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) {
//           if ( mGGg.getProb(GG_BOT,gAh0m)!=Prob() && mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1)!=Prob() ) {
//             mGe.setProb ( gAh0m, d, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) * mGlstarsum.getProb(gAh0m,d,gAh0);
//           }
//         }
//         */
//       }
//     }
//   }
//   /*
//   mBph.dump(cout,"Bph");
//   mBph.clear();
//   */
//   mGe.normalize();
//   mGe.dump(cout,"Ge");
//   mGe.clear();

//   ///// 2b. Define G-Transition-minus probs

//   GtmModel mGtm;
//   // For increasing depth d...
//   for ( int d=1; d<=D_MAX; d++ ) {
//     // For each given Ah...
//     G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//       cerr<<"2b,"<<d<<","<<gAh<<"...\n";
//       // For each Ah0, Ah1 by rule from Ah...
//       MdModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mMdR.setFirst(ggAh0_Ah1,d,gAh);
//                                           bAh0_Ah1; bAh0_Ah1=mMdR.setNext (ggAh0_Ah1,d,gAh) ) if ( GG_BOT != ggAh0_Ah1 ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         mGtm.setProb ( gAh1, d, gAh0, gAh ) += mMdR.getProb(ggAh0_Ah1,d,gAh);
//       }
//     }
//   }
//   mGtm.normalize();
//   mGtm.dump(cout,"Gtm");
//   mGtm.clear();

//   ///// 2c. Derive Q-Transition-plus probs

//   GtpModel mGtp;
//   // For increasing depth d...
//   for ( int d=1; d<=D_MAX; d++ ) {
//     // For each given Ah...
//     G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//       cerr<<"2c,"<<d<<","<<gAh<<"...\n";
//       // For each Ah0, Ah1 by rule from Ah (RC at d-1)...
//       MdModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mMdR.setFirst(ggAh0_Ah1,d-1,gAh);
//                                           bAh0_Ah1; bAh0_Ah1=mMdR.setNext (ggAh0_Ah1,d-1,gAh) ) if ( GG_BOT != ggAh0_Ah1 ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) if ( G_BOT != gAh0m ) {
//           // For each Ah0m0, Ah0m1 by rule from Ah0m...
//           MdModel::IterVal ggAh0m0_Ah0m1; for ( bool          bAh0m0_Ah0m1=mMdL.setFirst(ggAh0m0_Ah0m1,d,gAh0m);
//                                                   bAh0m0_Ah0m1; bAh0m0_Ah0m1=mMdL.setNext (ggAh0m0_Ah0m1,d,gAh0m) ) if ( GG_BOT != ggAh0m0_Ah0m1 ) {
//             G gAh0m0 = ggAh0m0_Ah0m1.first;
//             mGtp.setProb ( gAh0m, d, gAh0m0, gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlstarsum.getProb(gAh0m,d,gAh0) * mMdL.getProb(ggAh0m0_Ah0m1,d,gAh0m);
//               // / (mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlplussum.getProb(gAh0m0,d,gAh0));
//               ////* mGlzerosum.getProb(gAh0,d,gAh0) / mGlstarsum.getProb(gAh0,d,gAh0);  // * zero/star (this last part is relocated from end reduction)
//           }
//         }
//       }
//       /*
//       // For each Ah0, Ah1 by rule from Ah (RC at d-1)...
//       GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
//                                          bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) {
//           // For each Ah0m0, Ah0m1 by rule from Ah0m...
//           MdModel::IterVal ggAh0m0_Ah0m1; for ( bool          bAh0m0_Ah0m1=mMdL.setFirst(ggAh0m0_Ah0m1,d,gAh0m);
//                                                   bAh0m0_Ah0m1; bAh0m0_Ah0m1=mMdL.setNext (ggAh0m0_Ah0m1,d,gAh0m) ) {
//             G gAh0m0 = ggAh0m0_Ah0m1.first;
//             mGtp.setProb ( gAh0m, d, gAh0m0, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1)
//               * mGlstarsum.getProb(gAh0m,d,gAh0) * mMdL.getProb(ggAh0m0_Ah0m1,d,gAh0m);
//           }
//         }
//       }
//       */
//     }
//   }
//   mGtp.normalize();
//   mGtp.dump(cout,"Gtp");
//   mGtp.clear();

//   ///// 2d. Define G-Transition-plus-extra probs

//   GtqModel mGtq;
//   // For increasing depth d...
//   for ( int d=1; d<=D_MAX; d++ ) {
//     // For each given Ah...
//     G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//       cerr<<"2d,"<<d<<","<<gAh<<"...\n";
//       // For each Ah0, Ah1 by rule from Ah...
//       MdModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mMdL.setFirst(ggAh0_Ah1,d,gAh);
//                                           bAh0_Ah1; bAh0_Ah1=mMdL.setNext (ggAh0_Ah1,d,gAh) ) if ( GG_BOT != ggAh0_Ah1 ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         mGtq.setProb ( gAh1, d, gAh, gAh0 ) += mMdL.getProb(ggAh0_Ah1,d,gAh);
//       }
//     }
//   }
//   mGtq.normalize();
//   mGtq.dump(cout,"Gtq");
//   mGtq.clear();

//   ///// 2e. Derive F-Reduce probs

//   FrModel  mFr;
//   // For increasing depth d...
//   for ( int d=1; d<=D_MAX; d++ ) {
//     // For each given Ah...
//     G gAh; for ( bool bAh=gAh.setFirst(); bAh; bAh=gAh.setNext() ) {
//       cerr<<"2e,"<<d<<","<<gAh<<"...\n";
//       // For each Ah0, Ah1 by rule from Ah (RC at d-1)...
//       MdModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mMdR.setFirst(ggAh0_Ah1,d-1,gAh);
//                                           bAh0_Ah1; bAh0_Ah1=mMdR.setNext (ggAh0_Ah1,d-1,gAh) ) if ( GG_BOT != ggAh0_Ah1 ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) if ( G_BOT != gAh0m ) {
//           /*
//           if ( gAh0 == gAh0m ) {
//             mFr.setProb ( F(G(gAh0m).getString().c_str()), d, G(gAh0m), gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh)
//               / (mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlstarsum.getProb(gAh0m,d,gAh0));
//           }
//           if ( mGlplussum.getProb(G(gAh0m),d,gAh0) != Prob() ) {
//             mFr.setProb ( F_1,                             d, G(gAh0m), gAh ) = 1.0;
//           }
//           */
//           Prob pr;
//           if ( (pr = mGlzerosum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//             mFr.setProb ( F(G(gAh0m).getString().c_str()), d, G(gAh0m), gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * pr;
//               // / mGlstarsum.getProb(gAh0m,d,gAh0);
//           }
//           if ( (pr = mGlplussum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//             ////cerr<<"  "<<mFr.getProb(F_1,d,gAh0m,gAh)<<" * "<<pr<<" / "<<mGlstarsum.getProb(gAh0m,d,gAh0);
//             mFr.setProb ( F_1,                             d, G(gAh0m), gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * pr;
//               // / mGlstarsum.getProb(gAh0m,d,gAh0);
//             ////cerr<<" = "<<mFr.getProb(F_1,d,gAh0m,gAh)<<"\n";
//           }
//           /*
//           Prob pr;
//           if ( (pr = mGlzerosum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//             mFr.setProb ( F(G(gAh0m).getString().c_str()), d, G(gAh0m), gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * pr
//               / (mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlstarsum.getProb(gAh0m,d,gAh0));
//           }
//           if ( (pr = mGlstarsum.getProb(G(gAh0m),d,gAh0) - mGlzerosum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//             ////cerr<<"  "<<mFr.getProb(F_1,d,gAh0m,gAh)<<" * "<<pr<<" / "<<mGlstarsum.getProb(gAh0m,d,gAh0);
//             mFr.setProb ( F_1,                             d, G(gAh0m), gAh ) +=
//               mMdR.getProb(ggAh0_Ah1,d-1,gAh) * pr
//               / (mMdR.getProb(ggAh0_Ah1,d-1,gAh) * mGlstarsum.getProb(gAh0m,d,gAh0));
//             ////cerr<<" = "<<mFr.getProb(F_1,d,gAh0m,gAh)<<"\n";
//           }
//           */
//         }
//       }
//       /*
//       // For each Ah0, Ah1 by rule from Ah (RC at d-1)...
//       GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
//                                          bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
//         G gAh0 = ggAh0_Ah1.first;
//         G gAh1 = ggAh0_Ah1.second;
//         // For each Ah0m by star from Ah0...
//         GdgModel::IterVal gAh0m; for ( bool   bAh0m=mGlstarsum.setFirst(gAh0m,d,gAh0);
//                                        bAh0m; bAh0m=mGlstarsum.setNext (gAh0m,d,gAh0) ) {
//           if ( mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) != Prob() ) {
//             Prob pr;
//             if ( (pr = mGlzerosum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//               mFr.setProb ( F(G(gAh0m).getString().c_str()), d, G(gAh0m), gAh )
//                 += ( mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) * pr / mGlstarsum.getProb(G(gAh0m),d,gAh0) );
//             }
//             if ( (pr = mGlstarsum.getProb(G(gAh0m),d,gAh0) - mGlzerosum.getProb(G(gAh0m),d,gAh0)) != Prob() ) {
//               mFr.setProb ( F_1,                             d, G(gAh0m), gAh )
//                 += ( mGGg.getProb(ggAh0_Ah1,gAh) * mBboundedRC.getProb(B_1,K_MAX,d-1,gAh1) * pr / mGlstarsum.getProb(G(gAh0m),d,gAh0) );
//             }
//           }
//         }
//       }
//       */
//     }
//   }
//   mFr.normalize();
//   mFr.dump(cout,"Fr");
//   mFr.clear();
}

