import json
from typing import List, Dict

class SearchResult:
    def __init__(self, title, url, snip):
        """
        初始化一个Snippet对象，包含标题、URL和摘要三个属性。
        
        Args:
            title (str): 标题，类型为str。
            url (str): URL，类型为str。
            snip (str): 摘要，类型为str。
        
        Returns:
            None，无返回值。
        
        Raises:
            None，不引发任何异常。
        """
        self.title = title
        self.url = url
        self.snip = snip

    def dump(self):
        """
        将对象转换为字典，方便序列化
        
        Returns:
            dict: 包含标题、URL和摘要的字典，格式如下：
                {
                    "title": str,  # 标题
                    "url": str,  # URL
                    "snip": str  # 摘要
                }
        """
        return {
            "title": self.title,
            "url": self.url,
            "snip": self.snip
        }

    def __str__(self):
        """
        返回一个字符串，用于在控制台打印出对象的信息。
        
        Returns:
            str: 一个包含对象信息的字符串。
        """
        return json.dumps(self.dump())
    
class SearcherInterface:
    def search(self, query):
        """
        搜索指定的查询字符串，返回一个包含所有结果的列表。每个结果都是一个字典，包含以下键值对：
            - "title" (str)：标题
            - "url" (str)：URL地址
            - "content" (str, optional)：内容（如果可用）
            - "score" (float, optional)：相关性得分（如果可用）
            注意：这里的相关性得分并不是百分比，而是一个介于0和1之间的实数，其中1表示完全匹配，0表示没有匹配。
        如果无法找到任何结果，则应该返回一个空列表。
        
        Args:
            query (str): 要搜索的查询字符串
        
        Returns:
            List[Dict[str, Union[str, float]]]: 包含所有结果的列表，每个结果是一个字典
        
        Raises:
            NotImplementedError: 必须在子类中重写此方法才能正常工作
        """
        raise NotImplementedError()