using System;
using System.Linq;
using System.Collections.Generic;

namespace SpreadsheetLLM.Heuristic
{
    internal partial class TableDetectionHybrid
    {
        private Dictionary<Boundary, bool> _headerUpMap = new Dictionary<Boundary, bool>();
        private Dictionary<Boundary, bool> _headerLeftMap = new Dictionary<Boundary, bool>();

        public bool IsHeaderUp(Boundary header)
        {
            if (_headerUpMap.ContainsKey(header))
            {
                return _headerUpMap[header];
            }
            if (IsHeaderUpSimple(header))
            {
                Boundary HeaderDownSide = Utils.DownRow(header, start: -1);

                if (_sheet.ComputeSimilarRow(header, HeaderDownSide) < 0.15 && IsHeaderUpSimple(HeaderDownSide))
                {
                    _headerUpMap[header] = false;
                    return false;
                }
                else if (_sheet.sumContentExist.SubmatrixSum(header) == 2 && _sheet.sumContentExist.SubmatrixSum(HeaderDownSide) == 2
                    && HeaderRate(header, step: 0) == HeaderRate(HeaderDownSide, step: 0))
                {
                    return false;
                }
                else
                {
                    _headerUpMap[header] = true;
                    return true;
                }
            }
            _headerUpMap[header] = false;
            return false;
        }

        public bool IsHeaderLeft(Boundary header)
        {
            if (_headerLeftMap.ContainsKey(header))
            {
                return _headerLeftMap[header];
            }
            if (IsHeaderLeftSimple(header))
            {
                _headerLeftMap[header] = true;
                return true;
            }

            Boundary headerRight = Utils.RightCol(header, start: -1);
            if (_sheet.ContentExistValueDensity(headerRight) >= 1.5 * _sheet.ContentExistValueDensity(header)
                && _sheet.ContentExistValueDensity(headerRight) > 2 * 0.6
                && IsHeaderLeftSimple(headerRight))
            {
                _headerLeftMap[header] = true;
                return true;

            }
            _headerLeftMap[header] = false;
            return false;
        }

        private bool IsHeaderUpSimple(Boundary header)
        {
            // only consider this simple row, dont consider the relation between the next rows
            if (header.right == header.left)
            {
                return false;
            }
            if (_sheet.sumContentExist.SubmatrixSum(header) <= 4 && HeaderRate(header, step: 0) <= 0.5)
            {
                return false;
            }
            if ((Utils.AreaSize(header) > 4 && _sheet.TextDistinctCount(header) <= 2)
                 || (Utils.AreaSize(header) > 3 && _sheet.TextDistinctCount(header) < 2))
            {
                return false;
            }
            Boundary rightRegionOfHeader = new Boundary(header.top, header.bottom, Math.Min(header.left, header.right - 5) + 3, header.right);
            if (_sheet.ContentExistValueDensity(header) > 2 * 0.3
                && HeaderRate(header) > 0.4
                && HeaderRate(rightRegionOfHeader) > 0.3)
            {
                return true;
            }
            return false;
        }

        private bool IsHeaderLeftSimple(Boundary header)
        {
            // only consider this simple col, dont consider the relation between the next cols

            if (header.bottom - header.top == 0)
            {
                return false;
            }
            if (header.bottom - header.top == 1 && HeaderRate(header) >= 0.5)
            {
                return true;
            }
            if ((Utils.AreaSize(header) > 4 && _sheet.TextDistinctCount(header) <= 2)
                || (Utils.AreaSize(header) > 3 && _sheet.TextDistinctCount(header) < 2))
            {
                return false;
            }
            if (_sheet.sumContentExist.SubmatrixSum(header) <= 4 && HeaderRate(header) <= 0.5)
            {
                return false;
            }
            Boundary upRegionOfHeader = new Boundary(Math.Min(header.top, header.bottom - 5) + 3, header.bottom, header.left, header.right);
            if (_sheet.ContentExistValueDensity(header) > 2 * 0.3 &&
                HeaderRate(header) > 0.4 && HeaderRate(upRegionOfHeader) > 0.3)
            {
                return true;
            }
            return false;
        }

        private bool IsHeaderUpWithDataArea(Boundary upperBoundaryRow, Boundary box)
        {
            int up = upperBoundaryRow.top;
            if (!IsHeaderUp(upperBoundaryRow)) { return false; }
            // find the bottom of up header
            while (IsHeaderUpSimple(upperBoundaryRow) && upperBoundaryRow.top <= up + 2 && upperBoundaryRow.top < box.bottom)
            {
                upperBoundaryRow = new Boundary(upperBoundaryRow.top + 1, upperBoundaryRow.top + 1, upperBoundaryRow.left, upperBoundaryRow.right);
            }

            bool validHeaderMark = true;
            // verify the next rows under the header are not like headers
            for (int k = 1; k <= 3; k++)
            {
                Boundary nextRow = new Boundary(upperBoundaryRow.top + k, upperBoundaryRow.top + k, upperBoundaryRow.left, upperBoundaryRow.right);
                if (nextRow.top <= box.bottom && IsHeaderUp(nextRow))
                {
                    validHeaderMark = false;
                    break;
                }
            }
            return validHeaderMark;
        }

        private double HeaderRate(Boundary box, int step = 6)
        {
            // calculate the rate in the cells in the header proposal that not numerical
            int cntExistContent = 0;
            int cntAllCells = 0;
            int cntHeaderLikeCell = 0;

            for (int i = box.top; i <= box.bottom; i++)
            {
                for (int j = box.left; j <= box.right; j++)
                {
                    cntAllCells++;
                    Boundary headerControledSuroundingRegion = new Boundary(i, i, j, j);
                    if (box.top == box.bottom)
                    {
                        headerControledSuroundingRegion = new Boundary(i, i + step, j, j);
                    }
                    else if (box.top == box.bottom)
                    {
                        headerControledSuroundingRegion = new Boundary(i, i, j, j + step);
                    }
                    if (_sheet.sumContentExist.SubmatrixSum(headerControledSuroundingRegion) != 0)
                    {
                        cntExistContent++;
                        if (_sheet.featureMap[i - 1, j - 1].MarkText)
                        {
                            if ((_sheet.featureMap[i - 1, j - 1].AlphabetRatio >= _sheet.featureMap[i - 1, j - 1].NumberRatio && _sheet.featureMap[i - 1, j - 1].AlphabetRatio != 0)
                                || _sheet.featureMap[i - 1, j - 1].AlphabetRatio * _sheet.featureMap[i - 1, j - 1].TextLength > 2.5 || _sheet.featureMap[i - 1, j - 1].SpCharRatio > 0)
                            {
                                cntHeaderLikeCell++;
                            }
                        }
                    }
                }
            }

            if (cntAllCells == 0) return 0;
            return (double)cntHeaderLikeCell / Math.Max(cntExistContent, (double)cntAllCells / 3);
        }
    }
}