"""Utility function for positions."""


# %% position conversion functions
# string to coordinates

def col_str_to_index(col_str, base=ord('A')-1):
    """Translate alphabet column labels ('ABS') to integer column index. (base == 64)"""
    col_index = 0
    for char in col_str:
        col_index = col_index * 26
        col_index += ord(char) - base
    return col_index


def str_to_coords(ws_str):
    """Translate worksheet str position 'A2' to (1-index) coordinates (2,1)."""
    i, L = 0, len(ws_str)
    while i < L:
        if ws_str[i].isdigit(): break
        i += 1
    if i == L: 
        raise ValueError(f"{ws_str} cannot be translated to coordinates ...")

    row_index = int(ws_str[i:])
    col_index = col_str_to_index(ws_str[:i])
    return (row_index, col_index)
    

# coordinates to string
def col_index_to_str(col_index, base=ord('A')):
    char_list = []

    while col_index > 0:
        col_index -= 1

        r = col_index % 26
        char = chr(base + r)
        char_list.append(char)
        col_index = col_index // 26
    
    col_str = ''.join(reversed(char_list))
    return col_str


def coords_to_str(coords):
    """Translate 1-index row & column coordinates to worksheet position string."""
    row_index, col_index = coords
    col_str = col_index_to_str(col_index)
    return f'{col_str}{row_index}'


# 1-indexed coordinates to 0-indexed 
def ws_to_cm_coords(ws_coords, row_offset=3, col_offset=1):
    ws_row, ws_col = ws_coords
    return (ws_row - row_offset, ws_col - col_offset)

def cm_to_ws_coords(cm_coords, row_offset=3, col_offset=1):
    cm_row, cm_col = cm_coords
    return (cm_row + row_offset, cm_col + col_offset)



def stretch_range(strs):
    """Iteratively take all of the cell items within the range.
    Args:
        strs: List[] of len(*) 1 or 2, e.g. ['A1'], ['A1', 'C3']
    Returns:
        coords_list: List of all single cell items.
    """
    if len(strs) != 1 and len(strs) != 2: 
        raise ValueError(f"got unexpected range {strs} not of len 1or2")

    if len(strs) == 1:   # a single cell
        start_pos, end_pos = strs[0], strs[0]
    else:  # if len(coords) == 2     # a ws-str range  
        start_pos, end_pos = strs
    srow, scol = str_to_coords(start_pos)
    erow, ecol = str_to_coords(end_pos)
    coords_list = [(irow, icol) for irow in range(srow, erow+1) for icol in range(scol, ecol+1)]

    return coords_list


def get_formula_argument(argument_cells):
    """Convert a list of 'Cell' objects to 'LinkedPosition'-'Cell' pairs."""
    argument_pairs = []
    for cell in argument_cells:
        row_index = cell.span['start_row']
        col_index = cell.span['start_col']
        position = LinkedPosition.from_cm_coords( (row_index, col_index) )
        argument_pairs.append((position, cell))
    return argument_pairs

# %% Linked Position Class

class LinkedPosition(object):
    """A unified ds for in-worksheet and on-structure positions."""

    def __init__(self, ws_str=None, ws_coords=None, cm_coords=None):
        self.ws_str = ws_str
        self.ws_coords = ws_coords
        self.cm_coords = cm_coords

    @classmethod
    def from_ws_str(cls, ws_str):
        """Create an instance from available worksheet string."""
        ws_coords = str_to_coords(ws_str)
        cm_coords = ws_to_cm_coords(ws_coords)
        return cls(ws_str, ws_coords, cm_coords)
    
    @classmethod
    def from_ws_coords(cls, ws_coords):
        """Create an instance from available worksheet coordinates."""
        ws_str = coords_to_str(ws_coords)
        cm_coords = ws_to_cm_coords(ws_coords)
        return cls(ws_str, ws_coords, cm_coords)
    
    @classmethod
    def from_cm_coords(cls, cm_coords):
        ws_coords = cm_to_ws_coords(cm_coords)
        ws_str = coords_to_str(ws_coords)
        return cls(ws_str, ws_coords, cm_coords)
