import os
from dataclasses import dataclass, field
import torch
from typing import List, Optional, Union
from filelock import FileLock
import time
from enum import Enum
from torch.utils.data.dataset import Dataset
from utils import InputExample, InputFeatures
import numpy as np
import pandas as pd
from transformers import AutoTokenizer, AutoModelForSequenceClassification, HfArgumentParser, \
    TrainingArguments, set_seed, DataProcessor
from sklearn.metrics import accuracy_score, precision_recall_fscore_support
from sklearn.model_selection import train_test_split
import logging
from OptimTrainer import OptimTrainer, PREFIX_CHECKPOINT_DIR
logger = logging.getLogger(__name__)


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        default=None, metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None, metadata={"help": "Where do you want to store the pretrained models downloaded from s3"}
    )

@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.
    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """
    data_file: str = field(
        default="",
        metadata={"help": "The input data dir. Should contain the .tsv files (or other data files) for the task."}
    )
    task_name: str = field(default="Twitter")
    max_seq_length: int = field(
        default=128,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
            "than this will be truncated, sequences shorter will be padded."
        },
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached training and evaluation sets"}
    )

    def __post_init__(self):
        self.task_name = self.task_name.lower()

def convert_examples_to_features(
    examples: List[InputExample],
    tokenizer: AutoTokenizer,
    max_length: Optional[int] = None,
    label_list: List = None,
    output_mode="classification",
):
    if max_length is None:
        max_length = tokenizer.max_len
    label_map = {label: i for i, label in enumerate(label_list)}

    def label_from_example(example: InputExample) -> Union[int, float, None]:
        if example.label is None:
            return None
        if output_mode == "classification":
            return label_map[example.label]
        elif output_mode == "regression":
            return float(example.label)
        raise KeyError(output_mode)

    labels = [label_from_example(example) for example in examples]

    features = []
    for i, example in enumerate(examples):
        inputs = tokenizer(example.text_a, example.text_b, truncation=True, padding='max_length', max_length=max_length)

        feature = InputFeatures(**inputs, label=labels[i])
        features.append(feature)

    for i, example in enumerate(examples[:5]):
        logger.info("*** Example ***")
        logger.info("guid: %s" % (example.guid))
        logger.info("features: %s" % features[i])

    return features

class Processor(DataProcessor):
    """Processor for the CoLA data set (GLUE version)."""

    def __init__(self, labels, train, dev, test, allf=None, text_column="text",
                 label_column="label", id_column="text_id"):
        self.labels = labels
        self.train = train
        self.dev = dev
        self.test = test
        self.all = allf
        self.text_column = text_column
        self.label_column = label_column
        self.id_column = id_column

    def get_train_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.train), "train")

    def get_train_data(self):
        return self._read_tsv(self.train)
    def get_dev_data(self):
        return self._read_tsv(self.dev)
    def get_test_data(self):
        return self._read_tsv(self.test)

    def get_dev_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.dev), "dev")

    def get_test_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.test), "test")

    def get_all_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.all), "all")

    def get_test_labels(self):
        return self._read_tsv(self.test)

    def get_all_labels(self):
        return self._read_tsv(self.all)

    def get_labels(self):
        """See base class."""
        return self.labels

    def _read_tsv(self, input, quotechar=None):
        lines = []
        for i in input.itertuples():
            lines.append({
                'text': getattr(i, self.text_column),
                'sent_id': getattr(i, self.id_column),
                'label': getattr(i, self.label_column)
            })
        return lines

    def _create_examples(self, lines, set_type):
        """Creates examples for the training, dev and test sets."""
        test_mode = set_type == "test"
        examples = []
        for (i, line) in enumerate(lines):
            guid = "%s-%s" % (set_type, i)
            text_a = line["text"]
            #label = None if test_mode else line["label"]
            label = line["label"]
            examples.append(InputExample(guid=guid, text_a=text_a, text_b=None, label=label))
        return examples

class Split(Enum):
    train = "train"
    dev = "dev"
    test = "test"
    all = "all"

class TwitterDataset(Dataset):
    """
    This will be superseded by a framework-agnostic approach
    soon.
    """

    args: DataTrainingArguments
    output_mode: str
    features: List[InputFeatures]

    def __init__(
        self,
        args: DataTrainingArguments,
        tokenizer: AutoTokenizer,
        processor: DataProcessor,
        limit_length: Optional[int] = None,
        mode: Union[str, Split] = Split.train,
        cache_dir: Optional[str] = None,
        split_id: Optional[str] = "0"
    ):
        self.args = args
        self.processor = processor
        self.output_mode = "classification"
        if isinstance(mode, str):
            try:
                mode = Split[mode]
            except KeyError:
                raise KeyError("mode is not a valid split name")
        # Load data features from cache or dataset file
        cached_features_file = os.path.join(
            cache_dir if cache_dir is not None else args.data_dir,
            "cached_{}_{}_{}_{}_{}".format(
                mode.value,
                split_id,
                tokenizer.__class__.__name__,
                str(args.max_seq_length),
                args.task_name,
            ),
        )
        if cache_dir and not os.path.exists(cache_dir):
            os.mkdir(cache_dir)
        label_list = self.processor.get_labels()
        self.label_list = label_list

        # Make sure only the first process in distributed training processes the dataset,
        # and the others will use the cache.
        lock_path = cached_features_file + ".lock"
        with FileLock(lock_path):

            if os.path.exists(cached_features_file) and not args.overwrite_cache:
                start = time.time()
                self.features = torch.load(cached_features_file)
                logger.info(
                    f"Loading features from cached file {cached_features_file} [took %.3f s]", time.time() - start
                )
            else:
                logger.info(f"Creating features from dataset")

                if mode == Split.dev:
                    examples = self.processor.get_dev_examples(None)
                elif mode == Split.test:
                    examples = self.processor.get_test_examples(None)
                elif mode == Split.train:
                    examples = self.processor.get_train_examples(None)
                else:
                    examples = self.processor.get_all_examples(None)
                if limit_length is not None:
                    examples = examples[:limit_length]
                self.features = convert_examples_to_features(
                    examples,
                    tokenizer,
                    max_length=args.max_seq_length,
                    label_list=label_list,
                    output_mode=self.output_mode,
                )
                start = time.time()
                torch.save(self.features, cached_features_file)
                # ^ This seems to take a lot of time so I want to investigate why and how we can improve.
                logger.info(
                    "Saving features into cached file %s [took %.3f s]", cached_features_file, time.time() - start
                )
    def __len__(self):
        return len(self.features)

    def __getitem__(self, i) -> InputFeatures:
        return self.features[i]

# parameters
BERT_model = "bert-base-german-cased"
train_size = 0.8
test_size = 1.0 - train_size
expert_data = "../data/expertdata_resolved.tsv"
studentfile = "../data/studentdata_original.tsv"

model_args = ModelArguments(
    model_name_or_path=BERT_model,
    tokenizer_name=BERT_model,
    cache_dir="./cache",
    config_name=None,
)

training_args = TrainingArguments(
    logging_steps=100,
    per_device_train_batch_size=8,
    per_device_eval_batch_size=8,
    save_steps=1000,
#            evaluate_during_training=True,
    output_dir="./model_output/",
    overwrite_output_dir=True,
    do_train=True,
    do_eval=True,
    do_predict=True,
    learning_rate=0.00008,
    num_train_epochs=10
)

data_args = DataTrainingArguments(
    data_file="",
    overwrite_cache=True,
    max_seq_length=256
)

seeds = [2020, 2021, 2022, 2023, 2024, 2025, 2026, 2027, 2028, 2029]
fold_processors = {}
for fold in seeds:
    # create expert data processor
    label_column = "label"
    df = pd.read_csv(expert_data, sep="\t", dtype=str, encoding="utf-8")
    df["text"] = df["full_text"]
    df["text_id"] = df["id_str"]
    labels = df[label_column].unique()
    num_labels = len(labels)
    df_train, df_val = train_test_split(df, test_size=test_size, train_size=train_size, shuffle=True, stratify=df[[label_column]],
                                    random_state=fold)
    #df_train, df_val, df_test = split_stratified_into_train_val_test(df, label_column, random_state=fold)
    expert_processor = Processor(train=df_train, dev=df_val, test=None, allf=df, labels=list(labels),
                                 text_column="text", id_column="text_id", label_column=label_column)

    fold_processors[fold] = {
        'expert': expert_processor,
    }

    # create group 1 processor
    label_column = "studentannotation"
    text_column = ""
    df = pd.read_csv(studentfile, sep="\t", dtype=str, encoding="utf-8")

    for group in ["g1", "g2", "g3"]:
        # filter group
        student_df = df[df["group_id"]==group]
        student_df_train, student_df_val = train_test_split(student_df, test_size=test_size, train_size=train_size, shuffle=True,
                                            stratify=student_df[[label_column]], random_state=fold)
        student_processor = Processor(train=student_df_train, dev=student_df_val, test=None, allf=student_df,
                                 labels=list(labels), text_column="full_text", id_column="id_str",
                                 label_column=label_column)

        fold_processors[fold][group] = student_processor

transfer_results = {}
groups = ["expert", "g1", "g2", "g3"]
for fold in seeds:
    print('Fold:', fold)
    transfer_results[fold] = {}
    for group in groups:
        processor = fold_processors[fold][group]
        group_results = {}
        # Setup logging
        logging.basicConfig(
            format="%(asctime)s - %(levelname)s - %(name)s -   %(message)s",
            datefmt="%m/%d/%Y %H:%M:%S",
            level=logging.INFO if training_args.local_rank in [-1, 0] else logging.WARN,
            #level=logging.WARN,
        )
        logger.warning(
            "Process rank: %s, device: %s, n_gpu: %s, distributed training: %s, 16-bits training: %s",
            training_args.local_rank,
            training_args.device,
            training_args.n_gpu,
            bool(training_args.local_rank != -1),
            training_args.fp16,
        )
        logger.info("Training/evaluation parameters %s", training_args)

        # Set seed
        set_seed(fold)
        output_mode = "classification"

        # model
        tokenizer = AutoTokenizer.from_pretrained(BERT_model)
        model = AutoModelForSequenceClassification.from_pretrained(BERT_model, num_labels=num_labels)

        # datasets
        train_dataset = (
            TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, mode="train", cache_dir=model_args.cache_dir)
        )
        eval_dataset = (
            TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, mode="dev", cache_dir=model_args.cache_dir)
        )
        # test_dataset = (
        #     TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, mode="test", cache_dir=model_args.cache_dir)
        # )

        def compute_metrics(pred):
            labels = pred.label_ids
            preds = pred.predictions.argmax(-1)
            precision, recall, f1, _ = precision_recall_fscore_support(labels, preds, average='macro')
            acc = accuracy_score(labels, preds)
            return {
                'accuracy': acc,
                'f1': f1,
                'precision': precision,
                'recall': recall
            }

        trainer = OptimTrainer(
            model=model,
            args=training_args,
            train_dataset=train_dataset,
            eval_dataset=eval_dataset,
            compute_metrics=compute_metrics
        )

        # Training
        if training_args.do_train:
            trainer.train(eval_metric='f1')

        # load best model
        print('load best model')
        checkpoint_folder = f"{PREFIX_CHECKPOINT_DIR}-best"
        output_dir = os.path.join(trainer.args.output_dir, checkpoint_folder)
        model = AutoModelForSequenceClassification.from_pretrained(output_dir, num_labels=num_labels)
        trainer = OptimTrainer(
            model=model,
            args=training_args,
            train_dataset=train_dataset,
            eval_dataset=eval_dataset,
            compute_metrics=compute_metrics
        )
        # Evaluation
        if training_args.do_eval:
            logger.info("*** Evaluate ***")
            eval_result = trainer.evaluate()
            group_results['eval_result'] = eval_result
        # Prediction
        if training_args.do_predict:
            # make predictions for all other groups
            for g in fold_processors[fold].keys():
                if g == group:
                    continue
                proc = fold_processors[fold][g]
                all_dataset = TwitterDataset(data_args, tokenizer=tokenizer, processor=proc, mode="all",
                                   cache_dir=model_args.cache_dir)
                predictions = trainer.predict(test_dataset=all_dataset).predictions
                predictions = np.argmax(predictions, axis=1)
                label_list = proc.get_labels()
                label_map = {label: i for i, label in enumerate(label_list)}
                labels = [label_map[i['label']] for i in proc.get_all_labels()]
                precision, recall, f1, _ = precision_recall_fscore_support(labels, predictions, average='macro')
                accuracy = accuracy_score(labels, predictions)
                group_results[g + '_result'] = {'f1': f1, 'precision': precision, 'recall': recall, 'accuracy': accuracy}
        transfer_results[fold][group] = group_results

import json
with open('transfer_results_train_test_80_20.json', 'w') as of:
    json.dump(transfer_results, of)