///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "nl-cpt.h"
#include "nl-refrv.h"
#include "nl-fixedmatrix.h"
#include "nl-heap.h"
#include <list>

char psX[]="";
char psSlash[]="/";
char psComma[]=",";
char psSemi[]=";";
char psSemiSemi[]=";;";
char psDashDiamondDash[]="-<>-";
char psTilde[]="~";
//char psBar[]="|";
char psLParen[]="(";
char psRParen[]=")";
char psLBrace[]="{";
char psRBrace[]="}";
char psLAngle[]="<";
char psRAngle[]=">";
char psLBrack[]="[";
char psRBrack[]="]";

const char* BEG_STATE = "ID:S/ID:end;-:-/-:-;-:-/-:-;-:-/-:-";
const char* END_STATE = "ID:S/ID:end;-:-/-:-;-:-/-:-;-:-/-:-";

// for dynamic vector sizing
DiscreteDomain<int> domVSIZE;
class VSIZE : public DiscreteDomainRV<int,domVSIZE> {
 public: 
  VSIZE (  ) : DiscreteDomainRV<int,domVSIZE> ( ) { }
  VSIZE ( const char* ps ) : DiscreteDomainRV<int,domVSIZE> ( ps ) { }
  int getInt ( ) { return atoi(getString().c_str()); }
};
VSIZE M_SIZE("13");

////////////////////////////////////////////////////////////////////////////////
//
//  Random Variables
//
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////// Simple Variables

//// SEMANTICS
//// I: individual entity in vector/matrix...
DiscreteDomain<int> domI;
class I : public DiscreteDomainRV<int,domI> {
 public:
  I ( )                : DiscreteDomainRV<int,domI> ( )    { }
  I ( const char* ps ) : DiscreteDomainRV<int,domI> ( ps ) { }
};
//const I I_NIL("-");

//// {Indexed,TimeStamped}Vector
class IndexedVector : public Matrix<Prob> {
 private:
  bool transpose;
 public:
  IndexedVector ( )                        : Matrix<Prob>( )   { transpose=false; }
  IndexedVector ( bool o )                 : Matrix<Prob>( )   { transpose=true;  } // orientation on init
  IndexedVector ( int x, int y )           : Matrix<Prob>(x,y) { /*assert(x==M_SIZE.getInt()||x==1); assert(y==M_SIZE.getInt()||y==1);*/ transpose=false; }
  IndexedVector ( int x, int y, Prob p )   : Matrix<Prob>(x,y,p) { assert(x==M_SIZE.getInt()||x==1); assert(y==M_SIZE.getInt()||y==1); transpose=false; }
  IndexedVector ( const Matrix<Prob>& m )  : Matrix<Prob>(m)   { assert(m.xSize()==M_SIZE.getInt()||m.xSize()==1); assert(m.ySize()==M_SIZE.getInt()||m.ySize()==1); transpose=false; }
  IndexedVector ( const IndexedVector& m ) : Matrix<Prob>(m.xSize(),m.ySize()) {  transpose=false; //assert(m.xSize()==M_SIZE.getInt()||m.xSize()==1); assert(m.ySize()==M_SIZE.getInt()||m.ySize()==1);
    for(int i=0;i<xSize();i++) for(int j=0;j<ySize();j++) set(i,j)=m.get(i,j);
  }
  //IndexedVector ( const matrix<Prob>& m ) : Matrix<Prob>(m) { }

  void setTranspose( bool b ) { transpose=b; } // sets for reading in as transpose. does not calculate a transpose vector

  friend pair<StringInput,IndexedVector*> operator>> ( StringInput si, IndexedVector& m ) {
    return pair<StringInput,IndexedVector*>(si,&m);
  }
  friend StringInput operator>> ( pair<StringInput,IndexedVector*> si_m, const char* psD ) {
    if (StringInput(NULL)==si_m.first) return si_m.first;
    StringInput si; I i,j; Prob p;
    if (si_m.second->transpose) {
      if ((*si_m.second)==IndexedVector()) { 
	si_m.second->init(1,M_SIZE.getInt(),Prob()); 
      }
      si=si_m.first>>i>>" : ">>j>>" = ">>p>>psD;
      if ( si!=NULL ) si_m.second->set(0,j.getIndex())=p;
    } else {
      if ((*si_m.second)==IndexedVector()) { si_m.second->init(M_SIZE.getInt(),1,Prob()); cerr<<" vector index: -"; }
      si=si_m.first>>i>>" : ">>j>>" = ">>p>>psD;
      cerr<<(i.getIndex()==0?"":",")<<i<<(i.getIndex()>=(M_SIZE.getInt()-1) ? "\n":"");
      if ( si!=NULL ) si_m.second->set(i.getIndex(),0)=p;
    }
    return si;
  }
};
class TimeStampedVector : public IndexedVector {
 private:
  typedef IndexedVector Parent;
 public:
  int t;
  TimeStampedVector ( )                        : IndexedVector( ), t(-1) { }
  TimeStampedVector ( bool o )                 : IndexedVector( o ), t(-1) { }
  TimeStampedVector ( int x, int y, Prob p )   : IndexedVector(x,y,p), t(-1) { }
  TimeStampedVector ( const IndexedVector& m ) : IndexedVector(m.xSize(),m.ySize()), t(-1) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  TimeStampedVector ( int tA, const IndexedVector& m ) : IndexedVector(m), t(tA) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  TimeStampedVector ( const TimeStampedVector& m ) : IndexedVector(m.xSize(),m.ySize()), t(m.t) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  /*
  friend bool operator== ( const TimeStampedVector& m1, const TimeStampedVector& m2 ) { 
    if (m1.t!=m2.t) return false;
    if (m1.xSize()!=m2.xSize() || m1.ySize()!=m2.ySize()) return false;
    for(int i=0;i<m1.xSize();i++) for(int j=0;j<m1.ySize();j++) if(m1.get(i,j)!=m2.get(i,j)) return false;
    return true;
  }
  friend bool operator!= ( const TimeStampedVector& m1, const TimeStampedVector& m2 ) { return !(m1==m2); }
  */
};
//// {Indexed,TimeStamped}Matrix
class IndexedMatrix : public Matrix<Prob> {
 public:
  IndexedMatrix ( )                       : Matrix<Prob>( ) { }
  IndexedMatrix ( int x, int y )          : Matrix<Prob>(x,y) { /*assert(x==M_SIZE.getInt()||x==1); assert(y==M_SIZE.getInt()||y==1);*/ }
  IndexedMatrix ( int x, int y, Prob p )  : Matrix<Prob>(x,y,p) { assert(x==M_SIZE.getInt()||x==1); assert(y==M_SIZE.getInt()||y==1); }
  IndexedMatrix ( const Matrix<Prob>& m ) : Matrix<Prob>(m) { assert(m.xSize()==M_SIZE.getInt()||m.xSize()==1); assert(m.ySize()==M_SIZE.getInt()||m.ySize()==1); }
  IndexedMatrix ( const IndexedMatrix& m ) : Matrix<Prob>(m.xSize(),m.ySize()) { //assert(m.xSize()==M_SIZE.getInt()||m.xSize()==1); assert(m.ySize()==M_SIZE.getInt()||m.ySize()==1); 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  //IndexedMatrix ( const matrix<Prob>& m ) : Matrix<Prob>(m) { }
  friend pair<StringInput,IndexedMatrix*> operator>> ( StringInput si, IndexedMatrix& m ) {
    return pair<StringInput,IndexedMatrix*>(si,&m);
  }
  friend StringInput operator>> ( pair<StringInput,IndexedMatrix*> si_m, const char* psD ) {
    if (StringInput(NULL)==si_m.first) return si_m.first;
    StringInput si; I i,j; Prob p;
    if ((*si_m.second)==IndexedMatrix()) si_m.second->init(M_SIZE.getInt(),M_SIZE.getInt(),Prob());
    si=si_m.first>>i>>" : ">>j>>" = ">>p>>psD;
    if ( si!=NULL ) si_m.second->set(i.getIndex(),j.getIndex())=p;
    return si;
  }
};
class TimeStampedMatrix : public IndexedMatrix {
 private:
  typedef IndexedMatrix Parent;
 public:
  int t;
  TimeStampedMatrix ( ) : IndexedMatrix( ), t(-1) { }
  TimeStampedMatrix ( int x, int y, Prob p ) : IndexedMatrix(x,y,p), t(-1) { }
  TimeStampedMatrix ( const IndexedMatrix& m ) : IndexedMatrix(m.xSize(),m.ySize()), t(-1) {
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  TimeStampedMatrix ( int tA, const Matrix<Prob>& m )  : IndexedMatrix(m.xSize(),m.ySize()), t(tA) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  TimeStampedMatrix ( int tA, const IndexedMatrix& m ) : IndexedMatrix(m.xSize(),m.ySize()), t(tA) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  TimeStampedMatrix ( const TimeStampedMatrix& m ) : IndexedMatrix(m.xSize(),m.ySize()), t(m.t) { 
    for(int i=0;i<m.xSize();i++) for(int j=0;j<m.ySize();j++) set(i,j)=m.get(i,j);
  }
  //TimeStampedMatrix ( int tA, const matrix<Prob>& m ) : IndexedMatrix(m), t(tA) { }
};
TimeStampedMatrix M_ID;
TimeStampedMatrix Ms_NIL;
TimeStampedVector M_PRIOR(true);
TimeStampedVector M_1;
TimeStampedVector Mr_NIL;

//// Es: reference to E matrix at shift steps (incomplete constituents)...
std::map<TimeStampedMatrix,TimeStampedMatrix> domEs;
typedef RefRV<TimeStampedMatrix,domEs> Es;
Es Es_TOP; //(M_ID);
Es Es_BOT; //(Ms_NIL);

//// Er: reference to E vector at reduce steps (complete constituents)...
std::map<TimeStampedVector,TimeStampedVector> domEr;
typedef RefRV<TimeStampedVector,domEr> Er;
Er Er_1;   //(M_1);
Er Er_BOT; //(Mr_NIL);

//// A: reference to A matrix...
std::map<TimeStampedVector,TimeStampedVector> domA;
typedef RefRV<TimeStampedVector,domA> A;
A A_TOP; //(M_PRIOR); 
A A_BOT; //(Mr_NIL);


//// SYNTAX
//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
class D : public DiscreteDomainRV<char,domD> {
 public:
  D ( )                : DiscreteDomainRV<char,domD> ( )    { }
  D ( int i )          : DiscreteDomainRV<char,domD> ( i )  { }
  D ( const char* ps ) : DiscreteDomainRV<char,domD> ( ps ) { }
};
const D D_0("0");
const D D_1("1");
const D D_2("2");
const D D_3("3");
const D D_4("4");
const D D_5("5");


//// B: boolean
DiscreteDomain<char> domB;
class B : public DiscreteDomainRV<char,domB> {
 public:
  B ( )                : DiscreteDomainRV<char,domB> ( )    { }
  B ( const char* ps ) : DiscreteDomainRV<char,domB> ( ps ) { }
};
const B B_0 ("0");
const B B_1 ("1");


//// L: linear operator...
DiscreteDomain<int> domL;
class L : public DiscreteDomainRV<int,domL> {
 private:
  static SimpleHash<X,L> hXtoL;
  void calcDetModels ( string s ) {
    if ( !hXtoL.contains(X(s.c_str())) ) {
      hXtoL.set(X(s.c_str())) = *this;
    }
  }
 public:
  static const L L_UNK;
  L ( )                                      : DiscreteDomainRV<int,domL> ( )    { }
  L ( const DiscreteDomainRV<int,domL>& rv ) : DiscreteDomainRV<int,domL> ( rv ) { }
  L ( const char* ps )                       : DiscreteDomainRV<int,domL> ( ps ) { calcDetModels(ps); }
  L ( const X& x ) { *this=hXtoL.get(x); if(L()==*this) *this=L_UNK; }
  friend pair<StringInput,L*> operator>> ( StringInput si, L& x ) { return pair<StringInput,L*>(si,&x); }
  friend StringInput operator>> ( pair<StringInput,L*> si_x, const char* psD ) {
    if ( si_x.first == NULL ) return NULL;
    StringInput si=si_x.first>>(DiscreteDomainRV<int,domL>&)*si_x.second>>psD;
    si_x.second->calcDetModels(si_x.second->getString()); return si; }
};
SimpleHash<X,L> L::hXtoL;
const L L_BOT("-");
const L L_TOP("ID");
const L L::L_UNK("unk");


//// C: constituent category...
DiscreteDomain<int> domC;
class C : public DiscreteDomainRV<int,domC> {
 private:
  static SimpleHash<C,B> hToTerm;
  void calcDetModels ( string s ) {
    if (!hToTerm.contains(*this)) {
      hToTerm.set(*this) = (('A'<=s[0] && s[0]<='Z') || s.find('_')!=string::npos) ? B_0 : B_1;
    }
  }
 public:
  C ( )                                      : DiscreteDomainRV<int,domC> ( )    { }
  C ( const DiscreteDomainRV<int,domC>& rv ) : DiscreteDomainRV<int,domC>(rv) { }
  C ( const char* ps )                       : DiscreteDomainRV<int,domC> ( ps ) { calcDetModels(ps); }
  //C ( string s ) : DiscreteDomainRV<int,domC> ( s )  { calcDetModels(s); }
  B getTerm ( ) const { return hToTerm.get(*this); }
  friend pair<StringInput,C*> operator>> ( StringInput si, C& x ) { return pair<StringInput,C*>(si,&x); }
  friend StringInput operator>> ( pair<StringInput,C*> si_x, const char* psD ) {
    if ( si_x.first == NULL ) return NULL;
    StringInput si=si_x.first>>(DiscreteDomainRV<int,domC>&)*si_x.second>>psD;
    si_x.second->calcDetModels(si_x.second->getString()); return si; }
};
SimpleHash<C,B> C::hToTerm;
const C C_0("0");
const C C_1("1");
const C C_BOT("-");
const C C_TOP("ROOT");
const C C_RST("REST");


// NOTE: G should be of form L:C or L:C#X for terminal symbols where X is hypothesized word;
// so L* still goes from C to C (as currently), but output generated from X instead of C

//// G: grammar-rule consituent...
DiscreteDomain<int> domG;
class G : public DiscreteDomainRV<int,domG> {
 private:
  static SimpleHash<G,L> hToL;
  static SimpleHash<G,C> hToC;
  void calcDetModels ( string s ) {
    if (!hToL.contains(*this)) {
      size_t i=s.find(':');
      assert(i!=string::npos);
      hToL.set(*this) = L(s.substr(0,i).c_str());
    }
    if (!hToC.contains(*this)) {
      size_t i=s.find(':');
      assert(i!=string::npos);
      size_t j=s.find('#');     // HERE: this line and first case below cover terminals with X
      if (j!=string::npos)
        hToC.set(*this) = C(s.substr(i+1,j-i).c_str());
      else
        hToC.set(*this) = C(s.substr(i+1).c_str());
    }
  }
 public:
  G ( )                                      : DiscreteDomainRV<int,domG> ( )    { }
  G ( const DiscreteDomainRV<int,domG>& rv ) : DiscreteDomainRV<int,domG> ( rv ) { }
  G ( const char* ps )                       : DiscreteDomainRV<int,domG> ( ps ) { calcDetModels(ps); }
  L        getL ( )     const { return hToL.get(*this); }
  C        getC ( )     const { return hToC.get(*this); }
  friend pair<StringInput,G*> operator>> ( StringInput si, G& x ) { return pair<StringInput,G*>(si,&x); }
  friend StringInput operator>> ( pair<StringInput,G*> si_x, const char* psD ) {
    if ( si_x.first == NULL ) return NULL;
    StringInput si=si_x.first>>(DiscreteDomainRV<int,domG>&)*si_x.second>>psD;
    si_x.second->calcDetModels(si_x.second->getString()); return si; }
  B getTerm ( ) const { return getC().getTerm(); }
};
SimpleHash<G,L> G::hToL;
SimpleHash<G,C> G::hToC;
const G G_BOT("-:-");
const G G_TOP("h:ROOT");
const G G_RST("h:REST");


//// F: final (complete constituent) state...
DiscreteDomain<int> domF;
class F : public DiscreteDomainRV<int,domF> {
 private:
  static SimpleHash<F,L> hToL;
  static SimpleHash<F,C> hToC;
  static SimpleHash<F,G> hToG;
  static SimpleHash<G,F> hFromG;
  void calcDetModels ( string s ) {
    if (!hToL.contains(*this)) {
      size_t i=s.find(':');
      if ( i==string::npos ) cerr<<"yuck! "<<s<<"\n";
      assert(i!=string::npos);
      hToL.set(*this) = L(s.substr(0,i).c_str());
    }
    if (!hToC.contains(*this)) {
      size_t i=s.find(':');
      assert(i!=string::npos);
      hToC.set(*this) = C(s.substr(i+1).c_str());
    }
    if (!hToG.contains(*this)) {
      hToG.set(*this) = (this->toInt()>1) ? G(this->getString().c_str()) : G_BOT;
      hFromG.set(G(this->getString().c_str())) = *this;
    }
  }
 public:
  F ( )                                      : DiscreteDomainRV<int,domF> ( )    { }
  F ( const DiscreteDomainRV<int,domF>& rv ) : DiscreteDomainRV<int,domF> ( rv ) { }
  F ( const char* ps )                       : DiscreteDomainRV<int,domF> ( ps ) { calcDetModels(ps); }
  F ( const G& g )                                                               { *this = hFromG.get(g); }
  L        getL ( )     const { return hToL.get(*this); }
  C        getC ( )     const { return hToC.get(*this); }
  G        getG ( )     const { return hToG.get(*this); }
  static F getF ( G g )       { return hFromG.get(g); }
  friend pair<StringInput,F*> operator>> ( StringInput si, F& x ) { return pair<StringInput,F*>(si,&x); }
  friend StringInput operator>> ( pair<StringInput,F*> si_x, const char* psD ) {
    if ( si_x.first == NULL ) return NULL;
    StringInput si=si_x.first>>(DiscreteDomainRV<int,domF>&)*si_x.second>>psD;
    si_x.second->calcDetModels(si_x.second->getString());  return si; }
};
SimpleHash<F,L> F::hToL;
SimpleHash<F,C> F::hToC;
SimpleHash<F,G> F::hToG;
SimpleHash<G,F> F::hFromG;
const F F_0("-:0");
const F F_1("-:1");
const F F_BOT("-:-");


//// Q: syntactic state...
class Q : public DelimitedJoint2DRV<psX,G,psSlash,G,psX> {
  typedef DelimitedJoint2DRV<psX,G,psSlash,G,psX> Parent;
 public:
  Q ( )                          : Parent ( )        { }
  Q ( const Parent& rv )         : Parent ( rv )     { }
  Q ( const char* ps )           : Parent ( ps )     { }
  Q ( const G& g1, const G& g2 ) : Parent ( g1, g2 ) { }
  const G& getGac ( ) const { return first;  }
  const G& getGaw ( ) const { return second; }
  template<class P> class ArrayIterator : public Parent::ArrayIterator<P> {
   public:
    G::ArrayIterator<P>& setGac ( ) { return Parent::ArrayIterator<P>::first; }
    G::ArrayIterator<P>& setGaw ( ) { return Parent::ArrayIterator<P>::second; }
  };
};
#define Q_BOT Q(G_BOT,G_BOT)
#define Q_TOP Q(G_TOP,G_RST)


//////////////////////////////////////// Joint Variables

//// FE:
class FE : public DelimitedJoint2DRV<psX,F,psLBrace,Er,psRBrace> {
  typedef DelimitedJoint2DRV<psX,F,psLBrace,Er,psRBrace> Parent;
 public:
  FE ( )                                : Parent()    { }
  template<class P>
  FE ( const FE::ArrayIterator<P>& it )               { setVal(it); }
  FE ( const F& f, const Er& e )         : Parent(f,e) { }
  const F&  getF ( ) const { return first;  }
  const Er& getE ( ) const { return second; }
  template<class P> class ArrayIterator : public Parent::ArrayIterator<P> {
   public:
    F::ArrayIterator<P>& setF ( )    { return Parent::ArrayIterator<P>::first;  }
    Er::ArrayIterator<P>& setE ( )   { return Parent::ArrayIterator<P>::second; }
  };
};
FE FE_BOT(F_BOT,Er_BOT);

//// QE:
class QE : public DelimitedJoint3DRV<psX,Q,psLBrace,Es,psRBrace,A,psX> {
  typedef DelimitedJoint3DRV<psX,Q,psLBrace,Es,psRBrace,A,psX> Parent;
 public:
  QE ( )                                    : Parent()      { }
  template<class P>
  QE ( const QE::ArrayIterator<P>& it )                     { setVal(it); }
  QE ( const Q& q, const Es& e, const A& a ) : Parent(q,e,a) { }
  friend pair<StringInput,QE*> operator>> ( StringInput si, QE& m ) { return pair<StringInput,QE*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,QE*> si_m, const char* psD ) {
    return pair<StringInput, Parent*>(si_m) >> psD;
  }
  const Q& getQ   ( ) const { return first;  }
  const G& getGac ( ) const { return first.getGac(); }
  const G& getGaw ( ) const { return first.getGaw(); }
  const Es& getE  ( ) const { return second; }
  const A& getA   ( ) const { return third; }
  template<class P> class ArrayIterator : public Parent::ArrayIterator<P> {
   public:
    Q::ArrayIterator<P>& setQ   ( ) { return Parent::ArrayIterator<P>::first;  }
    G::ArrayIterator<P>& setGac ( ) { return Parent::ArrayIterator<P>::first.setGac();  }
    G::ArrayIterator<P>& setGaw ( ) { return Parent::ArrayIterator<P>::first.setGaw();  }
    Es::ArrayIterator<P>& setE  ( ) { return Parent::ArrayIterator<P>::second; }
    A::ArrayIterator<P>& setA   ( ) { return Parent::ArrayIterator<P>::third; }
  };
};
QE QE_BOT(Q_BOT,Es_BOT,A_BOT);
QE QE_TOP(Q_TOP,Es_TOP,A_TOP);


//// R: collection of syntactic variables at all depths in each `reduce' phase...
typedef DelimitedJointArrayRV<4,psSemi,FE> R;


//// S: collection of syntactic variables at all depths in each `shift' phase...
class S : public DelimitedJoint2DRV<psX,DelimitedJointArrayRV<4,psSemi,QE>,psSemi,FE,psX> {
  typedef DelimitedJoint2DRV<psX,DelimitedJointArrayRV<4,psSemi,QE>,psSemi,FE,psX> Parent;
 public:
  const QE& get ( int i ) const { return first.get(i); }
  template<class P> class ArrayIterator : public Parent::ArrayIterator<P> {
   public:
    QE::ArrayIterator<P>& set ( int i ) { return Parent::ArrayIterator<P>::first.set(i);  }
  };
  operator C()  const { return ( ( (second        !=FE_BOT) ? second.getF().getC()           :
                                   (first.get(4-1)!=QE_BOT) ? first.get(4-1).getGaw().getC() :
                                   (first.get(3-1)!=QE_BOT) ? first.get(3-1).getGaw().getC() :
                                   (first.get(2-1)!=QE_BOT) ? first.get(2-1).getGaw().getC() : first.get(1-1).getGaw().getC() ) ); }
  bool compareFinal ( const S& s ) const { return(*this==s); }
};


//// Y: the set of all (marginalized) reduce and (modeled) shift variables in the HHMM...
class Y : public DelimitedJoint2DRV<psX,R,psDashDiamondDash,S,psX>
{ public:
  operator R() const {return first;}
  operator S() const {return second;}
};


////////////////////////////////////////////////////////////////////////////////
//
//  Models
//
////////////////////////////////////////////////////////////////////////////////

//// Model of FE given D and FE and QE (from above) and QE (from previous)
class FEModel {
 private:
  int t;
  SimpleHash<L,IndexedMatrix >& hL;
  SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix> >& hhLstar;
  SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix> >& hhLzero;
/*   HidVarCPT4DModel<F,D,G,G,LogProb>  mFrd;  // Reduc model: F giv D, G (active cat from prev), G (awaited cat from above) (assume prev awa = reduc) */
  HidVarCPT3DModel<Er,Er,Es,LogProb> mErd;  // Cached reduc model of E (awaited cat from prev), E (from reduction)
  HidVarCPT5DModel<A,A,G,G,Er,LogProb>    mArd;
  HidVarCPT5DModel<F,D,FE,QE,QE,LogProb> mFrdBestFirst;
 public:
  static /*const*/ HidVarCPT1DModel<Er,LogProb> mEr_1;   // Fixed E_TOP model.
  static /*const*/ HidVarCPT1DModel<Er,LogProb> mEr_BOT; // Fixed Er_BOT model.
 private:
  static const HidVarCPT1DModel<F,LogProb>  mF_0;    // Fixed F_ZERO model.
  static const HidVarCPT1DModel<F,LogProb>  mF_BOT;  // Fixed F_BOT model.
 public:
  static bool F_ROOT_OBS;
  static bool greaterF(const pair<F,Prob>& fp1, const pair<F,Prob>& fp2) { return (fp1.second>fp2.second); }
  FEModel ( SimpleHash<L,IndexedMatrix >& h,
	    SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh,
            SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh0 ) : t(0), hL(h), hhLstar(hh), hhLzero(hh0) { }
  LogProb setIterProb ( FE::ArrayIterator<LogProb>& fe, const D& d, const FE& feD, const QE& qeP, const QE& qeU, int& vctr ) const {
    LogProb pr,p;
    //// G (bottom) case...
    if ( feD.getF()==F_BOT && (qeP.getGaw()==G_BOT) ) {
      pr  = mF_BOT .setIterProb(fe.setF(),vctr);
      pr *= mEr_BOT .setIterProb(fe.setE(),vctr);
    }
    //// G (middle) case...
    else if ( feD.getF()==F_BOT && qeP.getGaw().getTerm()==B_1 ) {
      if ( !mFrdBestFirst.contains(d,feD,qeP,qeU) ) {
        Heap<pair<F,Prob>,&greaterF> maxheap;
        Prob pr,prTot=0.0;

        // For each possible F, populate unordered heap for distribution renormalized by approx semantics w(a^T)...
        for ( bool f=false; f<=true; f++ ) {

          // Create referent matrix if does not already exist...
          Er e;
          if ( !mErd.contains( (Er_BOT==feD.getE())?Er_1:feD.getE(), qeP.getE() ) ) {
            // Define e if not in cache...
            e = TimeStampedVector( t, qeP.getE().getRef() * ((Er_BOT==feD.getE())?Er_1:feD.getE()).getRef() );
            const_cast<TimeStampedVector&>(e.getRef()).t = t;
            const_cast<FEModel*>(this)->mErd.setProb ( e, (Er_BOT==feD.getE())?Er_1:feD.getE(), qeP.getE() ) = 1.0;
          } else {
            // Update t, showing referent entity is still relevant...
            e = mErd.getDistrib ( (Er_BOT==feD.getE())?Er_1:feD.getE(), qeP.getE() ).get(0).first;
            const_cast<TimeStampedVector&>(e.getRef()).t = t;
          }

          // Create approximate transpose vector, if does not already exist...
          A a;
          if ( !mArd.contains(qeU.getA(), qeU.getGaw(), qeP.getGac(), e) ) {
            if ( f && IndexedMatrix() == hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(qeP.getGac()) )
              cerr<<"\nERROR: no L* "<<qeU.getGaw()<<" "<<qeP.getGac()<<"\n\n";
            a = (f) ?
              TimeStampedVector ( t, qeU.getA().getRef()
                                  * hhLzero.get(MapKey2D<D,G>(d,qeU.getGaw())).get(qeP.getGac())
                                  * e.getRef() ) :
              TimeStampedVector ( t, qeU.getA().getRef()
                                  * ( hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(qeP.getGac()) -
                                      hhLzero.get(MapKey2D<D,G>(d,qeU.getGaw())).get(qeP.getGac()) )
                                  * e.getRef() ) ;
            const_cast<TimeStampedVector&>(a.getRef()).t = t;
            const_cast<FEModel*>(this)->mArd.setProb ( a, qeU.getA(), qeU.getGaw(), qeP.getGac(), e ) = 1.0;
          } else {
            a = mArd.getDistrib ( qeU.getA(), qeU.getGaw(), qeP.getGac(), e ).get(0).first;
            const_cast<TimeStampedVector&>(a.getRef()).t = t;
          }
          // Store probability...
          pr  = a.getRef().infnorm();
          if ( pr > 0.0 )
            maxheap.enqueue ( pair<F,Prob>((f)?qeP.getGac():F_1,pr) );
          //prTot += pr;
        }
        // Populate renormalized distribution from heap...
        while ( maxheap.getSize()>0 ) {
          const_cast<FEModel*>(this)->mFrdBestFirst.setProb ( maxheap.getTop().first, d, feD, qeP, qeU ) = maxheap.getTop().second;  // / prTot;
          maxheap.dequeueTop();
        }
      }
      pr  = p = mFrdBestFirst.setIterProb(fe.setF(),d,feD,qeP,qeU,vctr);
      if ( vctr<-1 && p==LogProb() ) cerr<<"\nERROR: no condition FrdBestFirst "<<d<<" "<<feD<<" "<<qeP<<" "<<qeU<<" "<<qeP.getE()<<"\n\n";
      pr *= mErd.setIterProb(fe.setE(),(Er_BOT==feD.getE())?Er_1:feD.getE(),qeP.getE(),vctr);
    }
    //// 0 (top) case...
    else {
      pr  = mF_0   .setIterProb(fe.setF(),vctr);
      pr *= mEr_1  .setIterProb(fe.setE(),vctr);  // inconsequential. previously E_TOP
    }
    // Iterate again if result doesn't match root observation...
    if ( vctr>=-1 && d==D_1 && F_ROOT_OBS!=(F(fe.setF()).getC()>C_1) ) pr=LogProb();
    ////cerr<<"    F "<<d<<" "<<feD<<" "<<qeP<<" "<<qeU<<" : "<<fe<<" = "<<pr<<" ("<<vctr<<")\n";
    return pr;
  }
  friend pair<StringInput,FEModel*> operator>> ( StringInput si, FEModel& m ) { return pair<StringInput,FEModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,FEModel*> si_m, const char* psD ) {
    StringInput si;
    return ( false ) ? si : StringInput(NULL);
  }
  void update ( ) {
    for ( HidVarCPT3DModel<Er,Er,Es,LogProb>::iterator mit=mErd.begin(); mit!=mErd.end();  ) {
      if ( t != mit->first.getX1().getRef().t ||
           t != mit->first.getX2().getRef().t ||
           t != mit->second.get(0).first.getRef().t )
        mErd.erase ( mit++ );
      else
        ++mit;
    }
    for ( HidVarCPT5DModel<A,A,G,G,Er,LogProb>::iterator mit=mArd.begin(); mit!=mArd.end(); )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->first.getX4().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mArd.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT5DModel<F,D,FE,QE,QE,LogProb>::iterator mit=mFrdBestFirst.begin(); mit!=mFrdBestFirst.end();  ) {
      if ( t != mit->first.getX2().getE().getRef().t  ||
           t != mit->first.getX3().getE().getRef().t  ||
           t != mit->first.getX4().getE().getRef().t  )
        mFrdBestFirst.erase ( mit++ );
      else
        ++mit;
    }
    t++;
  }
  void writeCaches ( ) {
    mErd.dump(cout,"mErd");
    mArd.dump(cout,"mArd");
  }
};
/*const*/ HidVarCPT1DModel<Er,LogProb> FEModel::mEr_1;   //(Er_1) ;
/*const*/ HidVarCPT1DModel<Er,LogProb> FEModel::mEr_BOT; //(Er_BOT) ;
const HidVarCPT1DModel<F,LogProb>  FEModel::mF_0   (F_0)   ;
const HidVarCPT1DModel<F,LogProb>  FEModel::mF_BOT (F_BOT) ;
bool FEModel::F_ROOT_OBS = false;


//// Model of QE given D and FE and FE and QE(from prev) and QE(from above)
class QEModel {
 public:
/*   HidVarCPT3DModel<G,D,G,LogProb>   mGex;  // Expansion model of G given D, G (from above) */
  SimpleHash<MapKey4D<D,G,G,G>,IndexedMatrix> hM_L;
  SimpleHash<MapKey4D<D,G,G,G>,IndexedMatrix> hM_R;
  static bool greaterG(const pair<G,Prob>& gp1, const pair<G,Prob>& gp2) { return (gp1.second>gp2.second); }
  static bool greaterQ(const pair<Q,Prob>& qp1, const pair<Q,Prob>& qp2) { return (qp1.second>qp2.second); }
 private:
  int t;
  SimpleHash<L,IndexedMatrix >& hL;
  SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix> >& hhLstar;
/*   HidVarCPT4DModel<G,D,G,G,LogProb>   mGac;    // Active trans model of G (active) given D, G (active cat from prev), G (awaited cat from above) */
/*   HidVarCPT4DModel<G,D,G,G,LogProb>   mGad;    // Active trans completn of G (awaited) giv D, G (active cat from curr), G (active cat from prev) */
/*   HidVarCPT4DModel<G,D,G,G,LogProb>   mGaw;    // Awaited trans model of G (awaited) given D, G (awaited cat from prev), G (from reduc) */
  HidVarCPT2DModel<G,G,LogProb>       mG_COPY; // Cached COPY model  --  WARNING: const-cast indicates parts that are not thread safe!
  HidVarCPT2DModel<Es,L,LogProb>           mEex;    // Cached Expansion model of E (prior)
  HidVarCPT5DModel<Es,L,Er,L,L,LogProb>    mEac;    // Cached Active trans model of E given D, L,E (from current reduc), L (curr awaited)
  HidVarCPT6DModel<Es,Es,L,Er,L,L,LogProb> mEaw;    // Cached Awaited trans model of E given D, E (from prev), L,E (from below reduc), L (curr awaited)
  HidVarCPT2DModel<Es,Es,LogProb>          mE_COPY; // Cached COPY model for E
  HidVarCPT5DModel<A,A,G,G,Es,LogProb>     mAex;    // mAac is the same as mAex
  HidVarCPT5DModel<A,A,G,G,Es,LogProb>     mAac;    // same dependencies as mAex.
  HidVarCPT6DModel<A,A,L,Er,L,L,LogProb>  mAaw;
  HidVarCPT2DModel<A,A,LogProb>           mA_COPY;
  HidVarCPT3DModel<G,D,QE,LogProb>        mGexBestFirst;
  HidVarCPT5DModel<G,D,G,Er,QE,LogProb>   mGacBestFirst;
  HidVarCPT6DModel<G,D,G,G,Er,QE,LogProb> mGadBestFirst;
  HidVarCPT4DModel<G,D,FE,QE,LogProb>     mGawBestFirst;
 public:
  static /*const*/ HidVarCPT1DModel<Es,LogProb>   mEs_TOP;   // Fixed E_TOP model.
  static /*const*/ HidVarCPT1DModel<A,LogProb>    mA_BOT;    // Fixed A_BOT model.
  static /*const*/ HidVarCPT1DModel<Es,LogProb>   mEs_BOT;   // Fixed E_BOT model.
 private:
  static const HidVarCPT1DModel<G,LogProb>   mG_BOT;   // Fixed G_BOT model.
 public:
  static X WORD;
  QEModel ( SimpleHash<L,IndexedMatrix >& h,
            SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh ) : t(0), hL(h), hhLstar(hh) { }
  LogProb setIterProb ( QE::ArrayIterator<LogProb>& qe, const D& d, const FE& feD, const FE& fe, const QE& qeP, const QE& qeU, int& vctr ) const {
    LogProb pr,p;
    if (feD.getF().getC()>C_1) {
      if (fe.getF().getC()>=C_1) {
        if (fe.getF().getC()>C_1) {
          //// G G (expansion to null) case:
          if (qeU.getGaw().getTerm()==B_1 || qeU.getQ()==Q_BOT) {
            pr  = mG_BOT.setIterProb(qe.setGac(),vctr);
            pr *= mG_BOT.setIterProb(qe.setGaw(),vctr);
            pr *= mEs_BOT.setIterProb(qe.setE(),vctr);
            pr *= mA_BOT.setIterProb(qe.setA(),vctr);
          }
          //// G G (expansion) case:
          else {
            if ( !mGexBestFirst.contains(d,qeU) ) {
              Heap<pair<G,Prob>,&greaterG> maxheap;
              Prob pr,prTot=0.0;

              // For each result of L*, populate unordered heap for distribution renormalized by approx semantics w(a^T)...
              for ( SimpleHash<G,IndexedMatrix>::const_iterator it = hhLstar.get(MapKey2D<D,G>(d,qeP.getGaw())).begin(); it != hhLstar.get(MapKey2D<D,G>(d,qeP.getGaw())).end(); it++ ) {
                if ( it->first.getC().getTerm()==B_1 ) {
                  const G gh0 = it->first;

                  // Create referent matrix if does not already exist...
                  Es e;
                  if ( !mEex.contains( L(WORD) ) ) {
                    // Define e if not in cache...
                    e = TimeStampedMatrix ( t, (gh0.getC().getTerm()==B_0) ? Es_TOP.getRef() : hL.get(L(WORD)) );
                    const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
                    const_cast<QEModel*>(this)->mEex.setProb ( e, L(WORD) ) = 1.0;
                  } else {
                    // Update t, showing referent entity is still relevant...
                    e = mEex.getDistrib ( L(WORD) ).get(0).first;
                    const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
                  }

                  // Create approximate transpose vector, if does not already exist...
                  A a;
                  if ( !mAex.contains(qeU.getA(), qeU.getGaw(), gh0, e) ) {
                    if ( IndexedMatrix() == hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(gh0) )
                      cerr<<"\nERROR: no L* "<<qeU.getGaw()<<" "<<gh0<<"\n\n";
                    a = TimeStampedVector ( t, qeU.getA().getRef() * (hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(gh0)) * e.getRef() );
                    const_cast<TimeStampedVector&>(a.getRef()).t = t;
                    const_cast<QEModel*>(this)->mAex.setProb ( a, qeU.getA(), qeU.getGaw(), gh0, e ) = 1.0;
                  } else {
                    a = mAex.getDistrib ( qeU.getA(), qeU.getGaw(), gh0, e ).get(0).first;
                    const_cast<TimeStampedVector&>(a.getRef()).t = t;
                  }

                  // Store probability...
                  pr  = a.getRef().infnorm();
                  if ( pr>0.0 ) maxheap.enqueue ( pair<G,Prob>(gh0,pr) );
                  //prTot += pr;
                }
              }
              // Populate renormalized distribution from heap...
              if ( 0==maxheap.getSize() )
                const_cast<QEModel*>(this)->mGexBestFirst.setProb ( G_BOT, d, qeU ) =
                  //const_cast<QEModel*>(this)->mEex.setProb ( E_BOT, G_BOT, L(WORD) ) = 
                  const_cast<QEModel*>(this)->mAex.setProb ( A_BOT, qeU.getA(), qeU.getGaw(), G_BOT, Es_BOT ) = 1.0;
              while ( maxheap.getSize()>0 ) {
                const_cast<QEModel*>(this)->mGexBestFirst.setProb ( maxheap.getTop().first, d, qeU ) = maxheap.getTop().second;  // / prTot;
                maxheap.dequeueTop();
              }
            }
            pr  = p = mGexBestFirst.setIterProb(qe.setGac(),d,qeU,vctr);
            if ( vctr<-1 && p==LogProb() ) cerr<<"\nERROR: no condition GexBestFirst "<<d<<" "<<qeU<<"\n\n";
            if ( !mG_COPY.contains(G(qe.setGac())) )
              const_cast<QEModel*>(this)->mG_COPY.setProb(G(qe.setGac()),G(qe.setGac()))=1.0;
            pr *= mG_COPY.setIterProb(qe.setGaw(),G(qe.setGac()),vctr);
            pr *= mEex.setIterProb ( qe.setE(), L(WORD), vctr );
            pr *= mAex.setIterProb ( qe.setA(), qeU.getA(), qeU.getGaw(), G(qe.setGac()), Es(qe.setE()), vctr );
          }
        }
        //// G 1 (active transition) case:
        else {
          if ( !mGacBestFirst.contains(d,qeP.getGac(),fe.getE(),qeU) ) {
            Heap<pair<Q,Prob>,&greaterQ> maxheap;
            Prob p,pr,prTot=0.0;

            // For each d,c,l,c,l,c key in M_L, populate unordered heap for distribution renormalized by approx semantics w(a^T)...
            for ( SimpleHash<MapKey4D<D,G,G,G>,IndexedMatrix>::const_iterator it = hM_L.begin(); it != hM_L.end(); it++ ) {
              if ( it->first.getX1()==d && it->first.getX3()==qeP.getGac() ) {
                const G gh  = it->first.getX2();
                const G gh0 = it->first.getX3();
                const G gh1 = it->first.getX4();

                // Create referent matrix if does not already exist...
                Es e;
                if ( !mEac.contains(gh0.getL(), fe.getE(), gh1.getL(), L(WORD)) ) {
                  if ( IndexedMatrix() == hL.get(gh0.getL()) )
                    cerr<<"\nERROR: no L "<<gh0.getL()<<"\n\n";
                  if ( IndexedMatrix() == hL.get(gh1.getL()) )
                    cerr<<"\nERROR: no L "<<gh1.getL()<<"\n\n";
                  if ( gh1.getC().getTerm()==B_0 )
                    e = TimeStampedMatrix ( t, hM_L.get(it->first) * diag( hL.get(gh0.getL()) * fe.getE().getRef() ) *  hL.get(gh1.getL()) );
                  else
                    e = TimeStampedMatrix ( t, hM_L.get(it->first) * diag( hL.get(gh0.getL()) * fe.getE().getRef() ) * hL.get(gh1.getL()) * hL.get(L(WORD)) );
                  const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
                  const_cast<QEModel*>(this)->mEac.setProb ( e, gh0.getL(), fe.getE(), gh1.getL(), L(WORD) ) = 1.0;
                } else {
                  e = mEac.getDistrib ( gh0.getL(), fe.getE(), gh1.getL(), L(WORD) ).get(0).first;
                  const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
                }

                // Create approximate transpose vector, if does not already exist...
                A a;
                if ( !mAac.contains(qeU.getA(), qeU.getGaw(), gh1, e) ) {
                  if ( IndexedMatrix() == hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(gh1) )
                    cerr<<"\nERROR: no L* "<<qeU.getGaw()<<" "<<gh1<<"\n\n";
                  a = TimeStampedVector ( t, qeU.getA().getRef() * (hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(gh)) * e.getRef() );
                  const_cast<TimeStampedVector&>(a.getRef()).t = t;
                  const_cast<QEModel*>(this)->mAac.setProb ( a, qeU.getA(), qeU.getGaw(), gh1, e ) = 1.0;
                } else {
                  a = mAac.getDistrib ( qeU.getA(), qeU.getGaw(), gh1, e ).get(0).first;
                  const_cast<TimeStampedVector&>(a.getRef()).t = t;
                }

                // Store probability...
                pr  = a.getRef().infnorm();
                if ( pr>0.0 ) maxheap.enqueue ( pair<Q,Prob>(Q(gh,gh1),pr) );
                //prTot += pr;
              }
            }
            // Populate renormalized distribution from heap...
            if ( 0==maxheap.getSize() )
              const_cast<QEModel*>(this)->mGacBestFirst.setProb ( G_BOT, d, qeP.getGac(), fe.getE(), qeU ) =
                const_cast<QEModel*>(this)->mGadBestFirst.setProb ( G_BOT, d, G_BOT, qeP.getGac(), fe.getE(), qeU ) =
                const_cast<QEModel*>(this)->mEac.setProb ( Es_BOT, qeP.getGac().getL(), fe.getE(), G_BOT.getL(), L(WORD) ) = 
                const_cast<QEModel*>(this)->mAac.setProb ( A_BOT, qeU.getA(), qeU.getGaw(), G_BOT, Es_BOT ) = 1.0;
            while ( maxheap.getSize()>0 ) {
              const_cast<QEModel*>(this)->mGacBestFirst.setProb ( maxheap.getTop().first.getGac(), d, qeP.getGac(), fe.getE(), qeU )
                = 1.0; //maxheap.getTop().second / prTot;
              const_cast<QEModel*>(this)->mGadBestFirst.setProb ( maxheap.getTop().first.getGaw(), d,
                                                               maxheap.getTop().first.getGac(), qeP.getGac(), fe.getE(), qeU )
                = maxheap.getTop().second;  // / prTot;
              maxheap.dequeueTop();
            }
          }
          pr  = mGacBestFirst.setIterProb(qe.setGac(),d,qeP.getGac(),fe.getE(),qeU,vctr);
          if ( vctr<-1 && pr==LogProb() ) cerr<<"\nERROR: no condition GacBestFirst "<<d<<" "<<qeP.getGac()<<" "<<fe.getE()<<" "<<qeU<<"\n\n";
          pr *= p = mGadBestFirst.setIterProb(qe.setGaw(),d,G(qe.setGac()),qeP.getGac(),fe.getE(),qeU,vctr);
          if ( vctr<-1 && p==LogProb() ) cerr<<"\nERROR: no condition GadBestFirst "<<d<<" "<<G(qe.setGac())<<" "<<qeP.getGac()<<" "<<fe.getE()<<" "<<qeU<<"\n\n";
          pr *= mEac.setIterProb ( qe.setE(), qeP.getGac().getL(), fe.getE(), G(qe.setGaw()).getL(), L(WORD), vctr );
          pr *= mAac.setIterProb ( qe.setA(), qeU.getA(), qeU.getGaw(), G(qe.setGac()), Es(qe.setE()), vctr );
        }
      }
      //// G 0 (awaited transition) case:
      else {
        if ( !mGawBestFirst.contains(d,feD,qeP) ) {
          Heap<pair<G,Prob>,&greaterG> maxheap;
          Prob pr,prTot=0.0;

          // For each d,c,l,c,l,c key in M_R, populate unordered heap for distribution renormalized by approx semantics w(a^T)...
          for ( SimpleHash<MapKey4D<D,G,G,G>,IndexedMatrix>::const_iterator it = hM_R.begin(); it != hM_R.end(); it++ ) {
            if ( it->first.getX1()==d && it->first.getX2()==qeP.getGaw() && it->first.getX3()==feD.getF().getG() ) {
              const G gh  = it->first.getX2();
              const G gh0 = it->first.getX3();
              const G gh1 = it->first.getX4();

              // Create referent matrix if does not already exist...
              Es e;
              if ( !mEaw.contains(qeP.getE(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD)) ) {
                if ( IndexedMatrix() == hL.get(feD.getF().getG().getL()) )
                  cerr<<"\nERROR: no L "<<feD.getF().getG().getL()<<"\n\n";
                if ( IndexedMatrix() == hL.get(gh1.getL()) )
                  cerr<<"\nERROR: no L "<<gh1.getL()<<"\n\n";
                if (gh1.getC().getTerm()==B_0)
                  e = TimeStampedMatrix ( t, qeP.getE().getRef() * hM_R.get(it->first) * diag( hL.get(feD.getF().getG().getL()) * feD.getE().getRef() ) * hL.get(gh1.getL()) );
                else
                  e = TimeStampedMatrix ( t, qeP.getE().getRef() * hM_R.get(it->first) * diag( hL.get(feD.getF().getG().getL()) * feD.getE().getRef() ) * hL.get(gh1.getL()) * hL.get(L(WORD)) );
                const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
                const_cast<QEModel*>(this)->mEaw.setProb( e, qeP.getE(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD) ) = 1.0;
              } else {
                e = mEaw.getDistrib ( qeP.getE(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD) ).get(0).first;
                const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
              }

              // Create approximate transpose vector, if does not already exist...
              A a;
              if ( !mAaw.contains(qeP.getA(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD)) ) {
                if ( IndexedMatrix() == hL.get(feD.getF().getG().getL()) )
                  cerr<<"\nERROR: no L "<<feD.getF().getG().getL()<<"\n\n";
                if ( IndexedMatrix() == hL.get(gh1.getL()) )
                  cerr<<"\nERROR: no L "<<gh1.getL()<<"\n\n";
                a = TimeStampedVector ( t, qeU.getA().getRef() * (hhLstar.get(MapKey2D<D,G>(d,qeU.getGaw())).get(qeP.getGac())) * e.getRef() );
/*               if (gh1.getC().getTerm()==B_0) */
/*                 a = TimeStampedVector ( t, qeP.getA().getRef() * hM.get(it->first) * diag( hL.get(feD.getF().getG().getL()) * feD.getE().getRef() ) * hL.get(gh1.getL()) ); */
/*               else */
/*                 a = TimeStampedVector ( t, qeP.getA().getRef() * hM.get(it->first) * diag( hL.get(feD.getF().getG().getL()) * feD.getE().getRef() ) * (hL.get(gh1.getL())) * hL.get(L(WORD)) ); */
                const_cast<TimeStampedVector&>(a.getRef()).t = t;
                const_cast<QEModel*>(this)->mAaw.setProb( a, qeP.getA(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD) ) = 1.0;
              } else {
                a = mAaw.getDistrib ( qeP.getA(), feD.getF().getG().getL(), feD.getE(), gh1.getL(), L(WORD) ).get(0).first;
                const_cast<TimeStampedVector&>(a.getRef()).t = t;
              }

              // Store probability...
              pr  = a.getRef().infnorm();
              if ( pr>0.0 ) maxheap.enqueue ( pair<G,Prob>(gh1,pr) );
              //prTot += pr;
            }
            // Populate renormalized distribution from heap...
            if ( 0==maxheap.getSize() )
              const_cast<QEModel*>(this)->mGawBestFirst.setProb ( G_BOT, d, feD, qeP ) =
                const_cast<QEModel*>(this)->mEaw.setProb( Es_BOT, qeP.getE(), feD.getF().getG().getL(), feD.getE(), G_BOT.getL(), L(WORD) ) = 
                const_cast<QEModel*>(this)->mAaw.setProb( A_BOT, qeP.getA(), feD.getF().getG().getL(), feD.getE(), G_BOT.getL(), L(WORD) ) = 1.0;
            while ( maxheap.getSize()>0 ) {
              const_cast<QEModel*>(this)->mGawBestFirst.setProb ( maxheap.getTop().first, d, feD, qeP ) = maxheap.getTop().second;  // / prTot;
              maxheap.dequeueTop();
            }
          }
        }
        if ( !mG_COPY.contains(qeP.getGac()) )
          const_cast<QEModel*>(this)->mG_COPY.setProb(qeP.getGac(),qeP.getGac())=1.0;
        pr  = mG_COPY.setIterProb(qe.setGac(),qeP.getGac(),vctr);
        pr *= p = mGawBestFirst.setIterProb(qe.setGaw(),d,feD,qeP,vctr);
        if ( vctr<-1 && p==LogProb() ) cerr<<"\nERROR: no condition GawBestFirst "<<d<<" "<<feD<<" "<<qeP<<"\n\n";
        pr *= mEaw.setIterProb ( qe.setE(), qeP.getE(), feD.getF().getG().getL(), feD.getE(), G(qe.setGaw()).getL(), L(WORD), vctr );
        pr *= mAaw.setIterProb ( qe.setA(), qeP.getA(), feD.getF().getG().getL(), feD.getE(), G(qe.setGaw()).getL(), L(WORD), vctr );
      }
    }
    //// <=1 0 (copy) case:
    else {
      if ( !mE_COPY.contains(qeP.getE()) ) {
        // Define e if not in cache...
        Es e = //const_cast<QEModel*>(this)->estorem.add() = 
          TimeStampedMatrix ( t, qeP.getE().getRef() );
        const_cast<TimeStampedMatrix&>(e.getRef()).t = t;
        const_cast<QEModel*>(this)->mE_COPY.setProb ( e, qeP.getE() ) = 1.0;
      } else {
        // Update t, showing referent entity is still relevant...
        const_cast<TimeStampedMatrix&>(mE_COPY.getDistrib ( qeP.getE() ).get(0).first.getRef()).t = t;
      }
      if ( !mA_COPY.contains(qeP.getA()) ) {
        // Define a if not in cache...
        A a = //const_cast<QEModel*>(this)->estorev.add() = 
          TimeStampedVector ( t, qeP.getA().getRef() );
        const_cast<TimeStampedVector&>(a.getRef()).t = t;
        const_cast<QEModel*>(this)->mA_COPY.setProb ( a, qeP.getA() ) = 1.0;
      } else {
        // Update t, showing referent entity is still relevant...
        A a = mA_COPY.getDistrib ( qeP.getA() ).get(0).first;
        const_cast<TimeStampedVector&>(a.getRef()).t = t;
      }
      if ( !mG_COPY.contains(qeP.getGac()) )
        const_cast<QEModel*>(this)->mG_COPY.setProb(qeP.getGac(), qeP.getGac() )=1.0;
      pr  = mG_COPY.setIterProb(qe.setGac(), qeP.getGac(), vctr);
      if ( !mG_COPY.contains(qeP.getGaw()) )
        const_cast<QEModel*>(this)->mG_COPY.setProb(qeP.getGaw(),qeP.getGaw())=1.0;
      pr *= mG_COPY.setIterProb(qe.setGaw(),qeP.getGaw(),vctr);
      pr *= mE_COPY.setIterProb(qe.setE(),qeP.getE(),vctr);
      pr *= mA_COPY.setIterProb(qe.setA(),qeP.getA(),vctr);
    }
    ////cerr<<"    Q "<<d<<" "<<feD<<" "<<fe<<" "<<qeP<<" "<<qeU<<" : "<<qe<<" = "<<pr<<" ("<<vctr<<")\n";
    //if (A(qe.setA()).getRef()==TimeStampedVector() && QE(qe)!=QE_BOT) {
    //  cerr<<"ERROR: produced empty A. "<<d<<" "<<feD<<" "<<fe<<" "<<qeP<<" "<<qeU<<" : "<<qe<<" = "<<pr<<" ("<<vctr<<")\n";
    //} 
     return pr;
 }
  friend pair<StringInput,QEModel*> operator>> ( StringInput si, QEModel& m ) { return pair<StringInput,QEModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,QEModel*> si_m, const char* psD ) {
    StringInput si; D d; G gh, gh0, gh1;
    return ( (si=si_m.first>>"M_L ">>d>>" ">>gh>>" ">>gh0>>" ">>gh1>>" ")!=NULL && (si=si>>si_m.second->hM_L.set(MapKey4D<D,G,G,G>(d,gh,gh0,gh1))>>psD)!=NULL ||
             (si=si_m.first>>"M_R ">>d>>" ">>gh>>" ">>gh0>>" ">>gh1>>" ")!=NULL && (si=si>>si_m.second->hM_R.set(MapKey4D<D,G,G,G>(d,gh,gh0,gh1))>>psD)!=NULL ) ? si : StringInput(NULL);
  }
  void update ( ) {
    for ( HidVarCPT2DModel<Es,L,LogProb>::iterator mit=mEex.begin(); mit!=mEex.end();  )
      if ( t != mit->second.get(0).first.getRef().t  )
        mEex.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT5DModel<Es,L,Er,L,L,LogProb>::iterator mit=mEac.begin(); mit!=mEac.end();  )
      if ( t != mit->first.getX2().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mEac.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT6DModel<Es,Es,L,Er,L,L,LogProb>::iterator mit=mEaw.begin(); mit!=mEaw.end();  )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->first.getX3().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mEaw.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT2DModel<Es,Es,LogProb>::iterator mit=mE_COPY.begin(); mit!=mE_COPY.end();  )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mE_COPY.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT5DModel<A,A,G,G,Es,LogProb>::iterator mit=mAex.begin(); mit!=mAex.end(); )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->first.getX4().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mAex.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT5DModel<A,A,G,G,Es,LogProb>::iterator mit=mAac.begin(); mit!=mAac.end(); )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->first.getX4().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mAac.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT6DModel<A,A,L,Er,L,L,LogProb>::iterator mit=mAaw.begin(); mit!=mAaw.end();  )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->first.getX3().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mAaw.erase ( mit++ );
      else
        ++mit;
    for ( HidVarCPT2DModel<A,A,LogProb>::iterator mit=mA_COPY.begin(); mit!=mA_COPY.end();  )
      if ( t != mit->first.getX1().getRef().t  ||
           t != mit->second.get(0).first.getRef().t  )
        mA_COPY.erase ( mit++ );
      else
        ++mit;
    mGexBestFirst.clear();  // because L(WORD) different every step
    mGacBestFirst.clear();
    mGadBestFirst.clear();
    mGawBestFirst.clear();

    const_cast<TimeStampedMatrix&>(Es_TOP.getRef()).t = t;
    const_cast<TimeStampedMatrix&>(Es_BOT.getRef()).t = t;
    for ( map<TimeStampedMatrix,TimeStampedMatrix>::iterator mit=Es::setDomain().begin(); mit!=Es::setDomain().end();  ) {
      if ( mit->second.t != t )
        Es::setDomain().erase(mit++);
      else
        ++mit;
    }
    const_cast<TimeStampedVector&>(Er_1  .getRef()).t = t;
    const_cast<TimeStampedVector&>(Er_BOT.getRef()).t = t;
    for ( map<TimeStampedVector,TimeStampedVector>::iterator mit=Er::setDomain().begin(); mit!=Er::setDomain().end();  ) {
      if ( mit->second.t != t )
        Er::setDomain().erase(mit++);
      else
        ++mit;
    }
    const_cast<TimeStampedVector&>(A_TOP.getRef()).t = t;
    const_cast<TimeStampedVector&>(A_BOT.getRef()).t = t;
    for ( map<TimeStampedVector,TimeStampedVector>::iterator mit=A::setDomain().begin(); mit!=A::setDomain().end();  ) {
      if ( mit->second.t != t )
        A::setDomain().erase(mit++);
      else
        ++mit;
    }
    // update time step for QEModel
    t++;
 }
  void writeCaches ( ) {
    mEex.dump(cout,"mEex");
    mEac.dump(cout,"mEac");
    mEaw.dump(cout,"mEaw");
    mE_COPY.dump(cout,"mE_COPY");
    mAex.dump(cout,"mAex");
    mAaw.dump(cout,"mAaw");
    mA_COPY.dump(cout,"mA_COPY");

    for ( map<TimeStampedMatrix,TimeStampedMatrix>::iterator mit=Es::setDomain().begin(); mit!=Es::setDomain().end(); mit++ ) {
      cerr<<"domEs t="<<mit->second.t<<" x="<<mit->second.xSize()<<" y="<<mit->second.ySize()<<" addr="<<&(mit->second)<<mit->second<<endl;
    }
    for ( map<TimeStampedVector,TimeStampedVector>::iterator mit=Er::setDomain().begin(); mit!=Er::setDomain().end(); mit++ ) {
      cerr<<"domEr t="<<mit->second.t<<" x="<<mit->second.xSize()<<" y="<<mit->second.ySize()<<" addr="<<&(mit->second)<<mit->second<<endl;
    }
    for ( map<TimeStampedVector,TimeStampedVector>::iterator mit=A::setDomain().begin(); mit!=A::setDomain().end(); mit++ ) {
      cerr<<"domA t="<<mit->second.t<<" x="<<mit->second.xSize()<<" y="<<mit->second.ySize()<<" addr="<<&(mit->second)<<mit->second<<endl;
    }
  }
};
/*const*/ HidVarCPT1DModel<Es,LogProb>  QEModel::mEs_BOT; // (Es_BOT) ;
/*const*/ HidVarCPT1DModel<A,LogProb>   QEModel::mA_BOT;  // (A_BOT) ;
/*const*/ HidVarCPT1DModel<Es,LogProb>  QEModel::mEs_TOP; // (Es_TOP) ;
const HidVarCPT1DModel<G,LogProb>  QEModel::mG_BOT (G_BOT) ;
X QEModel::WORD;


//// Model of R given S
class RModel {
 private:
  FEModel mFE;
 public:
  RModel ( SimpleHash<L,IndexedMatrix >& h,
           SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh,
           SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh0 ) : mFE(h,hh,hh0) { }
  LogProb setIterProb ( R::ArrayIterator<LogProb>& r, const S& sP, int& vctr ) const {
    LogProb pr;
    pr  = mFE.setIterProb ( r.set(4-1), D(4), FE(sP.second),  sP.get(4-1), sP.get(3-1), vctr );
    pr *= mFE.setIterProb ( r.set(3-1), D(3), FE(r.get(4-1)), sP.get(3-1), sP.get(2-1), vctr );
    pr *= mFE.setIterProb ( r.set(2-1), D(2), FE(r.get(3-1)), sP.get(2-1), sP.get(1-1), vctr );
    pr *= mFE.setIterProb ( r.set(1-1), D(1), FE(r.get(2-1)), sP.get(1-1), QE_TOP     , vctr );
    return pr;
  }
  void update ( ) { mFE.update(); }
  void writeCaches ( ) { mFE.writeCaches(); }
  friend pair<StringInput,RModel*> operator>> ( StringInput si, RModel& m ) { return pair<StringInput,RModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,RModel*> si_m, const char* psD ) {
    StringInput si; return ( (si=si_m.first>>si_m.second->mFE>>psD)!=NULL ) ? si : NULL; }
};


//// Model of S given R and S
class SModel {
 private:
  int t;
  SimpleHash<L,IndexedMatrix >& hL;
  QEModel mQE;
  static const HidVarCPT1DModel<F,LogProb>   mF_BOT;
  HidVarCPT2DModel<Er,L,LogProb>             mEex;    // Cached Expansion model of E (prior)
  HidVarCPT2DModel<F,QE,LogProb>             mFexBestFirst;
  static bool greaterF(const pair<F,Prob>& gp1, const pair<F,Prob>& gp2) { return (gp1.second>gp2.second); }
  static X& WORD;
 public:
  SModel ( SimpleHash<L,IndexedMatrix >& h,
           SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > >& hh ) : t(0), hL(h), mQE(h,hh) { }
  LogProb setIterProb ( S::ArrayIterator<LogProb>& s, const R::ArrayIterator<LogProb>& r, const S& sP, int& vctr ) const {
    Heap<pair<F,Prob>,&greaterF> maxheap;
    const QE qeU = QE(s.set(4-1));

    if ( !mFexBestFirst.contains(qeU) ) {
      Heap<pair<F,Prob>,&greaterF> maxheap;
      Prob pr,prTot=0.0;

      // For each d,c,l,c,l,c key in M_R, populate unordered heap for distribution renormalized by approx semantics w(a^T)...
      for ( SimpleHash<MapKey4D<D,G,G,G>,IndexedMatrix>::const_iterator it = mQE.hM_R.begin(); it != mQE.hM_R.end(); it++ ) {
        if ( it->first.getX1()==4 && it->first.getX2()==qeU.getGaw() && it->first.getX3().getTerm()==B_1 ) {
          const G gh  = it->first.getX2();
          const G gh0 = it->first.getX3();
          const G gh1 = it->first.getX4();

          // Create referent matrix if does not already exist...
          Er e;
          if ( !mEex.contains( L(WORD)) ) {
            // Define e if not in cache...
            e = TimeStampedVector ( t, mQE.hM_R.get(it->first) * hL.get(L(WORD)) * Er_1.getRef() );
            const_cast<TimeStampedVector&>(e.getRef()).t = t;
            const_cast<SModel*>(this)->mEex.setProb( e, L(WORD) ) = 1.0;
          } else {
            // Update t, showing referent entity is still relevant...
            e = mEex.getDistrib ( L(WORD) ).get(0).first;
            const_cast<TimeStampedVector&>(e.getRef()).t = t;
          }

          pr  = ( qeU.getA().getRef() * e.getRef() ).infnorm();
          if ( pr > 0.0 ) maxheap.enqueue ( pair<G,Prob>(gh0,pr) );
        }
      }
      // Populate renormalized distribution from heap...
      if ( 0==maxheap.getSize() )
        const_cast<SModel*>(this)->mFexBestFirst.setProb ( F_BOT, qeU ) = 1.0;
      while ( maxheap.getSize()>0 ) {
        const_cast<SModel*>(this)->mFexBestFirst.setProb ( maxheap.getTop().first, qeU ) = maxheap.getTop().second;  // / prTot;
        maxheap.dequeueTop();
      }
    }

    LogProb pr;
    pr  = mQE.setIterProb ( s.set(1-1), D(1), FE(r.get(2-1)), FE(r.get(1-1)), sP.get(1-1), QE_TOP         ,vctr );
    pr *= mQE.setIterProb ( s.set(2-1), D(2), FE(r.get(3-1)), FE(r.get(2-1)), sP.get(2-1), QE(s.set(1-1)) ,vctr );
    pr *= mQE.setIterProb ( s.set(3-1), D(3), FE(r.get(4-1)), FE(r.get(3-1)), sP.get(3-1), QE(s.set(2-1)) ,vctr );
    pr *= mQE.setIterProb ( s.set(4-1), D(4), FE(sP.second),  FE(r.get(4-1)), sP.get(4-1), QE(s.set(3-1)) ,vctr );
    pr *= ( G(s.set(4-1).setGaw())!=G_BOT &&
            G(s.set(4-1).setGaw()).getTerm()!=B_1 )
      ? mFexBestFirst.setIterProb ( s.second.first, QE(s.set(4-1)), vctr )
      : mF_BOT.setIterProb ( s.second.first, vctr );
    pr *= mEex.setIterProb ( s.second.second, L(WORD), vctr ) ;
    ////cerr<<"  G "<<5<<" "<<G(q4.second)<<" : "<<g<<" = "<<pr<<" ("<<a<<")\n";
    return pr;
  }
  void update ( ) {
    mQE.update();

    for ( HidVarCPT2DModel<Er,L,LogProb>::iterator mit=mEex.begin(); mit!=mEex.end();  )
      if ( t != mit->second.get(0).first.getRef().t  )
        mEex.erase ( mit++ );
      else
        ++mit;

    mFexBestFirst.clear();  // because L(WORD) different every step

    const_cast<TimeStampedVector&>(Er_1  .getRef()).t = t;
    const_cast<TimeStampedVector&>(Er_BOT.getRef()).t = t;
    for ( map<TimeStampedVector,TimeStampedVector>::iterator mit=Er::setDomain().begin(); mit!=Er::setDomain().end();  ) {
      if ( mit->second.t != t )
        Er::setDomain().erase(mit++);
      else
        ++mit;
    }

    t++;
  }
  void writeCaches ( ) { mQE.writeCaches(); }
  friend pair<StringInput,SModel*> operator>> ( StringInput si, SModel& m ) { return pair<StringInput,SModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,SModel*> si_m, const char* psD ) {
    StringInput si; return ( (si=si_m.first>>si_m.second->mQE>>psD)!=NULL ) ? si : NULL; }
};
const HidVarCPT1DModel<F,LogProb> SModel::mF_BOT(F_BOT);
X&    SModel::WORD       = QEModel::WORD;


//// Model of Y=R,S given S
class YModel {
 private:
  SimpleHash<L,IndexedMatrix > hL;
  SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > > hhLstar;
  SimpleHash<MapKey2D<D,G>,SimpleHash<G,IndexedMatrix > > hhLzero;
  RModel mR;
  SModel mS;
 public:
  static X& WORD;
  static bool& F_ROOT_OBS;
  typedef Y::ArrayIterator<LogProb> IterVal;
  YModel ( ) : mR(hL,hhLstar,hhLzero), mS(hL,hhLstar) { }
  S& setTrellDat ( S& s, const Y::ArrayIterator<LogProb>& y ) const {
    s.setVal(y.second);
    return s;
  }
  R setBackDat ( const Y::ArrayIterator<LogProb>& y ) const {
    R r;
    for(int i=0;i<4;i++)
      r.set(i)=FE(y.first.get(i));
    return r;
  }
  LogProb setIterProb ( Y::ArrayIterator<LogProb>& y, const S& sP, int& vctr ) const {
    LogProb pr;
    // Capture bad matrices coming in from the stack
    for (int i=0; i<3; i++) {
      if ( (Es(sP.get(i).getE()).getRef().xSize()!=M_SIZE.getInt() && Es(sP.get(i).getE()).getRef().xSize()!=1 && Es(sP.get(i).getE()).getRef().xSize()!=0) ||
           (Es(sP.get(i).getE()).getRef().ySize()!=M_SIZE.getInt() && Es(sP.get(i).getE()).getRef().ySize()!=1 && Es(sP.get(i).getE()).getRef().ySize()!=0) ||
           (A(sP.get(i).getA()).getRef().xSize()!=M_SIZE.getInt() && A(sP.get(i).getA()).getRef().xSize()!=1 && A(sP.get(i).getA()).getRef().xSize()!=0) ||
           (A(sP.get(i).getA()).getRef().ySize()!=M_SIZE.getInt() && A(sP.get(i).getA()).getRef().ySize()!=1 && A(sP.get(i).getA()).getRef().ySize()!=0) ) {
        cerr<<" Retrieving bad matrix from beam: QE^"<<(i+1)<<", matching M_SIZE="<<M_SIZE.getInt()<<", in    ";
        cerr<<G(sP.get(0).getGac())<<"/"<<G(sP.get(0).getGaw());
        cerr<<G(sP.get(1).getGac())<<"/"<<G(sP.get(1).getGaw())<<";";
        cerr<<G(sP.get(2).getGac())<<"/"<<G(sP.get(2).getGaw())<<";";
        cerr<<G(sP.get(3).getGac())<<"/"<<G(sP.get(3).getGaw())<<";";
        cerr<<"\n "<<Es(sP.get(i).getE())<<" e.x="<<Es(sP.get(i).getE()).getRef().xSize()<<" e.y="<<Es(sP.get(i).getE()).getRef().ySize()<<endl;
        cerr<<"\n "<<A(sP.get(i).getA())<<" a.x="<<A(sP.get(i).getA()).getRef().xSize()<<" a.y="<<A(sP.get(i).getA()).getRef().ySize()<<endl;
        assert(false);
      }
    }
    pr  = mR.setIterProb ( y.first, sP, vctr );
    if ( LogProb()==pr ) return pr;
    pr *= mS.setIterProb ( y.second, y.first, sP, vctr );
    // Capture bad matrices being written out to the stack
    for (int i=0; i<3; i++) {
      if ( (Es(y.second.set(i).setE()).getRef().xSize()!=M_SIZE.getInt() && Es(y.second.set(i).setE()).getRef().xSize()!=1 && Es(y.second.set(i).setE()).getRef().xSize()!=0) ||
           (Es(y.second.set(i).setE()).getRef().ySize()!=M_SIZE.getInt() && Es(y.second.set(i).setE()).getRef().ySize()!=1 && Es(y.second.set(i).setE()).getRef().ySize()!=0) ||
           (A(y.second.set(i).setA()).getRef().xSize()!=M_SIZE.getInt() && A(y.second.set(i).setA()).getRef().xSize()!=1 && A(y.second.set(i).setA()).getRef().xSize()!=0) ||
           (A(y.second.set(i).setA()).getRef().ySize()!=M_SIZE.getInt() && A(y.second.set(i).setA()).getRef().ySize()!=1 && A(y.second.set(i).setA()).getRef().ySize()!=0)      ) {
        cerr<<" Trying to return bad matrix in y iter: QE^"<<(i+1)<<", matching M_SIZE="<<M_SIZE.getInt()<<", in    ";
        cerr<<G(y.second.set(0).setGac())<<"/"<<G(y.second.set(0).setGaw());
        cerr<<G(y.second.set(1).setGac())<<"/"<<G(y.second.set(1).setGaw())<<";";
        cerr<<G(y.second.set(2).setGac())<<"/"<<G(y.second.set(2).setGaw())<<";";
        cerr<<G(y.second.set(3).setGac())<<"/"<<G(y.second.set(3).setGaw())<<";";
        cerr<<"\n "<<Es(y.second.set(i).setE())<<" e.x="<<Es(y.second.set(i).setE()).getRef().xSize()<<" e.y="<<Es(y.second.set(i).setE()).getRef().ySize()<<endl;
        cerr<<"\n "<<A(y.second.set(i).setA())<<" a.x="<<A(y.second.set(i).setA()).getRef().xSize()<<" a.y="<<A(y.second.set(i).setA()).getRef().ySize()<<endl;
        assert(false);
      }
    }
    return pr;
  }
  void update ( ) const {
    const_cast<YModel*>(this)->mR.update();
    const_cast<YModel*>(this)->mS.update();
  }
  void writeCaches ( ) const {
    const_cast<YModel*>(this)->mR.writeCaches();
    const_cast<YModel*>(this)->mS.writeCaches();
  }
  static bool init ( ) {
    Es_TOP = M_ID;
    Es_BOT = Ms_NIL;
    Er_1   = M_1;
    Er_BOT = Mr_NIL;
    A_TOP  = M_PRIOR;
    A_BOT  = Mr_NIL;
    QE_BOT = QE(Q_BOT,Es_BOT,A_BOT);
    QE_TOP = QE(Q_TOP,Es_TOP,A_TOP);
    FEModel::mEr_1.setProb   ( Er_1   ) = 1.0;
    FEModel::mEr_BOT.setProb ( Er_BOT ) = 1.0;
    QEModel::mEs_BOT.setProb ( Es_BOT ) = 1.0;
    QEModel::mEs_TOP.setProb ( Es_TOP ) = 1.0;
    QEModel::mA_BOT.setProb  ( A_BOT  ) = 1.0;
    cerr<<"constant matrices initialized.\n";
    return true;
  }
  friend pair<StringInput,YModel*> operator>> ( StringInput si, YModel& m ) { return pair<StringInput,YModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,YModel*> si_m, const char* psD ) {
    StringInput si; D d; L l; G g1,g2;
    return ( (si=si_m.first>>"M_SIZE ">>M_SIZE>>psD)!=NULL ||
             (si=si_m.first>>"M_ID " >>M_ID >>psD)!=NULL ||
             (si=si_m.first>>"M_1 ">>M_1>>psD)!=NULL ||
             (si=si_m.first>>"M_PRIOR ">>M_PRIOR>>psD)!=NULL ||
             ((si=si_m.first>>"END_MATRICES">>psD)!=NULL && init()) ||
             (si=si_m.first>>"L ">>l>>" ")!=NULL && (si=si>>si_m.second->hL.set(l)>>psD)!=NULL ||
             (si=si_m.first>>"L* ">>d>>" ">>g1>>" ">>g2>>" ")!=NULL && (si=si>>si_m.second->hhLstar.set(MapKey2D<D,G>(d,g1)).set(g2)>>psD)!=NULL ||
             (si=si_m.first>>"L0 ">>d>>" ">>g1>>" ">>g2>>" ")!=NULL && (si=si>>si_m.second->hhLzero.set(MapKey2D<D,G>(d,g1)).set(g2)>>psD)!=NULL ||
             (si=si_m.first>>si_m.second->mR>>psD)!=NULL ||
             (si=si_m.first>>si_m.second->mS>>psD)!=NULL ) ? si : NULL; }
};
X&    YModel::WORD       = QEModel::WORD;
bool& YModel::F_ROOT_OBS = FEModel::F_ROOT_OBS;

typedef Er Evec;
typedef Es Emat;

