# Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.  
# SPDX-License-Identifier: CC-BY-NC-4.0
import dataclasses
from dataclasses import dataclass
from typing import List, Any, Iterable

from dataclasses_json import DataClassJsonMixin
from tqdm import tqdm

from da4er.augmentation import BaseAugmentation


@dataclass
class Sample(DataClassJsonMixin):
    query: str
    entities: List[str]


@dataclass
class AugmentedSamples(DataClassJsonMixin):
    """
    Only used as an input/output format
    """
    original: Sample
    augmented: List[Sample]


@dataclass
class InputSample(DataClassJsonMixin):
    raw_input: Any
    work_sample: AugmentedSamples


class InputConverter:
    def process(self, raw_input: str) -> InputSample:
        augmented_sample = InputConverter._convert_augmented_sample(raw_input)
        return InputSample(augmented_sample, augmented_sample)

    @staticmethod
    def _convert_augmented_sample(raw_input: str) -> AugmentedSamples:
        try:
            return AugmentedSamples.from_json(raw_input)
        except:
            pass
        try:
            sample = Sample.from_json(raw_input)
            return AugmentedSamples(sample, [])
        except:
            pass
        raise NotImplementedError("Support for this input type is not implemented")


class OutputConverter:
    def generate_output(self, input_sample: InputSample) -> Iterable[str]:
        return [augmented_sample.to_json() for augmented_sample in input_sample.work_sample.augmented]


class InputProcessor:
    def __init__(self,
                 augmentations: List[BaseAugmentation],
                 input_converter=None, output_converter=None,
                 skip_entities=False):
        self.augmentations = augmentations
        self.input_converter = InputConverter() if input_converter is None else input_converter
        self.output_converter = OutputConverter() if output_converter is None else output_converter
        self.skip_entities = skip_entities

    def augment_sample(self, sample: Sample) -> List[Sample]:
        output = []
        for augmentation in self.augmentations:
            augmented_sample = dataclasses.replace(sample)

            augmented_sample.query = augmentation.augment(augmented_sample.query)
            if not self.skip_entities:
                augmented_sample.entities = [augmentation.augment(entity) for entity in augmented_sample.entities]

            output.append(augmented_sample)
        return output

    def augment_file(self, input, output):
        with open(input, 'r') as reader, open(output, 'w') as writer:
            for line in tqdm(reader, desc="Augmenting input file '%s'" % input):
                input_sample = self.input_converter.process(line)
                augmented = input_sample.work_sample
                augmented.augmented.extend(self.augment_sample(augmented.original))

                output = self.output_converter.generate_output(input_sample)
                for output_sample in output:
                    writer.write(output_sample)
                    writer.write('\n')
