"""Check operation utilities. May be temporary."""

from openpyxl import load_workbook
from preprocess.message_utils import _ANNO_ANSWER, FILL_PATTERN, _FORMULA

# message objects
_OP_ANSWER = 'ANSWER'
_OP_CLASS = 'CLASS'
_OP_DIMENSION = 'DIMENSION'
_OP_DATA = 'DATA'

# error types 
_OP_ERROR_NONE = 'NONE'
_OP_ERROR_MULTI = 'MULTI'
_OP_ERROR_PLACE = 'PLACE'   # for non-top/left place of unique answer
_OP_ERROR_NUM = 'NUM'  # for unmatching number of data against header
_OP_ERROR_GENERAL = 'GENERAL'   # for general unspecified error types (maybe temporarily)


class CheckOpMsg(object):
    """Error message during per-operation checking."""

    def __init__(self, msg_object, error_type, sub_block_start=None):
        self.msg_object = msg_object
        self.error_type = error_type
        self.sub_start = sub_block_start
    
    def __str__(self):
        return f'(OP){self.msg_object}-{self.error_type}'
    
    def get_row(self, offset=_ANNO_ANSWER):
        if self.sub_start is None:
            print(f'please set the starting row index of sub-block first..')
            return
        return self.sub_start + offset



def linearize_msg_dict(msg_dict):
    """Linearize {'answer', 'class', 'dimension', 'data'} into a single message"""
    reads, rows = [], []
    for msgkey, msg in msg_dict.items():
        if msg is None: continue
        reads.append( str(msg) )
        rows.append( msg.get_row() )

    rows = [r for r in rows if r is not None]
    reads = [r for r in reads if r is not None]
    if len(rows) == 0 and len(reads) == 0: return None, None
    msgstr = '\t'.join(reads)
    return rows[0], msgstr



def write_operation(
    filename, fml_msg_pairs, 
    formula_col=3, error_col=4, pattern=FILL_PATTERN[_FORMULA]
):
    """Write the error messages into the annotated file."""
    # print(f'\n[{filename}]')

    # copy and create a new annotated worksheet
    wb = load_workbook(filename)
    anno_ws = wb['labeling']

    # remove previous tempts
    old_check_names = [name for name in wb.sheetnames if name.startswith('check')]
    old_check_sheets = [wb[name] for name in old_check_names]
    for ocs in old_check_sheets:
        wb.remove(ocs)

    ws = wb.copy_worksheet(anno_ws)
    ws.title = 'check'

    for fml, msg_dict in fml_msg_pairs:
        row_index = fml.get_row()
        fml.convert_to_data(ws)    # convert non-numeric data cells to 'n' type
        ws.cell(row_index, formula_col).value = fml.excel_reads
        ws.cell(row_index, formula_col).fill = pattern

        _, msgstr = linearize_msg_dict(msg_dict)
        if (msgstr is None): continue
        ws.cell(row_index, error_col).value = msgstr

    wb.save(filename)

