import React, { Component } from 'react';
import { withTracker } from 'meteor/react-meteor-data';
import { Redirect } from "react-router-dom";

import ReactModal from 'react-modal';

import Instructions from './Instructions.jsx';
import TextDoc from './TextDoc.jsx';
import SentimentTask from './SentimentTask.jsx';
import MTurk from './MTurk.jsx';
import { Texts, Annotations, Assignments } from '../api/documents';
import { gup } from '../util/gup.js';

export class Annotate extends Component {
  constructor(props) {
    super(props);

    this.markDone = this.markDone.bind(this);
    this.handleSubmit = this.handleSubmit.bind(this);

    this.handleOpenModal = this.handleOpenModal.bind(this);
    this.handleCloseModal = this.handleCloseModal.bind(this);

    this.state = {
      message: "Task incomplete - Please check your answers",
      done: false,
      submit: false,
      showModal: false,
    };
  }

  handleOpenModal(message) {
    this.setState({ 
      message: message,
      showModal: true
    });
  }

  handleCloseModal() {
    this.setState({ showModal: false });
  }

  markDone() {
    this.setState({
      done: true,
    });
  }

  handleSubmit() {
    if (gup("workerId") == "") {
      // Ignore, this is just a preview
    } else if (this.state.done) {
      const user_id = gup("workerId");
      const assignment_id = gup("internalAssignment");

      // Compare time
      const assignment = this.props.assignments.find(assignment => {
        return assignment._id == assignment_id;
      });
      if (assignment.minTime != undefined) {
        const text = this.props.texts.find(text => {
          return text._id == assignment.text_id;
        });
        const annotation = this.props.annotations.find(ann => {
          return ann.user == user_id && ann.text_id == assignment.text_id;
        });
        const now = new Date();
        const time_spent = now - annotation.createdAt;
        const min_time = assignment.minTime * 1000 * 60;
        if (time_spent < 0) {
          Meteor.call('logError', user_id, `Time error, spent ${time_spent} based on ${now} - ${annotation.createdAt}`);
        } else if (time_spent < min_time) {
          Meteor.call('logError', user_id, `Warned: Insufficient time spent ${time_spent} < ${min_time}`);
          this.handleOpenModal("Task incomplete - Please check your answers");
          return;
        }
      }

      Meteor.call('assignments.complete', assignment_id, user_id);
      this.setState({
        submit: true,
      });
    } else {
      this.handleOpenModal(this.state.message);
    }
  }

  render() {
    ReactModal.setAppElement(document.getElementById('react-target'));

    if (this.state.submit) {
      // When done, go to feedback page
      const target = "/feedback/" + this.props.location.search;
      return (
        <Redirect to={target} />
      );
    } else if (this.props.areReady) {
      const assign_id = gup("internalAssignment");
      const user_id = gup("workerId");
      const assignment = this.props.assignments.find(assignment => {
        return assignment._id == assign_id;
      });
      const text = this.props.texts.find(text => {
        return text._id == assignment.text_id;
      });
      const annotation = this.props.annotations.find(ann => {
        return ann.user == user_id && ann.text_id == assignment.text_id;
      });
      if (text != undefined && annotation != undefined) {
        if (this.props.match.params.ui == "sentiment") {
          return (
            <div>
              <ReactModal 
                 isOpen={this.state.showModal}
                 contentLabel="Task Incomplete Warning"
              >
                <div className="center-buttons">
                  <h1>
                    {this.state.message}
                  </h1>
                  <button className="noselect button big" onClick={this.handleCloseModal}>Close Message</button>
                </div>
              </ReactModal>

              <SentimentTask
                text={text}
                ann_id={annotation._id}
                clusters={annotation.clusters}
                validate={this.markDone}
                readOnly={user_id == ""}
                showModal={this.handleOpenModal}
              />
              <div className="clear">
              </div>
              <div className="center-buttons">
                <button className="noselect button big" onClick={this.handleSubmit}>
                  Done
                </button>
              </div>
            </div>
          );
        } else {
          return (
            <div>
              <ReactModal 
                 isOpen={this.state.showModal}
                 contentLabel="Task Incomplete Warning"
              >
                <div className="center-buttons">
                  <h1>
                    {this.state.message}
                  </h1>
                  <button className="noselect button big" onClick={this.handleCloseModal}>Close Message</button>
                </div>
              </ReactModal>

              <Instructions 
                ui={this.props.match.params.ui}
                labels={annotation.labels}
              />
              <TextDoc
                text={text}
                clusters={annotation.clusters}
                checks={annotation.checks}
                key={annotation._id}
                ann_id={annotation._id}
                labelOptions={annotation.labels}
                ui={this.props.match.params.ui}
                validate={this.markDone}
                readOnly={user_id == ""}
                focusMention={assignment.focus_mention}
                showModal={this.handleOpenModal}
              />
              <div className="clear">
              </div>
              <div className="center-buttons">
                <button className="noselect button big" onClick={this.handleSubmit}>
                  Done
                </button>
              </div>
            </div>
          );
        }
      }
    }

    return (
      <p>
      Loading...
      </p>
    );
  }
}

export default AnnotationContainer = withTracker(() => {
  const text_id = gup("textId");
  const user_id = gup("workerId");
  const assign_id = gup("internalAssignment");

  const handles = [
    Meteor.subscribe('texts-single', text_id),
    Meteor.subscribe('annotations-user', user_id),
    Meteor.subscribe('assignments-single', assign_id),
  ];
 
  return {
    texts: Texts.find().fetch(),
    annotations: Annotations.find().fetch(),
    assignments: Assignments.find().fetch(),
    areReady: handles.every(handle => handle.ready()),
  };
})(Annotate);
