;; Textual/Linguistic utility functions.
;;
;; Most of this file was taken from the Lore project. The defpackage was
;; refactored and some comments have been added by Gene (2018-11-15). Some new
;; functions have been added since.
;;
;; ```Original Lore Header Comment```
;; Textual/Linguistic Utility Functions
;; Jonathan Gordon, Benjamin Van Durme, Phil Michalak, Fabrizio Morbini
;;   The functions for the plural and singular processing are derived
;;   from the Perl package Lingua-EN-Inflect-1.90 by Damian Conway.

(in-package :util)

;; Give cl-ppcre a nickname.
(add-nickname "CL-PPCRE" "RE")

;; Words with a silent initial 'h': use 'an'.
(define-constant +consonant-exceptions+
  '(honor honest heir hour honorable hourly heiress honored hour-long
    x-ray xray))

;; Words with an initial vowel that take 'a'.
(define-constant +vowel-exceptions+
  '(one once onetime un us))


(defun vowel? (char)
  "Non-nil if a given character is a vowel."
  (member char (list #\A #\O #\U #\I #\E)))


(defun consonant? (char)
  "Non-nil if a given character is a consonant."
  (not (vowel? char)))


(defun add-indefinite (common-noun)
  "Concatenation of the common-noun with the proper indefinite article."
  (declare (inline))
  (mkstr (indefinite-article common-noun) " " common-noun))


;; Originally this returned "some" for plural terms, but 'plural?' is not
;; currently reliable, and it's not clear that this is desirable behavior
;; for all the places I want to use this function; I want this as a
;; replacement to the use of Knext's 'starts-with-vowel'.
(defun indefinite-article (common-noun)
  "Returns 'a or 'an as appropriate."
  (let* ((common-noun (tree-from-string (string common-noun) :util))
         (chars (coerce (string common-noun) 'list)))
    (cond ((member common-noun +consonant-exceptions+)
           'an)
          ((member common-noun +vowel-exceptions+)
           'a)
          ;; 'u'-initial words...
          ((and (eq (first chars) #\U)
                (> (length chars) 2))
           (cond ((and (consonant? (second chars))
                       (not (eq (second chars) #\S))
                       (vowel? (third chars)))
                  ;; 'u' + consonant + vowel == 'a'
                  'a)
                 ((and (eql (second chars) #\S)
                       (or (eql (third chars) #\-)
                           (vowel? (third chars))))
                  'a)
                 (t
                  ;; 'an' otherwise
                  'an)))
          ;; 'e'-initial words...
          ((and (eql (first chars) #\E)
                (> (length chars) 1))
           (if (eql (second chars) #\U)
               ;; "eu..." == 'a'
               'a
               ;; 'an' otherwise
               'an))
          ;; otherwise default to 'an' for vowels
          ((vowel? (first chars))
           'an)
          ;; and 'a' for consonants
          (t 'a))))


(define-constant +number-regexps+
  (mapcar #'(lambda (x)
              (apply
               #'(lambda (sg-pattern pl-pattern sg-parameters pl-parameters)
                   (list (re:create-scanner sg-pattern)
                         (re:create-scanner pl-pattern)
                         sg-parameters
                         pl-parameters)) x))
          '(("^adonis$" "^adonises$" (:post "adonises") (:post "adonis"))
            ("^fly-by" "^fly-bys$" (:post "fly-bys") (:post "fly-by"))
            ("^(.* )?iris$" "^(.* )?irises$"
             (:pre 1 :post "irises") (:pre 1 :post "iris"))
            ("^(.*)penis$" "^(.*)penes$"
             (:pre 1 :post "penes") (:pre 1 :post "penis"))
            ("^day off$" "^days off$" (:post "days off") (:post "day off"))
            ("^part to whole relation$" "^part to whole relations$"
             (:post "part to whole relations")
             (:post "part to whole relation"))
            ("^(.* )?staff$" "^(.* )?staff$" (:pre 0) (:pre 0))
            ("^degree (celsisus|fahrenheit|kelvin)$"
             "^degrees (celsisus|fahrenheit|kelvin)$"
             (:pre "degrees " :post 1) (:pre "degree " :post 1))
            ("^(.*)hertz$" "^(.*)hertz$" (:pre 0) (:pre 0))
            ("^casanova$" "^casanovas$"
             (:post "casanovas") (:post "casanova"))
            ("^christmas$" "^christmases$"
             (:post "christmases") (:post "christmas"))
            ("^(.*)beau$" "^(.*)beaux$"
             (:pre 1 :post "beaux") (:pre 1 :post "beau"))
            ("^built in bed$" "^built in beds$"
             (:post "built in beds") (:post "built in bed"))
            ("^bachelor-at-arms$" "^bachelors-at-arms$"
             (:post "bachelors-at-arms") (:post "bachelor-at-arms"))
            ("^serjeant-at-arms$" "^serjeants-at-arms$"
             (:post "serjeants-at-arms") (:post "serjeant-at-arms"))
            ("^serjeant-at-law$" "^serjeants-at-law$"
             (:post "serjeants-at-law") (:post "serjeant-at-law"))
            ("^sergeant-at-arms$" "^sergeants-at-arms$"
             (:post "sergeants-at-arms") (:post "sergeant-at-arms"))
            ("^sergeant-at-law$" "^sergeants-at-law$"
             (:post "sergeants-at-law") (:post "sergeant-at-law"))
            ("(.*)man-at-arms$" "(.*)men-at-arms$"
             (:pre 1 :post "men-at-arms") (:pre 1 :post "man-at-arms"))
            ("^counselor-at-law$" "^counselors-at-law$"
             (:post "counselors-at-law") (:post "counselor-at-law"))
            ("^heir-at-law$" "^heirs-at-law$"
             (:post "heirs-at-law") (:post "heir-at-law"))
            ("^groom-to-be$" "^grooms-to-be$"
             (:post "grooms-to-be") (:post "groom-to-be"))
            ("^man-about-town$" "^men-about-town$"
             (:post "men-about-town") (:post "man-about-town"))
            ("^hanger-on$" "^hangers-on$"
             (:post "hangers-on") (:post "hanger-on"))
            ("^commander-in-chief$" "^commanders-in-chief$"
             (:post "commanders-in-chief") (:post "commander-in-chief"))
            ("^(.* )?love$" "^(.* )?love$" (:pre 0) (:pre 0))
            ("wildebeest$" "wildebeest$" (:pre 0) (:pre 0))
            ("^accused$" "^accused$" (:pre 0) (:pre 0))
            ("swine$" "swine$" (:pre 0) (:pre 0))
            ("^eland$" "^eland$" (:pre 0) (:pre 0))
            ("bison$" "bison$" (:pre 0) (:pre 0))
            ("buffalo$" "buffalo$" (:pre 0) (:pre 0))
            ("^(.* )?elk$" "^(.* )?elk$" (:pre 0) (:pre 0))
            ("^moose$" "^moose$" (:pre 0) (:pre 0))
            ("^rhinoceros$" "^rhinoceros$" (:pre 0) (:pre 0))
            ("fish$" "fish$" (:pre 0) (:pre 0))
            ("^tuna$" "^tuna$" (:pre 0) (:pre 0))
            ("^salmon$" "^salmon$" (:pre 0) (:pre 0))
            ("^mackerel$" "^mackerel$" (:pre 0) (:pre 0))
            ("^trout$" "^trout$" (:pre 0) (:pre 0))
            ("^bream$" "^bream$" (:pre 0) (:pre 0))
            ("^sea[- ]bass$" "^sea[- ]bass$" (:pre 0) (:pre 0))
            ("^carp$" "^carp$" (:pre 0) (:pre 0))
            ("^cod$" "^cod$" (:pre 0) (:pre 0))
            ("^flounder" "^flounder" (:pre 0) (:pre 0))
            ("^whiting" "^whiting" (:pre 0) (:pre 0))
            ("^(.*)deer$" "^(.*)deer$" (:pre 0) (:pre 0))
            ("^(.*)sheep$" "^(.*)sheep$" (:pre 0) (:pre 0))
            ("^portuguese$" "^portuguese$" (:pre 0) (:pre 0))
            ("^amoyese$" "^amoyese$" (:pre 0) (:pre 0))
            ("^borghese$" "^borghese$" (:pre 0) (:pre 0))
            ("^congoese$" "^congoese$" (:pre 0) (:pre 0))
            ("^faroese$" "^faroese$" (:pre 0) (:pre 0))
            ("^foochowese$" "^foochowese$" (:pre 0) (:pre 0))
            ("^genevese$" "^genevese$" (:pre 0) (:pre 0))
            ("^genoese$" "^genoese$" (:pre 0) (:pre 0))
            ("^gilbertese$" "^gilbertese$" (:pre 0) (:pre 0))
            ("^hottentotese$" "^hottentotese$" (:pre 0) (:pre 0))
            ("^kiplingese$" "^kiplingese$" (:pre 0) (:pre 0))
            ("^kongoese$" "^kongoese$" (:pre 0) (:pre 0))
            ("^lucchese$" "^lucchese$" (:pre 0) (:pre 0))
            ("^maltese$" "^maltese$" (:pre 0) (:pre 0))
            ("^nankingese$" "^nankingese$" (:pre 0) (:pre 0))
            ("^niasese$" "^niasese$" (:pre 0) (:pre 0))
            ("^pekingese$" "^pekingese$" (:pre 0) (:pre 0))
            ("^piedmontese$" "^piedmontese$" (:pre 0) (:pre 0))
            ("^pistoiese$" "^pistoiese$" (:pre 0) (:pre 0))
            ("^sarawakese$" "^sarawakese$" (:pre 0) (:pre 0))
            ("^shavese$" "^shavese$" (:pre 0) (:pre 0))
            ("^vermontese$" "^vermontese$" (:pre 0) (:pre 0))
            ("^wenchowese$" "^wenchowese$" (:pre 0) (:pre 0))
            ("^yengeese$" "^yengeese$" (:pre 0) (:pre 0))
            ("^(.*)[nrlm]ese$" "^(.*)[nrlm]ese$" (:pre 0) (:pre 0))
            ("breeches$" "breeches$" (:pre 0) (:pre 0))
            ("britches$" "britches$" (:pre 0) (:pre 0))
            ("clippers$" "clippers$" (:pre 0) (:pre 0))
            ("gallows$" "gallows$" (:pre 0) (:pre 0))
            ("^hijinks" "^hijinks" (:pre 0) (:pre 0))
            ("headquarters$" "headquarters$" (:pre 0) (:pre 0))
            ("pliers$" "pliers$" (:pre 0) (:pre 0))
            ("scissors$" "scissors$" (:pre 0) (:pre 0))
            ("^testes$" "^testes$" (:pre 0) (:pre 0))
            ("^herpes$" "^herpes$" (:pre 0) (:pre 0))
            ("^pincers$" "^pincers$" (:pre 0) (:pre 0))
            ("^shears$" "^shears$" (:pre 0) (:pre 0))
            ("proceedings$" "proceedings$" (:pre 0) (:pre 0))
            ("trousers$" "trousers$" (:pre 0) (:pre 0))
            ("cantus$" "cantus$" (:pre 0) (:pre 0))
            ("coitus$" "coitus$" (:pre 0) (:pre 0))
            ("clothing$" "clothing$" (:pre 0) (:pre 0))
            ("nexus$" "nexus$" (:pre 0) (:pre 0))
            ("^contretemps$" "^contretemps$" (:pre 0) (:pre 0))
            ("corps$" "corps$" (:pre 0) (:pre 0))
            ("^tapping$" "^tapping$" (:pre 0) (:pre 0))
            ("media$" "media$" (:pre 0) (:pre 0))
            ("^debris$" "^debris$" (:pre 0) (:pre 0))
            ("^(.*)ois$" "^(.*)ois$" (:pre 0) (:pre 0))
            ("^siemens" "^siemens" (:pre 0) (:pre 0))
            ("^(.*)measles$" "^(.*)measles$" (:pre 0) (:pre 0))
            ("^mumps" "^mumps" (:pre 0) (:pre 0))
            ("diabetes$" "diabetes$" (:pre 0) (:pre 0))
            ("^jackanapes" "^jackanapes" (:pre 0) (:pre 0))
            ("series$" "series$" (:pre 0) (:pre 0))
            ("species$" "species$" (:pre 0) (:pre 0))
            ("^rabies$" "^rabies$" (:pre 0) (:pre 0))
            ("^chassis$" "^chassis$" (:pre 0) (:pre 0))
            ("^innings$" "^innings$" (:pre 0) (:pre 0))
            ("(.*)literati$" "(.*)literati$" (:pre 0) (:pre 0))
            ("news$" "news$" (:pre 0) (:pre 0))
            ("^mews" "^mews" (:pre 0) (:pre 0))
            ("personnel$" "personnel$" (:pre 0) (:pre 0))
            ("^(.*)pox$" "^(.*)pox$" (:pre 0) (:pre 0))
            ("graffiti$" "graffiti$" (:pre 0) (:pre 0))
            ("^djinn" "^djinn" (:pre 0) (:pre 0))
            ("(.*)child$" "(.*)children$"
             (:pre 1 :post "children") (:pre 1 :post "child"))
            ("^person$" "^people$" (:post "people") (:pre "person"))
            ("(.*)brother$" "(.*)(brothers|brethren)$"
             (:pre 1 :post "brothers")
             (:pre 1 :post "brother"))
            ("(.*)loaf$" "(.*)loaves$"
             (:pre 1 :post "loaves") (:pre 1 :post "loaf"))
            ("(.*)hoof$" "(.*)(hoofs|hooves)$"
             (:pre 1 :post "hoofs") (:pre 1 :post "hoof"))
            ("^(.* )?beef$" "^(.* )?(beefs|beeves)$"
             (:pre 1 :post "beefs") (:pre 1 :post "beef"))
            ("(.*)money$" "(.*)monies$"
             (:pre 1 :post "monies") (:pre 1 :post "money"))
            ("(.*)mongoose$" "(.*)mongooses$"
             (:pre 1 :post "mongooses") (:pre 1 :post "mongoose"))
            ("^(.* )?ox$" "^(.* )?oxen$"
             (:pre 1 :post "oxen") (:pre 1 :post "ox"))
            ("^cow$" "^(cows|kine)" (:post "cows") (:post "cow"))
            ("^soliloquy" "^soliloquies" (:post "soliloquies")
             (:post "soliloquy"))
            ("^graffito$" "^graffiti$" (:post "graffiti")
             (:post "graffito"))
            ("^prima donna$" "^(prima donnas|prime donne)$"
             (:post "prime donne") (:post "prima donna"))
            ("^octopus" "^(octopuses|octopodes)" (:post "octopuses")
             (:post "octopus"))
            ("^genie" "^(genies|genii)" (:post "genies") (:post "genie"))
            ("^ganglion" "^(ganglions|ganglia)" (:post "ganglions")
             (:post "ganglion"))
            ("^trilby$" "^trilbys$" (:post "trilbys") (:post "trilby"))
            ("^turf$" "^turfs|turves$" (:post "turfs") (:post "turf"))
            ("^numen$" "^numina$" (:post "numina") (:post "numen"))
            ("^occiput$" "^(occiputs|occipita)$"
             (:post "occiputs") (:post "occiput"))
            ("^human$" "^humans$" (:post "humans") (:post "human"))
            ("^alabaman$" "^alabamans$" (:post "alabamans") (:post "alabaman"))
            ("^bahaman$" "^bahamans$" (:post "bahamans") (:post "bahaman"))
            ("^burman$" "^burmans$" (:post "burmans") (:post "burman"))
            ("^german$" "^germans$" (:post "germans") (:post "german"))
            ("^hiroshiman$" "^hiroshimans$" (:post "hiroshimans")
             (:post "hiroshiman"))
            ("^liman$" "^limans$" (:post "limans") (:post "liman"))
            ("^nakayaman$" "^nakayamans$" (:post "nakayamans")
             (:post "nakayaman"))
            ("^oklahoman$" "^oklahomans$" (:post "oklahomans")
             (:post "oklahoman"))
            ("^panaman$" "^panamans$" (:post "panamans") (:post "panaman"))
            ("^selman$" "^selmans$" (:post "selmans") (:post "selman"))
            ("^sonaman$" "^sonamans$" (:post "sonamans") (:post "sonaman"))
            ("^tacoman$" "^tacomans$" (:post "tacomans") (:post "tacoman"))
            ("^yakiman$" "^yakimans$" (:post "yakimans") (:post "yakiman"))
            ("^yokohaman$" "^yokohamans$" (:post "yokohamans")
             (:post "yokohaman"))
            ("^yuman" "^yumans" (:post "yumans") (:post "yuman"))
            ("^ottoman$" "^ottomans$" (:post "ottomans") (:post "ottoman"))
            ("^abdomen$" "^abdomens$" (:post "abdomens") (:post "abdomen"))
            ("(.*)man$" "(.*)men$" (:pre 1 :post "men") (:pre 1 :post "man"))
            ("(.*)mouse$" "(.*)mice$" (:pre 1 :post "mice")
             (:pre 1 :post "mouse"))
            ("(.*)louse$" "(.*)lice$" (:pre 1 :post "lice")
             (:pre 1 :post "louse"))
            ("(.*)goose$" "(.*)geese$" (:pre 1 :post "geese")
             (:pre 1 :post "goose"))
            ("(.*)tooth$" "(.*)teeth$" (:pre 1 :post "teeth")
             (:pre 1 :post "tooth"))
            ("(.*)foot$" "(.*)feet$" (:pre 1 :post "feet")
             (:pre 1 :post "foot"))
            ("(.*)thesis$" "(.*)theses$" (:pre 1 :post "theses")
             (:pre 1 :post "thesis"))
            ("(.*(symposi|consorti|curricul|dict|honorari|lustr|maxim|minim))um$"
             "(.*(symposi|consorti|curricul|dict|honorari|lustr|maxim|minim))a$"
             (:pre 1 :post "a")
             (:pre 1 :post "um"))
            ("(.*)ceps$" "(.*)ceps$" (:pre 0) (:pre 0))
            ("(.*)zoon$" "(.*)zoa$" (:pre 1 :post "zoa") (:pre 1 :post "zoon"))
            ("(.*)rose$" "(.*)roses$"
             (:pre 1 :post "roses") (:pre 1 :post "rose"))
            ("(.*)sis$" "(.*)ses$" (:pre 1 :post "ses")
             (:pre 1 :post "sis"))
            ("(.*[cx])is$" "(.*[cx])es$" (:pre 1 :post "es")
             (:pre 1 :post "is"))
            ("(.*)quiz$" "(.*)quizzes" (:pre 1 :post "quizzes")
             (:pre 1 :post "quiz"))
            ("(.*(cod|mur|sil))ex$" "(.*(cod|mur|sil))ices$"
             (:pre 1 :post "ices")
             (:pre 1 :post "ex"))
            ("(.*(rad|hel))ix$" "(.*(rad|hel))ices$" (:pre 1 :post "ices")
             (:pre 1 :post "ix"))
            ("(.*agenda)$" "(.*agendas)$" (:post "agendas") (:post "agenda"))
            ("(.*(bacteri|agend|desiderat|errat|strat|dat|ov|extrem|candelabr))um$"
             "(.*(bacteri|agend|desiderat|errat|strat|dat|ov|extrem|candelabr))a$"
             (:pre 1 :post "a")
             (:pre 1 :post "um"))
            ("(.*(alumn|alveol|bacill|bronch|cumul|loc|nucle|stimul|menisc|re|fung|modul))us$"
             "(.*(alumn|alveol|bacill|bronch|cumul|loc|nucle|stimul|menisc|re|fung|modul))i$"
             (:pre 1 :post "i")
             (:pre 1 :post "us"))
            ("(.*(criteri|periheli|apheli|phenomen|prolegomen|noumen|organ|asyndet|hyperbat))on$"
             "(.*(criteri|periheli|apheli|phenomen|prolegomen|noumen|organ|asyndet|hyperbat))a$"
             (:pre 1 :post "a")
             (:pre 1 :post "on"))
            ("(.*(alumna|alga|vertebra|persona))$"
             "(.*(alumna|alga|vertebra|persona))e$"
             (:pre 0 :post "e")
             (:pre 1))
            ("(.*)genus$" "(.*)genera$"
             (:pre 1 :post "genera")
             (:pre 1 :post "genus"))
            ("^(.*ss|acropolis|aegis|alias|asbestos|bathos|bias|bronchitis|bursitis|caddis|cannabis|canvas|chaos|cosmos|dais|digitalis|epidermis|ethos|eyas|gas|glottis|hubris|ibis|lens|mantis|marquis|metropolis|pathos|pelvis|polis|rhinoceros|sassafras|trellis|.*us|ephemeris|iris|clitoris|chrysalis|epididymis|.*itis)$"
             "^(.*ss|acropolis|aegis|alias|asbestos|bathos|bias|bronchitis|bursitis|caddis|cannabis|canvas|chaos|cosmos|dais|digitalis|epidermis|ethos|eyas|gas|glottis|hubris|ibis|lens|mantis|marquis|metropolis|pathos|pelvis|polis|rhinoceros|sassafras|trellis|.*us|ephemeris|iris|clitoris|chrysalis|epididymis|.*itis)es$"
             (:pre 0 :post "es")
             (:pre 1))
            ("^$" "^([a-z].*s)es$" (:pre 0 :post "es") (:pre 1))
            ("((.*)([cs]h|[zx]))$" "((.*)([cs]h|[zx]))es$" (:pre 0 :post "es")
             (:pre 1))
            (".*us$" "(.*us)es$" (:pre 0 :post "es") (:pre 1))
            ("((.*[eao]l)|(.*[^d]ea)|((.*)ar))f$" "((.*[eao]l)|(.*[^d]ea)|((.*)ar))ves$" (:pre 1 :post "ves") (:pre 1 :post "f")) ;;problem
            ("(.*[nlw]i)fe$" "(.*[nlw]i)ves$" (:pre 1 :post "ves")
             (:pre 1 :post "fe"))
            ("(.*[aeiou])y$" "$(.*[aeiou])ys$" (:pre 1 :post "ys")
             (:pre 1 :post "y"))
            ("(.+)y$" "(.+)ies$" (:pre 1 :post "ies") (:pre 1 :post "y"))
            ("^(albino|archipelago|armadillo|commando|crescendo|fiasco|ditto|dynamo|embryo|ghetto|guano|inferno|jumbo|lumbago|magneto|manifesto|medico|octavo|photo|pro|quarto|canto|lingo|generalissimo|stylo|rhino|solo|soprano|basso|alto|contralto|tempo|piano|virtuoso|accelerando|allegretto|allegro|alter[ -]ego|auto|autogiro|autogyro|bambino|beano|biro|bistro|bolero|bonito|bravo|bronco|casino|cello|coco|concerto|credo|cruzeiro|curio|demo|disco|ego|escudo|espresso|expo|gaucho|gigolo|gumbo|gyro|homo|intermezzo|kakemono|kilo|kimono|kobo|largo|lean-to|libido|libretto|lido|limbo|lino|maestro|major-domo|maraschino|memo|merino|mikado|morello|nympho|obbligato|palmetto|peso|piccolo|placebo|pseudo|pueblo|quattrocento|recto|risotto|rondo)$"
             "^(albino|archipelago|armadillo|commando|crescendo|fiasco|ditto|dynamo|embryo|ghetto|guano|inferno|jumbo|lumbago|magneto|manifesto|medico|octavo|photo|pro|quarto|canto|lingo|generalissimo|stylo|rhino|solo|soprano|basso|alto|contralto|tempo|piano|virtuoso|accelerando|allegretto|allegro|alter[ -]ego|auto|autogiro|autogyro|bambino|beano|biro|bistro|bolero|bonito|bravo|bronco|casino|cello|coco|concerto|credo|cruzeiro|curio|demo|disco|ego|escudo|espresso|expo|gaucho|gigolo|gumbo|gyro|homo|intermezzo|kakemono|kilo|kimono|kobo|largo|lean-to|libido|libretto|lido|limbo|lino|maestro|major-domo|maraschino|memo|merino|mikado|morello|nympho|obbligato|palmetto|peso|piccolo|placebo|pseudo|pueblo|quattrocento|recto|risotto|rondo)s$"
             (:pre 0 :post "s") (:pre 1))
            (".*[aeiou]o$" "(.*[aeiou]o)s$" (:pre 0 :post "s") (:pre 1))
            (".*o$" "(.*o)es$" (:pre 0 :post "es") (:pre 1))
            (".*[^s]$" "(.*)s$" (:pre 0 :post "s") (:pre 1)))))


(defun plural? (common-noun)
  "Returns multiple values:
   t iff common-noun is plural,
   the entry from +number-regexps+ that tipped us off"

  (let ((common-noun (if (stringp common-noun) common-noun
                         (string-downcase (format nil "~a" common-noun)))))
    (flet ((singular (x) (first x))
           (plural (x) (second x)))
      (loop for e in +number-regexps+
                do
                (if (re:scan (plural e) common-noun)
                    (return (values t e)))
                (if (re:scan (singular e) common-noun)
                    (return (values nil e)))))))


(defun singular? (common-noun)
  "Returns multiple values:
   t iff common-noun is singular,
   the entry from +number-regexps+ that tipped us off"
  (multiple-value-bind (is-plural e)
      (plural? common-noun)
    (values (not is-plural) e)))


(defun plural-of (common-noun)
  "Returns plural of common-noun as a string. If already plural then returns
   as-is."
  (let ((common-noun (if (stringp common-noun) common-noun
                         (string-downcase (format nil "~a" common-noun)))))

    (re:register-groups-bind (word etc)
        ("(.+) ((?:of|before|during|after|under|in|for|on|to|from) (?:(?:an?|the) )?.+)$" common-noun)
      (return-from plural-of (mkstr (plural-of word) " " etc)))

    (multiple-value-bind (is-plural e)
        (plural? common-noun)
      (if is-plural
          common-noun
        (transform-number 'plural common-noun e)))))


(defun singular-of (common-noun)
  "Returns singular form of common-noun as a string. If already singular then
   returns as-is."
  (let ((common-noun (if (stringp common-noun) common-noun
                         (string-downcase (string common-noun)))))
    (multiple-value-bind (is-singular e)
        (singular? common-noun)
      (if is-singular common-noun
          (transform-number 'singular common-noun e)))))


(defun transform-number (number common-noun number-entry)
  "Transforms the given common-noun from singular to plural or vice-versa
   depending on the value of number ('singular or 'plural).
   number-entry refers to the entry in +number-regexps+ that we will be using
   to do the transformation."
  (let (scanner parameters)
    (case number
      ;; if common-noun is becoming plural, then it currently matches singular,
      ;; and vice versa
      (plural (setf scanner (first number-entry))
               (setf parameters (third number-entry)))
      (singular (setf scanner (second number-entry))
                 (setf parameters (fourth number-entry)))
      (otherwise
       (error "transform-number requires 'number' to be either 'singular or 'plural")))

    (multiple-value-bind (whole-string substring-match-array)
        (re:scan-to-strings scanner common-noun)

      (let ((substring (if (> (length substring-match-array) 0)
                           (aref substring-match-array 0)
                         nil))
            (pre (second (member ':pre parameters)))
            (post (second (member ':post parameters))))

        (concatenate 'string
                     ;; Get the prefix (potentially the whole word, or maybe
                     ;; nothing at all)
                     (cond
                      ((numberp pre)
                       (if (= pre 0) whole-string substring))
                      ((stringp pre) pre)
                      (t ""))

                     ;; Get the postfix (potentially the whole word, or maybe
                     ;; nothing at all)
                     (cond
                      ((numberp post)
                       (if (= post 0) whole-string substring))
                      ((stringp post) post)
                      (t "")))))))


;; This is the function that was used in the Epik utils package, but it fails
;; to distinguish between simple past tense and past participles.
(defun apply-ed (verb-string)
  "Takes a verb, as a string, applies -ED morphology, returns string."
  (let ((vs (string-upcase (format nil "~a" verb-string))))
    (let* ((tokens (re:split " " vs))
           (last-word-tokens (re:split "_" (first (last tokens))))
           (head (first (last tokens))))
      (setf tokens (reverse (cdr (reverse tokens))))
      (if (> (length last-word-tokens) 1)
          (prog ()
             (setf tokens (append tokens
                                  (reverse (cdr (reverse last-word-tokens)))))
             (setf head (first (last last-word-tokens)))))
      (let ((morphed
             (case (aref head (- (length head) 1))
               (#\Y
                (if (and (> (length head) 1)
                         (member (aref head (- (length head) 2))
                                 '(#\A #\E #\I #\O #\U)))
                    (format nil "~aED" head)
                    (format nil "~aIED" (subseq head 0 (- (length head) 1)))))
               ((#\W #\X #\Z)
                (format nil "~aED" head))
               (#\E
                (if (and (> (length head) 1)
                         (member (aref head (- (length head) 2)) '(#\T))
                         (string-equal "NA" (subseq head
                                                    (- (length head) 4)
                                                    (- (length head) 2))))
                    (format nil "~aNATED" (subseq head 0
                                                  (- (length head) 4)))
                    (format nil "~aD" head)))
               (t
                (format nil "~aED" head)))))
        (if (> (length last-word-tokens) 1)
            (format nil "~{~A ~}~A-~A"
                    (reverse (cdr (reverse tokens)))
                    (first (last tokens))
                    morphed)
            (format nil "~{~A ~}~A" tokens morphed))))))


(defparameter *simple-pasts* (make-hash-table :test #'equal))
(mapc #'(lambda (x)
          (setf (gethash (first x) *simple-pasts*)
                (second x)))
      '(("abacinate" "abacinised")
        ("abet" "abetted")
        ("abhor" "abhorred")
        ("abide by" "abided by")
        ("abound in" "abounded in")
        ("about-face" "about-faced")
        ("abut" "abutted")
        ("aby" "abought")
        ("abye" "abought")
        ("accompany" "accompanied")
        ("account for" "accounted for")
        ("acetify" "acetified")
        ("acidify" "acidified")
        ("acquit" "acquitted")
        ("act as" "acted as")
        ("act involuntarily" "acted involuntarily")
        ("act on" "acted on")
        ("act out" "acted out")
        ("act reflexively" "acted reflexively")
        ("act superior" "acted superior")
        ("act up" "acted up")
        ("act upon" "acted upon")
        ("ad-lib" "adlibbed")
        ("add on" "added on")
        ("add to" "added to")
        ("add together" "added together")
        ("add up" "added up")
        ("admit" "admitted")
        ("advertise" "advertized")
        ("aerify" "aerified")
        ("air out" "aired out")
        ("air-condition" "airconditioned")
        ("air-cool" "aircooled")
        ("align" "alined")
        ("alkalify" "alkalified")
        ("allot" "allotted")
        ("allow for" "allowed for")
        ("allow in" "allowed in")
        ("ally" "allied")
        ("ally with" "allied with")
        ("ammonify" "ammonified")
        ("amnesty" "amnestied")
        ("amplify" "amplified")
        ("anaesthetize" "anesthetized")
        ("analyse" "analyzed")
        ("annul" "annulled")
        ("answer for" "answered for")
        ("ante up" "anteed up")
        ("antic" "anticked")
        ("apply" "applied")
        ("arc" "arcked")
        ("arch over" "arched over")
        ("argufy" "argufied")
        ("arise" "arose")
        ("arrive at" "arrived at")
        ("arse about" "arsed about")
        ("arse around" "arsed around")
        ("ask for" "asked for")
        ("ask for it" "asked for it")
        ("ask for trouble" "asked for trouble")
        ("ask in" "asked in")
        ("ask out" "asked out")
        ("ask over" "asked over")
        ("ask round" "asked round")
        ("assert oneself" "asserted oneself")
        ("assonate" "assonised")
        ("atrophy" "atrophied")
        ("attach to" "attached to")
        ("attend to" "attended to")
        ("auction off" "auctioned off")
        ("aurify" "aurified")
        ("autopsy" "autopsied")
        ("aver" "averred")
        ("average out" "averaged out")
        ("awake" "awoke")
        ("babble out" "babbled out")
        ("baby" "babied")
        ("baby-sit" "babysat")
        ("back away" "backed away")
        ("back down" "backed down")
        ("back off" "backed off")
        ("back out" "backed out")
        ("back up" "backed up")
        ("backbite" "backbit")
        ("backslide" "backslid")
        ("bag" "bagged")
        ("bail out" "bailed out")
        ("bale out" "baled out")
        ("ball over" "balled over")
        ("ball up" "balled up")
        ("ballyrag" "ballyragged")
        ("ban" "banned")
        ("band oneself" "banded oneself")
        ("band together" "banded together")
        ("bandy" "bandied")
        ("bandy about" "bandied about")
        ("bang out" "banged out")
        ("bang up" "banged up")
        ("bar" "barred")
        ("bar hop" "barred hop")
        ("bar mitzvah" "barred mitzvah")
        ("bargain down" "bargained down")
        ("barge in" "barged in")
        ("barrage jam" "barraged jam")
        ("barter away" "bartered away")
        ("basify" "basified")
        ("bat" "batted")
        ("bat mitzvah" "batted mitzvah")
        ("batten down" "battened down")
        ("bawl out" "bawled out")
        ("be" "was")
        ("be active" "was active")
        ("be adrift" "was adrift")
        ("be after" "was after")
        ("be amiss" "was amiss")
        ("be at pains" "was at pains")
        ("be born" "was born")
        ("be due" "was due")
        ("be full" "was full")
        ("be given" "was given")
        ("be intimate" "was intimate")
        ("be on" "was on")
        ("be on cloud nine" "was on cloud nine")
        ("be on the ball" "was on the ball")
        ("be quiet" "was quiet")
        ("be sick" "was sick")
        ("be well" "was well")
        ("be with it" "was with it")
        ("bear" "bore")
        ("bear away" "bore away")
        ("bear down" "bore down")
        ("bear down on" "bore down on")
        ("bear down upon" "bore down upon")
        ("bear in mind" "bore in mind")
        ("bear off" "bore off")
        ("bear on" "bore on")
        ("bear out" "bore out")
        ("bear up" "bore up")
        ("bear upon" "bore upon")
        ("bear witness" "bore witness")
        ("beat" "beat")
        ("beat a retreat" "beat a retreat")
        ("beat about" "beat about")
        ("beat around the bush" "beat around the bush")
        ("beat back" "beat back")
        ("beat down" "beat down")
        ("beat in" "beat in")
        ("beat out" "beat out")
        ("beat up" "beat up")
        ("beatify" "beatified")
        ("beautify" "beautified")
        ("beaver away" "beavered away")
        ("become" "became")
        ("become flat" "became flat")
        ("bed" "bedded")
        ("bed down" "bedded down")
        ("bedevil" "bedevilled")
        ("bedight" "bedight")
        ("bedim" "bedimmed")
        ("beef up" "beefed up")
        ("beetle off" "beetled off")
        ("befall" "befell")
        ("befit" "befitted")
        ("befog" "befogged")
        ("beg" "begged")
        ("beg off" "begged off")
        ("beget" "begot")
        ("begin" "began")
        ("behold" "beheld")
        ("believe in" "believed in")
        ("bell the cat" "belled the cat")
        ("belly" "bellied")
        ("belly dance" "bellied dance")
        ("belly out" "bellied out")
        ("belly-land" "bellylanded")
        ("belong to" "belonged to")
        ("belt along" "belted along")
        ("belt down" "belted down")
        ("belt out" "belted out")
        ("belt up" "belted up")
        ("bend" "bent")
        ("bend over backwards" "bent over backwards")
        ("berry" "berried")
        ("beseech" "besought")
        ("beset" "beset")
        ("besot" "besotted")
        ("bespeak" "bespoke")
        ("bestead" "besteaded")
        ("bestir" "bestirred")
        ("bestir oneself" "bestirred oneself")
        ("bestride" "bestrode")
        ("bet" "bet")
        ("bet on" "bet on")
        ("betake oneself" "betook oneself")
        ("bethink" "bethought")
        ("bid" "bid")
        ("bin" "binned")
        ("bind" "bound")
        ("bind off" "bound off")
        ("bind over" "bound over")
        ("bite" "bit")
        ("bite off" "bit off")
        ("bite out" "bit out")
        ("bivouac" "bivouacked")
        ("blab" "blabbed")
        ("blab out" "blabbed out")
        ("black market" "blacked market")
        ("black marketeer" "blacked marketeer")
        ("black out" "blacked out")
        ("blackberry" "blackberried")
        ("blacken out" "blackened out")
        ("blackleg" "blacklegged")
        ("blank out" "blanked out")
        ("blanket jam" "blanketed jam")
        ("blare out" "blared out")
        ("blast off" "blasted off")
        ("blat" "blatted")
        ("blat out" "blatted out")
        ("blaze away" "blazed away")
        ("blaze out" "blazed out")
        ("blaze up" "blazed up")
        ("blazon out" "blazoned out")
        ("bleach out" "bleached out")
        ("bleed" "bled")
        ("blend in" "blended in")
        ("blink away" "blinked away")
        ("blob" "blobbed")
        ("block off" "blocked off")
        ("block out" "blocked out")
        ("block up" "blocked up")
        ("bloody" "bloodied")
        ("blossom forth" "blossomed forth")
        ("blossom out" "blossomed out")
        ("blot" "blotted")
        ("blot out" "blotted out")
        ("blow" "blew")
        ("blow a fuse" "blew a fuse")
        ("blow off" "blew off")
        ("blow one's stack" "blew one's stack")
        ("blow out" "blew out")
        ("blow out of the water" "blew out of the water")
        ("blow over" "blew over")
        ("blow up" "blew up")
        ("blow-dry" "blow-dried")
        ("blub" "blubbed")
        ("blubber out" "blubbered out")
        ("blue-pencil" "bluepenciled")
        ("bluff out" "bluffed out")
        ("blunder out" "blundered out")
        ("blur" "blurred")
        ("blurt out" "blurted out")
        ("board up" "boarded up")
        ("bob" "bobbed")
        ("bob about" "bobbed about")
        ("bob around" "bobbed around")
        ("bob under" "bobbed under")
        ("bob up" "bobbed up")
        ("body" "bodied")
        ("body forth" "bodied forth")
        ("body guard" "bodied guard")
        ("bog" "bogged")
        ("bog down" "bogged down")
        ("bogey" "bogied")
        ("boil down" "boiled down")
        ("boil over" "boiled over")
        ("bollix up" "bollixed up")
        ("bollocks up" "bollocksed up")
        ("bolster up" "bolstered up")
        ("bolt down" "bolted down")
        ("bolt out" "bolted out")
        ("bomb out" "bombed out")
        ("bomb up" "bombed up")
        ("bone up" "boned up")
        ("book up" "booked up")
        ("boom out" "boomed out")
        ("boost up" "boosted up")
        ("boot out" "booted out")
        ("bootleg" "bootlegged")
        ("bop" "bopped")
        ("border on" "bordered on")
        ("boss around" "bossed around")
        ("botch up" "botched up")
        ("bottle up" "bottled up")
        ("bottle-feed" "bottle-fed")
        ("bottlefeed" "bottlefed")
        ("bottom out" "bottomed out")
        ("bounce back" "bounced back")
        ("bounce out" "bounced out")
        ("bound off" "bounded off")
        ("bouse" "bowsed")
        ("bow down" "bowed down")
        ("bow out" "bowed out")
        ("bowl over" "bowled over")
        ("box in" "boxed in")
        ("box up" "boxed up")
        ("brace oneself for" "braced oneself for")
        ("brace up" "braced up")
        ("bracket out" "bracketed out")
        ("brag" "bragged")
        ("branch out" "branched out")
        ("brave out" "braved out")
        ("break" "broke")
        ("break apart" "broke apart")
        ("break away" "broke away")
        ("break bread" "broke bread")
        ("break camp" "broke camp")
        ("break dance" "broke dance")
        ("break down" "broke down")
        ("break even" "broke even")
        ("break in" "broke in")
        ("break into" "broke into")
        ("break loose" "broke loose")
        ("break off" "broke off")
        ("break one's back" "broke one's back")
        ("break open" "broke open")
        ("break out" "broke out")
        ("break short" "broke short")
        ("break someone's heart" "broke someone's heart")
        ("break through" "broke through")
        ("break up" "broke up")
        ("break water" "broke water")
        ("break wind" "broke wind")
        ("break with" "broke with")
        ("breastfeed" "breastfed")
        ("breathalyse" "breathalyzed")
        ("breathe in" "breathed in")
        ("breathe out" "breathed out")
        ("breed" "bred")
        ("breeze through" "breezed through")
        ("brevet" "brevetted")
        ("brick in" "bricked in")
        ("brick over" "bricked over")
        ("brick up" "bricked up")
        ("bridge over" "bridged over")
        ("bridle at" "bridled at")
        ("bridle up" "bridled up")
        ("brim" "brimmed")
        ("brim over" "brimmed over")
        ("bring" "brought")
        ("bring about" "brought about")
        ("bring around" "brought around")
        ("bring back" "brought back")
        ("bring down" "brought down")
        ("bring forth" "brought forth")
        ("bring forward" "brought forward")
        ("bring home" "brought home")
        ("bring home the bacon" "brought home the bacon")
        ("bring in" "brought in")
        ("bring off" "brought off")
        ("bring on" "brought on")
        ("bring oneself" "brought oneself")
        ("bring out" "brought out")
        ("bring outside" "brought outside")
        ("bring round" "brought round")
        ("bring through" "brought through")
        ("bring to" "brought to")
        ("bring to bear" "brought to bear")
        ("bring together" "brought together")
        ("bring up" "brought up")
        ("brisk up" "brisked up")
        ("bristle at" "bristled at")
        ("bristle up" "bristled up")
        ("browbeat" "browbeat")
        ("brush aside" "brushed aside")
        ("brush down" "brushed down")
        ("brush off" "brushed off")
        ("brush on" "brushed on")
        ("brush up" "brushed up")
        ("bubble over" "bubbled over")
        ("bubble up" "bubbled up")
        ("buck up" "bucked up")
        ("bucket along" "bucketed along")
        ("buckle down" "buckled down")
        ("buckle under" "buckled under")
        ("bud" "budded")
        ("budget for" "budgeted for")
        ("bug" "bugged")
        ("bug out" "bugged out")
        ("bugger off" "buggered off")
        ("build" "built")
        ("build in" "built in")
        ("build on" "built on")
        ("build up" "built up")
        ("build upon" "built upon")
        ("bulge out" "bulged out")
        ("bulk large" "bulked large")
        ("bull through" "bulled through")
        ("bullet vote" "bulleted vote")
        ("bullshit" "bullshitted")
        ("bully" "bullied")
        ("bully off" "bullied off")
        ("bullyrag" "bullyragged")
        ("bum" "bummed")
        ("bum about" "bummed about")
        ("bum around" "bummed around")
        ("bump around" "bumped around")
        ("bump into" "bumped into")
        ("bump off" "bumped off")
        ("bump up" "bumped up")
        ("bunch together" "bunched together")
        ("bunch up" "bunched up")
        ("bunco" "bunkoed")
        ("bundle off" "bundled off")
        ("bundle up" "bundled up")
        ("bunk down" "bunked down")
        ("bunk off" "bunked off")
        ("buoy up" "buoyed up")
        ("bur" "burred")
        ("burgeon forth" "burgeoned forth")
        ("burn" "burnt")
        ("burn down" "burnt down")
        ("burn off" "burnt off")
        ("burn out" "burnt out")
        ("burn up" "burnt up")
        ("burst" "burst")
        ("burst forth" "burst forth")
        ("burst in on" "burst in on")
        ("burst out" "burst out")
        ("burst upon" "burst upon")
        ("bury" "buried")
        ("bus" "bussed")
        ("bush out" "bushed out")
        ("bust up" "busted up")
        ("bustle about" "bustled about")
        ("busy" "busied")
        ("butt against" "butted against")
        ("butt in" "butted in")
        ("butt on" "butted on")
        ("butter up" "buttered up")
        ("butterfly" "butterflied")
        ("button up" "buttoned up")
        ("buy" "bought")
        ("buy at" "bought at")
        ("buy back" "bought back")
        ("buy food" "bought food")
        ("buy in" "bought in")
        ("buy into" "bought into")
        ("buy it" "bought it")
        ("buy off" "bought off")
        ("buy out" "bought out")
        ("buy the farm" "bought the farm")
        ("buy time" "bought time")
        ("buy up" "bought up")
        ("buzz off" "buzzed off")
        ("cabal" "caballed")
        ("caddy" "caddied")
        ("cage in" "caged in")
        ("calcify" "calcified")
        ("call at" "called at")
        ("call attention" "called attention")
        ("call back" "called back")
        ("call down" "called down")
        ("call for" "called for")
        ("call forth" "called forth")
        ("call in" "called in")
        ("call into question" "called into question")
        ("call it a day" "called it a day")
        ("call it quits" "called it quits")
        ("call off" "called off")
        ("call on" "called on")
        ("call on the carpet" "called on the carpet")
        ("call one's bluff" "called one's bluff")
        ("call out" "called out")
        ("call the shots" "called the shots")
        ("call the tune" "called the tune")
        ("call to order" "called to order")
        ("call up" "called up")
        ("calm down" "calmed down")
        ("camp down" "camped down")
        ("camp out" "camped out")
        ("can" "could")
        ("canal" "canalled")
        ("cancel" "cancelled")
        ("cancel out" "cancelled out")
        ("candy" "candied")
        ("cannonball along" "cannonballed along")
        ("cannulate" "canulated")
        ("canopy" "canopied")
        ("cant over" "canted over")
        ("cap" "capped")
        ("cap off" "capped off")
        ("caravan" "caravanned")
        ("care a hang" "cared a hang")
        ("care for" "cared for")
        ("carnify" "carnified")
        ("carpet bomb" "carpeted bomb")
        ("carry" "carried")
        ("carry away" "carried away")
        ("carry back" "carried back")
        ("carry forward" "carried forward")
        ("carry off" "carried off")
        ("carry on" "carried on")
        ("carry out" "carried out")
        ("carry over" "carried over")
        ("carry through" "carried through")
        ("carry to term" "carried to term")
        ("carry weight" "carried weight")
        ("carry-the can" "carry-thed can")
        ("cart away" "carted away")
        ("cart off" "carted off")
        ("carve out" "carved out")
        ("carve up" "carved up")
        ("cascade down" "cascaded down")
        ("cash in" "cashed in")
        ("cash in on" "cashed in on")
        ("cash in one's chips" "cashed in one's chips")
        ("cash out" "cashed out")
        ("cast" "cast")
        ("cast about" "cast about")
        ("cast anchor" "cast anchor")
        ("cast around" "cast around")
        ("cast aside" "cast aside")
        ("cast away" "cast away")
        ("cast down" "cast down")
        ("cast off" "cast off")
        ("cast on" "cast on")
        ("cast out" "cast out")
        ("cat" "catted")
        ("catalyse" "catalyzed")
        ("catch" "caught")
        ("catch a glimpse" "caught a glimpse")
        ("catch a wink" "caught a wink")
        ("catch cold" "caught cold")
        ("catch fire" "caught fire")
        ("catch it" "caught it")
        ("catch on" "caught on")
        ("catch one's breath" "caught one's breath")
        ("catch out" "caught out")
        ("catch sight" "caught sight")
        ("catch some z's" "caught some z's")
        ("catch up" "caught up")
        ("catch up with" "caught up with")
        ("catnap" "catnapped")
        ("cause to be perceived" "caused to be perceived")
        ("cause to sleep" "caused to sleep")
        ("cave in" "caved in")
        ("cavern out" "caverned out")
        ("cell phone" "celled phone")
        ("center" "centred")
        ("center on" "centred on")
        ("center punch" "centred punch")
        ("certify" "certified")
        ("chain up" "chained up")
        ("chain-smoke" "chainsmoked")
        ("chalk out" "chalked out")
        ("chalk up" "chalked up")
        ("chance on" "chanced on")
        ("chance upon" "chanced upon")
        ("change by reversal" "changed by reversal")
        ("change course" "changed course")
        ("change form" "changed form")
        ("change hands" "changed hands")
        ("change integrity" "changed integrity")
        ("change intensity" "changed intensity")
        ("change magnitude" "changed magnitude")
        ("change over" "changed over")
        ("change owners" "changed owners")
        ("change posture" "changed posture")
        ("change shape" "changed shape")
        ("change state" "changed state")
        ("change surface" "changed surface")
        ("change taste" "changed taste")
        ("channel" "channelled")
        ("chap" "chapped")
        ("char" "charred")
        ("charge up" "charged up")
        ("chase after" "chased after")
        ("chase away" "chased away")
        ("chat" "chatted")
        ("chat up" "chatted up")
        ("cheat on" "cheated on")
        ("check in" "checked in")
        ("check into" "checked into")
        ("check off" "checked off")
        ("check out" "checked out")
        ("check over" "checked over")
        ("check up on" "checked up on")
        ("cheer up" "cheered up")
        ("chemisorb" "chemosorbed")
        ("chevvy" "chevvied")
        ("chevy" "chevied")
        ("chew out" "chewed out")
        ("chew over" "chewed over")
        ("chew the fat" "chewed the fat")
        ("chew up" "chewed up")
        ("chicken out" "chickened out")
        ("chide" "chid")
        ("chill out" "chilled out")
        ("chime in" "chimed in")
        ("chin" "chinned")
        ("chin up" "chinned up")
        ("chip" "chipped")
        ("chip at" "chipped at")
        ("chip away" "chipped away")
        ("chip away at" "chipped away at")
        ("chip in" "chipped in")
        ("chip off" "chipped off")
        ("chirk up" "chirked up")
        ("chisel in" "chiseled in")
        ("chit-chat" "chit-chatted")
        ("chitchat" "chitchatted")
        ("chivvy" "chivvied")
        ("chivy" "chivied")
        ("chock up" "chocked up")
        ("choke back" "choked back")
        ("choke down" "choked down")
        ("choke off" "choked off")
        ("choke up" "choked up")
        ("chondrify" "chondrified")
        ("choose" "chose")
        ("choose up" "chose up")
        ("chop" "chopped")
        ("chop down" "chopped down")
        ("chop off" "chopped off")
        ("chop up" "chopped up")
        ("chuck out" "chucked out")
        ("chuck up the sponge" "chucked up the sponge")
        ("chug" "chugged")
        ("chum" "chummed")
        ("chum up" "chummed up")
        ("churn out" "churned out")
        ("churn up" "churned up")
        ("circle around" "circled around")
        ("circle round" "circled round")
        ("citify" "cityfied")
        ("clam" "clammed")
        ("clam up" "clammed up")
        ("clamp down" "clamped down")
        ("clangor" "clangoured")
        ("clap" "clapped")
        ("clap on" "clapped on")
        ("clap together" "clapped together")
        ("clap up" "clapped up")
        ("clarify" "clarified")
        ("classify" "classified")
        ("clean house" "cleaned house")
        ("clean out" "cleaned out")
        ("clean up" "cleaned up")
        ("clear away" "cleared away")
        ("clear off" "cleared off")
        ("clear out" "cleared out")
        ("clear the air" "cleared the air")
        ("clear the throat" "cleared the throat")
        ("clear up" "cleared up")
        ("cleave" "cleft")
        ("click off" "clicked off")
        ("click open" "clicked open")
        ("climb down" "climbed down")
        ("climb on" "climbed on")
        ("climb up" "climbed up")
        ("cling" "clung")
        ("cling to" "clung to")
        ("clip" "clipped")
        ("clock in" "clocked in")
        ("clock off" "clocked off")
        ("clock on" "clocked on")
        ("clock out" "clocked out")
        ("clock up" "clocked up")
        ("clog" "clogged")
        ("clog up" "clogged up")
        ("clop" "clopped")
        ("close down" "closed down")
        ("close in" "closed in")
        ("close off" "closed off")
        ("close out" "closed out")
        ("close up" "closed up")
        ("clot" "clotted")
        ("cloud over" "clouded over")
        ("cloud up" "clouded up")
        ("clown around" "clowned around")
        ("club" "clubbed")
        ("clue in" "clued in")
        ("clutter up" "cluttered up")
        ("co-occur with" "co-occured with")
        ("co-star" "costarred")
        ("co-vary" "co-varied")
        ("cobble together" "cobbled together")
        ("cobble up" "cobbled up")
        ("cock up" "cocked up")
        ("cod" "codded")
        ("codify" "codified")
        ("cog" "cogged")
        ("coif" "coiffed")
        ("cold work" "colded work")
        ("cold-shoulder" "coldshouldered")
        ("collide with" "collided with")
        ("collocate with" "collocated with")
        ("colly" "collied")
        ("color in" "colored in")
        ("colour in" "coloured in")
        ("comb out" "combed out")
        ("come" "came")
        ("come about" "came about")
        ("come across" "came across")
        ("come after" "came after")
        ("come alive" "came alive")
        ("come along" "came along")
        ("come apart" "came apart")
        ("come around" "came around")
        ("come away" "came away")
        ("come back" "came back")
        ("come before" "came before")
        ("come by" "came by")
        ("come close" "came close")
        ("come down" "came down")
        ("come forth" "came forth")
        ("come forward" "came forward")
        ("come home" "came home")
        ("come in" "came in")
        ("come in for" "came in for")
        ("come in handy" "came in handy")
        ("come into" "came into")
        ("come into being" "came into being")
        ("come near" "came near")
        ("come of age" "came of age")
        ("come off" "came off")
        ("come on" "came on")
        ("come out" "came out")
        ("come out of the closet" "came out of the closet")
        ("come over" "came over")
        ("come round" "came round")
        ("come short" "came short")
        ("come through" "came through")
        ("come to" "came to")
        ("come to grips" "came to grips")
        ("come to hand" "came to hand")
        ("come to life" "came to life")
        ("come to light" "came to light")
        ("come to mind" "came to mind")
        ("come to the fore" "came to the fore")
        ("come together" "came together")
        ("come up" "came up")
        ("come up to" "came up to")
        ("come upon" "came upon")
        ("come with" "came with")
        ("comminate" "comminised")
        ("commit" "committed")
        ("commit suicide" "committed suicide")
        ("company" "companied")
        ("compel" "compelled")
        ("complexify" "complexified")
        ("complot" "complotted")
        ("comply" "complied")
        ("con" "conned")
        ("conceive of" "conceived of")
        ("concentrate on" "concentrated on")
        ("concur" "concurred")
        ("condole with" "condoled with")
        ("confab" "confabbed")
        ("confer" "conferred")
        ("confer with" "conferred with")
        ("conform to" "conformed to")
        ("conjure up" "conjured up")
        ("conk out" "conked out")
        ("connive at" "connived at")
        ("consonate" "consonised")
        ("construe with" "construed with")
        ("contra danse" "contraed danse")
        ("contract in" "contracted in")
        ("contract out" "contracted out")
        ("control" "controlled")
        ("cooccur with" "cooccured with")
        ("cook out" "cooked out")
        ("cook up" "cooked up")
        ("cool down" "cooled down")
        ("cool it" "cooled it")
        ("cool off" "cooled off")
        ("cool one's heels" "cooled one's heels")
        ("coop in" "cooped in")
        ("coop up" "cooped up")
        ("cop" "copped")
        ("cop out" "copped out")
        ("cope with" "coped with")
        ("copy" "copied")
        ("copy out" "copied out")
        ("copyread" "copyread")
        ("coquet" "coquetted")
        ("corbel" "corbelled")
        ("cordon off" "cordoned off")
        ("core out" "cored out")
        ("cork up" "corked up")
        ("coronate" "coronised")
        ("cost" "cost")
        ("cotton on" "cottoned on")
        ("cotton up" "cottoned up")
        ("cough out" "coughed out")
        ("cough up" "coughed up")
        ("count down" "counted down")
        ("count off" "counted off")
        ("count on" "counted on")
        ("count out" "counted out")
        ("counterplot" "counterplotted")
        ("countersink" "countersank")
        ("couple on" "coupled on")
        ("couple up" "coupled up")
        ("court favor" "courted favor")
        ("court favour" "courted favour")
        ("court-martial" "courtmartialed")
        ("cover for" "covered for")
        ("cover up" "covered up")
        ("cover-up" "covered-up")
        ("cozy up" "cozied up")
        ("crab" "crabbed")
        ("crack down" "cracked down")
        ("crack up" "cracked up")
        ("cram" "crammed")
        ("crank out" "cranked out")
        ("crank up" "cranked up")
        ("crap" "crapped")
        ("crap up" "crapped up")
        ("crash land" "crashed land")
        ("crawfish out" "crawfished out")
        ("crawl in" "crawled in")
        ("cream off" "creamed off")
        ("create by mental act" "created by mental act")
        ("create from raw material" "created from raw material")
        ("create from raw stuff" "created from raw stuff")
        ("create mentally" "created mentally")
        ("create verbally" "created verbally")
        ("creep" "crept")
        ("creep in" "crept in")
        ("creep up" "crept up")
        ("crib" "cribbed")
        ("crock up" "crocked up")
        ("crop" "cropped")
        ("crop out" "cropped out")
        ("crop up" "cropped up")
        ("cross examine" "crossed examine")
        ("cross off" "crossed off")
        ("cross oneself" "crossed oneself")
        ("cross out" "crossed out")
        ("cross question" "crossed question")
        ("cross-check" "crosschecked")
        ("cross-index" "crossindexed")
        ("cross-pollinate" "crosspollinated")
        ("cross-refer" "crossrefered")
        ("crosscut" "crosscut")
        ("crowd out" "crowded out")
        ("crowd together" "crowded together")
        ("crucify" "crucified")
        ("crush out" "crushed out")
        ("cry" "cried")
        ("cry for" "cried for")
        ("cry out" "cried out")
        ("cry out for" "cried out for")
        ("cub" "cubbed")
        ("cull out" "culled out")
        ("cup" "cupped")
        ("curl up" "curled up")
        ("curry" "curried")
        ("curry favor" "curried favor")
        ("curry favour" "curried favour")
        ("curtain off" "curtained off")
        ("curtsy" "curtsied")
        ("cut" "cut")
        ("cut across" "cut across")
        ("cut away" "cut away")
        ("cut back" "cut back")
        ("cut corners" "cut corners")
        ("cut down" "cut down")
        ("cut in" "cut in")
        ("cut into" "cut into")
        ("cut off" "cut off")
        ("cut out" "cut out")
        ("cut short" "cut short")
        ("cut through" "cut through")
        ("cut to" "cut to")
        ("cut to ribbons" "cut to ribbons")
        ("cut up" "cut up")
        ("cycle on" "cycled on")
        ("dab" "dabbed")
        ("dally" "dallied")
        ("dam" "dammed")
        ("dam up" "dammed up")
        ("dandify" "dandified")
        ("dash down" "dashed down")
        ("dash off" "dashed off")
        ("date back" "dated back")
        ("date from" "dated from")
        ("date stamp" "dated stamp")
        ("de-emphasize" "deemphasized")
        ("de-escalate" "deescalated")
        ("de-ice" "deiced")
        ("de-iodinate" "de-iodinised")
        ("de-ionate" "de-ionised")
        ("deal" "dealt")
        ("deal out" "dealt out")
        ("deaminate" "deaminized")
        ("debar" "debarred")
        ("debug" "debugged")
        ("decalcify" "decalcified")
        ("decertify" "decertified")
        ("dechlorinate" "dechlorinised")
        ("deck out" "decked out")
        ("deck up" "decked up")
        ("declare oneself" "declared oneself")
        ("declassify" "declassified")
        ("decontrol" "decontrolled")
        ("decry" "decried")
        ("deed over" "deeded over")
        ("deep freeze" "deeped freeze")
        ("deep-fat-fry" "deep fat-fried")
        ("deep-fry" "deepfried")
        ("deep-six" "deepsixed")
        ("default on" "defaulted on")
        ("defer" "deferred")
        ("defibrinate" "defibrinised")
        ("defuse" "defuzed")
        ("defy" "defied")
        ("degas" "degassed")
        ("degust" "degustated")
        ("dehumidify" "dehumidified")
        ("deify" "deified")
        ("deliver the goods" "delivered the goods")
        ("demob" "demobbed")
        ("demulsify" "demulsified")
        ("demur" "demurred")
        ("demyelinate" "demyelinised")
        ("demystify" "demystified")
        ("denature" "denaturised")
        ("denazify" "denazified")
        ("denitrify" "denitrified")
        ("deny" "denied")
        ("deoxygenate" "deoxygenized")
        ("depend on" "depended on")
        ("depend upon" "depended upon")
        ("descant on" "descanted on")
        ("descry" "descried")
        ("deter" "deterred")
        ("detoxify" "detoxified")
        ("devitrify" "devitrified")
        ("devolve on" "devolved on")
        ("diagram" "diagrammed")
        ("dialyse" "dialyzed")
        ("die away" "died away")
        ("die back" "died back")
        ("die down" "died down")
        ("die hard" "died hard")
        ("die off" "died off")
        ("die out" "died out")
        ("dig" "dug")
        ("dig in" "dug in")
        ("dig into" "dug into")
        ("dig out" "dug out")
        ("dig up" "dug up")
        ("dignify" "dignified")
        ("dilly-dally" "dillydallied")
        ("dillydally" "dillydallied")
        ("dim" "dimmed")
        ("din" "dinned")
        ("dine in" "dined in")
        ("dine out" "dined out")
        ("dip" "dipped")
        ("dip into" "dipped into")
        ("dip solder" "dipped solder")
        ("dirty" "dirtied")
        ("disagree with" "disagreed with")
        ("disannul" "disannulled")
        ("disbar" "disbarred")
        ("disbud" "disbudded")
        ("disembody" "disembodied")
        ("disembowel" "disembowelled")
        ("dish out" "dished out")
        ("dish the dirt" "dished the dirt")
        ("dish up" "dished up")
        ("disincarnate" "disincarnised")
        ("disinter" "disinterred")
        ("disorientate" "disoriented")
        ("dispel" "dispelled")
        ("dispense with" "dispensed with")
        ("display" "displayed")
        ("dispose of" "disposed of")
        ("disqualify" "disqualified")
        ("dissatisfy" "dissatisfied")
        ("dissonate" "dissonised")
        ("distil" "distilled")
        ("disunify" "disunified")
        ("dive-bomb" "divebombed")
        ("diversify" "diversified")
        ("divvy up" "divvied up")
        ("dizzy" "dizzied")
        ("do" "did")
        ("do a job on" "did a job on")
        ("do away with" "did away with")
        ("do by" "did by")
        ("do drugs" "did drugs")
        ("do good" "did good")
        ("do in" "did in")
        ("do it" "did it")
        ("do justice" "did justice")
        ("do one's best" "did one's best")
        ("do the dishes" "did the dishes")
        ("do the honors" "did the honors")
        ("do up" "did up")
        ("do well" "did well")
        ("do well by" "did well by")
        ("do work" "did work")
        ("doctor up" "doctored up")
        ("dog" "dogged")
        ("dole out" "doled out")
        ("doll up" "dolled up")
        ("don" "donned")
        ("dope off" "doped off")
        ("dope up" "doped up")
        ("doss down" "dossed down")
        ("dot" "dotted")
        ("double back" "doubled back")
        ("double birdie" "doubled birdie")
        ("double bogey" "doubled bogey")
        ("double crochet" "doubled crochet")
        ("double cross" "doubled cross")
        ("double dye" "doubled dye")
        ("double over" "doubled over")
        ("double stitch" "doubled stitch")
        ("double tongue" "doubled tongue")
        ("double up" "doubled up")
        ("double-check" "doublechecked")
        ("double-park" "doubleparked")
        ("double-space" "doublespaced")
        ("doze off" "dozed off")
        ("drag" "dragged")
        ("drag a bunt" "dragged a bunt")
        ("drag down" "dragged down")
        ("drag in" "dragged in")
        ("drag on" "dragged on")
        ("drag one's feet" "dragged one's feet")
        ("drag one's heels" "dragged one's heels")
        ("drag out" "dragged out")
        ("drag through the mud" "dragged through the mud")
        ("drag up" "dragged up")
        ("drain the cup" "drained the cup")
        ("draw" "drew")
        ("draw a bead on" "drew a bead on")
        ("draw a blank" "drew a blank")
        ("draw a line" "drew a line")
        ("draw and quarter" "drew and quarter")
        ("draw away" "drew away")
        ("draw back" "drew back")
        ("draw close" "drew close")
        ("draw in" "drew in")
        ("draw near" "drew near")
        ("draw off" "drew off")
        ("draw out" "drew out")
        ("draw rein" "drew rein")
        ("draw the line" "drew the line")
        ("draw together" "drew together")
        ("draw up" "drew up")
        ("dream up" "dreamed up")
        ("dredge up" "dredged up")
        ("dress down" "dressed down")
        ("dress out" "dressed out")
        ("dress ship" "dressed ship")
        ("dress up" "dressed up")
        ("drift apart" "drifted apart")
        ("drift away" "drifted away")
        ("drift off" "drifted off")
        ("drill in" "drilled in")
        ("drink" "drank")
        ("drink down" "drank down")
        ("drink in" "drank in")
        ("drink up" "drank up")
        ("drip" "dripped")
        ("drip-dry" "drip-dried")
        ("drive" "drove")
        ("drive around" "drove around")
        ("drive away" "drove away")
        ("drive back" "drove back")
        ("drive home" "drove home")
        ("drive in" "drove in")
        ("drive off" "drove off")
        ("drive out" "drove out")
        ("drive up" "drove up")
        ("drone on" "droned on")
        ("drool over" "drooled over")
        ("drop" "dropped")
        ("drop a line" "dropped a line")
        ("drop anchor" "dropped anchor")
        ("drop away" "dropped away")
        ("drop back" "dropped back")
        ("drop behind" "dropped behind")
        ("drop by" "dropped by")
        ("drop by the wayside" "dropped by the wayside")
        ("drop dead" "dropped dead")
        ("drop down" "dropped down")
        ("drop in" "dropped in")
        ("drop like flies" "dropped like flies")
        ("drop off" "dropped off")
        ("drop one's serve" "dropped one's serve")
        ("drop open" "dropped open")
        ("drop out" "dropped out")
        ("drop the ball" "dropped the ball")
        ("drop-kick" "dropkicked")
        ("drown out" "drowned out")
        ("drowse off" "drowsed off")
        ("drub" "drubbed")
        ("drug" "drugged")
        ("drum" "drummed")
        ("drum out" "drummed out")
        ("drum up" "drummed up")
        ("dry" "dried")
        ("dry clean" "dried clean")
        ("dry out" "dried out")
        ("dry up" "dried up")
        ("dry-dock" "drydocked")
        ("dub" "dubbed")
        ("duel" "duelled")
        ("dulcify" "dulcified")
        ("dumbfound" "dumfounded")
        ("dummy" "dummied")
        ("dummy up" "dummied up")
        ("dun" "dunned")
        ("dwell on" "dwelled on")
        ("dwindle away" "dwindled away")
        ("dwindle down" "dwindled down")
        ("e-mail" "e-mailed")
        ("earth up" "earthed up")
        ("earwig" "earwigged")
        ("ease off" "eased off")
        ("ease up" "eased up")
        ("eat" "ate")
        ("eat at" "ate at")
        ("eat away" "ate away")
        ("eat in" "ate in")
        ("eat into" "ate into")
        ("eat on" "ate on")
        ("eat out" "ate out")
        ("eat up" "ate up")
        ("eavesdrop" "eavesdropped")
        ("ebb away" "ebbed away")
        ("ebb down" "ebbed down")
        ("ebb off" "ebbed off")
        ("ebb out" "ebbed out")
        ("eddy" "eddied")
        ("edge in" "edged in")
        ("edge up" "edged up")
        ("edify" "edified")
        ("edit out" "edited out")
        ("egg on" "egged on")
        ("egotrip" "egotripped")
        ("eke out" "eked out")
        ("electrify" "electrified")
        ("embark on" "embarked on")
        ("embed" "embedded")
        ("embody" "embodied")
        ("emit" "emitted")
        ("empanel" "empanelled")
        ("employ" "employed")
        ("empty" "emptied")
        ("emulsify" "emulsified")
        ("encapsulate" "incapsulated")
        ("encase" "incased")
        ("enclose" "inclosed")
        ("encroach upon" "encroached upon")
        ("encrust" "incrusted")
        ("end up" "ended up")
        ("enfold" "infolded")
        ("englut" "englutted")
        ("engraft" "ingrafted")
        ("enmesh" "inmeshed")
        ("ensky" "enskied")
        ("ensnare" "insnared")
        ("enter upon" "entered upon")
        ("entrap" "entrapped")
        ("entwine" "entwintwined")
        ("envy" "envied")
        ("enwind" "enwound")
        ("enwrap" "enwrapped")
        ("epoxy" "epoxied")
        ("equip" "equipped")
        ("eruct" "eructated")
        ("escape from" "escaped from")
        ("espy" "espied")
        ("esterify" "esterified")
        ("eternalize" "eternized")
        ("etherify" "etherified")
        ("even off" "evened off")
        ("even out" "evened out")
        ("even up" "evened up")
        ("excel" "excelled")
        ("excel at" "excelled at")
        ("exemplify" "exemplified")
        ("expel" "expelled")
        ("explode a bombshell" "exploded a bombshell")
        ("express emotion" "expressed emotion")
        ("express feelings" "expressed feelings")
        ("express joy" "expressed joy")
        ("express mirth" "expressed mirth")
        ("extend oneself" "extended oneself")
        ("extend to" "extended to")
        ("extol" "extolled")
        ("face off" "faced off")
        ("face the music" "faced the music")
        ("face up" "faced up")
        ("face-harden" "facehardened")
        ("factor analyse" "factored analyse")
        ("factor analyze" "factored analyze")
        ("factor in" "factored in")
        ("factor out" "factored out")
        ("fade away" "faded away")
        ("fade out" "faded out")
        ("fag" "fagged")
        ("fag out" "fagged out")
        ("faggot up" "faggoted up")
        ("fall" "fell")
        ("fall all over" "fell all over")
        ("fall apart" "fell apart")
        ("fall asleep" "fell asleep")
        ("fall away" "fell away")
        ("fall back" "fell back")
        ("fall behind" "fell behind")
        ("fall by the wayside" "fell by the wayside")
        ("fall down" "fell down")
        ("fall flat" "fell flat")
        ("fall for" "fell for")
        ("fall from grace" "fell from grace")
        ("fall in" "fell in")
        ("fall in line" "fell in line")
        ("fall in love" "fell in love")
        ("fall into" "fell into")
        ("fall into place" "fell into place")
        ("fall off" "fell off")
        ("fall open" "fell open")
        ("fall out" "fell out")
        ("fall over" "fell over")
        ("fall over backwards" "fell over backwards")
        ("fall short" "fell short")
        ("fall short of" "fell short of")
        ("fall through" "fell through")
        ("fall under" "fell under")
        ("fall upon" "fell upon")
        ("falsify" "falsified")
        ("fan" "fanned")
        ("fan out" "fanned out")
        ("fancify" "fancified")
        ("fancy" "fancied")
        ("fancy up" "fancied up")
        ("fantasy" "phantasied")
        ("farm out" "farmed out")
        ("fast dye" "fasted dye")
        ("fasten on" "fastened on")
        ("fat" "fatted")
        ("fatten out" "fattened out")
        ("fatten up" "fattened up")
        ("feast one's eyes" "feasted one's eyes")
        ("feather one's nest" "feathered one's nest")
        ("featherbed" "featherbedded")
        ("feed" "fed")
        ("feed back" "fed back")
        ("feed in" "fed in")
        ("feed on" "fed on")
        ("feed upon" "fed upon")
        ("feel" "felt")
        ("feel for" "felt for")
        ("feel like" "felt like")
        ("feel like a million" "felt like a million")
        ("feel like a million dollars" "felt like a million dollars")
        ("feel out" "felt out")
        ("felt up" "felted up")
        ("fence in" "fenced in")
        ("fend for" "fended for")
        ("fend off" "fended off")
        ("ferret out" "ferreted out")
        ("ferry" "ferried")
        ("fess up" "fessed up")
        ("festinate" "festinised")
        ("fetch up" "fetched up")
        ("fib" "fibbed")
        ("fiddle with" "fiddled with")
        ("fig out" "figged out")
        ("fig up" "figged up")
        ("fight" "fought")
        ("fight back" "fought back")
        ("fight down" "fought down")
        ("fight off" "fought off")
        ("figure out" "figured out")
        ("file away" "filed away")
        ("file in" "filed in")
        ("file out" "filed out")
        ("fill again" "filled again")
        ("fill in" "filled in")
        ("fill out" "filled out")
        ("fill the bill" "filled the bill")
        ("fill up" "filled up")
        ("film over" "filmed over")
        ("filter out" "filtered out")
        ("fin" "finned")
        ("find" "found")
        ("find fault" "found fault")
        ("find oneself" "found oneself")
        ("find out" "found out")
        ("finish off" "finished off")
        ("finish out" "finished out")
        ("finish up" "finished up")
        ("fire up" "fired up")
        ("firm up" "firmed up")
        ("fit" "fitted")
        ("fit in" "fitted in")
        ("fit out" "fitted out")
        ("fit the bill" "fitted the bill")
        ("fix up" "fixed up")
        ("fizzle out" "fizzled out")
        ("flag" "flagged")
        ("flag down" "flagged down")
        ("flake off" "flaked off")
        ("flake out" "flaked out")
        ("flame up" "flamed up")
        ("flap" "flapped")
        ("flap down" "flapped down")
        ("flare out" "flared out")
        ("flare up" "flared up")
        ("flash back" "flashed back")
        ("flatten out" "flattened out")
        ("flee" "fled")
        ("flesh out" "fleshed out")
        ("fling" "flung")
        ("fling off" "flung off")
        ("flip" "flipped")
        ("flip one's lid" "flipped one's lid")
        ("flip one's wig" "flipped one's wig")
        ("flip out" "flipped out")
        ("flip over" "flipped over")
        ("flirt with" "flirted with")
        ("flit" "flitted")
        ("flog" "flogged")
        ("flood in" "flooded in")
        ("flood out" "flooded out")
        ("floodlight" "floodlit")
        ("flop" "flopped")
        ("flow away" "flowed away")
        ("flow from" "flowed from")
        ("flow off" "flowed off")
        ("flow out" "flowed out")
        ("fluff up" "fluffed up")
        ("flump down" "flumped down")
        ("flurry" "flurried")
        ("flush down" "flushed down")
        ("flush it" "flushed it")
        ("fly" "flew")
        ("fly blind" "flew blind")
        ("fly by" "flew by")
        ("fly contact" "flew contact")
        ("fly high" "flew high")
        ("fly in the face of" "flew in the face of")
        ("fly in the teeth of" "flew in the teeth of")
        ("fly off the handle" "flew off the handle")
        ("fly on" "flew on")
        ("fly open" "flew open")
        ("fly the coop" "flew the coop")
        ("foam at the mouth" "foamed at the mouth")
        ("fob" "fobbed")
        ("fob off" "fobbed off")
        ("focus on" "focused on")
        ("fog" "fogged")
        ("fog up" "fogged up")
        ("foist off" "foisted off")
        ("fold up" "folded up")
        ("folk dance" "folked dance")
        ("follow out" "followed out")
        ("follow suit" "followed suit")
        ("follow through" "followed through")
        ("follow up" "followed up")
        ("follow up on" "followed up on")
        ("fool around" "fooled around")
        ("fool away" "fooled away")
        ("foot up" "footed up")
        ("foray into" "forayed into")
        ("forbear" "forbore")
        ("forbid" "forbade")
        ("force back" "forced back")
        ("force out" "forced out")
        ("force-feed" "forcefed")
        ("forego" "forewent")
        ("foreknow" "foreknew")
        ("foresee" "foresaw")
        ("foreswear" "foreswore")
        ("foretell" "foretold")
        ("forget" "forgot")
        ("forgive" "forgave")
        ("forgo" "forwent")
        ("fork out" "forked out")
        ("fork over" "forked over")
        ("fork up" "forked up")
        ("form bubbles" "formed bubbles")
        ("format" "formatted")
        ("forsake" "forsook")
        ("forswear" "forswore")
        ("fort up" "forted up")
        ("fortify" "fortified")
        ("foul out" "fouled out")
        ("foul up" "fouled up")
        ("frame in" "framed in")
        ("frame up" "framed up")
        ("frap" "frapped")
        ("freak out" "freaked out")
        ("freeze" "froze")
        ("freeze down" "froze down")
        ("freeze off" "froze off")
        ("freeze out" "froze out")
        ("freeze-dry" "freezedried")
        ("french-fry" "french-fried")
        ("frenchify" "frenchified")
        ("freshen up" "freshened up")
        ("fret" "fretted")
        ("frig around" "frigged around")
        ("frighten away" "frightened away")
        ("frighten off" "frightened off")
        ("fritter away" "frittered away")
        ("frivol" "frivolled")
        ("frivol away" "frivolled away")
        ("frog" "frogged")
        ("frolic" "frolicked")
        ("frost over" "frosted over")
        ("froth at the mouth" "frothed at the mouth")
        ("frown on" "frowned on")
        ("frown upon" "frowned upon")
        ("fructify" "fructified")
        ("fry" "fried")
        ("fuck off" "fucked off")
        ("fuck up" "fucked up")
        ("fudge together" "fudged together")
        ("fuel" "fuelled")
        ("fund raise" "funded raise")
        ("funnel" "funnelled")
        ("furbish up" "furbished up")
        ("gab" "gabbed")
        ("gad" "gadded")
        ("gag" "gagged")
        ("gain ground" "gained ground")
        ("gain vigor" "gained vigor")
        ("gainsay" "gainsaid")
        ("gambol" "gambolled")
        ("gang up" "ganged up")
        ("garbage down" "garbaged down")
        ("garotte" "garrotted")
        ("gas" "gassed")
        ("gas up" "gassed up")
        ("gasify" "gasified")
        ("gather in" "gathered in")
        ("gather up" "gathered up")
        ("gear up" "geared up")
        ("gentrify" "gentrified")
        ("get" "got")
        ("get a line" "got a line")
        ("get a load" "got a load")
        ("get a look" "got a look")
        ("get a noseful" "got a noseful")
        ("get a whiff" "got a whiff")
        ("get about" "got about")
        ("get across" "got across")
        ("get ahead" "got ahead")
        ("get along" "got along")
        ("get along with" "got along with")
        ("get around" "got around")
        ("get around to" "got around to")
        ("get at" "got at")
        ("get away" "got away")
        ("get back" "got back")
        ("get behind" "got behind")
        ("get by" "got by")
        ("get cracking" "got cracking")
        ("get down" "got down")
        ("get dressed" "got dressed")
        ("get even" "got even")
        ("get going" "got going")
        ("get hitched with" "got hitched with")
        ("get hold" "got hold")
        ("get hold of" "got hold of")
        ("get in" "got in")
        ("get in touch" "got in touch")
        ("get into" "got into")
        ("get it" "got it")
        ("get it on" "got it on")
        ("get laid" "got laid")
        ("get married" "got married")
        ("get moving" "got moving")
        ("get off" "got off")
        ("get off the ground" "got off the ground")
        ("get on" "got on")
        ("get on with" "got on with")
        ("get one's lumps" "got one's lumps")
        ("get onto" "got onto")
        ("get out" "got out")
        ("get over" "got over")
        ("get rid of" "got rid of")
        ("get rolling" "got rolling")
        ("get started" "got started")
        ("get stuck" "got stuck")
        ("get the best" "got the best")
        ("get the better of" "got the better of")
        ("get the goods" "got the goods")
        ("get the hang" "got the hang")
        ("get the jump" "got the jump")
        ("get the picture" "got the picture")
        ("get through" "got through")
        ("get to" "got to")
        ("get to grips" "got to grips")
        ("get together" "got together")
        ("get under one's skin" "got under one's skin")
        ("get up" "got up")
        ("get weaving" "got weaving")
        ("get well" "got well")
        ("get wind" "got wind")
        ("get wise" "got wise")
        ("get word" "got word")
        ("get worse" "got worse")
        ("ghostwrite" "ghostwrote")
        ("gild the lily" "gilded the lily")
        ("gin" "ginned")
        ("ginger up" "gingered up")
        ("gird" "girt")
        ("give" "gave")
        ("give a damn" "gave a damn")
        ("give a hang" "gave a hang")
        ("give a hoot" "gave a hoot")
        ("give and take" "gave and take")
        ("give away" "gave away")
        ("give back" "gave back")
        ("give birth" "gave birth")
        ("give care" "gave care")
        ("give chase" "gave chase")
        ("give ear" "gave ear")
        ("give forth" "gave forth")
        ("give full measure" "gave full measure")
        ("give in" "gave in")
        ("give it a try" "gave it a try")
        ("give it a whirl" "gave it a whirl")
        ("give it the deep six" "gave it the deep six")
        ("give notice" "gave notice")
        ("give off" "gave off")
        ("give one's best" "gave one's best")
        ("give out" "gave out")
        ("give rise" "gave rise")
        ("give suck" "gave suck")
        ("give thanks" "gave thanks")
        ("give the axe" "gave the axe")
        ("give the bounce" "gave the bounce")
        ("give the eye" "gave the eye")
        ("give the gate" "gave the gate")
        ("give the glad eye" "gave the glad eye")
        ("give the once over" "gave the once over")
        ("give the sack" "gave the sack")
        ("give tongue to" "gave tongue to")
        ("give up" "gave up")
        ("give vent" "gave vent")
        ("give voice" "gave voice")
        ("give way" "gave way")
        ("give-up the ghost" "gave-up the ghost")
        ("glam up" "glamed up")
        ("glamorize" "glamourized")
        ("glance over" "glanced over")
        ("glass in" "glassed in")
        ("glass over" "glassed over")
        ("glaze over" "glazed over")
        ("glide by" "glided by")
        ("globe-trot" "globe-trotted")
        ("glorify" "glorified")
        ("glory" "gloried")
        ("gloss over" "glossed over")
        ("glut" "glutted")
        ("gnaw at" "gnawed at")
        ("go" "went")
        ("go a long way" "went a long way")
        ("go about" "went about")
        ("go across" "went across")
        ("go after" "went after")
        ("go against" "went against")
        ("go ahead" "went ahead")
        ("go all out" "went all out")
        ("go along" "went along")
        ("go around" "went around")
        ("go away" "went away")
        ("go back" "went back")
        ("go back on" "went back on")
        ("go bad" "went bad")
        ("go ballistic" "went ballistic")
        ("go by" "went by")
        ("go deep" "went deep")
        ("go down" "went down")
        ("go down on" "went down on")
        ("go dutch" "went dutch")
        ("go far" "went far")
        ("go for" "went for")
        ("go for broke" "went for broke")
        ("go forth" "went forth")
        ("go forward" "went forward")
        ("go home" "went home")
        ("go in" "went in")
        ("go into" "went into")
        ("go off" "went off")
        ("go off at half-cock" "went off at half-cock")
        ("go off half-cocked" "went off half-cocked")
        ("go on" "went on")
        ("go out" "went out")
        ("go over" "went over")
        ("go past" "went past")
        ("go steady" "went steady")
        ("go through" "went through")
        ("go through the motions" "went through the motions")
        ("go to" "went to")
        ("go to bed" "went to bed")
        ("go to pieces" "went to pieces")
        ("go to pot" "went to pot")
        ("go to sleep" "went to sleep")
        ("go to the dogs" "went to the dogs")
        ("go to war" "went to war")
        ("go under" "went under")
        ("go up" "went up")
        ("go with" "went with")
        ("go wrong" "went wrong")
        ("gobble up" "gobbled up")
        ("gold plate" "golded plate")
        ("goose step" "goosed step")
        ("gouge out" "gouged out")
        ("grab" "grabbed")
        ("grass over" "grassed over")
        ("gratify" "gratified")
        ("grease one's palms" "greased one's palms")
        ("grin" "grinned")
        ("grind" "ground")
        ("grind away" "ground away")
        ("grind down" "ground down")
        ("grind out" "ground out")
        ("grind to a halt" "ground to a halt")
        ("grip" "gripped")
        ("grit" "gritted")
        ("grope for" "groped for")
        ("gross out" "grossed out")
        ("ground out" "grounded out")
        ("grow" "grew")
        ("grow over" "grew over")
        ("grow up" "grew up")
        ("grub" "grubbed")
        ("grub out" "grubbed out")
        ("grub up" "grubbed up")
        ("guide on" "guided on")
        ("gum" "gummed")
        ("gum up" "gummed up")
        ("gun" "gunned")
        ("gun down" "gunned down")
        ("gussy up" "gussied up")
        ("gut" "gutted")
        ("gyp" "gypped")
        ("hack on" "hacked on")
        ("had best" "had best")
        ("ham" "hammed")
        ("ham it up" "hammed it up")
        ("hammer in" "hammered in")
        ("hammer out" "hammered out")
        ("hamstring" "hamstrung")
        ("hand down" "handed down")
        ("hand out" "handed out")
        ("hand over" "handed over")
        ("hand-pick" "handpicked")
        ("handicap" "handicapped")
        ("handle with kid gloves" "handled with kid gloves")
        ("hang" "hung")
        ("hang around" "hung around")
        ("hang back" "hung back")
        ("hang by a hair" "hung by a hair")
        ("hang by a thread" "hung by a thread")
        ("hang glide" "hung glide")
        ("hang in" "hung in")
        ("hang on" "hung on")
        ("hang out" "hung out")
        ("hang together" "hung together")
        ("hang up" "hung up")
        ("hap" "happed")
        ("happen upon" "happened upon")
        ("hard surface" "harded surface")
        ("hark back" "harked back")
        ("harry" "harried")
        ("harsh on" "harshed on")
        ("hash out" "hashed out")
        ("hash over" "hashed over")
        ("hat" "hatted")
        ("haul away" "hauled away")
        ("haul off" "hauled off")
        ("haul up" "hauled up")
        ("have" "had")
        ("have a ball" "had a ball")
        ("have a bun in the oven" "had a bun in the oven")
        ("have a fit" "had a fit")
        ("have a go" "had a go")
        ("have a go at it" "had a go at it")
        ("have a good time" "had a good time")
        ("have a look" "had a look")
        ("have down" "had down")
        ("have got" "had got")
        ("have in mind" "had in mind")
        ("have intercourse" "had intercourse")
        ("have it away" "had it away")
        ("have it coming" "had it coming")
        ("have it off" "had it off")
        ("have kittens" "had kittens")
        ("have on" "had on")
        ("have sex" "had sex")
        ("have the best" "had the best")
        ("have words" "had words")
        ("have young" "had young")
        ("have-to doe with" "have-toed doe with")
        ("haze over" "hazed over")
        ("head for the hills" "headed for the hills")
        ("head home" "headed home")
        ("head off" "headed off")
        ("head up" "headed up")
        ("heap up" "heaped up")
        ("hear" "heard")
        ("hear out" "heard out")
        ("heat up" "heated up")
        ("heave up" "heaved up")
        ("hedge in" "hedged in")
        ("hedge-hop" "hedge-hopped")
        ("hedgehop" "hedgehopped")
        ("heft up" "hefted up")
        ("help oneself" "helped oneself")
        ("help out" "helped out")
        ("hem" "hemmed")
        ("hem and haw" "hemmed and haw")
        ("hem in" "hemmed in")
        ("hero-worship" "hero-worshipped")
        ("hew out" "hewed out")
        ("hide" "hid")
        ("hide out" "hid out")
        ("high-hat" "high-hatted")
        ("hightail it" "hightailed it")
        ("hike up" "hiked up")
        ("hinge on" "hinged on")
        ("hinge upon" "hinged upon")
        ("hire out" "hired out")
        ("hit" "hit")
        ("hit home" "hit home")
        ("hit it up" "hit it up")
        ("hit the books" "hit the books")
        ("hit the ceiling" "hit the ceiling")
        ("hit the deck" "hit the deck")
        ("hit the dirt" "hit the dirt")
        ("hit the hay" "hit the hay")
        ("hit the jackpot" "hit the jackpot")
        ("hit the roof" "hit the roof")
        ("hit the sack" "hit the sack")
        ("hitch up" "hitched up")
        ("hive away" "hived away")
        ("hive off" "hived off")
        ("hive up" "hived up")
        ("hob" "hobbed")
        ("hobnob" "hobnobbed")
        ("hog" "hogged")
        ("hold" "held")
        ("hold back" "held back")
        ("hold close" "held close")
        ("hold dear" "held dear")
        ("hold down" "held down")
        ("hold fast" "held fast")
        ("hold firm" "held firm")
        ("hold forth" "held forth")
        ("hold in" "held in")
        ("hold off" "held off")
        ("hold on" "held on")
        ("hold one's own" "held one's own")
        ("hold open" "held open")
        ("hold out" "held out")
        ("hold over" "held over")
        ("hold still for" "held still for")
        ("hold sway" "held sway")
        ("hold the line" "held the line")
        ("hold tight" "held tight")
        ("hold up" "held up")
        ("hold water" "held water")
        ("hole out" "holed out")
        ("hole up" "holed up")
        ("holler out" "hollered out")
        ("hollow out" "hollowed out")
        ("home in" "homed in")
        ("honey" "honied")
        ("hoof it" "hoofed it")
        ("hook on" "hooked on")
        ("hook up" "hooked up")
        ("hook-up" "hooked-up")
        ("hook up with" "hooked up with")
        ("hop" "hopped")
        ("hop on" "hopped on")
        ("hop out" "hopped out")
        ("hop up" "hopped up")
        ("horn in" "horned in")
        ("horrify" "horrified")
        ("horse around" "horsed around")
        ("horsewhip" "horsewhipped")
        ("hose down" "hosed down")
        ("hot up" "hoted up")
        ("hot-dog" "hotdoged")
        ("huddle together" "huddled together")
        ("hug" "hugged")
        ("hugger mugger" "huggered mugger")
        ("hum" "hummed")
        ("humbug" "humbugged")
        ("humidify" "humidified")
        ("humify" "humified")
        ("hunch forward" "hunched forward")
        ("hunch over" "hunched over")
        ("hunker down" "hunkered down")
        ("hunt down" "hunted down")
        ("hurry" "hurried")
        ("hurt" "hurt")
        ("hush up" "hushed up")
        ("hydrolyse" "hydrolyzed")
        ("hype up" "hyped up")
        ("hypertrophy" "hypertrophied")
        ("hyphenate" "hyphened")
        ("ice over" "iced over")
        ("ice skate" "iced skate")
        ("ice up" "iced up")
        ("identify" "identified")
        ("ill-treat" "illtreated")
        ("ill-use" "illused")
        ("imbed" "imbedded")
        ("impel" "impelled")
        ("impinge on" "impinged on")
        ("imply" "implied")
        ("impulse-buy" "impulse-bought")
        ("inconvenience oneself" "inconvenienced oneself")
        ("incur" "incurred")
        ("indemnify" "indemnified")
        ("indwell" "indwelt")
        ("infer" "inferred")
        ("inhere in" "inhered in")
        ("initial" "initialled")
        ("inlay" "inlaid")
        ("inmarry" "inmarried")
        ("inset" "inset")
        ("inspan" "inspanned")
        ("intensify" "intensified")
        ("inter" "interred")
        ("interbreed" "interbred")
        ("intermarry" "intermarried")
        ("intermit" "intermitted")
        ("interstratify" "interstratified")
        ("interweave" "interwove")
        ("intromit" "intromitted")
        ("intrude on" "intruded on")
        ("inventory" "inventoried")
        ("invite out" "invited out")
        ("inweave" "inwove")
        ("iodinate" "iodinised")
        ("ionate" "ionised")
        ("iron out" "ironed out")
        ("island hop" "islanded hop")
        ("issue forth" "issued forth")
        ("jab" "jabbed")
        ("jack off" "jacked off")
        ("jack up" "jacked up")
        ("jag" "jagged")
        ("jam" "jammed")
        ("japan" "japanned")
        ("jar" "jarred")
        ("jar against" "jarred against")
        ("jazz around" "jazzed around")
        ("jazz up" "jazzed up")
        ("jellify" "jellified")
        ("jelly" "jellied")
        ("jerk off" "jerked off")
        ("jest at" "jested at")
        ("jet" "jetted")
        ("jib" "jibbed")
        ("jig" "jigged")
        ("jimmy" "jimmied")
        ("job" "jobbed")
        ("jog" "jogged")
        ("join battle" "joined battle")
        ("join forces" "joined forces")
        ("jollify" "jollified")
        ("jolly" "jollied")
        ("jolly along" "jollied along")
        ("jolly up" "jollied up")
        ("jot" "jotted")
        ("jot down" "jotted down")
        ("jug" "jugged")
        ("juice up" "juiced up")
        ("jump for joy" "jumped for joy")
        ("jump off" "jumped off")
        ("jump on" "jumped on")
        ("jump out" "jumped out")
        ("jump-start" "jumpstarted")
        ("justify" "justified")
        ("jut" "jutted")
        ("jut out" "jutted out")
        ("keel over" "keeled over")
        ("keep" "kept")
        ("keep abreast" "kept abreast")
        ("keep an eye on" "kept an eye on")
        ("keep apart" "kept apart")
        ("keep away" "kept away")
        ("keep back" "kept back")
        ("keep company" "kept company")
        ("keep down" "kept down")
        ("keep going" "kept going")
        ("keep guard" "kept guard")
        ("keep in" "kept in")
        ("keep in line" "kept in line")
        ("keep mum" "kept mum")
        ("keep note" "kept note")
        ("keep off" "kept off")
        ("keep on" "kept on")
        ("keep one's distance" "kept one's distance")
        ("keep one's eyes off" "kept one's eyes off")
        ("keep one's eyes open" "kept one's eyes open")
        ("keep one's eyes peeled" "kept one's eyes peeled")
        ("keep one's eyes skinned" "kept one's eyes skinned")
        ("keep one's hands off" "kept one's hands off")
        ("keep one's mouth shut" "kept one's mouth shut")
        ("keep one's nose to the grindstone" "kept one's nose to the grindstone")
        ("keep one's shoulder to the wheel" "kept one's shoulder to the wheel")
        ("keep open" "kept open")
        ("keep out" "kept out")
        ("keep pace" "kept pace")
        ("keep quiet" "kept quiet")
        ("keep step" "kept step")
        ("keep tabs on" "kept tabs on")
        ("keep to oneself" "kept to oneself")
        ("keep track" "kept track")
        ("keep up" "kept up")
        ("key out" "keyed out")
        ("kick about" "kicked about")
        ("kick around" "kicked around")
        ("kick back" "kicked back")
        ("kick down" "kicked down")
        ("kick downstairs" "kicked downstairs")
        ("kick in" "kicked in")
        ("kick off" "kicked off")
        ("kick one's heels" "kicked one's heels")
        ("kick out" "kicked out")
        ("kick the bucket" "kicked the bucket")
        ("kick up" "kicked up")
        ("kick upstairs" "kicked upstairs")
        ("kid" "kidded")
        ("kidnap" "kidnapped")
        ("kill off" "killed off")
        ("kill oneself" "killed oneself")
        ("kink up" "kinked up")
        ("kip" "kipped")
        ("kip down" "kipped down")
        ("kit" "kitted")
        ("kit out" "kitted out")
        ("kit up" "kitted up")
        ("knap" "knapped")
        ("kneel" "knelt")
        ("knit" "knit")
        ("knock about" "knocked about")
        ("knock against" "knocked against")
        ("knock back" "knocked back")
        ("knock cold" "knocked cold")
        ("knock down" "knocked down")
        ("knock off" "knocked off")
        ("knock out" "knocked out")
        ("knock over" "knocked over")
        ("knock up" "knocked up")
        ("knot" "knotted")
        ("know" "knew")
        ("know apart" "knew apart")
        ("know the score" "knew the score")
        ("know what's going on" "knew what's going on")
        ("know what's what" "knew what's what")
        ("knuckle down" "knuckled down")
        ("knuckle under" "knuckled under")
        ("lace into" "laced into")
        ("lace up" "laced up")
        ("laden" "laden")
        ("lag" "lagged")
        ("lallygag" "lallygagged")
        ("lam" "lammed")
        ("lam into" "lammed into")
        ("land up" "landed up")
        ("lap" "lapped")
        ("lap up" "lapped up")
        ("lapidify" "lapidified")
        ("lark about" "larked about")
        ("lash out" "lashed out")
        ("lash together" "lashed together")
        ("last out" "lasted out")
        ("latch on" "latched on")
        ("laugh at" "laughed at")
        ("laugh away" "laughed away")
        ("laugh loudly" "laughed loudly")
        ("laugh off" "laughed off")
        ("laugh softly" "laughed softly")
        ("lay" "laid")
        ("lay aside" "laid aside")
        ("lay away" "laid away")
        ("lay claim" "laid claim")
        ("lay down" "laid down")
        ("lay eyes on" "laid eyes on")
        ("lay hands on" "laid hands on")
        ("lay in" "laid in")
        ("lay into" "laid into")
        ("lay off" "laid off")
        ("lay on the line" "laid on the line")
        ("lay out" "laid out")
        ("lay over" "laid over")
        ("lay to rest" "laid to rest")
        ("lay up" "laid up")
        ("lay waste to" "laid waste to")
        ("lead" "led")
        ("lead astray" "led astray")
        ("lead by the nose" "led by the nose")
        ("lead off" "led off")
        ("lead on" "led on")
        ("lead up" "led up")
        ("league together" "leagued together")
        ("leak out" "leaked out")
        ("lean against" "leaned against")
        ("lean back" "leaned back")
        ("lean on" "leaned on")
        ("leap out" "leaped out")
        ("leapfrog" "leapfrogged")
        ("learn" "learnt")
        ("leave" "left")
        ("leave alone" "left alone")
        ("leave behind" "left behind")
        ("leave no stone unturned" "left no stone unturned")
        ("leave off" "left off")
        ("leave office" "left office")
        ("leave out" "left out")
        ("lech after" "letched after")
        ("leech onto" "leeched onto")
        ("leg it" "legged it")
        ("lend" "lent")
        ("lend oneself" "lent oneself")
        ("lenify" "lenified")
        ("let" "let")
        ("let down" "let down")
        ("let drive" "let drive")
        ("let fly" "let fly")
        ("let go" "let go")
        ("let go of" "let go of")
        ("let in" "let in")
        ("let it go" "let it go")
        ("let loose" "let loose")
        ("let off" "let off")
        ("let on" "let on")
        ("let out" "let out")
        ("let the cat out of the bag" "let the cat out of the bag")
        ("let up" "let up")
        ("letter bomb" "lettered bomb")
        ("level off" "leveled off")
        ("levy" "levied")
        ("lie about" "lied about")
        ("lie around" "lay around")
        ("lie awake" "lay awake")
        ("lie dormant" "lay dormant")
        ("lie down" "lay down")
        ("lie in" "lay in")
        ("lie in wait" "lay in wait")
        ("lie low" "lay low")
        ("lie with" "lay with")
        ("lift off" "lifted off")
        ("lift out" "lifted out")
        ("lift up" "lifted up")
        ("light up" "lighted up")
        ("light upon" "lighted upon")
        ("lighten up" "lightened up")
        ("lignify" "lignified")
        ("limber up" "limbered up")
        ("line one's pockets" "lined one's pockets")
        ("line up" "lined up")
        ("linger over" "lingered over")
        ("link up" "linked up")
        ("lip off" "lipped off")
        ("lip-read" "lip-read")
        ("lipread" "lipread")
        ("liquefy" "liquefied")
        ("liquify" "liquified")
        ("listen in" "listened in")
        ("live down" "lived down")
        ("live in" "lived in")
        ("live it up" "lived it up")
        ("live on" "lived on")
        ("live out" "lived out")
        ("live over" "lived over")
        ("live together" "lived together")
        ("live up to" "lived up to")
        ("live with" "lived with")
        ("liven up" "livened up")
        ("load down" "loaded down")
        ("load up" "loaded up")
        ("lob" "lobbed")
        ("lobby" "lobbied")
        ("lock away" "locked away")
        ("lock in" "locked in")
        ("lock out" "locked out")
        ("lock up" "locked up")
        ("lodge in" "lodged in")
        ("log" "logged")
        ("log in" "logged in")
        ("log off" "logged off")
        ("log on" "logged on")
        ("log out" "logged out")
        ("log up" "logged up")
        ("log z's" "logged z's")
        ("loll around" "lolled around")
        ("look across" "looked across")
        ("look after" "looked after")
        ("look around" "looked around")
        ("look at" "looked at")
        ("look away" "looked away")
        ("look back" "looked back")
        ("look backward" "looked backward")
        ("look down on" "looked down on")
        ("look for" "looked for")
        ("look forward" "looked forward")
        ("look into" "looked into")
        ("look like" "looked like")
        ("look on" "looked on")
        ("look out" "looked out")
        ("look out on" "looked out on")
        ("look out over" "looked out over")
        ("look sharp" "looked sharp")
        ("look to" "looked to")
        ("look up" "looked up")
        ("look up to" "looked up to")
        ("look upon" "looked upon")
        ("loose off" "loosed off")
        ("loosen up" "loosened up")
        ("lop" "lopped")
        ("lop off" "lopped off")
        ("lord it over" "lorded it over")
        ("lose" "lost")
        ("lose it" "lost it")
        ("lose one's temper" "lost one's temper")
        ("lose sight of" "lost sight of")
        ("lose track" "lost track")
        ("lose weight" "lost weight")
        ("lot" "lotted")
        ("lounge about" "lounged about")
        ("lounge around" "lounged around")
        ("louse up" "loused up")
        ("lower oneself" "lowered oneself")
        ("luck into" "lucked into")
        ("luck it" "lucked it")
        ("luck out" "lucked out")
        ("luck through" "lucked through")
        ("lug" "lugged")
        ("lust after" "lusted after")
        ("machine gun" "machined gun")
        ("machine wash" "machined wash")
        ("magnify" "magnified")
        ("mail out" "mailed out")
        ("make" "made")
        ("make a clean breast of" "made a clean breast of")
        ("make a face" "made a face")
        ("make a motion" "made a motion")
        ("make a point" "made a point")
        ("make a stink" "made a stink")
        ("make as if" "made as if")
        ("make believe" "made believe")
        ("make bold" "made bold")
        ("make clean" "made clean")
        ("make do" "made do")
        ("make for" "made for")
        ("make full" "made full")
        ("make fun" "made fun")
        ("make good" "made good")
        ("make grow" "made grow")
        ("make happy" "made happy")
        ("make hay" "made hay")
        ("make headway" "made headway")
        ("make it" "made it")
        ("make love" "made love")
        ("make merry" "made merry")
        ("make no bones about" "made no bones about")
        ("make noise" "made noise")
        ("make off" "made off")
        ("make out" "made out")
        ("make over" "made over")
        ("make pass" "made pass")
        ("make peace" "made peace")
        ("make pure" "made pure")
        ("make relaxed" "made relaxed")
        ("make sense" "made sense")
        ("make sure" "made sure")
        ("make unnecessary" "made unnecessary")
        ("make up" "made up")
        ("make up one's mind" "made up one's mind")
        ("make vibrant sounds" "made vibrant sounds")
        ("make water" "made water")
        ("make way" "made way")
        ("make whoopie" "made whoopie")
        ("man" "manned")
        ("manumit" "manumitted")
        ("map" "mapped")
        ("map out" "mapped out")
        ("mar" "marred")
        ("marcel" "marcelled")
        ("march on" "marched on")
        ("march out" "marched out")
        ("mark down" "marked down")
        ("mark off" "marked off")
        ("mark out" "marked out")
        ("mark up" "marked up")
        ("marry" "married")
        ("mass-produce" "massproduced")
        ("mat" "matted")
        ("mat up" "matted up")
        ("matte up" "matted up")
        ("matter to" "mattered to")
        ("max out" "maxed out")
        ("mean" "meant")
        ("measure out" "measured out")
        ("measure up" "measured up")
        ("medal" "medalled")
        ("meet" "met")
        ("meet up with" "met up with")
        ("mellow out" "mellowed out")
        ("melt down" "melted down")
        ("melt off" "melted off")
        ("meow" "miaoued")
        ("mess about" "messed about")
        ("mess around" "messed around")
        ("mess up" "messed up")
        ("mete out" "meted out")
        ("metrify" "metrified")
        ("microcopy" "microcopied")
        ("mill about" "milled about")
        ("mill around" "milled around")
        ("mimic" "mimicked")
        ("minify" "minified")
        ("misally" "misallied")
        ("misapply" "misapplied")
        ("miscarry" "miscarried")
        ("miscast" "miscast")
        ("miscegenate" "miscegenised")
        ("misdeal" "misdealt")
        ("misdo" "misdid")
        ("misgive" "misgave")
        ("misidentify" "misidentified")
        ("mislay" "mislaid")
        ("mislead" "misled")
        ("mismarry" "mismarried")
        ("misread" "misread")
        ("misspend" "misspent")
        ("mist over" "misted over")
        ("mistake" "mistook")
        ("misunderstand" "misunderstood")
        ("mix in" "mixed in")
        ("mix up" "mixed up")
        ("mob" "mobbed")
        ("mock up" "mocked up")
        ("modify" "modified")
        ("mollify" "mollified")
        ("monkey around" "monkeyed around")
        ("moon about" "mooned about")
        ("moon around" "mooned around")
        ("moon on" "mooned on")
        ("mop" "mopped")
        ("mop up" "mopped up")
        ("mope around" "moped around")
        ("mortify" "mortified")
        ("mound over" "mounded over")
        ("mount up" "mounted up")
        ("mouth off" "mouthed off")
        ("move around" "moved around")
        ("move back" "moved back")
        ("move back and forth" "moved back and forth")
        ("move in" "moved in")
        ("move in on" "moved in on")
        ("move into" "moved into")
        ("move involuntarily" "moved involuntarily")
        ("move on" "moved on")
        ("move out" "moved out")
        ("move over" "moved over")
        ("move reflexively" "moved reflexively")
        ("move through" "moved through")
        ("move up" "moved up")
        ("mow down" "mowed down")
        ("muck about" "mucked about")
        ("muck around" "mucked around")
        ("muck up" "mucked up")
        ("mud" "mudded")
        ("muddy" "muddied")
        ("muddy up" "muddied up")
        ("mug" "mugged")
        ("mug up" "mugged up")
        ("mull over" "mulled over")
        ("multiply" "multiplied")
        ("mummify" "mummified")
        ("muscle into" "muscled into")
        ("muster in" "mustered in")
        ("muster out" "mustered out")
        ("muster up" "mustered up")
        ("mutiny" "mutinied")
        ("mystify" "mystified")
        ("nab" "nabbed")
        ("nag" "nagged")
        ("nail down" "nailed down")
        ("nap" "napped")
        ("narrow down" "narrowed down")
        ("nazify" "nazified")
        ("net" "netted")
        ("net fish" "netted fish")
        ("nickel" "nickelled")
        ("nigrify" "nigrified")
        ("nip" "nipped")
        ("nip off" "nipped off")
        ("nitrify" "nitrified")
        ("nod" "nodded")
        ("nod off" "nodded off")
        ("nolle pros" "nolled pros")
        ("nolle prosequi" "nolled prosequi")
        ("nonplus" "nonplussed")
        ("nose out" "nosed out")
        ("notify" "notified")
        ("nullify" "nullified")
        ("nut" "nutted")
        ("nutrify" "nutrified")
        ("objectify" "objectified")
        ("obstinate" "obstinised")
        ("obtrude upon" "obtruded upon")
        ("occupy" "occupied")
        ("occur" "occurred")
        ("offer up" "offered up")
        ("offset" "offset")
        ("omit" "omitted")
        ("ooze out" "oozed out")
        ("ooze through" "oozed through")
        ("opacify" "opacified")
        ("open fire" "opened fire")
        ("open up" "opened up")
        ("operate on" "operated on")
        ("opt out" "opted out")
        ("ordinate" "ordinised")
        ("originate in" "originated in")
        ("ossify" "ossified")
        ("out in" "outed in")
        ("outbid" "outbid")
        ("outcrop" "outcropped")
        ("outcry" "outcried")
        ("outdo" "outdid")
        ("outfight" "outfought")
        ("outfit" "outfitted")
        ("outgo" "outwent")
        ("outgrow" "outgrew")
        ("outmanoeuvre" "outmaneuvred")
        ("outride" "outrode")
        ("outrival" "outrivalled")
        ("outrun" "outran")
        ("outsell" "outsold")
        ("outshine" "outshone")
        ("outspan" "outspanned")
        ("outstrip" "outstripped")
        ("outwear" "outwore")
        ("outwit" "outwitted")
        ("oven broil" "ovened broil")
        ("overbear" "overbore")
        ("overbid" "overbade")
        ("overblow" "overblew")
        ("overcome" "overcame")
        ("overcrop" "overcropped")
        ("overdo" "overdid")
        ("overdraw" "overdrew")
        ("overdrive" "overdrove")
        ("overeat" "overate")
        ("overexert oneself" "overexerted oneself")
        ("overfeed" "overfed")
        ("overfly" "overflew")
        ("overgrow" "overgrew")
        ("overhang" "overhung")
        ("overlap" "overlapped")
        ("overlay" "overlaid")
        ("overleap" "overleapt")
        ("overlie" "overlay")
        ("overpay" "overpaid")
        ("override" "overrode")
        ("overrun" "overran")
        ("oversee" "oversaw")
        ("overshoot" "overshot")
        ("oversimplify" "oversimplified")
        ("oversleep" "overslept")
        ("overspend" "overspent")
        ("overstep" "overstepped")
        ("oversupply" "oversupplied")
        ("overtake" "overtook")
        ("overthrow" "overthrew")
        ("overtop" "overtopped")
        ("overweary" "overwearied")
        ("overwrite" "overwrote")
        ("own up" "owned up")
        ("pacify" "pacified")
        ("pack on" "packed on")
        ("pack together" "packed together")
        ("pad" "padded")
        ("paint a picture" "painted a picture")
        ("paint the lily" "painted the lily")
        ("pair off" "paired off")
        ("pal" "palled")
        ("pal up" "palled up")
        ("palm off" "palmed off")
        ("palsy" "palsied")
        ("pan" "panned")
        ("pan off" "panned off")
        ("pan out" "panned out")
        ("pan roast" "panned roast")
        ("pan-fry" "pan-fried")
        ("panic" "panicked")
        ("paralyse" "paralyzed")
        ("parcel out" "parceled out")
        ("pare down" "pared down")
        ("parody" "parodied")
        ("parry" "parried")
        ("part with" "parted with")
        ("partake" "partook")
        ("partake in" "partook in")
        ("partition off" "partitioned off")
        ("partner off" "partnered off")
        ("party" "partied")
        ("pass across" "passed across")
        ("pass along" "passed along")
        ("pass around" "passed around")
        ("pass away" "passed away")
        ("pass by" "passed by")
        ("pass judgment" "passed judgment")
        ("pass off" "passed off")
        ("pass on" "passed on")
        ("pass out" "passed out")
        ("pass over" "passed over")
        ("pass through" "passed through")
        ("pass up" "passed up")
        ("pass water" "passed water")
        ("pass with flying colors" "passed with flying colors")
        ("pat" "patted")
        ("patch up" "patched up")
        ("patinate" "patinised")
        ("patrol" "patrolled")
        ("pay" "paid")
        ("pay back" "paid back")
        ("pay cash" "paid cash")
        ("pay for" "paid for")
        ("pay heed" "paid heed")
        ("pay off" "paid off")
        ("pay out" "paid out")
        ("pay up" "paid up")
        ("peck at" "pecked at")
        ("peel off" "peeled off")
        ("peer review" "peered review")
        ("peg" "pegged")
        ("peg away" "pegged away")
        ("peg down" "pegged down")
        ("pelt along" "pelted along")
        ("pen" "penned")
        ("pen up" "penned up")
        ("pension off" "pensioned off")
        ("pep up" "pepped up")
        ("perk up" "perked up")
        ("permit" "permitted")
        ("persist in" "persisted in")
        ("personify" "personified")
        ("pet" "petted")
        ("peter out" "petered out")
        ("petrify" "petrified")
        ("pettifog" "pettifogged")
        ("phase in" "phased in")
        ("phase out" "phased out")
        ("photocopy" "photocopied")
        ("photoset" "photoset")
        ("photostat" "photostatted")
        ("pick apart" "picked apart")
        ("pick at" "picked at")
        ("pick off" "picked off")
        ("pick out" "picked out")
        ("pick over" "picked over")
        ("pick up" "picked up")
        ("pick up the gauntlet" "picked up the gauntlet")
        ("picnic" "picnicked")
        ("piddle away" "piddled away")
        ("pig" "pigged")
        ("pig it" "pigged it")
        ("pig out" "pigged out")
        ("pile up" "piled up")
        ("pillory" "pilloried")
        ("pin" "pinned")
        ("pin down" "pinned down")
        ("pin up" "pinned up")
        ("pine away" "pined away")
        ("pinkify" "pinkified")
        ("pip" "pipped")
        ("pip out" "pipped out")
        ("pipe down" "piped down")
        ("pipe in" "piped in")
        ("pipe up" "piped up")
        ("pistol-whip" "pistolwhipped")
        ("pit" "pitted")
        ("pitch in" "pitched in")
        ("pitch into" "pitched into")
        ("pitter-patter" "pitterpattered")
        ("pity" "pitied")
        ("place down" "placed down")
        ("place upright" "placed upright")
        ("place-kick" "placekicked")
        ("plan" "planned")
        ("plank down" "planked down")
        ("plank over" "planked over")
        ("plaster over" "plastered over")
        ("plastinate" "plastinised")
        ("plat" "platted")
        ("play a joke on" "played a joke on")
        ("play a trick on" "played a trick on")
        ("play along" "played along")
        ("play around" "played around")
        ("play back" "played back")
        ("play down" "played down")
        ("play false" "played false")
        ("play hooky" "played hooky")
        ("play it by ear" "played it by ear")
        ("play off" "played off")
        ("play out" "played out")
        ("play possum" "played possum")
        ("play tricks" "played tricks")
        ("play up" "played up")
        ("plead" "pled")
        ("plod" "plodded")
        ("plonk down" "plonked down")
        ("plop" "plopped")
        ("plot" "plotted")
        ("plough on" "ploughed on")
        ("plow ahead" "plowed ahead")
        ("pluck at" "plucked at")
        ("plug" "plugged")
        ("plug away" "plugged away")
        ("plug in" "plugged in")
        ("plug into" "plugged into")
        ("plump down" "plumped down")
        ("plump for" "plumped for")
        ("plump in" "plumped in")
        ("plump out" "plumped out")
        ("plump up" "plumped up")
        ("plunk down" "plunked down")
        ("plunk for" "plunked for")
        ("ply" "plied")
        ("pod" "podded")
        ("point jam" "pointed jam")
        ("point out" "pointed out")
        ("point the way" "pointed the way")
        ("point up" "pointed up")
        ("poke at" "poked at")
        ("poke fun" "poked fun")
        ("poke into" "poked into")
        ("poke out" "poked out")
        ("polish off" "polished off")
        ("polish up" "polished up")
        ("pollenate" "pollenised")
        ("pony up" "ponied up")
        ("pooch out" "pooched out")
        ("pooh-pooh" "poohpoohed")
        ("poop out" "pooped out")
        ("pop" "popped")
        ("pop in" "popped in")
        ("pop off" "popped off")
        ("pop out" "popped out")
        ("pop the question" "popped the question")
        ("pop up" "popped up")
        ("portion out" "portioned out")
        ("portray" "portrayed")
        ("pot" "potted")
        ("potter around" "pottered around")
        ("pound off" "pounded off")
        ("pound up" "pounded up")
        ("pour cold water on" "poured cold water on")
        ("pour down" "poured down")
        ("pour forth" "poured forth")
        ("pour out" "poured out")
        ("power-dive" "powerdived")
        ("practice bundling" "practiced bundling")
        ("prang up" "pranged up")
        ("preachify" "preachified")
        ("prefer" "preferred")
        ("preoccupy" "preoccupied")
        ("prepare for" "prepared for")
        ("prepay" "prepaid")
        ("press down" "pressed down")
        ("press down on" "pressed down on")
        ("press home" "pressed home")
        ("press on" "pressed on")
        ("press out" "pressed out")
        ("pretermit" "pretermitted")
        ("prettify" "prettified")
        ("pretty up" "prettied up")
        ("preview" "prevued")
        ("prick up" "pricked up")
        ("pride oneself" "prided oneself")
        ("prim" "primmed")
        ("prim out" "primmed out")
        ("prim up" "primmed up")
        ("print over" "printed over")
        ("prod" "prodded")
        ("progress to" "progressed to")
        ("proofread" "proofread")
        ("prop" "propped")
        ("prop up" "propped up")
        ("propel" "propelled")
        ("prophesy" "prophesied")
        ("prove oneself" "proved oneself")
        ("pry" "pried")
        ("psych up" "psyched up")
        ("psychoanalyse" "psychoanalyzed")
        ("pub-crawl" "pubcrawled")
        ("puff out" "puffed out")
        ("puff up" "puffed up")
        ("pull a face" "pulled a face")
        ("pull a fast one on" "pulled a fast one on")
        ("pull ahead" "pulled ahead")
        ("pull along" "pulled along")
        ("pull at" "pulled at")
        ("pull away" "pulled away")
        ("pull back" "pulled back")
        ("pull down" "pulled down")
        ("pull in" "pulled in")
        ("pull in one's horns" "pulled in one's horns")
        ("pull off" "pulled off")
        ("pull one's weight" "pulled one's weight")
        ("pull out" "pulled out")
        ("pull out all the stops" "pulled out all the stops")
        ("pull over" "pulled over")
        ("pull round" "pulled round")
        ("pull someone's leg" "pulled someone's leg")
        ("pull strings" "pulled strings")
        ("pull the leg of" "pulled the leg of")
        ("pull the plug" "pulled the plug")
        ("pull the wool over someone's eyes"
         "pulled the wool over someone's eyes")
        ("pull through" "pulled through")
        ("pull together" "pulled together")
        ("pull up" "pulled up")
        ("pull up short" "pulled up short")
        ("pull up stakes" "pulled up stakes")
        ("pull wires" "pulled wires")
        ("pullulate with" "pullulated with")
        ("pummel" "pummelled")
        ("pun" "punned")
        ("punch in" "punched in")
        ("punch out" "punched out")
        ("pup" "pupped")
        ("purify" "purified")
        ("purl stitch" "purled stitch")
        ("push around" "pushed around")
        ("push aside" "pushed aside")
        ("push away" "pushed away")
        ("push back" "pushed back")
        ("push down" "pushed down")
        ("push forward" "pushed forward")
        ("push on" "pushed on")
        ("push out" "pushed out")
        ("push through" "pushed through")
        ("push up" "pushed up")
        ("put" "put")
        ("put across" "put across")
        ("put aside" "put aside")
        ("put away" "put away")
        ("put back" "put back")
        ("put behind bars" "put behind bars")
        ("put down" "put down")
        ("put forward" "put forward")
        ("put in" "put in")
        ("put off" "put off")
        ("put on" "put on")
        ("put on airs" "put on airs")
        ("put on the line" "put on the line")
        ("put one across" "put one across")
        ("put one over" "put one over")
        ("put out" "put out")
        ("put out feelers" "put out feelers")
        ("put over" "put over")
        ("put right" "put right")
        ("put through" "put through")
        ("put to death" "put to death")
        ("put to sleep" "put to sleep")
        ("put to work" "put to work")
        ("put together" "put together")
        ("put under" "put under")
        ("put up" "put up")
        ("putrefy" "putrefied")
        ("putter around" "puttered around")
        ("putty" "puttied")
        ("puzzle out" "puzzled out")
        ("puzzle over" "puzzled over")
        ("quail at" "quailed at")
        ("qualify" "qualified")
        ("quantify" "quantified")
        ("quarry" "quarried")
        ("queen it over" "queened it over")
        ("query" "queried")
        ("quest after" "quested after")
        ("quest for" "quested for")
        ("queue up" "queueed up")
        ("quick-freeze" "quickfrozen")
        ("quickstep" "quickstepped")
        ("quiet down" "quieted down")
        ("quip" "quipped")
        ("quit" "quit")
        ("quiz" "quizzed")
        ("rabbit on" "rabbited on")
        ("rachet up" "racheted up")
        ("rack up" "racked up")
        ("raffle off" "raffled off")
        ("rag" "ragged")
        ("rail in" "railed in")
        ("rail off" "railed off")
        ("rain buckets" "rained buckets")
        ("rain cats and dogs" "rained cats and dogs")
        ("rain down" "rained down")
        ("rain out" "rained out")
        ("raise a stink" "raised a stink")
        ("raise hell" "raised hell")
        ("raise the roof" "raised the roof")
        ("raise up" "raised up")
        ("rake in" "raked in")
        ("rake off" "raked off")
        ("rake up" "raked up")
        ("rally" "rallied")
        ("ram" "rammed")
        ("ram down" "rammed down")
        ("ram home" "rammed home")
        ("ramble on" "rambled on")
        ("ramify" "ramified")
        ("ramp up" "ramped up")
        ("range in" "ranged in")
        ("rap" "rapped")
        ("rappel" "rappelled")
        ("rarefy" "rarefied")
        ("rarify" "rarified")
        ("rat" "ratted")
        ("ratchet down" "ratcheted down")
        ("ratify" "ratified")
        ("ration out" "rationed out")
        ("rationalise away" "rationalised away")
        ("rationalize away" "rationalized away")
        ("rattle down" "rattled down")
        ("rattle off" "rattled off")
        ("rattle on" "rattled on")
        ("ravel out" "raveled out")
        ("reach into" "reached into")
        ("reach one's nostrils" "reached one's nostrils")
        ("reach out" "reached out")
        ("read" "read")
        ("read between the lines" "read between the lines")
        ("ready" "readied")
        ("rear back" "reared back")
        ("reason out" "reasoned out")
        ("rebel" "rebelled")
        ("rebuild" "rebuilt")
        ("rebury" "reburied")
        ("rebut" "rebutted")
        ("recast" "recast")
        ("reclassify" "reclassified")
        ("recommit" "recommitted")
        ("recopy" "recopied")
        ("rectify" "rectified")
        ("recur" "recurred")
        ("redo" "redid")
        ("reel off" "reeled off")
        ("refer" "referred")
        ("refit" "refitted")
        ("refuel" "refuelled")
        ("regard as" "regarded as")
        ("regret" "regretted")
        ("reify" "reified")
        ("rein in" "reined in")
        ("rejig" "rejigged")
        ("relieve oneself" "relieved oneself")
        ("rely" "relied")
        ("rely on" "relied on")
        ("rely upon" "relied upon")
        ("remain down" "remained down")
        ("remain firm" "remained firm")
        ("remake" "remade")
        ("remarry" "remarried")
        ("remedy" "remedied")
        ("remember oneself" "remembered oneself")
        ("remit" "remitted")
        ("rend" "rent")
        ("renege" "renegued")
        ("renege on" "renegued on")
        ("renegue on" "renegued on")
        ("rent out" "rented out")
        ("repay" "repaid")
        ("repel" "repelled")
        ("reply" "replied")
        ("report out" "reported out")
        ("repose on" "reposed on")
        ("repot" "repotted")
        ("reread" "reread")
        ("rerun" "reran")
        ("resell" "resold")
        ("reset" "reset")
        ("rest on" "rested on")
        ("ret" "retted")
        ("retake" "retook")
        ("retell" "retold")
        ("rethink" "rethought")
        ("retrofit" "retrofitted")
        ("retry" "retried")
        ("reunify" "reunified")
        ("rev" "revved")
        ("rev up" "revved up")
        ("revel" "revelled")
        ("revet" "revetted")
        ("revivify" "revivified")
        ("revolve about" "revolved about")
        ("revolve around" "revolved around")
        ("rewrite" "rewrote")
        ("rib" "ribbed")
        ("ricochet" "ricochetted")
        ("rid" "rid")
        ("rid of" "rid of")
        ("ride" "rode")
        ("ride away" "rode away")
        ("ride herd" "rode herd")
        ("ride horseback" "rode horseback")
        ("ride off" "rode off")
        ("ride out" "rode out")
        ("ride roughshod" "rode roughshod")
        ("ride the bench" "rode the bench")
        ("rig" "rigged")
        ("rig out" "rigged out")
        ("rig up" "rigged up")
        ("rigidify" "rigidified")
        ("rim" "rimmed")
        ("ring" "rang")
        ("ring out" "rang out")
        ("ring up" "rang up")
        ("rinse off" "rinsed off")
        ("rip" "ripped")
        ("rip off" "ripped off")
        ("rip out" "ripped out")
        ("rip up" "ripped up")
        ("rise" "rose")
        ("rise up" "rose up")
        ("rival" "rivalled")
        ("roar off" "roared off")
        ("rob" "robbed")
        ("roll around" "rolled around")
        ("roll down" "rolled down")
        ("roll in" "rolled in")
        ("roll in the hay" "rolled in the hay")
        ("roll off" "rolled off")
        ("roll out" "rolled out")
        ("roll over" "rolled over")
        ("roll up" "rolled up")
        ("roller skate" "rollered skate")
        ("root for" "rooted for")
        ("root on" "rooted on")
        ("root out" "rooted out")
        ("rope down" "roped down")
        ("rope in" "roped in")
        ("rope off" "roped off")
        ("rope up" "roped up")
        ("rot" "rotted")
        ("rough in" "roughed in")
        ("rough out" "roughed out")
        ("rough up" "roughed up")
        ("rough-cast" "rough-cast")
        ("rough-dry" "rough-dried")
        ("rough-hew" "rough-hewn")
        ("roughcast" "roughcast")
        ("round down" "rounded down")
        ("round off" "rounded off")
        ("round out" "rounded out")
        ("round up" "rounded up")
        ("rout out" "routed out")
        ("rout up" "routed up")
        ("rub" "rubbed")
        ("rub along" "rubbed along")
        ("rub down" "rubbed down")
        ("rub off" "rubbed off")
        ("rub out" "rubbed out")
        ("rubify" "rubified")
        ("ruck up" "rucked up")
        ("ruffle up" "ruffled up")
        ("rule in" "ruled in")
        ("rule out" "ruled out")
        ("run" "ran")
        ("run a risk" "ran a risk")
        ("run across" "ran across")
        ("run afoul" "ran afoul")
        ("run aground" "ran aground")
        ("run along" "ran along")
        ("run around" "ran around")
        ("run away" "ran away")
        ("run bases" "ran bases")
        ("run by" "ran by")
        ("run down" "ran down")
        ("run dry" "ran dry")
        ("run for" "ran for")
        ("run into" "ran into")
        ("run low" "ran low")
        ("run off" "ran off")
        ("run on" "ran on")
        ("run out" "ran out")
        ("run over" "ran over")
        ("run roughshod" "ran roughshod")
        ("run short" "ran short")
        ("run through" "ran through")
        ("run up" "ran up")
        ("rush along" "rushed along")
        ("rush away" "rushed away")
        ("rush off" "rushed off")
        ("rush out" "rushed out")
        ("rut" "rutted")
        ("saccharify" "saccharified")
        ("sack out" "sacked out")
        ("sack up" "sacked up")
        ("sag" "sagged")
        ("sag down" "sagged down")
        ("sail through" "sailed through")
        ("salinate" "salinised")
        ("sally forth" "sallied forth")
        ("sally out" "sallied out")
        ("salt away" "salted away")
        ("sanctify" "sanctified")
        ("sand cast" "sanded cast")
        ("sandbag" "sandbagged")
        ("sap" "sapped")
        ("saponify" "saponified")
        ("satisfy" "satisfied")
        ("save up" "saved up")
        ("savvy" "savvied")
        ("saw logs" "sawed logs")
        ("saw wood" "sawed wood")
        ("say" "said")
        ("say farewell" "said farewell")
        ("scab" "scabbed")
        ("scale down" "scaled down")
        ("scale up" "scaled up")
        ("scan" "scanned")
        ("scar" "scarred")
        ("scare away" "scared away")
        ("scare off" "scared off")
        ("scarf out" "scarfed out")
        ("scarify" "scarified")
        ("scat" "scatted")
        ("scend" "sended")
        ("scent out" "scented out")
        ("schlep" "schlepped")
        ("scoop out" "scooped out")
        ("scoop up" "scooped up")
        ("score out" "scored out")
        ("scotch tape" "scotched tape")
        ("scrag" "scragged")
        ("scram" "scrammed")
        ("scrap" "scrapped")
        ("scrape along" "scraped along")
        ("scrape by" "scraped by")
        ("scrape up" "scraped up")
        ("scratch along" "scratched along")
        ("scratch off" "scratched off")
        ("scratch out" "scratched out")
        ("scratch up" "scratched up")
        ("screen off" "screened off")
        ("screen out" "screened out")
        ("screw up" "screwed up")
        ("scrub" "scrubbed")
        ("scrub up" "scrubbed up")
        ("scrunch up" "scrunched up")
        ("scry" "scried")
        ("scud" "scudded")
        ("scum" "scummed")
        ("scurry" "scurried")
        ("seal in" "sealed in")
        ("seal off" "sealed off")
        ("secernate" "secernised")
        ("second-guess" "secondguessed")
        ("see" "saw")
        ("see double" "saw double")
        ("see eye to eye" "saw eye to eye")
        ("see red" "saw red")
        ("see the light" "saw the light")
        ("see through" "saw through")
        ("see to it" "saw to it")
        ("seek" "sought")
        ("seek out" "sought out")
        ("seize on" "seized on")
        ("seize with teeth" "seized with teeth")
        ("sell" "sold")
        ("sell off" "sold off")
        ("sell out" "sold out")
        ("sell short" "sold short")
        ("sell up" "sold up")
        ("send" "sent")
        ("send a message" "sent a message")
        ("send around" "sent around")
        ("send away" "sent away")
        ("send back" "sent back")
        ("send down" "sent down")
        ("send for" "sent for")
        ("send in" "sent in")
        ("send off" "sent off")
        ("send on" "sent on")
        ("send out" "sent out")
        ("send packing" "sent packing")
        ("send word" "sent word")
        ("sensify" "sensified")
        ("separate off" "separated off")
        ("separate out" "separated out")
        ("serve up" "served up")
        ("serve well" "served well")
        ("set" "set")
        ("set ablaze" "set ablaze")
        ("set about" "set about")
        ("set afire" "set afire")
        ("set aflame" "set aflame")
        ("set ahead" "set ahead")
        ("set apart" "set apart")
        ("set ashore" "set ashore")
        ("set aside" "set aside")
        ("set back" "set back")
        ("set down" "set down")
        ("set forth" "set forth")
        ("set free" "set free")
        ("set in" "set in")
        ("set in motion" "set in motion")
        ("set off" "set off")
        ("set on" "set on")
        ("set on fire" "set on fire")
        ("set out" "set out")
        ("set to music" "set to music")
        ("set up" "set up")
        ("set upon" "set upon")
        ("settle down" "settled down")
        ("settle on" "settled on")
        ("sew together" "sewed together")
        ("sex up" "sexed up")
        ("shack up" "shacked up")
        ("shade off" "shaded off")
        ("shag" "shagged")
        ("shake" "shook")
        ("shake hands" "shook hands")
        ("shake off" "shook off")
        ("shake up" "shook up")
        ("sham" "shammed")
        ("shape up" "shaped up")
        ("she-bop" "she-bopped")
        ("shed" "shed")
        ("shed blood" "shed blood")
        ("shed light on" "shed light on")
        ("shell out" "shelled out")
        ("shell stitch" "shelled stitch")
        ("shellac" "shellacked")
        ("shillyshally" "shillyshallied")
        ("shimmy" "shimmied")
        ("shin" "shinned")
        ("shine" "shone")
        ("shine at" "shone at")
        ("shine up" "shone up")
        ("shinny" "shinnied")
        ("ship" "shipped")
        ("shit" "shat")
        ("shoe" "shod")
        ("shoo away" "shooed away")
        ("shoo off" "shooed off")
        ("shoot" "shot")
        ("shoot a line" "shot a line")
        ("shoot craps" "shot craps")
        ("shoot down" "shot down")
        ("shoot for" "shot for")
        ("shoot one's mouth off" "shot one's mouth off")
        ("shoot the breeze" "shot the breeze")
        ("shoot up" "shot up")
        ("shop" "shopped")
        ("shop at" "shopped at")
        ("shore up" "shored up")
        ("short-change" "shortchanged")
        ("short-circuit" "shortcircuited")
        ("shotgun" "shutgunned")
        ("shoulder in" "shouldered in")
        ("shout down" "shouted down")
        ("shout out" "shouted out")
        ("shove along" "shoved along")
        ("shove off" "shoved off")
        ("shovel in" "shoveled in")
        ("show off" "showed off")
        ("show the door" "showed the door")
        ("show up" "showed up")
        ("shower down" "showered down")
        ("shred" "shred")
        ("shrink" "shrank")
        ("shrink back" "shrank back")
        ("shrink from" "shrank from")
        ("shrinkwrap" "shrinkwrapped")
        ("shrive" "shrove")
        ("shrivel up" "shriveled up")
        ("shrug" "shrugged")
        ("shrug off" "shrugged off")
        ("shun" "shunned")
        ("shut" "shut")
        ("shut away" "shut away")
        ("shut down" "shut down")
        ("shut in" "shut in")
        ("shut off" "shut off")
        ("shut one's mouth" "shut one's mouth")
        ("shut out" "shut out")
        ("shut up" "shut up")
        ("shy" "shied")
        ("shy away from" "shied away from")
        ("sic" "sicked")
        ("side-slip" "sideslipped")
        ("sidestep" "sidestepped")
        ("sidle up" "sidled up")
        ("sieve out" "sieved out")
        ("sight-read" "sight-read")
        ("sight-sing" "sight-sang")
        ("sightread" "sightread")
        ("sightsee" "sightsaw")
        ("sightsing" "sightsang")
        ("sign away" "signed away")
        ("sign in" "signed in")
        ("sign off" "signed off")
        ("sign on" "signed on")
        ("sign over" "signed over")
        ("sign up" "signed up")
        ("signify" "signified")
        ("silt up" "silted up")
        ("simmer down" "simmered down")
        ("simplify" "simplified")
        ("sin" "sinned")
        ("sing" "sung")
        ("sing along" "sung along")
        ("single crochet" "singled crochet")
        ("single out" "singled out")
        ("single stitch" "singled stitch")
        ("single-foot" "singlefooted")
        ("sink" "sank")
        ("sink in" "sank in")
        ("sip" "sipped")
        ("siphon" "syphoned")
        ("siphon off" "syphoned off")
        ("sit" "sat")
        ("sit around" "sat around")
        ("sit back" "sat back")
        ("sit by" "sat by")
        ("sit down" "sat down")
        ("sit in" "sat in")
        ("sit out" "sat out")
        ("sit tight" "sat tight")
        ("sit up" "sat up")
        ("size up" "sized up")
        ("skate over" "skated over")
        ("sken" "skenned")
        ("skid" "skidded")
        ("skim" "skimmed")
        ("skim off" "skimmed off")
        ("skim over" "skimmed over")
        ("skimp over" "skimped over")
        ("skin" "skinned")
        ("skin over" "skinned over")
        ("skin pop" "skinned pop")
        ("skinny-dip" "skinnydipped")
        ("skip" "skipped")
        ("skip over" "skipped over")
        ("skip town" "skipped town")
        ("skunk" "skunksed")
        ("sky" "skied")
        ("sky dive" "skied dive")
        ("slab" "slabbed")
        ("slack off" "slacked off")
        ("slack up" "slacked up")
        ("slacken off" "slackened off")
        ("slag" "slagged")
        ("slam" "slammed")
        ("slam dance" "slammed dance")
        ("slam on" "slammed on")
        ("slap" "slapped")
        ("slap on" "slapped on")
        ("slap together" "slapped together")
        ("slat" "slatted")
        ("sled" "sledded")
        ("sleek down" "sleeked down")
        ("sleek over" "sleeked over")
        ("sleep" "slept")
        ("sleep around" "slept around")
        ("sleep in" "slept in")
        ("sleep late" "slept late")
        ("sleep off" "slept off")
        ("sleep out" "slept out")
        ("sleep over" "slept over")
        ("sleep together" "slept together")
        ("sleep with" "slept with")
        ("slice into" "sliced into")
        ("slice through" "sliced through")
        ("slice up" "sliced up")
        ("slick down" "slicked down")
        ("slick up" "slicked up")
        ("slide" "slid")
        ("slide by" "slid by")
        ("slide down" "slid down")
        ("slim down" "slimed down")
        ("sling" "slung")
        ("slink" "slunk")
        ("slip" "slipped")
        ("slip away" "slipped away")
        ("slip by" "slipped by")
        ("slip in" "slipped in")
        ("slip off" "slipped off")
        ("slip on" "slipped on")
        ("slip one's mind" "slipped one's mind")
        ("slip up" "slipped up")
        ("slit" "slit")
        ("slobber over" "slobbered over")
        ("slog" "slogged")
        ("slop" "slopped")
        ("slosh around" "sloshed around")
        ("slot" "slotted")
        ("slough off" "sloughed off")
        ("slow down" "slowed down")
        ("slow up" "slowed up")
        ("slug" "slugged")
        ("sluice down" "sluiced down")
        ("slum" "slummed")
        ("slur" "slurred")
        ("slur over" "slurred over")
        ("slush around" "slushed around")
        ("smarten up" "smartened up")
        ("smash up" "smashed up")
        ("smell out" "smelled out")
        ("smell up" "smelled up")
        ("smite" "smote")
        ("smoke out" "smoked out")
        ("smooth out" "smoothed out")
        ("smooth over" "smoothed over")
        ("smut" "smutted")
        ("snag" "snagged")
        ("snap" "snapped")
        ("snap at" "snapped at")
        ("snap back" "snapped back")
        ("snap off" "snapped off")
        ("snap up" "snapped up")
        ("snarl up" "snarled up")
        ("snatch up" "snatched up")
        ("sneak" "snuck")
        ("sneak away" "snuck away")
        ("sneak in" "snuck in")
        ("sneak off" "snuck off")
        ("sneak out" "snuck out")
        ("sneak up" "snuck up")
        ("sniff out" "sniffed out")
        ("snip" "snipped")
        ("snip off" "snipped off")
        ("snivel" "snivelled")
        ("snog" "snogged")
        ("snub" "snubbed")
        ("snuff it" "snuffed it")
        ("snuff out" "snuffed out")
        ("soak through" "soaked through")
        ("soak up" "soaked up")
        ("soar up" "soared up")
        ("soar upwards" "soared upwards")
        ("sob" "sobbed")
        ("sober up" "sobered up")
        ("sod" "sodded")
        ("soft-pedal" "softpedalled")
        ("soft-soap" "softsoaped")
        ("solidify" "solidified")
        ("sop" "sopped")
        ("sop up" "sopped up")
        ("sort out" "sorted out")
        ("sound off" "sounded off")
        ("sound out" "sounded out")
        ("soup up" "souped up")
        ("sow in" "sowed in")
        ("sow one's oats" "sowed one's oats")
        ("sow one's wild oats" "sowed one's wild oats")
        ("span" "spanned")
        ("spar" "sparred")
        ("spark off" "sparked off")
        ("spat" "spatted")
        ("speak" "spoke")
        ("speak for" "spoke for")
        ("speak in tongues" "spoke in tongues")
        ("speak of the devil" "spoke of the devil")
        ("speak out" "spoke out")
        ("speak up" "spoke up")
        ("spear up" "speared up")
        ("specify" "specified")
        ("speechify" "speechified")
        ("speed" "sped")
        ("speed up" "sped up")
        ("spell out" "spelled out")
        ("spellbind" "spellbound")
        ("spend" "spent")
        ("spend a penny" "spent a penny")
        ("spew out" "spewed out")
        ("spice up" "spiced up")
        ("spiel off" "spieled off")
        ("spiff up" "spiffed up")
        ("spike out" "spiked out")
        ("spill out" "spilled out")
        ("spill over" "spilled over")
        ("spill the beans" "spilled the beans")
        ("spin" "spun")
        ("spin around" "spun around")
        ("spin off" "spun off")
        ("spin out" "spun out")
        ("spin-dry" "spin-dried")
        ("spiral" "spiralled")
        ("spirit away" "spirited away")
        ("spirit off" "spirited off")
        ("spirit up" "spirited up")
        ("spit" "spat")
        ("spit out" "spat out")
        ("spit up" "spat up")
        ("splash around" "splashed around")
        ("split" "split")
        ("split up" "split up")
        ("sponge down" "sponged down")
        ("sponge off" "sponged off")
        ("sponge on" "sponged on")
        ("sponge up" "sponged up")
        ("spot" "spotted")
        ("spot jam" "spot jammed")
        ("spotlight" "spotlit")
        ("spray-dry" "spray-dried")
        ("spread" "spread")
        ("spread out" "spread out")
        ("spread over" "spread over")
        ("spread-eagle" "spreadeagled")
        ("spring" "sprang")
        ("spring to mind" "sprang to mind")
        ("spring up" "sprang up")
        ("spruce up" "spruced up")
        ("spud" "spudded")
        ("spur" "spurred")
        ("spy" "spied")
        ("square away" "squared away")
        ("square dance" "squared dance")
        ("square off" "squared off")
        ("square up" "squared up")
        ("squat" "squatted")
        ("squeak by" "squeaked by")
        ("squeak through" "squeaked through")
        ("squeeze by" "squeezed by")
        ("squeeze for" "squeezed for")
        ("squeeze out" "squeezed out")
        ("squirrel away" "squirreled away")
        ("stab" "stabbed")
        ("stack away" "stacked away")
        ("stack up" "stacked up")
        ("stage direct" "staged direct")
        ("stall-feed" "stall-fed")
        ("stamp down" "stamped down")
        ("stamp out" "stamped out")
        ("stand" "stood")
        ("stand back" "stood back")
        ("stand by" "stood by")
        ("stand fast" "stood fast")
        ("stand firm" "stood firm")
        ("stand for" "stood for")
        ("stand guard" "stood guard")
        ("stand in" "stood in")
        ("stand out" "stood out")
        ("stand pat" "stood pat")
        ("stand sentinel" "stood sentinel")
        ("stand still" "stood still")
        ("stand up" "stood up")
        ("stand watch" "stood watch")
        ("star" "starred")
        ("stare down" "stared down")
        ("start out" "started out")
        ("start up" "started up")
        ("stash away" "stashed away")
        ("stave in" "staved in")
        ("stave off" "staved off")
        ("stay at" "stayed at")
        ("stay away" "stayed away")
        ("stay fresh" "stayed fresh")
        ("stay in place" "stayed in place")
        ("stay off" "stayed off")
        ("stay on" "stayed on")
        ("stay over" "stayed over")
        ("stay put" "stayed put")
        ("stay together" "stayed together")
        ("stay up" "stayed up")
        ("steady" "steadied")
        ("steady down" "steadied down")
        ("steal" "stole")
        ("steal away" "stole away")
        ("steam clean" "steamed clean")
        ("steel oneself against" "steeled oneself against")
        ("steel onself for" "steeled onself for")
        ("stem" "stemmed")
        ("step" "stepped")
        ("step down" "stepped down")
        ("step forward" "stepped forward")
        ("step in" "stepped in")
        ("step on" "stepped on")
        ("step on it" "stepped on it")
        ("step out" "stepped out")
        ("step to the fore" "stepped to the fore")
        ("step up" "stepped up")
        ("stet" "stetted")
        ("stick" "stuck")
        ("stick about" "stuck about")
        ("stick around" "stuck around")
        ("stick by" "stuck by")
        ("stick in" "stuck in")
        ("stick on" "stuck on")
        ("stick out" "stuck out")
        ("stick to" "stuck to")
        ("stick together" "stuck together")
        ("stick up" "stuck up")
        ("stick with" "stuck with")
        ("still-hunt" "stillhunted")
        ("sting" "stung")
        ("stink" "stunk")
        ("stink out" "stunk out")
        ("stink up" "stunk up")
        ("stir" "stirred")
        ("stir fry" "stirred fry")
        ("stir up" "stirred up")
        ("stock up" "stocked up")
        ("stoop to" "stooped to")
        ("stop" "stopped")
        ("stop dead" "stopped dead")
        ("stop over" "stopped over")
        ("stop up" "stopped up")
        ("stow away" "stowed away")
        ("straighten out" "straightened out")
        ("straighten up" "straightened up")
        ("strap" "strapped")
        ("stratify" "stratified")
        ("stretch along" "stretched along")
        ("stretch forth" "stretched forth")
        ("stretch out" "stretched out")
        ("stride" "strode")
        ("strike" "struck")
        ("strike a blow" "struck a blow")
        ("strike a chord" "struck a chord")
        ("strike a note" "struck a note")
        ("strike back" "struck back")
        ("strike down" "struck down")
        ("strike dumb" "struck dumb")
        ("strike hard" "struck hard")
        ("strike home" "struck home")
        ("strike off" "struck off")
        ("strike out" "struck out")
        ("strike up" "struck up")
        ("string" "strung")
        ("string along" "strung along")
        ("string out" "strung out")
        ("string up" "strung up")
        ("strip" "stripped")
        ("strip down" "stripped down")
        ("strip mine" "stripped mine")
        ("strive" "strove")
        ("strong-arm" "strongarmed")
        ("strop" "stropped")
        ("strum" "strummed")
        ("strut" "strutted")
        ("stub" "stubbed")
        ("stub out" "stubbed out")
        ("stud" "studded")
        ("study" "studied")
        ("stultify" "stultified")
        ("stun" "stunned")
        ("stupefy" "stupefied")
        ("stymy" "stymied")
        ("sub" "subbed")
        ("sublet" "sublet")
        ("submit" "submitted")
        ("subscribe to" "subscribed to")
        ("suck in" "sucked in")
        ("suck out" "sucked out")
        ("suck up" "sucked up")
        ("sully" "sullied")
        ("sum" "summed")
        ("sum up" "summed up")
        ("sun" "sunned")
        ("sup" "supped")
        ("supervene upon" "supervened upon")
        ("supply" "supplied")
        ("surface mine" "surfaced mine")
        ("suss out" "sussed out")
        ("swab" "swabbed")
        ("swag" "swagged")
        ("swallow up" "swallowed up")
        ("swap" "swapped")
        ("swat" "swatted")
        ("swear" "swore")
        ("swear in" "swore in")
        ("swear off" "swore off")
        ("swear out" "swore out")
        ("sweat" "sweat")
        ("sweat off" "sweat off")
        ("sweep" "swept")
        ("sweep away" "swept away")
        ("sweep off" "swept off")
        ("sweep over" "swept over")
        ("sweep through" "swept through")
        ("sweep under the rug" "swept under the rug")
        ("sweep up" "swept up")
        ("sweet-talk" "sweettalked")
        ("swell up" "swelled up")
        ("swig" "swigged")
        ("swill down" "swilled down")
        ("swim" "swam")
        ("swing" "swang")
        ("swing about" "swang about")
        ("swing around" "swang around")
        ("swing out" "swang out")
        ("swing over" "swang over")
        ("switch off" "switched off")
        ("switch on" "switched on")
        ("switch over" "switched over")
        ("swob" "swobbed")
        ("swoop up" "swooped up")
        ("swop" "swopped")
        ("swot" "swotted")
        ("swot up" "swotted up")
        ("syllabify" "syllabified")
        ("sympathize with" "sympathized with")
        ("tack on" "tacked on")
        ("tack together" "tacked together")
        ("tag" "tagged")
        ("tag along" "tagged along")
        ("tag on" "tagged on")
        ("tail" "tawayed")
        ("take" "took")
        ("take a bow" "took a bow")
        ("take a breath" "took a breath")
        ("take a breather" "took a breather")
        ("take a chance" "took a chance")
        ("take a crap" "took a crap")
        ("take a dare" "took a dare")
        ("take a dive" "took a dive")
        ("take a firm stand" "took a firm stand")
        ("take a hit" "took a hit")
        ("take a hop" "took a hop")
        ("take a joke" "took a joke")
        ("take a leak" "took a leak")
        ("take a look" "took a look")
        ("take a powder" "took a powder")
        ("take a shit" "took a shit")
        ("take aback" "took aback")
        ("take account" "took account")
        ("take advantage" "took advantage")
        ("take after" "took after")
        ("take aim" "took aim")
        ("take apart" "took apart")
        ("take arms" "took arms")
        ("take away" "took away")
        ("take back" "took back")
        ("take care" "took care")
        ("take chances" "took chances")
        ("take charge" "took charge")
        ("take control" "took control")
        ("take down" "took down")
        ("take effect" "took effect")
        ("take exception" "took exception")
        ("take fire" "took fire")
        ("take five" "took five")
        ("take flight" "took flight")
        ("take for" "took for")
        ("take for granted" "took for granted")
        ("take form" "took form")
        ("take heart" "took heart")
        ("take heed" "took heed")
        ("take hold" "took hold")
        ("take hold of" "took hold of")
        ("take home" "took home")
        ("take in" "took in")
        ("take in charge" "took in charge")
        ("take in vain" "took in vain")
        ("take in water" "took in water")
        ("take into account" "took into account")
        ("take issue" "took issue")
        ("take it easy" "took it easy")
        ("take it on the chin" "took it on the chin")
        ("take kindly to" "took kindly to")
        ("take leave" "took leave")
        ("take lying down" "took lying down")
        ("take note" "took note")
        ("take notice" "took notice")
        ("take off" "took off")
        ("take office" "took office")
        ("take on" "took on")
        ("take one's lumps" "took one's lumps")
        ("take orders" "took orders")
        ("take out" "took out")
        ("take over" "took over")
        ("take pains" "took pains")
        ("take part" "took part")
        ("take place" "took place")
        ("take pride" "took pride")
        ("take root" "took root")
        ("take shape" "took shape")
        ("take stage" "took stage")
        ("take stock" "took stock")
        ("take ten" "took ten")
        ("take the air" "took the air")
        ("take the bull by the horns" "took the bull by the horns")
        ("take the cake" "took the cake")
        ("take the count" "took the count")
        ("take the field" "took the field")
        ("take the fifth" "took the fifth")
        ("take the fifth amendment" "took the fifth amendment")
        ("take the floor" "took the floor")
        ("take the road" "took the road")
        ("take the stage" "took the stage")
        ("take the stand" "took the stand")
        ("take the veil" "took the veil")
        ("take time by the forelock" "took time by the forelock")
        ("take time off" "took time off")
        ("take to" "took to")
        ("take to be" "took to be")
        ("take to heart" "took to heart")
        ("take to task" "took to task")
        ("take to the woods" "took to the woods")
        ("take turns" "took turns")
        ("take up" "took up")
        ("take up arms" "took up arms")
        ("take water" "took water")
        ("talk about" "talked about")
        ("talk down" "talked down")
        ("talk into" "talked into")
        ("talk of" "talked of")
        ("talk out of" "talked out of")
        ("talk over" "talked over")
        ("talk shop" "talked shop")
        ("talk terms" "talked terms")
        ("talk through one's hat" "talked through one's hat")
        ("talk turkey" "talked turkey")
        ("tally" "tallied")
        ("tamp down" "tamped down")
        ("tan" "tanned")
        ("tangle with" "tangled with")
        ("tap" "tapped")
        ("tap dance" "tapped dance")
        ("tap out" "tapped out")
        ("tape record" "taped record")
        ("taper off" "tapered off")
        ("tar" "tarred")
        ("tar-and-feather" "tar and-feathered")
        ("tarmac" "tarmacked")
        ("tarry" "tarried")
        ("tart up" "tarted up")
        ("tat" "tatted")
        ("teach" "taught")
        ("team up" "teamed up")
        ("tear" "tore")
        ("tear apart" "tore apart")
        ("tear away" "tore away")
        ("tear down" "tore down")
        ("tear into" "tore into")
        ("tear off" "tore off")
        ("tear up" "tore up")
        ("tease apart" "teased apart")
        ("tee off" "teed off")
        ("tee up" "teed up")
        ("teem in" "teemed in")
        ("telecast" "telecasteded")
        ("tell" "told")
        ("tell apart" "told apart")
        ("tell off" "told off")
        ("tell on" "told on")
        ("tense up" "tensed up")
        ("terrify" "terrified")
        ("test drive" "tested drive")
        ("test fly" "tested fly")
        ("testify" "testified")
        ("thin" "thinned")
        ("thin out" "thinned out")
        ("think" "thought")
        ("think about" "thought about")
        ("think back" "thought back")
        ("think of" "thought of")
        ("think out" "thought out")
        ("think over" "thought over")
        ("think the world of" "thought the world of")
        ("think twice" "thought twice")
        ("think up" "thought up")
        ("thrash about" "thrashed about")
        ("thrash out" "thrashed out")
        ("thresh about" "threshed about")
        ("throb" "throbbed")
        ("throw" "threw")
        ("throw a fit" "threw a fit")
        ("throw away" "threw away")
        ("throw back" "threw back")
        ("throw cold water on" "threw cold water on")
        ("throw in" "threw in")
        ("throw in the towel" "threw in the towel")
        ("throw off" "threw off")
        ("throw out" "threw out")
        ("throw out of kilter" "threw out of kilter")
        ("throw overboard" "threw overboard")
        ("throw together" "threw together")
        ("throw up" "threw up")
        ("thrum" "thrummed")
        ("thrust" "thrust")
        ("thrust ahead" "thrust ahead")
        ("thrust out" "thrust out")
        ("thud" "thudded")
        ("thump out" "thumped out")
        ("thurify" "thurified")
        ("tick off" "ticked off")
        ("tick over" "ticked over")
        ("tickle pink" "tickled pink")
        ("tide over" "tided over")
        ("tidy" "tidied")
        ("tidy up" "tidied up")
        ("tie down" "tied down")
        ("tie in" "tied in")
        ("tie up" "tied up")
        ("tighten one's belt" "tightened one's belt")
        ("tighten up" "tightened up")
        ("tin" "tinned")
        ("tip" "tipped")
        ("tip off" "tipped off")
        ("tip over" "tipped over")
        ("tire out" "tired out")
        ("tittle-tattle" "tittletattled")
        ("toady" "toadied")
        ("toe the line" "toed the line")
        ("tog" "togged")
        ("tog out" "togged out")
        ("tog up" "togged up")
        ("tone down" "toned down")
        ("tone up" "toned up")
        ("tool around" "tooled around")
        ("top" "topped")
        ("top off" "topped off")
        ("top out" "topped out")
        ("toss away" "tossed away")
        ("toss back" "tossed back")
        ("toss in" "tossed in")
        ("toss off" "tossed off")
        ("toss out" "tossed out")
        ("tot" "totted")
        ("tot up" "totted up")
        ("tote up" "toted up")
        ("touch a chord" "touched a chord")
        ("touch base" "touched base")
        ("touch down" "touched down")
        ("touch off" "touched off")
        ("touch on" "touched on")
        ("touch up" "touched up")
        ("touch-type" "touchtyped")
        ("towel" "towelled")
        ("toy with" "toyed with")
        ("track down" "tracked down")
        ("trade in" "traded in")
        ("traffic" "trafficked")
        ("tram" "trammed")
        ("tramp down" "tramped down")
        ("transaminate" "transaminised")
        ("transfer" "transferred")
        ("transmit" "transmitted")
        ("transmogrify" "transmogrified")
        ("transship" "transshipped")
        ("trap" "trapped")
        ("travel along" "traveled along")
        ("travel by" "traveled by")
        ("travel purposefully" "traveled purposefully")
        ("travel rapidly" "traveled rapidly")
        ("travel to" "traveled to")
        ("travesty" "travestied")
        ("tread" "trod")
        ("tread down" "trod down")
        ("tread on" "trod on")
        ("trek" "trekked")
        ("trepan" "trepanned")
        ("trice up" "triced up")
        ("trick out" "tricked out")
        ("trick up" "tricked up")
        ("trifle away" "trifled away")
        ("trigger off" "triggered off")
        ("trim" "trimmed")
        ("trim back" "trimmed back")
        ("trim down" "trimmed down")
        ("trip" "tripped")
        ("trip out" "tripped out")
        ("trip the light fantastic" "tripped the light fantastic")
        ("trip up" "tripped up")
        ("trot" "trotted")
        ("trot out" "trotted out")
        ("trouble oneself" "troubled oneself")
        ("trowel" "trowelled")
        ("true up" "trued up")
        ("trump out" "trumped out")
        ("trump up" "trumped up")
        ("try" "tried")
        ("try for" "tried for")
        ("try on" "tried on")
        ("try out" "tried out")
        ("tuck away" "tucked away")
        ("tuck in" "tucked in")
        ("tucker out" "tuckered out")
        ("tug" "tugged")
        ("tumble dry" "tumbled dry")
        ("tumefy" "tumefied")
        ("tump over" "tumped over")
        ("tune in" "tuned in")
        ("tune up" "tuned up")
        ("turf out" "turfed out")
        ("turn a blind eye" "turned a blind eye")
        ("turn a loss" "turned a loss")
        ("turn a nice dime" "turned a nice dime")
        ("turn a nice dollar" "turned a nice dollar")
        ("turn a nice penny" "turned a nice penny")
        ("turn a profit" "turned a profit")
        ("turn a trick" "turned a trick")
        ("turn around" "turned around")
        ("turn away" "turned away")
        ("turn back" "turned back")
        ("turn down" "turned down")
        ("turn in" "turned in")
        ("turn off" "turned off")
        ("turn on" "turned on")
        ("turn on a dime" "turned on a dime")
        ("turn one's stomach" "turned one's stomach")
        ("turn out" "turned out")
        ("turn over" "turned over")
        ("turn tail" "turned tail")
        ("turn the tables" "turned the tables")
        ("turn the tide" "turned the tide")
        ("turn thumbs down" "turned thumbs down")
        ("turn to" "turned to")
        ("turn turtle" "turned turtle")
        ("turn up" "turned up")
        ("turn up the heat" "turned up the heat")
        ("turn up the pressure" "turned up the pressure")
        ("tut" "tutted")
        ("tut-tut" "tut-tutted")
        ("twig" "twigged")
        ("twin" "twinned")
        ("twist around" "twisted around")
        ("twit" "twitted")
        ("two-time" "twotimed")
        ("typecast" "typecast")
        ("typeset" "typeset")
        ("typewrite" "typewrote")
        ("typify" "typified")
        ("uglify" "uglified")
        ("unbar" "unbarred")
        ("unbend" "unbent")
        ("unbind" "unbound")
        ("unclog" "unclogged")
        ("underbid" "underbid")
        ("underbuy" "underbought")
        ("undercut" "undercut")
        ("undergo" "underwent")
        ("underlay" "underlaid")
        ("underlete" "underlet")
        ("underlie" "underlay")
        ("underpay" "underpaid")
        ("underpin" "underpinned")
        ("undersell" "undersold")
        ("undershoot" "undershot")
        ("underspend" "underspent")
        ("understand" "understood")
        ("understudy" "understudied")
        ("undertake" "undertook")
        ("underwrite" "underwrote")
        ("undo" "undid")
        ("unfit" "unfitted")
        ("unfreeze" "unfroze")
        ("unify" "unified")
        ("unlearn" "unlearnt")
        ("unloose" "unloosened")
        ("unmake" "unmade")
        ("unman" "unmanned")
        ("unpin" "unpinned")
        ("unplug" "unplugged")
        ("unravel" "unravelled")
        ("unsanctify" "unsanctified")
        ("unsay" "unsaid")
        ("unsnap" "unsnapped")
        ("unstring" "unstrung")
        ("unteach" "untaught")
        ("unwind" "unwound")
        ("unwrap" "unwrapped")
        ("unzip" "unzipped")
        ("up" "upped")
        ("uphold" "upheld")
        ("uprise" "uprose")
        ("upset" "upset")
        ("urge on" "urged on")
        ("use up" "used up")
        ("usher in" "ushered in")
        ("usher out" "ushered out")
        ("vamp up" "vamped up")
        ("vary" "varied")
        ("vege out" "veged out")
        ("verbify" "verbified")
        ("verify" "verified")
        ("versify" "versified")
        ("vet" "vetted")
        ("view as" "viewed as")
        ("vilify" "vilified")
        ("vinify" "vinified")
        ("vitrify" "vitrified")
        ("vivify" "vivified")
        ("vomit up" "vomited up")
        ("vote down" "voted down")
        ("vote in" "voted in")
        ("vote out" "voted out")
        ("wad" "wadded")
        ("wag" "wagged")
        ("wait on" "waited on")
        ("wake" "woke")
        ("wake up" "woke up")
        ("walk about" "walked about")
        ("walk around" "walked around")
        ("walk away" "walked away")
        ("walk in" "walked in")
        ("walk off" "walked off")
        ("walk on air" "walked on air")
        ("walk out" "walked out")
        ("walk out of" "walked out of")
        ("walk over" "walked over")
        ("walk through" "walked through")
        ("wall in" "walled in")
        ("wall up" "walled up")
        ("waltz around" "waltzed around")
        ("wan" "wanned")
        ("wanton away" "wantoned away")
        ("war" "warred")
        ("ward off" "warded off")
        ("warm the bench" "warmed the bench")
        ("warm to" "warmed to")
        ("warm up" "warmed up")
        ("wash away" "washed away")
        ("wash down" "washed down")
        ("wash off" "washed off")
        ("wash one's hands" "washed one's hands")
        ("wash out" "washed out")
        ("wash up" "washed up")
        ("waste one's time" "wasted one's time")
        ("watch out" "watched out")
        ("watch over" "watched over")
        ("water down" "watered down")
        ("water ski" "watered ski")
        ("wave off" "waved off")
        ("waylay" "waylaid")
        ("wear" "wore")
        ("wear away" "wore away")
        ("wear down" "wore down")
        ("wear off" "wore off")
        ("wear on" "wore on")
        ("wear out" "wore out")
        ("wear round" "wore round")
        ("wear ship" "wore ship")
        ("wear the trousers" "wore the trousers")
        ("wear thin" "wore thin")
        ("wear upon" "wore upon")
        ("weary" "wearied")
        ("weave" "wove")
        ("web" "webbed")
        ("wed" "wed")
        ("weed out" "weeded out")
        ("weep" "wept")
        ("weigh anchor" "weighed anchor")
        ("weigh down" "weighed down")
        ("weigh on" "weighed on")
        ("weigh the anchor" "weighed the anchor")
        ("weight down" "weighted down")
        ("well out" "welled out")
        ("well over" "welled over")
        ("well up" "welled up")
        ("wet" "wet")
        ("wet nurse" "wet nursed")
        ("wham" "whammed")
        ("wheel around" "wheeled around")
        ("whet" "whetted")
        ("while away" "whiled away")
        ("whinny" "whinnied")
        ("whip" "whipped")
        ("whip through" "whipped through")
        ("whip up" "whipped up")
        ("whir" "whirred")
        ("whirl around" "whirled around")
        ("whisk away" "whisked away")
        ("whisk by" "whisked by")
        ("whisk off" "whisked off")
        ("white out" "whited out")
        ("whittle away" "whittled away")
        ("whittle down" "whittled down")
        ("whiz" "whizzed")
        ("whizz along" "whizzed along")
        ("whomp up" "whomped up")
        ("whoop it up" "whooped it up")
        ("whoosh" "wooshed")
        ("whop" "whopped")
        ("wigwag" "wigwagged")
        ("win" "won")
        ("win back" "won back")
        ("win over" "won over")
        ("wind" "wound")
        ("wind off" "wound off")
        ("wind up" "wound up")
        ("window-shop" "windowshopped")
        ("wine and dine" "wined and dine")
        ("wink at" "winked at")
        ("winkle out" "winkled out")
        ("winnow out" "winnowed out")
        ("wipe away" "wiped away")
        ("wipe off" "wiped off")
        ("wipe out" "wiped out")
        ("wipe up" "wiped up")
        ("wise up" "wised up")
        ("wish well" "wished well")
        ("withdraw" "withdrew")
        ("withhold" "withheld")
        ("withstand" "withstood")
        ("wolf down" "wolfed down")
        ("wolf-whistle" "wolfwhistled")
        ("work at" "worked at")
        ("work in" "worked in")
        ("work off" "worked off")
        ("work on" "worked on")
        ("work out" "worked out")
        ("work over" "worked over")
        ("work through" "worked through")
        ("work up" "worked up")
        ("worry" "worried")
        ("wrap" "wrapped")
        ("wrap up" "wrapped up")
        ("wring" "wrung")
        ("wring from" "wrung from")
        ("wring out" "wrung out")
        ("write" "wrote")
        ("write about" "wrote about")
        ("write copy" "wrote copy")
        ("write down" "wrote down")
        ("write in" "wrote in")
        ("write in code" "wrote in code")
        ("write of" "wrote of")
        ("write off" "wrote off")
        ("write on" "wrote on")
        ("write out" "wrote out")
        ("write up" "wrote up")
        ("yack away" "yacked away")
        ("yap" "yapped")
        ("yap away" "yapped away")
        ("yield up" "yielded up")
        ("zap" "zapped")
        ("zero in" "zeroed in")
        ("zigzag" "zigzagged")
        ("zip" "zipped")
        ("zip by" "zipped by")
        ("zip up" "zipped up")
        ("zonk out" "zonked out")
        ("zoom along" "zoomed along")
        ("zoom in" "zoomed in")))


(defun simple-past (verb-string)
  (cond ((gethash (string-downcase verb-string) *simple-pasts*)
         (string-upcase (gethash (string-downcase verb-string)
                                 *simple-pasts*)))
        ((re:scan " " verb-string)
         (format nil "~a~{ ~a~}"
                 (simple-past (car (re:split " " verb-string)))
                 (cdr (re:split " " verb-string))))
        (t
         ;; If we don't know an irregular simple past tense, use the phonetic
         ;; rules in 'apply-ed'
         (apply-ed verb-string))))


;; These are my hand-verified past participles. For compactness, I should
;; remove any that could be formed by just adding 'ed' to the end and only keep
;; the irregulars.
(defparameter *past-participles* (make-hash-table :test #'equal))
(mapc #'(lambda (x)
          (setf (gethash (first x) *past-participles*)
                (second x)))
      '(("arise" "arisen")
        ("backslide" "backslidden")
        ("be" "been")
        ("be active" "been active")
        ("be adrift" "been adrift")
        ("be after" "been after")
        ("be amiss" "been amiss")
        ("be at pains" "been at pains")
        ("be born" "been born")
        ("be due" "been due")
        ("be full" "been full")
        ("be given" "been given")
        ("be intimate" "been intimate")
        ("be on" "been on")
        ("be on cloud nine" "been on cloud nine")
        ("be on the ball" "been on the ball")
        ("be quiet" "been quiet")
        ("be sick" "been sick")
        ("be well" "been well")
        ("be with it" "been with it")
        ("bear" "born")
        ("bear away" "born away")
        ("bear down" "born down")
        ("bear down on" "born down on")
        ("bear down upon" "born down upon")
        ("bear in mind" "born in mind")
        ("bear off" "born off")
        ("bear on" "born on")
        ("bear out" "born out")
        ("bear up" "born up")
        ("bear upon" "born upon")
        ("bear witness" "born witness")
        ("become" "become")
        ("become flat" "become flat")
        ("befall" "befallen")
        ("beget" "begotten")
        ("begin" "begun")
        ("bespeak" "bespoken")
        ("bestead" "bestead")
        ("bestrew" "bestrewn")
        ("betake oneself" "betaken oneself")
        ("bid" "bidden")
        ("bite" "bitten")
        ("bite off" "bitten off")
        ("bite out" "bitten out")
        ("blow" "blown")
        ("blow a fuse" "blown a fuse")
        ("blow off" "blown off")
        ("blow one's stack" "blown one's stack")
        ("blow out" "blown out")
        ("blow out of the water" "blown out of the water")
        ("blow over" "blown over")
        ("blow up" "blown up")
        ("break" "broken")
        ("break apart" "broken apart")
        ("break away" "broken away")
        ("break bread" "broken bread")
        ("break camp" "broken camp")
        ("break dance" "broken dance")
        ("break down" "broken down")
        ("break even" "broken even")
        ("break in" "broken in")
        ("break into" "broken into")
        ("break loose" "broken loose")
        ("break off" "broken off")
        ("break one's back" "broken one's back")
        ("break open" "broken open")
        ("break out" "broken out")
        ("break short" "broken short")
        ("break someone's heart" "broken someone's heart")
        ("break through" "broken through")
        ("break up" "broken up")
        ("break water" "broken water")
        ("break wind" "broken wind")
        ("break with" "broken with")
        ("browbeat" "browbeaten")
        ("can" "canned")
        ("choose" "chosen")
        ("choose up" "chosen up")
        ("come" "come")
        ("come about" "come about")
        ("come across" "come across")
        ("come after" "come after")
        ("come alive" "come alive")
        ("come along" "come along")
        ("come apart" "come apart")
        ("come around" "come around")
        ("come away" "come away")
        ("come back" "come back")
        ("come before" "come before")
        ("come by" "come by")
        ("come close" "come close")
        ("come down" "come down")
        ("come forth" "come forth")
        ("come forward" "come forward")
        ("come home" "come home")
        ("come in" "come in")
        ("come in for" "come in for")
        ("come in handy" "come in handy")
        ("come into" "come into")
        ("come into being" "come into being")
        ("come near" "come near")
        ("come of age" "come of age")
        ("come off" "come off")
        ("come on" "come on")
        ("come out" "come out")
        ("come out of the closet" "come out of the closet")
        ("come over" "come over")
        ("come round" "come round")
        ("come short" "come short")
        ("come through" "come through")
        ("come to" "come to")
        ("come to grips" "come to grips")
        ("come to hand" "come to hand")
        ("come to life" "come to life")
        ("come to light" "come to light")
        ("come to mind" "come to mind")
        ("come to the fore" "come to the fore")
        ("come together" "come together")
        ("come up" "come up")
        ("come up to" "come up to")
        ("come upon" "come upon")
        ("come with" "come with")
        ("countersink" "countersunk")
        ("do" "done")
        ("do a job on" "done a job on")
        ("do away with" "done away with")
        ("do by" "done by")
        ("do drugs" "done drugs")
        ("do good" "done good")
        ("do in" "done in")
        ("do it" "done it")
        ("do justice" "done justice")
        ("do one's best" "done one's best")
        ("do the dishes" "done the dishes")
        ("do the honors" "done the honors")
        ("do up" "done up")
        ("do well" "done well")
        ("do well by" "done well by")
        ("do work" "done work")
        ("draw" "drawn")
        ("draw a bead on" "drawn a bead on")
        ("draw a blank" "drawn a blank")
        ("draw a line" "drawn a line")
        ("draw and quarter" "drawn and quarter")
        ("draw away" "drawn away")
        ("draw back" "drawn back")
        ("draw close" "drawn close")
        ("draw in" "drawn in")
        ("draw near" "drawn near")
        ("draw off" "drawn off")
        ("draw out" "drawn out")
        ("draw rein" "drawn rein")
        ("draw the line" "drawn the line")
        ("draw together" "drawn together")
        ("draw up" "drawn up")
        ("drink" "drunk")
        ("drink down" "drunk down")
        ("drink in" "drunk in")
        ("drink up" "drunk up")
        ("drive" "driven")
        ("drive around" "driven around")
        ("drive away" "driven away")
        ("drive back" "driven back")
        ("drive home" "driven home")
        ("drive in" "driven in")
        ("drive off" "driven off")
        ("drive out" "driven out")
        ("drive up" "driven up")
        ("e-mail" "e-mailed")
        ("eat" "eaten")
        ("eat at" "eaten at")
        ("eat away" "eaten away")
        ("eat in" "eaten in")
        ("eat into" "eaten into")
        ("eat on" "eaten on")
        ("eat out" "eaten out")
        ("eat up" "eaten up")
        ("fall" "fallen")
        ("fall all over" "fallen all over")
        ("fall apart" "fallen apart")
        ("fall asleep" "fallen asleep")
        ("fall away" "fallen away")
        ("fall back" "fallen back")
        ("fall behind" "fallen behind")
        ("fall by the wayside" "fallen by the wayside")
        ("fall down" "fallen down")
        ("fall flat" "fallen flat")
        ("fall for" "fallen for")
        ("fall from grace" "fallen from grace")
        ("fall in" "fallen in")
        ("fall in line" "fallen in line")
        ("fall in love" "fallen in love")
        ("fall into" "fallen into")
        ("fall into place" "fallen into place")
        ("fall off" "fallen off")
        ("fall open" "fallen open")
        ("fall out" "fallen out")
        ("fall over" "fallen over")
        ("fall over backwards" "fallen over backwards")
        ("fall short" "fallen short")
        ("fall short of" "fallen short of")
        ("fall through" "fallen through")
        ("fall under" "fallen under")
        ("fall upon" "fallen upon")
        ("fly" "flown")
        ("fly blind" "flown blind")
        ("fly by" "flown by")
        ("fly contact" "flown contact")
        ("fly high" "flown high")
        ("fly in the face of" "flown in the face of")
        ("fly in the teeth of" "flown in the teeth of")
        ("fly off the handle" "flown off the handle")
        ("fly on" "flown on")
        ("fly open" "flown open")
        ("fly the coop" "flown the coop")
        ("forbear" "forborne")
        ("forbid" "forbidden")
        ("forego" "foregone")
        ("foreknow" "foreknown")
        ("foresee" "foreseen")
        ("foreswear" "foresworn")
        ("forget" "forgotten")
        ("forgive" "forgiven")
        ("forgo" "forgone")
        ("forsake" "forsaken")
        ("forswear" "forsworn")
        ("freeze" "frozen")
        ("freeze down" "frozen down")
        ("freeze off" "frozen off")
        ("freeze out" "frozen out")
        ("get" "gotten")
        ("ghostwrite" "ghostwritten")
        ("give" "given")
        ("give a damn" "given a damn")
        ("give a hang" "given a hang")
        ("give a hoot" "given a hoot")
        ("give and take" "given and take")
        ("give away" "given away")
        ("give back" "given back")
        ("give birth" "given birth")
        ("give care" "given care")
        ("give chase" "given chase")
        ("give ear" "given ear")
        ("give forth" "given forth")
        ("give full measure" "given full measure")
        ("give in" "given in")
        ("give it a try" "given it a try")
        ("give it a whirl" "given it a whirl")
        ("give it the deep six" "given it the deep six")
        ("give notice" "given notice")
        ("give off" "given off")
        ("give one's best" "given one's best")
        ("give out" "given out")
        ("give rise" "given rise")
        ("give suck" "given suck")
        ("give thanks" "given thanks")
        ("give the axe" "given the axe")
        ("give the bounce" "given the bounce")
        ("give the eye" "given the eye")
        ("give the gate" "given the gate")
        ("give the glad eye" "given the glad eye")
        ("give the once over" "given the once over")
        ("give the sack" "given the sack")
        ("give tongue to" "given tongue to")
        ("give up" "given up")
        ("give vent" "given vent")
        ("give voice" "given voice")
        ("give way" "given way")
        ("give-up the ghost" "give-uped the ghost")
        ("go" "gone")
        ("go a long way" "gone a long way")
        ("go about" "gone about")
        ("go across" "gone across")
        ("go after" "gone after")
        ("go against" "gone against")
        ("go ahead" "gone ahead")
        ("go all out" "gone all out")
        ("go along" "gone along")
        ("go around" "gone around")
        ("go away" "gone away")
        ("go back" "gone back")
        ("go back on" "gone back on")
        ("go bad" "gone bad")
        ("go ballistic" "gone ballistic")
        ("go by" "gone by")
        ("go deep" "gone deep")
        ("go down" "gone down")
        ("go down on" "gone down on")
        ("go dutch" "gone dutch")
        ("go far" "gone far")
        ("go for" "gone for")
        ("go for broke" "gone for broke")
        ("go forth" "gone forth")
        ("go forward" "gone forward")
        ("go home" "gone home")
        ("go in" "gone in")
        ("go into" "gone into")
        ("go off" "gone off")
        ("go off at half-cock" "gone off at half-cock")
        ("go off half-cocked" "gone off half-cocked")
        ("go on" "gone on")
        ("go out" "gone out")
        ("go over" "gone over")
        ("go past" "gone past")
        ("go steady" "gone steady")
        ("go through" "gone through")
        ("go through the motions" "gone through the motions")
        ("go to" "gone to")
        ("go to bed" "gone to bed")
        ("go to pieces" "gone to pieces")
        ("go to pot" "gone to pot")
        ("go to sleep" "gone to sleep")
        ("go to the dogs" "gone to the dogs")
        ("go to war" "gone to war")
        ("go under" "gone under")
        ("go up" "gone up")
        ("go with" "gone with")
        ("go wrong" "gone wrong")
        ("grow" "grown")
        ("grow over" "grown over")
        ("grow up" "grown up")
        ("hew out" "hewn out")
        ("hide" "hidden")
        ("hide out" "hidden out")
        ("interweave" "interwoven")
        ("inweave" "inwoven")
        ("know" "known")
        ("know apart" "known apart")
        ("know the score" "known the score")
        ("know what's going on" "known what's going on")
        ("know what's what" "known what's what")
        ("lie around" "lain around")
        ("lie awake" "lain awake")
        ("lie dormant" "lain dormant")
        ("lie down" "lain down")
        ("lie in" "lain in")
        ("lie in wait" "lain in wait")
        ("lie low" "lain low")
        ("lie with" "lain with")
        ("misgive" "misgiven")
        ("mistake" "mistaken")
        ("outdo" "outdone")
        ("outgo" "outgone")
        ("outgrow" "outgrown")
        ("outride" "outridden")
        ("outrun" "outrun")
        ("outwear" "outworn")
        ("overbear" "overborne")
        ("overbid" "overbidden")
        ("overblow" "overblown")
        ("overcome" "overcome")
        ("overdo" "overdone")
        ("overdraw" "overdrawn")
        ("overdrive" "overdriven")
        ("overeat" "overeaten")
        ("overfly" "overflown")
        ("overgrow" "overgrown")
        ("overlie" "overlain")
        ("override" "overridden")
        ("overrun" "overrun")
        ("oversee" "overseen")
        ("overtake" "overtaken")
        ("overthrow" "overthrown")
        ("overwrite" "overwritten")
        ("partake" "partaken")
        ("partake in" "partaken in")
        ("play" "played")
        ("redo" "redone")
        ("rerun" "rerun")
        ("retake" "retaken")
        ("rewrite" "rewritten")
        ("ride" "ridden")
        ("ride away" "ridden away")
        ("ride herd" "ridden herd")
        ("ride horseback" "ridden horseback")
        ("ride off" "ridden off")
        ("ride out" "ridden out")
        ("ride roughshod" "ridden roughshod")
        ("ride the bench" "ridden the bench")
        ("ring" "rung")
        ("ring out" "rung out")
        ("ring up" "rung up")
        ("rise" "risen")
        ("rise up" "risen up")
        ("run" "run")
        ("run a risk" "run a risk")
        ("run across" "run across")
        ("run afoul" "run afoul")
        ("run aground" "run aground")
        ("run along" "run along")
        ("run around" "run around")
        ("run away" "run away")
        ("run bases" "run bases")
        ("run by" "run by")
        ("run down" "run down")
        ("run dry" "run dry")
        ("run for" "run for")
        ("run into" "run into")
        ("run low" "run low")
        ("run off" "run off")
        ("run on" "run on")
        ("run out" "run out")
        ("run over" "run over")
        ("run roughshod" "run roughshod")
        ("run short" "run short")
        ("run through" "run through")
        ("run up" "run up")
        ("see" "seen")
        ("see double" "seen double")
        ("see eye to eye" "seen eye to eye")
        ("see red" "seen red")
        ("see the light" "seen the light")
        ("see through" "seen through")
        ("see to it" "seen to it")
        ("shake" "shaken")
        ("shake hands" "shaken hands")
        ("shake off" "shaken off")
        ("shake up" "shaken up")
        ("shear" "shorn")
        ("shit" "shit")
        ("show" "shown")
        ("show off" "shown off")
        ("show the door" "shown the door")
        ("show up" "shown up")
        ("shrink" "shrunk")
        ("shrink back" "shrunk back")
        ("shrink from" "shrunk from")
        ("shrive" "shriven")
        ("sightsee" "sightseen")
        ("sightsing" "sightsung")
        ("sink" "sunk")
        ("sink in" "sunk in")
        ("slide" "slidden")
        ("slide by" "slidden by")
        ("slide down" "slidden down")
        ("smite" "smitten")
        ("sow" "sown")
        ("sow in" "sown in")
        ("sow one's oats" "sown one's oats")
        ("sow one's wild oats" "sown one's wild oats")
        ("speak" "spoken")
        ("speak for" "spoken for")
        ("speak in tongues" "spoken in tongues")
        ("speak of the devil" "spoken of the devil")
        ("speak out" "spoken out")
        ("speak up" "spoken up")
        ("spit" "spit")
        ("spit out" "spit out")
        ("spit up" "spit up")
        ("spray" "sprayed")
        ("spring" "sprung")
        ("spring to mind" "sprung to mind")
        ("spring up" "sprung up")
        ("steal" "stolen")
        ("steal away" "stolen away")
        ("stride" "stridden")
        ("strive" "striven")
        ("swear" "sworn")
        ("swear in" "sworn in")
        ("swear off" "sworn off")
        ("swear out" "sworn out")
        ("sweat off" "sweated off")
        ("swell up" "swollen up")
        ("swim" "swum")
        ("swing" "swung")
        ("swing about" "swung about")
        ("swing around" "swung around")
        ("swing out" "swung out")
        ("swing over" "swung over")
        ("take" "taken")
        ("take a bow" "taken a bow")
        ("take a breath" "taken a breath")
        ("take a breather" "taken a breather")
        ("take a chance" "taken a chance")
        ("take a crap" "taken a crap")
        ("take a dare" "taken a dare")
        ("take a dive" "taken a dive")
        ("take a firm stand" "taken a firm stand")
        ("take a hit" "taken a hit")
        ("take a hop" "taken a hop")
        ("take a joke" "taken a joke")
        ("take a leak" "taken a leak")
        ("take a look" "taken a look")
        ("take a powder" "taken a powder")
        ("take a shit" "taken a shit")
        ("take aback" "taken aback")
        ("take account" "taken account")
        ("take advantage" "taken advantage")
        ("take after" "taken after")
        ("take aim" "taken aim")
        ("take apart" "taken apart")
        ("take arms" "taken arms")
        ("take away" "taken away")
        ("take back" "taken back")
        ("take care" "taken care")
        ("take chances" "taken chances")
        ("take charge" "taken charge")
        ("take control" "taken control")
        ("take down" "taken down")
        ("take effect" "taken effect")
        ("take exception" "taken exception")
        ("take fire" "taken fire")
        ("take five" "taken five")
        ("take flight" "taken flight")
        ("take for" "taken for")
        ("take for granted" "taken for granted")
        ("take form" "taken form")
        ("take heart" "taken heart")
        ("take heed" "taken heed")
        ("take hold" "taken hold")
        ("take hold of" "taken hold of")
        ("take home" "taken home")
        ("take in" "taken in")
        ("take in charge" "taken in charge")
        ("take in vain" "taken in vain")
        ("take in water" "taken in water")
        ("take into account" "taken into account")
        ("take issue" "taken issue")
        ("take it easy" "taken it easy")
        ("take it on the chin" "taken it on the chin")
        ("take kindly to" "taken kindly to")
        ("take leave" "taken leave")
        ("take lying down" "taken lying down")
        ("take note" "taken note")
        ("take notice" "taken notice")
        ("take off" "taken off")
        ("take office" "taken office")
        ("take on" "taken on")
        ("take one's lumps" "taken one's lumps")
        ("take orders" "taken orders")
        ("take out" "taken out")
        ("take over" "taken over")
        ("take pains" "taken pains")
        ("take part" "taken part")
        ("take place" "taken place")
        ("take pride" "taken pride")
        ("take root" "taken root")
        ("take shape" "taken shape")
        ("take stage" "taken stage")
        ("take stock" "taken stock")
        ("take ten" "taken ten")
        ("take the air" "taken the air")
        ("take the bull by the horns" "taken the bull by the horns")
        ("take the cake" "taken the cake")
        ("take the count" "taken the count")
        ("take the field" "taken the field")
        ("take the fifth" "taken the fifth")
        ("take the fifth amendment" "taken the fifth amendment")
        ("take the floor" "taken the floor")
        ("take the road" "taken the road")
        ("take the stage" "taken the stage")
        ("take the stand" "taken the stand")
        ("take the veil" "taken the veil")
        ("take time by the forelock" "taken time by the forelock")
        ("take time off" "taken time off")
        ("take to" "taken to")
        ("take to be" "taken to be")
        ("take to heart" "taken to heart")
        ("take to task" "taken to task")
        ("take to the woods" "taken to the woods")
        ("take turns" "taken turns")
        ("take up" "taken up")
        ("take up arms" "taken up arms")
        ("take water" "taken water")
        ("tear" "torn")
        ("tear apart" "torn apart")
        ("tear away" "torn away")
        ("tear down" "torn down")
        ("tear into" "torn into")
        ("tear off" "torn off")
        ("tear up" "torn up")
        ("throw" "thrown")
        ("throw a fit" "thrown a fit")
        ("throw away" "thrown away")
        ("throw back" "thrown back")
        ("throw cold water on" "thrown cold water on")
        ("throw in" "thrown in")
        ("throw in the towel" "thrown in the towel")
        ("throw off" "thrown off")
        ("throw out" "thrown out")
        ("throw out of kilter" "thrown out of kilter")
        ("throw overboard" "thrown overboard")
        ("throw together" "thrown together")
        ("throw up" "thrown up")
        ("tread" "trodden")
        ("tread down" "trodden down")
        ("tread on" "trodden on")
        ("typewrite" "typewritten")
        ("undergo" "undergone")
        ("underlie" "underlain")
        ("undertake" "undertaken")
        ("underwrite" "underwritten")
        ("undo" "undone")
        ("unfreeze" "unfrozen")
        ("uprise" "uprisen")
        ("wake" "woken")
        ("wake up" "waked up")
        ("wear" "worn")
        ("wear away" "worn away")
        ("wear down" "worn down")
        ("wear off" "worn off")
        ("wear on" "worn on")
        ("wear out" "worn out")
        ("wear round" "worn round")
        ("wear ship" "worn ship")
        ("wear the trousers" "worn the trousers")
        ("wear thin" "worn thin")
        ("wear upon" "worn upon")
        ("weave" "woven")
        ("withdraw" "withdrawn")
        ("write" "written")
        ("write about" "written about")
        ("write copy" "written copy")
        ("write down" "written down")
        ("write in" "written in")
        ("write in code" "written in code")
        ("write of" "written of")
        ("write off" "written off")
        ("write on" "written on")
        ("write out" "written out")
        ("write up" "written up")))

(defun past-participle (verb-string)
  (cond ((gethash (string-downcase verb-string) *past-participles*)
         (string-upcase (gethash (string-downcase verb-string)
                                 *past-participles*)))
        ((re:scan " " verb-string)
         (format nil "~a~{ ~a~}"
                 (past-participle (car (re:split " " verb-string)))
                 (cdr (re:split " " verb-string))))
        (t
         ;; If we don't know a special past-participle form for the verb,
         ;; return the simple past tense.
         (simple-past verb-string))))


;; This is a quick-to-write Lisp verison of Lingua-ENG-Inflect's PART_PRES.
(defun apply-ing (verb)
  "Takes a verb, as a string or symbol, and applies -ING morphology."
  (let ((vs (string-upcase verb)))
    (setf vs (re:regex-replace "IE$" vs "Y"))
    (setf vs (re:regex-replace "UE$" vs "U"))
    (setf vs (re:regex-replace "([AUYS])E$" vs "\\1"))
    (setf vs (re:regex-replace "SKI$" vs "SKI"))
    (setf vs (re:regex-replace "[^B]I$" vs ""))
    (setf vs (re:regex-replace "^(?:ARE|WERE)$" vs "BE"))
    (setf vs (re:regex-replace "^(?:HAD)$" vs "HAV"))
    (setf vs (re:regex-replace "(HOE)$" vs "\\1"))
    (setf vs (re:regex-replace "([^E])E$" vs "\\1"))
    (unless (re:scan "ER$" vs)
      (setf vs (re:regex-replace "([^AEIOU][AEIOUY]([BDGMNPRST]))$" vs
                                 "\\1\\2")))
    (setf vs (format nil "~AING" vs))
    ;; Exceptions
    (setf vs (sub "OPERATTING$" "OPERATING" vs))
    (setf vs (sub "STRATTING$" "STRATING" vs))
    (setf vs (sub "VIOLATTING$" "VIOLATING" vs))
    (setf vs (sub "RISSING$" "RISING" vs))
    (setf vs (sub "COMMING$" "COMING" vs))
    (setf vs (sub "RRING$" "RING" vs))
    (setf vs (sub "TENNING$" "TENING" vs))
    (if (symbolp verb)
        (read-from-string vs)
        vs)))


(defun present-singular (word)
  (if (and (symbolp word)
           (member '#\_ (coerce (string (the symbol word)) 'list)))
      (let ((words (re:split "_" (string word))))
        (symb (format nil "~{~A~^_~}"
                      (cons (present-singular (intern (car words)))
                            (cdr words)))))
      (let ((vs (string-upcase (string word))))
        (setf vs (sub "([^AEIOU])Y$" "\\1IES" vs))
        (setf vs (sub "([XHO])$" "\\1ES" vs))
        (setf vs (sub "SS$" "SSES" vs))
        (setf vs (sub "([^S])$" "\\1S" vs))
        (setf vs (sub "^BES$" "IS" vs))
        (setf vs (sub "^HAVES$" "HAS" vs))
        (setf vs (sub "GHES$" "GHS" vs))
        (if (symbolp word)
            (read-from-string vs)
            vs))))


(defun capitalize-words (s)
  (let ((cap (string-capitalize (string-downcase s :start 1))))
    (dolist (x '(ii iii iv vi vii viii ix xi xii xiii xiv xv xvi xvii xviii
                 xix xx xxx us uk usa ussr un ss))
      (setf cap (sub (mkstr "([_ ])" (string-capitalize (mkstr x)) "\\.?([_ ])")
                     (mkstr "\\1" (string-upcase (mkstr x)) "\\2")
                     cap))
      (setf cap (sub (mkstr "([_ ])" (string-capitalize (mkstr x)) "$")
                     (mkstr "\\1" (string-upcase (mkstr x)))
                     cap))
      (setf cap (sub (mkstr "^" (string-capitalize (mkstr x)) "([_ ])")
                     (mkstr (string-upcase (mkstr x)) "\\1")
                     cap)))
    (dolist (x '(on of and the to or for from in di de))
      (setf cap (sub (mkstr "([_ ])" (string-capitalize (mkstr x)) "([_ ])")
                     (mkstr "\\1" (string-downcase (mkstr x)) "\\2")
                     cap)))
    (setf cap (sub "'S" "'s" cap))
    cap))


;; Gene Kim (2-14-2019): Below two functions are from unifgram.lisp of Peter
;; Norvig's Paradigms of AI Programming.

(defun punctuation-p (char) (find char "*_.,;:`!?#-()\\\""))

(defun remove-punctuation (string)
  "Replace punctuation with spaces in string."
  (substitute-if #\space #'punctuation-p string))

;; List of contractions is from
;; https://github.com/ian-beaver/pycontractions/blob/master/pycontractions/contractions.py
(defvar *contraction-regex-alist*
  '(("\\bare\\s\+not\\b" . "aren't")
    ("\\bcannot\\b" . "can't")
    ("\\bcould\\s\+have\\b" . "could've")
    ("\\bcould\\s\+not\\b" . "couldn't")
    ("\\bdid\\s\+not\\b" . "didn't")
    ("\\bdoes\\s\+not\\b" . "doesn't")
    ("\\bdo\\s\+not\\b" . "don't")
    ("\\bgot\\s\+to\\b" . "gotta")
    ("\\bhad\\s\+not\\b" . "hadn't")
    ("\\bhas\\s\+not\\b" . "hasn't")
    ("\\bhave\\s\+not\\b" . "haven't")
    ("\\bhe\\s\+had\\b" . "he'd")
    ("\\bhe\\s\+would\\b" . "he'd")
    ("\\bhe\\s\+shall\\b" . "he'll")
    ("\\bhe\\s\+will\\b" . "he'll")
    ("\\bhe\\s\+has\\b" . "he's")
    ("\\bhe\\s\+is\\b" . "he's")
    ("\\bhow\\s\+did\\b" . "how'd")
    ("\\bhow\\s\+would\\b" . "how'd")
    ("\\bhow\\s\+will\\b" . "how'll")
    ("\\bhow\\s\+has\\b" . "how's")
    ("\\bhow\\s\+is\\b" . "how's")
    ("\\bhow\\s\+does\\b" . "how's")
    ("\\bI\\s\+had\\b" . "I'd")
    ("\\bI\\s\+would\\b" . "I'd")
    ("\\bI\\s\+shall\\b" . "I'll")
    ("\\bI\\s\+will\\b" . "I'll")
    ("\\bI\\s\+am\\b" . "I'm")
    ("\\bI\\s\+have\\b" . "I've")
    ("\\bis\\s\+not\\b" . "isn't")
    ("\\bit\\s\+would\\b" . "it'd")
    ("\\bit\\s\+shall\\b" . "it'll")
    ("\\bit\\s\+will\\b" . "it'll")
    ("\\bit\\s\+has\\b" . "it's")
    ("\\bit\\s\+is\\b" . "it's")
    ("\\bmay\\s\+have\\b" . "may've")
    ("\\bmight\\s\+not\\b" . "mightn't")
    ("\\bmight\\s\+have\\b" . "might've")
    ("\\bmust\\s\+not\\b" . "mustn't")
    ("\\bmust\\s\+have\\b" . "must've")
    ("\\bneed\\s\+not\\b" . "needn't")
    ("\\bof\\s\+the\\s\+clock\\b" . "o'clock")
    ("\\bought\\s\+not\\b" . "oughtn't")
    ("\\bshall\\s\+not\\b" . "shan't")
    ("\\bshe\\s\+had\\b" . "she'd")
    ("\\bshe\\s\+would\\b" . "she'd")
    ("\\bshe\\s\+shall\\b" . "she'll")
    ("\\bshe\\s\+will\\b" . "she'll")
    ("\\bshe\\s\+has\\b" . "she's")
    ("\\bshe\\s\+is\\b" . "she's")
    ("\\bshould\\s\+have\\b" . "should've")
    ("\\bshould\\s\+not\\b" . "shouldn't")
    ("\\bsomebody\\s\+has\\b" . "somebody's")
    ("\\bsomebody\\s\+is\\b" . "somebody's")
    ("\\bsomeone\\s\+has\\b" . "someone's")
    ("\\bsomeone\\s\+is\\b" . "someone's")
    ("\\bsomething\\s\+has\\b" . "something's")
    ("\\bsomething\\s\+is\\b" . "something's")
    ("\\bthat\\s\+shall\\b" . "that'll")
    ("\\bthat\\s\+will\\b" . "that'll")
    ("\\bthat\\s\+are\\b" . "that're")
    ("\\bthat\\s\+has\\b" . "that's")
    ("\\bthat\\s\+is\\b" . "that's")
    ("\\bthat\\s\+would\\b" . "that'd")
    ("\\bthat\\s\+had\\b" . "that'd")
    ("\\bthere\\s\+had\\b" . "there'd")
    ("\\bthere\\s\+would\\b" . "there'd")
    ("\\bthere\\s\+are\\b" . "there're")
    ("\\bthere\\s\+has\\b" . "there's")
    ("\\bthere\\s\+is\\b" . "there's")
    ("\\bthese\\s\+are\\b" . "these're")
    ("\\bthey\\s\+had\\b" . "they'd")
    ("\\bthey\\s\+would\\b" . "they'd")
    ("\\bthey\\s\+shall\\b" . "they'll")
    ("\\bthey\\s\+will\\b" . "they'll")
    ("\\bthey\\s\+are\\b" . "they're")
    ("\\bthey\\s\+have\\b" . "they've")
    ("\\bthis\\s\+has\\b" . "this's")
    ("\\bthis\\s\+is\\b" . "this's")
    ("\\bthose\\s\+are\\b" . "those're")
    ("\\bwas\\s\+not\\b" . "wasn't")
    ("\\bwe\\s\+had\\b" . "we'd")
    ("\\bwe\\s\+would\\b" . "we'd")
    ("\\bwe\\s\+would\\s\+have\\b" . "we'd've")
    ("\\bwe\\s\+will\\b" . "we'll")
    ("\\bwe\\s\+are\\b" . "we're")
    ("\\bwe\\s\+have\\b" . "we've")
    ("\\bwere\\s\+not\\b" . "weren't")
    ("\\bwhat\\s\+did\\b" . "what'd")
    ("\\bwhat\\s\+shall\\b" . "what'll")
    ("\\bwhat\\s\+will\\b" . "what'll")
    ("\\bwhat\\s\+are\\b" . "what're")
    ("\\bwhat\\s\+has\\b" . "what's")
    ("\\bwhat\\s\+is\\b" . "what's")
    ("\\bwhat\\s\+does\\b" . "what's")
    ("\\bwhat\\s\+have\\b" . "what've")
    ("\\bwhen\\s\+has\\b" . "when's")
    ("\\bwhen\\s\+is\\b" . "when's")
    ("\\bwhere\\s\+did\\b" . "where'd")
    ("\\bwhere\\s\+are\\b" . "where're")
    ("\\bwhere\\s\+has\\b" . "where's")
    ("\\bwhere\\s\+is\\b" . "where's")
    ("\\bwhere\\s\+does\\b" . "where's")
    ("\\bwhere\\s\+have\\b" . "where've")
    ("\\bwhich\\s\+has\\b" . "which's")
    ("\\bwhich\\s\+is\\b" . "which's")
    ("\\bwho\\s\+would\\b" . "who'd")
    ("\\bwho\\s\+had\\b" . "who'd")
    ("\\bwho\\s\+did\\b" . "who'd")
    ("\\bwho\\s\+would\\s\+have\\b" . "who'd've")
    ("\\bwho\\s\+shall\\b" . "who'll")
    ("\\bwho\\s\+will\\b" . "who'll")
    ("\\bwho\\s\+are\\b" . "who're")
    ("\\bwho\\s\+has\\b" . "who's")
    ("\\bwho\\s\+is\\b" . "who's")
    ("\\bwho\\s\+does\\b" . "who's")
    ("\\bwho\\s\+have\\b" . "who've")
    ("\\bwhy\\s\+did\\b" . "why'd")
    ("\\bwhy\\s\+are\\b" . "why're")
    ("\\bwhy\\s\+has\\b" . "why's")
    ("\\bwhy\\s\+is\\b" . "why's")
    ("\\bwhy\\s\+does\\b" . "why's")
    ("\\bwill\\s\+not\\b" . "won't")
    ("\\bwould\\s\+have\\b" . "would've")
    ("\\bwould\\s\+not\\b" . "wouldn't")
    ("\\byou\\s\+had\\b" . "you'd")
    ("\\byou\\s\+would\\b" . "you'd")
    ("\\byou\\s\+shall\\b" . "you'll")
    ("\\byou\\s\+will\\b" . "you'll")
    ("\\byou\\s\+are\\b" . "you're")
    ("\\byou\\s\+have\\b" . "you've")))


(defvar *contraction-regex-scanner-alist*
  (regex-alist-to-scanner-alist
    *contraction-regex-alist* :case-insensitive-mode t))


;; Applies all common contractions to the given string, left-to-right.
;; e.g.
;;  "I cannot wait, you will not stop me" -> "I can't wait, so you won't stop me"
(defun add-contractions (str &key (custom-mappings nil))
  (reduce #'(lambda (acc cur)
              (let ((regex (car cur))
                    (result (cdr cur)))
                (re:regex-replace-all regex acc result)))
          (append (regex-alist-to-scanner-alist
                    custom-mappings :case-insensitive-mode t)
                  *contraction-regex-scanner-alist*)
          :initial-value str))


;; Generates a list of possible rewritings of the given sentence by adding
;; contractions. It assumes that possible contraction points don't overlap.
;; If there are overlaps, the portion that matches earlier in the string
;; is used.
;;
;; Custom contraction choices can be supplied via `custom-mappings` as an
;; alist mapping regular expressions to the contracted form.
;;
;; e.g.
;; "I cannot run, but I will not give up"
;; -> ("I cannot run, but I will not give up"
;;     "I can't run, but I will not give up"
;;     "I cannot run, but I'll not give up"
;;     "I can't run, but I'll not give up")
(defun contraction-possibilities-eager (str &key (custom-mappings nil))
  (let*
    ((all-mappings (append custom-mappings *contraction-regex-alist*))
     (all-matcher (cl-strings:join (mapcar #'car all-mappings) :separator "\|"))
     (unmatched (re:split all-matcher str))
     (matches (re:all-matches-as-strings all-matcher str))
     (contracted (mapcar #'(lambda (match)
                             (add-contractions match :custom-mappings custom-mappings))
                         matches))
     (paired (mapcar #'list matches contracted))
     (possibilities (cartesian-product paired)))
    (mapcar #'(lambda (fillers)
                (cl-strings:join (interleave unmatched fillers)))
      possibilities)))


;; Generates a list of possible rewritings of the given sentence by adding
;; contractions. If possible contraction points overlap, then possibilities
;; will be handled for each of these choices.
;;
;; Custom contraction choices can be supplied via `custom-mappings` as an
;; alist mapping regular expressions to the contracted form.
;;
;; e.g.
;; "I cannot run, but I will not give up"
;; -> ("I cannot run, but I will not give up" "I can't run, but I will not give up"
;;     "I cannot run, but I'll not give up" "I can't run, but I'll not give up"
;;     "I cannot run, but I won't give up" "I can't run, but I won't give up")
(defun contraction-possibilities-overlap (str &key (custom-mappings nil))
  (labels
    ;; Recursive helper function.
    ;; Generates an exhaustive list include-exclude combinations of the given
    ;; spans with no span overlap. Both inputs and outputs are sorted by span
    ;; starts.
    ;; e.g.
    ;;  ((2 8) (18 24) (20 28))
    ;;  -> (() ((2 8)) ((18 24)) ((20 28))
    ;;      ((2 8) (18 24)) ((2 8) (20 28)))
    ((in-out-combos (spans start acc)
       (cond
         ;; Reached the end.
         ((null spans) (list (reverse acc)))
         ;; We identified a span overlap so ignore current span.
         ((< (caar spans) start) (in-out-combos (cdr spans) start acc))
         ;; Recurse into both including and excluding the current span.
         (t (append
              ;; include
              (in-out-combos (cdr spans) (cadar spans) (cons (car spans) acc))
              ;; exclude
              (in-out-combos (cdr spans) start acc)))))
     ;; (2 8 18 24) ("can't" "I'll") "I cannot run, but I will not give up"
     ;; -> "I can't run, but I'll not give up"
     (apply-span-mappings (flat-spans map-strs cleanstr)
       (let* ((raw-spans (pair-up-list (append '(0) flat-spans (list (length cleanstr)))))
              (untouched-spans (mapcar
                                 #'(lambda (x) (subseq cleanstr (first x) (second x)))
                                 raw-spans)))
         (cl-strings:join (interleave untouched-spans map-strs))))
       ) ; end of labels definitions
    ;; Main body.
    (let*
      ((all-mappings (append custom-mappings *contraction-regex-alist*))
       (all-matcher (cl-strings:join (mapcar #'car all-mappings) :separator "\|"))
       (match-spans (pair-up-list (overlap-regex-matches all-matcher str)))
       (match-strs (overlap-regex-matches-as-strings all-matcher str))
       (match-maps (mapcar #'(lambda (match-str)
                               (add-contractions match-str :custom-mappings custom-mappings))
                           match-strs))
       (match-info (mapcar #'append match-spans (mapcar #'list match-strs match-maps)))
       (possible-spans (in-out-combos match-info 0 nil)))
      (mapcar
        #'(lambda (span-info-lst)
            (let ((flat-spans (interleave (mapcar #'first span-info-lst)
                                          (mapcar #'second span-info-lst)))
                  (map-strs (mapcar #'fourth span-info-lst)))
              (apply-span-mappings flat-spans map-strs str)))
        possible-spans))))

;; TODO: we could make the function below more efficient by turning it into a regex and doing
;; a single comparison..
;; == example above ==
;; "I cannot run, but I will not give up"
;; -> ("I cannot run, but I will not give up"
;;     "I can't run, but I will not give up"
;;     "I cannot run, but I won't give up"
;;     "I can't run, but I won't give up")
;;
;; == becomes ==
;; "I cannot run, but I will not give up"
;; -> "I (cannot|can't) run, but I (will not|won't) give up"
;;
;; == a bit more complicated for regex... ==
;;  might consider using the Aho-Corasick algorithm, available http://clstringmatch.sourceforge.net/
;;

;; Generates a list of possible rewritings of the given sentence by adding
;; contractions. If possible contraction points overlap, then possibilities
;; will be handled for each of these choices.
;;
;; Custom contraction choices can be supplied via `custom-mappings` as an
;; alist mapping regular expressions to the contracted form.
;;
;; If `overlap` is nil, the we assume no overlap in contractions possibilies,
;; which allows a more efficient version of the function.
(defun contraction-possibilities (str &key (custom-mappings nil) (overlap t))
  (if overlap
    (contraction-possibilities-overlap str :custom-mappings custom-mappings)
    (contraction-possibilities-eager str :custom-mappings custom-mappings)))

