#pragma once

#include <array>
#include <bitset>
#include <boost/functional/hash.hpp>
#include <unordered_set>
#include <vector>

#include "edsgraph.hpp"

namespace shrg {

const int MAX_GRAPH_EDGE_COUNT = 256;
const int MAX_GRAPH_NODE_COUNT = 256;
const int MAX_GRAMMAR_BOUNDARY_NODE_COUNT = 16;

using EdgeSet = std::bitset<MAX_GRAPH_EDGE_COUNT>;
using NodeSet = std::bitset<MAX_GRAPH_NODE_COUNT>;
union NodeMapping {
    using T1 = std::array<uint8_t, MAX_GRAMMAR_BOUNDARY_NODE_COUNT>;
    using T4 = std::array<uint32_t, MAX_GRAMMAR_BOUNDARY_NODE_COUNT / 4>;
    using T8 = std::array<uint64_t, MAX_GRAMMAR_BOUNDARY_NODE_COUNT / 8>;
    T1 m1;
    T4 m4;
    T8 m8;

    uint8_t &operator[](int index) { return m1[index]; }

    uint8_t operator[](int index) const { return m1[index]; }

    bool operator==(const NodeMapping &other) const { return m8 == other.m8; }

    constexpr std::size_t size() const noexcept { return MAX_GRAMMAR_BOUNDARY_NODE_COUNT; }
};

template <typename T> class Ref {
  public:
    Ref() : ptr_(nullptr) {}
    explicit Ref(const T &t) : ptr_(std::addressof(t)) {}
    Ref(T &&t) = delete;

    operator T &() const { return *ptr_; }
    const T &get() const { return *ptr_; }
    const T *get_pointer() const { return ptr_; }

  private:
    const T *ptr_;
};

template <typename T> Ref<T> GetRef(const T &t) { return Ref<T>(t); }

struct GrammarAttributes;
class ChartItem {
  public:
    static const int kEmpty = -1;
    static const int kExpanded = -100;
    static const int kVisited = -1000;

  public:
    GrammarAttributes *attrs_ptr; // the attributes of the grammar that the item belongs to

    ChartItem *next_ptr = nullptr;
    ChartItem *left_ptr = nullptr;
    ChartItem *right_ptr = nullptr;

    EdgeSet edge_set; // edge set of EdsGraph::Edge
    // mapping from boundary nodes of SHRG (SHRG::Node) to boundary nodes of EDS (EdsGraph::Node,
    // the index starts from 1)
    NodeMapping boundary_node_mapping;

    float score = 1.0; // initially above zero
    int status = kEmpty;

    ChartItem() : attrs_ptr(nullptr), boundary_node_mapping{} {}

    explicit ChartItem(GrammarAttributes *attrs_ptr, //
                       const EdgeSet &edge_set_ = 0, //
                       const NodeMapping &node_mapping_ = {})
        : attrs_ptr(attrs_ptr), edge_set(edge_set_), boundary_node_mapping(node_mapping_) {}

    void Swap(ChartItem &other) {
        std::swap(attrs_ptr, other.attrs_ptr);
        std::swap(left_ptr, other.left_ptr);
        std::swap(right_ptr, other.right_ptr);
        std::swap(score, other.score);
        std::swap(status, other.status);
    }

    ChartItem *Pop() {
        ChartItem *ptr = next_ptr;
        next_ptr = nullptr;
        return ptr;
    }

    void Push(ChartItem *chart_item_ptr) {
        chart_item_ptr->next_ptr = next_ptr;
        next_ptr = chart_item_ptr;
    }
};

template <typename T> inline bool operator==(const Ref<T> &ref1, const Ref<T> &ref2) {
    return ref1.get() == ref2.get();
}

inline bool operator==(const ChartItem &v1, const ChartItem &v2) {
    return v1.edge_set == v2.edge_set && v1.boundary_node_mapping == v2.boundary_node_mapping;
}

} // namespace shrg

namespace std {

using shrg::ChartItem;
using shrg::NodeMapping;
using shrg::Ref;
template <> struct hash<Ref<ChartItem>> {
    // !!! the hash function must be mark as const
    std::size_t operator()(const Ref<ChartItem> &ref) const {
        auto &v = ref.get();
        auto hash_value = std::hash<decltype(v.edge_set)>()(v.edge_set);
        boost::hash_combine(hash_value, v.boundary_node_mapping);
        return hash_value;
    }
};

template <> struct hash<NodeMapping> {
    // !!! the hash function must be mark as const
    std::size_t operator()(const NodeMapping &key) const { return boost::hash_value(key.m8); }
};

} // namespace std

namespace boost {

using shrg::NodeMapping;
template <> struct hash<NodeMapping> {
    // !!! the hash function must be mark as const
    std::size_t operator()(const NodeMapping &key) const { return boost::hash_value(key.m8); }
};

} // namespace boost

REGISTER_TYPE_NAME(shrg::ChartItem);
