import timeit
import nltk

from CTestGeneratorStrategy import CTestGeneratorStrategy

class GreedyStrategy(CTestGeneratorStrategy):
    '''
    This is a implemenation of a ctest generation startegy called "Greedy".
    This Strategy generates c-tests of different texts until one text hits
    the desired proficiency level and gets returned.
    '''
    def generateContextSents(self,ctest, sentence):
        '''
        This method appends a sentece which should not be mutilated by gaps in the DKPro TC Format.
        :param ctest:Token-List of ctest in DKPro TC format
        :param sentence: Sentece to add
        :return:List with added sentece
        '''
        token = nltk.word_tokenize(sentence)

        if not ctest:
            if len(token) > 20:
                [ctest.append(t + "\n") for t in token[-20:]]
            else:
                [ctest.append(t + "\n") for t in token]
        else:
            if len(token) > 20:
                [ctest.append(t + "\n") for t in token[:20]]
            else:
                [ctest.append(t + "\n") for t in token]
        ctest.append("----\n")

    def generate_ctest(self, txt):
        '''
        This method generates a ctest in the DKPro TC Format according to the greedy strategy.
        The startegy suggests to generate a ctest containing always 20 gaps with one candidate in between.
        Candidates are all normal words with more than two character.
        The first and the last sentece have no mutilations to provide context information.
        :param txt: Text to produce ctest from
        :return:None if text is not appropriate for the test generation or ctest in TC Format if successful
        '''
        ctest_tc = []
        ctest =[]
        gapId = 1;
        tN = 0
        sents = [s.replace("\n", "") for s in nltk.sent_tokenize(txt)[:30]]

        if len(sents) < 2:
            print("input should contain at least 2 sentences, but only ", len(sents), "were detected")
            return

        self.generateContextSents(ctest_tc, sents[0])
        ctest.append(sents[0])
        for n, sent in enumerate(sents[1:]):
            if sents[n + 1] == None or gapId > 20:
                self.generateContextSents(ctest_tc, sent)
                ctest.append(sent)
                return ("".join(ctest_tc),"".join(ctest))
            else:
                for i, t in enumerate(nltk.word_tokenize(sent)):
                    if t.isalpha() and not t.isupper() and len(t) > 1:
                        if tN % 2 == 1 and not gapId > 20:
                            cut = int(len(t) / 2)
                            ctest_tc.append(t + "\t" + str(gapId) + "\t" + t[0:cut] + "\t" + "0.0" + "\n")
                            ctest.append(" "+t[0:cut]+"_"*(len(t)-cut))
                            gapId += 1
                        else:
                            ctest_tc.append(t + "\n")
                            ctest.append(" "+t)
                        tN += 1
                    else:
                        ctest_tc.append(t + "\n")
                        if t not in [".",",","\'",'\'','?','!',';','&','(',')']:
                            ctest.append(" "+t)
                        else: ctest.append(t)
                ctest_tc.append("----\n")
        return None

    def retrieve_ctest(self):
        '''
        This method produces a c-test with the desired difficulty level. If the exact difficulty level can not be hit,
        the closes c-test gets retrieved.
        :return:ctest as tuple of tc-Format, Human readable and the predicted difficulty.
        '''
        start = timeit.default_timer()
        for n,text in enumerate(self.texts):
            ctestTemp = self.generate_ctest(text)
            if ctestTemp is not None:
                prediction = self.predict(ctestTemp[0])
                if self.evaluator(prediction[0],self.targetLevel)==0:
                    print("Test generation took", timeit.default_timer() - start)
                    return ctestTemp+(prediction[0],)
                elif self.ctest is None or abs(self.evaluator(prediction[0])) < self.delta:
                    self.ctest = ctestTemp+(prediction[0],)
                    self.delta = abs(self.evaluator(prediction[0]))

        print("Test generation took",timeit.default_timer() - start)
        print("Delta",self.delta)
        return self.ctest
