import argparse
import json
import os
from scipy.stats import ttest_rel


def co_storm_extract_cited_snippets(log_file_path):
    urls = set()
    with open(log_file_path) as f:
        data = json.load(f)
        for citation_idx, citation_data in data["knowledge_base"]["info_uuid_to_info_dict"].items():
            urls.add(citation_data["url"])
    return urls

def baseline_extract_cited_snippets(log_file_path):
    urls = set()
    with open(log_file_path) as f:
        data = json.load(f)
        for url, url_data in data["url_to_info"].items():
            urls.add(url)
    return urls
     

def main(args):
    base_dir = args.base_dir
    all_urls = {}
    for method_name in os.listdir(base_dir):
        method_path = os.path.join(base_dir, method_name)
        for article_name in os.listdir(method_path):
            article_path = os.path.join(method_path, article_name)
            cleaned_article_name = article_name.replace("_", " ").strip()
            urls = None
            if os.path.exists(os.path.join(article_path, "round_table_dump.json")):
                urls = co_storm_extract_cited_snippets(os.path.join(article_path, "round_table_dump.json"))
            elif os.path.exists(os.path.join(article_path, "url_to_info.json")):
                urls = baseline_extract_cited_snippets(os.path.join(article_path, "url_to_info.json"))
            if urls is not None:
                if method_name not in all_urls:
                    all_urls[method_name] = []
                all_urls[method_name].append(len(urls))
    
    for method_name, lengths in sorted(all_urls.items()):
        print(f"{method_name}: {round(sum(lengths) / len(lengths), 3)}")
    
    new_method_lengths = all_urls['new_method']

    for method_name, lengths in sorted(all_urls.items()):
        if method_name != 'new_method':
            t_stat, p_value = ttest_rel(new_method_lengths, lengths)
            print(f"\nPaired t-test results for new_method vs {method_name}:")
            print(f"t-statistic: {t_stat:.3f}")
            print(f"p-value: {p_value:.3f}")


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Preapre discourse trace for auto evaluation')
    parser.add_argument('-b', '--base-dir', type=str, default='../evaluation_dataset/raw_experiments_data/',
                        help='Base directory for raw experiments data')

    args = parser.parse_args()
    main(args)
