#include <ctime>
#include <memory>
#include <fstream>
#include <iostream>
#include <algorithm>
#include <cstdlib>

#include "io/edsgraph.h"
#include "io/rule.h"
#include "baseline_transducer.h"
#include "baseline_run.h"

#include "io/command_line.h"

namespace baseline {

void Run::initialize() const {
    auto& vm = commandLineVariables();
    if (vm.count("beam_size"))
        g_beam_size = vm["beam_size"].as< int >();
    if (vm.count("add_gold_answer_when_beam_is_empty"))
        g_add_gold_answer_when_beam_is_empty =
            vm["add_gold_answer_when_beam_is_empty"].as< int >();
}

void Run::train(const std::string& input_file,
                const std::string& rules_file,
                const std::string& feature_input,
                const std::string& feature_output) const {
    std::vector< EdsGraph > graphs;
    std::vector< int > random;

    initialize();

    loadGraphsFromFile(input_file, graphs, random);
    int graph_count = random.size();

    RuleSet rule_set;
    loadRulesFromFile(rules_file, rule_set);

    std::unique_ptr< BeamSearchTransducer >
        transducer(new BeamSearchTransducer(feature_input, feature_output,
                                            rule_set, TransducerState::TRAIN,
                                            g_beam_size));


    for (int round = 1; round <= graph_count; ++round)
        transducer->train(graphs[random[round - 1]], round);

    transducer->printInformation();
    transducer->finishTraining();
}

void Run::transduce(const std::string& input_file,
                    const std::string& output_file,
                    const std::string& rules_file,
                    const std::string& feature_file,
                    bool with_detail) const {
    EdsGraph graph;
    RuleSet rule_set;

    initialize();

    loadRulesFromFile(rules_file, rule_set);

    std::ifstream is(input_file);
    std::ofstream os(output_file, std::ios::out);
    if (!is) {
        LOG_ERROR(<< "Can't find file " << input_file);
        return;
    }

    std::unique_ptr< BeamSearchTransducer >
        transducer(new BeamSearchTransducer(feature_file, feature_file,
                                            rule_set, TransducerState::TRANSDUCE,
                                            g_beam_size));

    int round = 0, count = 0;
    while (is >> graph && graph.nodes.size() < MAX_GRAPH_NODE_COUNT) {
        os << graph.filename << ": ";
        if (transducer->transduce(graph, with_detail)) {
            os << transducer->result();
            ++count;
        }
        os << '\n';
        if (with_detail && transducer->hasResultComputed()) {
            transducer->printResultToStream(os, true);
            os << '\n';
        }
        if (++round % 200 == 0)
            LOG_INFO(<< "Transduced " << ( float )count / round << ' '
                     << count << '/' << round);
    }

    LOG_INFO(<< "Transduced " << ( float )count / round << ' '
             << count << '/' << round);

    os.close();
}

void Run::goldTest(const std::string& input_file,
                   const std::string& rules_file,
                   const std::string& feature_input) const {
    initialize();
}
}
