///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <getopt.h>

#include "nl-iomacros.h"
#include "nl-cpt.h"


////////////////////////////////////////////////////////////////////////////////
//
//  Given Variables
//
////////////////////////////////////////////////////////////////////////////////

char psX[]   = "";
char psSpc[] = " ";

//// G: grammar rule symbol
DiscreteDomain<int> domG;
typedef DiscreteDomainRV<int,domG> G;
const G G_BOT ("-");
const G G_TOP ("REST");

//// GG: pair of grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,G,psSpc,G,psX> GG;
const GG GG_BOT ( G_BOT, G_BOT );

//// P: pos symbol
DiscreteDomain<int> domP;
typedef DiscreteDomainRV<int,domP> P;
const P P_BOT ("-");

//// W: pos symbol
DiscreteDomain<int> domW;
typedef DiscreteDomainRV<int,domW> W;
const W W_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Given Models
//
////////////////////////////////////////////////////////////////////////////////

//// Model of G (prior)
typedef CPT1DModel<G,Prob> GModel;
//// Model of GG given G (binary branching expansion)
typedef CPT2DModel<GG,G,Prob> GGgModel;

//// Model of term P given G
typedef CPT2DModel<P,G,Prob> PgModel;
//// Model of term P given W
typedef CPT2DModel<P,W,Prob> PwModel;
//// Model of term P
typedef CPT1DModel<P,Prob> PModel;
//// Model of word W
typedef CPT1DModel<W,Prob> WModel;


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Variables
//
////////////////////////////////////////////////////////////////////////////////

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
typedef DiscreteDomainRV<char,domD> D;

//// K: length (input only, to HHMM models)...
DiscreteDomain<char> domK;
typedef DiscreteDomainRV<char,domK> K;

//// Q: incomplete category state
DiscreteDomain<int> domQ;
typedef DiscreteDomainRV<int,domQ> Q;
const Q Q_BOT ("-/-");

//// B: boolean
DiscreteDomain<int> domB;
typedef DiscreteDomainRV<int,domB> B;
const B B_0 ("0");
const B B_1 ("1");

//// F: final state
DiscreteDomain<int> domF;
typedef DiscreteDomainRV<int,domF> F;
const F F_0 ("0");
const F F_1 ("1");
const F F_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Models
//
////////////////////////////////////////////////////////////////////////////////

/*
//// Placeholder model of B given D and G (from above)
typedef CPT3DModel<B,D,G,Prob> BphModel;
*/
//// Expansion model of G given D and G (from above)
typedef CPT3DModel<G,D,G,Prob> GeModel;
//// Transition model of G given D and G (awaited cat from previous) and G (from reduction)
typedef CPT4DModel<G,D,G,G,Prob> GtmModel;
//// Transition model of G (active cat) given D and G (active cat from previous) and G (awaited cat from above)
typedef CPT4DModel<G,D,G,G,Prob> GtpModel;
//// Transition model of G (awaited cat) given D and G (active cat from current) and G (active cat from previous)
typedef CPT4DModel<G,D,G,G,Prob> GtqModel;
//// Reduction model of F given D and G (active cat from previous) and G (awaited cat from above) (only check if previous awaited = reduction)
typedef CPT4DModel<F,D,G,G,Prob> FrModel;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
///
///  Main function
///
////////////////////////////////////////////////////////////////////////////////

static struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {0, 0, 0, 0}};


void printUsage(char* progname) {
  //fprintf(stderr,"Usage: cat <RAW_FILE> | %s [OPTIONS]... [MODEL_FILE1] [MODEL_FILE2] ...\n",progname);
  //fprintf(stderr,"Runs the speech recognition system on RAW_FILE using the models in MODEL_FILE1, 2, etc.\n");
  fprintf(stderr," System variable Settings:\n");
  fprintf(stderr,"  -h, --help\t\tPrint this message\n");
}



void writeRand ( G& g_mu, GGgModel& mGGg ) {
  if ( mGGg.contains(g_mu) ) {
    Prob p = double(rand())/double(RAND_MAX);
    Prob pr = p;
//    cerr<<p<<"\n";
    // For each g_mu0, g_mu1...
    GGgModel::IterVal gg_mu0_mu1; for ( bool b_mu0_mu1=mGGg.setFirst(gg_mu0_mu1,g_mu); b_mu0_mu1; b_mu0_mu1=mGGg.setNext(gg_mu0_mu1,g_mu) ) {
      p -= mGGg.getProb(gg_mu0_mu1,g_mu);
      if ( p<=0.0 ) break;
    }
    if ( gg_mu0_mu1 != GG_BOT ) {
      cout<<" ("<<g_mu;
      writeRand(gg_mu0_mu1.first,mGGg);
      writeRand(gg_mu0_mu1.second,mGGg);
      cout<<")";
    } else {
      cout<<" ("<<g_mu<<" "<<g_mu<<"#"<<g_mu<<")";
    }
  } else {
    cout<<" ("<<g_mu<<" "<<g_mu<<"#"<<g_mu<<")";
  }
}


int main (int nArgs, char* argv[]) {

  // Given models...
  GModel   mGr;
  GGgModel mGGg;
  PgModel  mPg;
  PwModel  mPw;
  PModel   mP;
  WModel   mW;


  //////////////////// PART 0: COMMAND-LINE OPTIONS

  // Parse command-line options
  char* progname = strdup(argv[0]);
  int option_index = 0;
  char opt;
  while( (opt = getopt_long(nArgs, argv, "hp:q:Vv", long_options, &option_index)) != -1 ) {
    switch(opt) {
    case 'h': printUsage(progname);
              exit(0);
    //case 'p': UPDATE_PORT_NUM  = atoi(optarg); break;
    //case 'q': QUERY_PORT_NUM = atoi(optarg); break;
    //case 'V': WorldModel::OUTPUT_VERY_VERBOSE = true; break; // do break.
    //case 'v': WorldModel::OUTPUT_VERBOSE = true; break;
    default: break;
    }
  }
  // Complain if too few args...
  if (optind > nArgs ) {
    printUsage(progname);
    exit(0);
  }


  //////////////////// PART I. LOAD GIVEN MODELS (FROM STDIN)

  FILE* pf = stdin;

  int c=' '; int i=0; int line=1; String sBuff(1000);                          // Lookahead/ctrs/buffers
  CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                   // Get to first record
  while ( c!=-1 && c!='\0' && c!='\5' ) {                                      // For each record
    CONSUME_STR ( pf, c, (c!='\n' && c!='\0' && c!='\5'), sBuff, i, line );    //   Consume line
    StringInput si(sBuff.c_array());
    if ( !( sBuff[0]=='#' ||                                                   //   Accept comments/fields
	    (si>>"Gr ">>mGr >>"\0"!=NULL) ||
	    (si>>"GG ">>mGGg>>"\0"!=NULL) ||
            (si>>"Pg ">>mPg >>"\0"!=NULL) ||
	    (si>>"Pw ">>mPw >>"\0"!=NULL) ||
	    (si>>"P " >>mP  >>"\0"!=NULL) ||
	    (si>>"W " >>mW  >>"\0"!=NULL) ) )
      cerr<<"\nERROR: unable to process line "<<line<<": "<<sBuff<<"\n";
    CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                 //   Consume whitespace
  }

//  mPg.dump(cout,"Pg");
//  mPw.dump(cout,"Pw");
//  mP.dump(cout,"P");
//  mW.dump(cout,"W");

  mPg.clear();
  mPw.clear();
  mP.clear();
  mW.clear();

  //////////////////// PART II. BUILD DERIVED MODELS

  for ( int i=0; i<39883; i++ ) {
    cerr<<i<<"\n";
    Prob p = float(rand())/RAND_MAX;
    // For each g_mu0, g_mu1...
    GModel::IterVal g_mu; for ( bool b_mu=mGr.setFirst(g_mu); b_mu; b_mu=mGr.setNext(g_mu) ) {
      p -= mGr.getProb(g_mu);
      if ( p<=0.0 ) break;
    }
    writeRand(g_mu,mGGg);
    cout<<"\n";
  }
}

