# Copyright (c) 2021 doXray B.V., Neede, Netherlands
#
# Code is a supplementary material for the paper:
#   Title: Speeding Up Transformer Training By Using Dataset Subsampling - An Exploratory Analysis
#   Authors: Lovre Torbarina, Velimir Mihelcic, Bruno sarlija, Lukasz Roguski and Zeljko Kraljevic
#   Venue: SustaiNLP @ EMNLP 2021
# 
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
# 
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import numpy as np
import pandas as pd
import math
import torch
from tqdm import tqdm
from abc import ABC, abstractmethod

from kneed import KneeLocator
from sklearn.cluster import KMeans
from sklearn.preprocessing import Normalizer
from sklearn.decomposition import TruncatedSVD
from sklearn.feature_extraction.text import TfidfVectorizer
from sklearn.pipeline import make_pipeline
from transformers import AutoModelForSequenceClassification, AutoTokenizer, AutoConfig

pd.options.mode.chained_assignment = None  # default='warn'


class TransformerDatasetForSequenceClassification:

    def __init__(self, dataset, tokenizer, max_seq_length):
        self.tokenizer = tokenizer
        self.dataset = []
        self.sample_counter = 0
        self.dataset = dataset
        self.max_seq_length = max_seq_length
        self.dataset_len = len(dataset)

    def __len__(self):
        return self.dataset_len

    def __getitem__(self, idx):
        raw_tokens = self.dataset[idx]
        input_ids = np.array(self.tokenizer.encode(' '.join(raw_tokens), truncation=True, add_special_tokens=True, max_length=self.max_seq_length))
        ids = np.full(self.max_seq_length, self.tokenizer.pad_token_id, dtype=np.int)
        mask = np.zeros(self.max_seq_length, dtype=np.int)
        ids[:input_ids.shape[0]] = input_ids
        mask[:input_ids.shape[0]] = 1
        return torch.tensor(ids), torch.tensor(mask)


class BaseSampler(ABC):

    @abstractmethod
    def sample(self, dfs):
        '''dfs: list of pandas dataframes for each time segment, sorted from the oldest to the most recent time segment.'''
        pass

    @staticmethod
    def _read_csv(file_path):
        return pd.read_csv(file_path)


class FullDataSampler(BaseSampler):

    def sample(self, dfs):
        df_size = sum([len(d) for d in dfs])
        print(f'Size before sampling: {df_size}')
        df = pd.concat(dfs, ignore_index=True)
        print(f'Size after sampling: {len(df)}. Perc of original: {len(df)/df_size*100}%')
        return df


class LatestExamplesSampler(BaseSampler):

    def __init__(self, sampling_ratio):
        super().__init__()
        self.sampling_ratio = sampling_ratio

    def sample(self, dfs):
        for i, df in enumerate(dfs):
            df['timesegment'] = i
        df = pd.concat(dfs, ignore_index=True)
        df_size = len(df)
        print(f'Size before sampling: {df_size}')
        dfs_t = []
        gp = df.groupby(['category'])
        for k in gp.groups.keys():
            df_g = gp.get_group(k)
            df_g = df_g.sort_values('timesegment', ascending=False)
            new_size = math.ceil(self.sampling_ratio*len(df_g))
            df_g = df_g.iloc[:new_size]
            dfs_t.append(df_g.copy())
        df = pd.concat(dfs_t, ignore_index=True)
        del df['timesegment']
        print(f'Size after sampling: {len(df)}. Perc of original: {len(df)/df_size*100}%')
        return df


class RandomSampler(BaseSampler):

    def __init__(self, sampling_ratio):
        super().__init__()
        self.sampling_ratio = sampling_ratio

    def sample(self, dfs):
        df = pd.concat(dfs)
        df_size = len(df)
        print(f'Size before sampling: {df_size}')
        df = df.groupby('category').sample(
            frac=self.sampling_ratio,
            replace=False,
            weights=None,
            random_state=42
        )
        print(f'Size after sampling: {len(df)}. Perc of original: {len(df)/df_size*100}%')
        return df


class ClusteringSampler(BaseSampler):
    # for C1, C2, C3, C4 set use_weights and use_transformer_embeddings flags to get specific sampler
    def __init__(
            self,
            sampling_ratio,
            max_clusters=10,
            noise_perc=0.01,
            use_transformer_embeddings=False,
            use_weights=False,
            model_name='',
            max_seq_length=-1,
            batch_size=-1,
            init_weights_path='',
            num_labels=4,
            gpu_devices=[],
            num_workers=1
    ):
        super().__init__()
        self.sampling_ratio = sampling_ratio
        self.max_clusters = max_clusters
        self.noise_perc = noise_perc
        self.use_transformer_embeddings = use_transformer_embeddings
        self.use_weights = use_weights
        self.batch_size = batch_size
        self.num_workers = num_workers
        if self.use_transformer_embeddings:
            self.max_seq_length = max_seq_length
            self.device = 'cuda:0' if torch.cuda.is_available() else 'cpu'
            self.tokenizer = AutoTokenizer.from_pretrained(model_name, do_lower_case=False, do_basic_tokenize=False)
            self.config = AutoConfig.from_pretrained(model_name, return_dict=False, output_hidden_states=False, num_labels=num_labels)
            self.config.return_cls_embeddings=True
            self.model = AutoModelForSequenceClassification.from_pretrained(model_name, config=self.config)
            self.model.to(self.device)
            self.model.load_state_dict(torch.load(init_weights_path, map_location=self.device), strict=True)
            self.model = torch.nn.DataParallel(self.model, device_ids=gpu_devices)
        else:
            self.vectorizer = TfidfVectorizer(
                max_df=0.8,
                max_features=20000,
                min_df=0.1,
                ngram_range=(1,3),
            )
        self.svd = TruncatedSVD(n_components=5, n_iter=7, random_state=42)
        self.normalizer = Normalizer(copy=False)
        self.lsa = make_pipeline(self.svd, self.normalizer)
        self.kmeans_kwargs = {       
            "init": "k-means++",
            "n_init": 100,
            "max_iter": 1000,
            "random_state": 42,
        }

    def sample(self, dfs):
        rng = np.random.default_rng(42)
        for i, df in enumerate(reversed(dfs)):
            df['weight'] = (1.0 / (i+1)) if self.use_weights else 1.0
        df = pd.concat(dfs, ignore_index=True)
        df_size = len(df)
        print(f'Size before sampling: {df_size}')
        examples_per_category = {}
        if self.use_transformer_embeddings:
            self.model.eval()
            with torch.no_grad():
                texts = df['parsed_text'].astype('str').tolist()
                tokens = []
                for text in texts:
                    tokens.append(text.split()[:self.max_seq_length])
                embeddings = np.zeros((len(texts), 768))
                dataset = TransformerDatasetForSequenceClassification(tokens, self.tokenizer, self.max_seq_length)
                dataloader = torch.utils.data.DataLoader(dataset, batch_size=self.batch_size, shuffle=False, num_workers=self.num_workers, pin_memory=True)
                for i, (ids, mask) in enumerate(tqdm(dataloader)):
                    cls_embeddings = self.model(input_ids=ids.to(self.device), attention_mask=mask.to(self.device))
                    embeddings[i*self.batch_size:(i+1)*self.batch_size, :] = cls_embeddings.cpu().numpy()
                for c in tqdm(set(df['category'].tolist())):
                    weights = df['weight'][df['category']==c]
                    ids = weights.index.values
                    cat_embeddings = embeddings[ids, :]
                    examples_per_category[c] = [ids, cat_embeddings, weights.tolist()]
        else:
            for c in tqdm(set(df['category'].tolist())):
                texts = df['parsed_text'][df['category']==c]
                weights = df['weight'][df['category']==c].tolist()
                ids = texts.index.values
                texts = texts.tolist()
                examples_per_category[c] = [ids, texts, weights]
        sampled_ids = []
        for _, v in tqdm(examples_per_category.items()):
            ids = v[0]
            n_examples = len(ids)
            sample_size = math.ceil(n_examples*self.sampling_ratio)
            if sample_size > 0:
                max_clusters = sample_size if sample_size < self.max_clusters else self.max_clusters
                texts = v[1]
                weights = v[2]
                if self.use_transformer_embeddings:
                    X = texts
                else:
                    X = self.vectorizer.fit_transform(texts)
                X_lsa = self.lsa.fit_transform(X)
                sse = []
                id_labels = []
                for k in range(1, max_clusters+1):
                    kmeans = KMeans(n_clusters=k, **self.kmeans_kwargs)
                    kmeans.fit(X_lsa)
                    sse.append(kmeans.inertia_)
                    id_labels.append(kmeans.labels_)
                kl = KneeLocator(range(1, max_clusters+1), sse, curve="convex", direction="decreasing")
                n_clusters_best = kl.elbow if kl.elbow != None else max_clusters
                examples_per_cluster = [[] for i in range(n_clusters_best)]
                for id, cluster, weight in zip(ids, id_labels[n_clusters_best-1], weights):
                    examples_per_cluster[cluster].append((id, weight))
                cluster_weights = []
                examples_avaliable_per_cluster = []
                for c in range(n_clusters_best):
                    n_examples = len(examples_per_cluster[c])
                    weights = np.array([w for _, w in examples_per_cluster[c]])
                    avg_weight = np.sum(weights) / n_examples
                    cluster_weights.append(avg_weight)
                    examples_avaliable_per_cluster.append(n_examples)
                examples_avaliable_per_cluster = np.array(examples_avaliable_per_cluster)
                cluster_rel_size = examples_avaliable_per_cluster / np.sum(examples_avaliable_per_cluster)
                examples_avaliable_per_cluster[cluster_rel_size < self.noise_perc] = 0
                clusters_sample_size = [0 for _ in range(n_clusters_best)]
                for i in range(sample_size):
                    a = [c for c in range(n_clusters_best) if examples_avaliable_per_cluster[c] > 0]
                    weights = np.array([cluster_weights[c] for c in a])
                    p = weights / np.sum(weights)
                    picked_cluster = rng.choice(a, size=1, replace=False, p=p).item()
                    clusters_sample_size[picked_cluster] += 1
                    examples_avaliable_per_cluster[picked_cluster] -= 1
                for c in range(n_clusters_best):
                    cluster_sample_size = clusters_sample_size[c]
                    if cluster_sample_size > 0:
                        ids = [i for i, _ in examples_per_cluster[c]]
                        weights = np.array([w for _, w in examples_per_cluster[c]])
                        p = weights / np.sum(weights)
                        sample = rng.choice(ids, size=cluster_sample_size, replace=False, p=p)
                        sampled_ids += list(sample)
            else:
                print('Empty category!')
        df = df.iloc[sampled_ids]
        del df['weight']
        print(f'Size after sampling: {len(df)}. Perc of original: {len(df)/df_size*100}%')
        return df
