import React, { Component } from 'react';
import { withTracker } from 'meteor/react-meteor-data';
import { Random } from 'meteor/random'

import Token from './Token.jsx';
import Mention from './Mention.jsx';
import { gup } from '../util/gup.js';

function getClusterWithMention(clusters, mention) {
  return clusters.find(cluster => {
    return cluster.mentions.some(other => {
      return other.start == mention.start && other.end == mention.end && other.label == mention.label;
    });
  });
}

function ignoreMention(mention, checks) {
  const decision = checks.some(check => {
    if (
      check.decision == "one-entity-at-a-time" ||
      check.decision == "full" ||
      check.decision == "freq-ents"
    ) {
      return false;
    } else if (
      check.decision != 'remove' &&
      (mention.start != check.mention.start ||
        check.mention.end != mention.end) &&
      ((mention.start <= check.mention.start &&
        check.mention.start <= mention.end) ||
       (check.mention.start <= mention.start &&
         mention.start <= check.mention.end))
    ) {
      return true;
    } else if ((
      check.decision == 'remove' || check.decision == 'hide') &&
      mention.start == check.mention.start &&
      mention.end == check.mention.end &&
      mention.label == check.mention.label
    ) {
      return true;
    }
  });
  return decision;
}

function getSmallestMatchingMention(clusters, token_id, checks) {
  const matching = clusters.map(cluster => {
    // Get the smallest in this cluster that matches
    const smallest = cluster.mentions.reduce((best, mention) => {
      if (ignoreMention(mention, checks)) return best;

      if (mention.start <= token_id && token_id <= mention.end) {
        if (best != null) {
          const bLength = best.end - best.start;
          const mLength = mention.end - mention.start;
          return bLength < mLength ? best : mention;
        } else {
          return mention;
        }
      } else {
        return best;
      }
    }, null);
    return {mention: smallest, cluster: cluster};
  }).filter(pair => {
    return pair.mention != null;
  }).sort((a, b) => {
    const aLength = a.mention.end - a.mention.start;
    const bLength = b.mention.end - b.mention.start;
    return aLength < bLength ? -1 : 1;
  });
  return matching.length == 0 ? null : matching[0];
}

function getMatchingMention(start, end, label, clusters) {
  const options = clusters.map(cluster => {
    return cluster.mentions.filter(mention => {
      return mention.start == start && mention.end == end && mention.label == label;
    });
  }).filter(mentions => mentions.length > 0);
  if (options.length > 0) return options[0][0];
  else return null;
}

function validMention(start, end, clusters, ref_clusters, checks, allowNesting=true) {
  return clusters.every(cluster => {
    return cluster.mentions.every(other => {
      if (ignoreMention(other, checks)) return true;
      const inside = (other.start <= start && end <= other.end);
      const covers = (start <= other.start && other.end <= end);
      const left = (end < other.start);
      const right = (other.end < start);
      const match = (other.start == start && other.end == end);
      if (! allowNesting) {
        return (! inside) && (! covers) && (left || right) && (! match);
      } else {
        return (inside || covers || left || right) && (! match);
      }
    });
  }) && (
    ref_clusters == undefined || 
    ref_clusters.every(cluster => {
      return cluster.mentions.every(other => {
        const inside = (other.start <= start && end <= other.end);
        const covers = (start <= other.start && other.end <= end);
        const left = (end < other.start);
        const right = (other.end < start);
        return (inside || covers || left || right);
      });
    })
  );
}

function getMentionFromMap(mentionMap, token_id, focus, before_only, after_only) {
  if (token_id in mentionMap) {
    return mentionMap[token_id].reduce((largest, mention) => {
      const start = mention.mention.start;
      const end = mention.mention.end;
      if (focus != undefined && focus != null &&
        start <= focus.start && focus.end <= end &&
        (start != focus.start || focus.end != end)
      ) {
        return largest;
      } else if (focus != null && before_only && focus.start < start) {
        return largest;
      } else if (focus != null && after_only && end < focus.end) {
        return largest;
      } else if (largest != null && end < largest.mention.end) {
        return largest;
      } else {
        return mention;
      }
    }, null);
  } else {
    return null;
  }
}

function getClusterWithId(queryId, clusters) {
  return clusters.find(cluster => { return cluster._id == queryId; });
}

export default class TextDoc extends Component {
  constructor(props) {
    super(props);
    this.handleEnter = this.handleEnter.bind(this);
    this.handleLeave = this.handleLeave.bind(this);
    this.handleDown = this.handleDown.bind(this);
    this.handleUp = this.handleUp.bind(this);
    this.handleClick = this.handleClick.bind(this);
    this.handleEnterMention = this.handleEnterMention.bind(this);
    this.handleLeaveMention = this.handleLeaveMention.bind(this);
    this.handleDownMention = this.handleDownMention.bind(this);
    this.handleUpMention = this.handleUpMention.bind(this);
    this.handleClickMention = this.handleClickMention.bind(this);
    this.handleDoubleClick = this.handleDoubleClick.bind(this);
    this.handleDownLabel = this.handleDownLabel.bind(this);
    this.handleUpLabel = this.handleUpLabel.bind(this);

    this.noPuncSpan = this.noPuncSpan.bind(this);
    this.mergeClusters = this.mergeClusters.bind(this);
    this.splitMentionFromCluster = this.splitMentionFromCluster.bind(this);
    this.removeCluster = this.removeCluster.bind(this);
    this.removeMention = this.removeMention.bind(this);
    this.addMentionToCluster = this.addMentionToCluster.bind(this);
    this.createClusterFromMention = this.createClusterFromMention.bind(this);
    this.updateValidation = this.updateValidation.bind(this);

    // Form https://sashat.me/2017/01/11/list-of-20-simple-distinct-colors/
    const colors = [
      "#e6194B", "#3cb44b", "#ffe119", "#4363d8", "#f58231", "#911eb4",
      "#42d4f4", "#f032e6", "#bfef45", "#fabebe", "#469990", "#e6beff",
      "#9A6324", "#fffac8", "#800000", "#aaffc3", "#808000", "#ffd8b1",
      "#000075", "#a9a9a9",
    ];

    const newColors = colors.filter((color) => {
      return ! this.props.clusters.some((cluster) => {
        return cluster.color == color;
      });
    });

    const focusClusterId = (
      this.props.focusMention === undefined ||
      this.props.focusMention === null
    ) ? null : getClusterWithMention(this.props.clusters, this.props.focusMention)._id;

    this.state = {
      startSelected: -1,
      endSelected: -1,
      mouseIsDown: false,
      current: -1,
      selectedClusterId: focusClusterId,
      selectedMention: [],
      availableColors: newColors,
      actions: 0,
      warnings: 0,
      user: gup("workerId"),
    };
  }

  ////////////////////////////////////////////////////
  // General actions (ie. not UI specific)

  updateValidation() {
    this.setState((state, props) => {
      this.props.validate();
      return {
        actions: state.actions + 1
      };
    });
  }

  mergeClusters(cluster_keep, cluster_merge) {
    if (cluster_keep &&
      cluster_merge &&
      cluster_keep._id == cluster_merge._id
    ) {
    } else {
      const new_cluster = {
        _id: cluster_keep._id,
        mentions: [...cluster_keep.mentions, ...cluster_merge.mentions],
        color: cluster_keep.color,
      };
      Meteor.call('clusters.merge', this.props.text._id, this.props.ann_id, cluster_merge._id, cluster_keep._id, new_cluster, this.state.user);
      this.updateValidation();
      this.setState({
        selectedClusterId: new_cluster._id,
      });
    }
  }

  splitMentionFromCluster(cluster, mention, color="") {
    const new_cluster0 = {
      _id: cluster._id,
      mentions: cluster.mentions.filter(other => {
        return other.start != mention.start || other.end != mention.end;
      }),
      color: cluster.color,
    };
    const new_cluster1 = {
      _id: Random.id(),
      mentions: [mention],
      color: color != "" ? color : this.state.availableColors[0],
    };
    Meteor.call('clusters.split', this.props.text._id, this.props.ann_id, cluster._id, new_cluster0, new_cluster1, this.state.user);
    this.updateValidation();
    this.setState({
      selectedClusterId: new_cluster0._id,
    });

    if (color == "" && this.state.availableColors.length > 1) {
      const remainingColors = this.state.availableColors.slice(1);
      this.setState({
        availableColors: remainingColors,
      });
    }
  }

  removeMention(cluster, mention) {
    if (cluster.mentions.length == 1) {
      this.removeCluster(cluster);
    } else {
      Meteor.call('clusters.contract', this.props.text._id, this.props.ann_id, cluster._id, mention.start, mention.end, mention.label, this.state.user);
      this.updateValidation();
    }
  }

  removeCluster(cluster) {
    Meteor.call('clusters.remove', this.props.text._id, this.props.ann_id, cluster._id, cluster.mentions, this.state.user);
    this.updateValidation();
    if (this.state.selectedClusterId == cluster._id) {
      this.setState({
        selectedClusterId: null,
      });
    }
  }

  addMentionToCluster(cluster, start, end, label="") {
    Meteor.call('clusters.extend', this.props.text._id, this.props.ann_id, cluster._id, start, end, this.state.user);
    this.updateValidation();
    const nCluster = {
      _id: cluster._id,
      mentions: [...cluster.mentions, {start: start, end:end, label:label}],
      color: cluster.color,
    }
    this.setState({
      selectedClusterId: nCluster._id,
    });
  }

  createClusterFromMention(start, end, color, label="") {
    const cluster = {
      _id: Random.id(),
      mentions: [{start: start, end:end, label:label}],
      color: color != null ? color : this.state.availableColors[0],
    };
    if (color == null && this.state.availableColors.length > 1) {
      const remainingColors = this.state.availableColors.slice(1);
      this.setState({
        availableColors: remainingColors,
      });
    }
    Meteor.call('clusters.insert', this.props.text._id, this.props.ann_id, cluster, this.state.user);
    this.updateValidation();
  }

  ////////////////////////////////////////////////////
  // UI Specific processing

  handleEnter(token_id, shift, alt) {
    if (this.props.readOnly) return;

    if (
      this.props.ui == "conditionals-from-actions" &&
      this.props.focusMention != null &&
      this.props.focusMention.start <= token_id &&
      this.props.focusMention.end >= token_id
    ) {
      // No update
    } else if (this.props.ui == "freq-ents") {
      // No update
    } else if (this.props.ui.endsWith("editable") && this.state.mouseIsDown) {
      this.setState({
        endSelected: token_id, 
        current: token_id,
      });
    } else if (
      this.props.ui.startsWith("check-mentions") ||
      this.props.ui.startsWith("one-entity-at-a-time") ||
      this.props.ui.startsWith('link-to-one')
    ) {
      // Find the smallest mention we are inside and set it
      const matching = getSmallestMatchingMention(this.props.clusters, token_id, this.props.checks);
      if (matching) {
        this.setState(prevState => ({
          selectedMention: [...prevState.selectedMention.filter, matching.mention]
        }));
      } else if (this.props.ui.endsWith("editable")) {
        this.setState({
          startSelected: token_id,
          endSelected: token_id, 
          current: token_id,
        });
      }
    } else {
      if (! this.state.mouseIsDown) {
        this.setState({ startSelected: token_id });
      }
      this.setState({
        endSelected: token_id, 
        current: token_id,
      });
    }
  }
  handleEnterMention(mention, shift, alt) {
    if (this.props.readOnly) return;
    
    if (this.props.ui.endsWith("editable") && this.state.mouseIsDown) {
      //
    } else if (
      this.props.ui.startsWith("check-mentions") ||
      this.props.ui.startsWith("one-entity-at-a-time") ||
      this.props.ui == "freq-ents" ||
      this.props.ui.startsWith('link-to-one')
    ) {
      this.setState(prevState => ({
        selectedMention: [...prevState.selectedMention, mention]
      }));
    }
  }
  handleLeave(token_id, shift, alt) {
    if (this.props.readOnly) return;
    
    if (
      this.props.ui.startsWith("one-entity-at-a-time") ||
      this.props.ui.startsWith('link-to-one') ||
      this.props.ui == "freq-ents"
    ) {
      // No update
    } else if (this.props.ui.startsWith("check-mentions") &&
      (!this.props.ui.endsWith("editable"))) {
      //
    } else {
      if (this.state.current == token_id) {
        this.setState({ current: -1 });
      }
      if (this.state.startSelected == token_id &&
        this.state.endSelected == token_id &&
        this.state.mouseIsDown == false
      ) {
        this.setState({
          startSelected: -1,
          endSelected: -1,
        });
      }
    }
  }
  handleLeaveMention(mention, shift, alt) {
    if (this.props.readOnly) return;

    if (
      this.props.ui.startsWith("check-mentions") ||
      this.props.ui.startsWith("one-entity-at-a-time") ||
      this.props.ui == "freq-ents" ||
      this.props.ui.startsWith('link-to-one')
    ) {
      const isNested = this.props.clusters.some(cluster => {
        return cluster.mentions.some(other => {
          const inside = (other.start <= mention.start && mention.end <= other.end);
        });
      })
      if (isNested) {
        this.setState({
            selectedMention: [],
        });
      } else {
        this.setState(prevState => {
          const nList = prevState.selectedMention.slice(0, prevState.selectedMention.length - 1);
          return {
            selectedMention: nList,
          };
        });
      }
    }
  }
  handleDown(token_id, shift, alt) {
    if (this.props.readOnly) return;

    this.setState({ mouseIsDown: true, });
  }
  handleDownMention(mention, shift, alt) {
    if (this.props.readOnly) return;

    this.setState({ mouseIsDown: true, });
  }
  handleDownLabel(label, mention, shift, alt) {
    if (this.props.readOnly) return;

    this.setState({ mouseIsDown: true, });
  }
  handleUp(token_id, shift, alt) {
    if (this.props.readOnly) return;

    const overlap = getSmallestMatchingMention(this.props.clusters, token_id, this.props.checks);
    if (overlap && this.state.startSelected == this.state.endSelected) {
      return this.handleUpMention(overlap.mention, shift, alt, token_id);
    }

    this.setState({ mouseIsDown: false, });
    const start = Math.min(this.state.startSelected, this.state.endSelected);
    const end = Math.max(this.state.startSelected, this.state.endSelected);

    if (
      this.props.ui.startsWith("one-entity-at-a-time") ||
      this.props.ui.startsWith('link-to-one') ||
      (this.props.ui.startsWith("check-mentions") &&
       (! this.props.ui.endsWith("editable")))
    ) {
      // Do nothing
    } else {
      if (shift) {
      } else if (alt) {
        if (this.props.ui.startsWith("check-mentions")) {
        } else if (this.state.current != -1 && start != end) {
          this.props.clusters.forEach(cluster => {
            if (this.props.ui != "conditionals-from-actions" &&
              cluster.mentions.every(mention => {
              return start <= mention.start && mention.end <= end;
            })) {
              this.removeCluster(cluster);
            } else {
              cluster.mentions.forEach(mention => {
                if (this.props.ui == "conditionals-from-actions" &&
                  mention.start == this.props.focusMention.start &&
                  mention.end == this.props.focusMention.end
                ) {
                  // Ignore attempts to remove the focus mention
                } else if (start <= mention.start && mention.end <= end) {
                  this.removeMention(cluster, mention);
                }
              });
            }
          });
        }
      } else {
        if (this.state.current != -1) {
          if (this.props.ui.startsWith("check-mentions")) {
            // Go through and mark any overlapping mentions as removed
            this.props.clusters.forEach(cluster => {
              cluster.mentions.forEach(other => {
                const left = (end < other.start);
                const right = (other.end < start);
                const match = (other.start == start && other.end == end);
                if (! (left || right || match)) {
                  const toCheck = this.props.checks.filter(check => {
                    return check.mention.start == other.start && check.mention.end == other.end && check.mention.label == other.label;
                  });
                  if (toCheck.length == 0) {
                    Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, other, "remove");
                  } else if (toCheck[0].decision != "remove") {
                    Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, other, "remove");
                  }
                }
              });
            });
            const current = getMatchingMention(start, end, "", this.props.clusters);
            if (current != null) {
              const uncheck = this.props.checks.find(check => {
                return check.decision == 'remove' &&
                  start == check.mention.start &&
                  end == check.mention.end &&
                  check.mention.label == "";
              });
              if (uncheck) {
                if (this.props.ui.includes("include")) {
                  Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, {start: start, end: end, label: ""}, "keep");
                } else {
                  Meteor.call('mentions.uncheck', this.props.text._id, this.state.user, this.props.ann_id, current, "remove");
                }
                const nMention = {start: start, end: end, label: ""};
                this.setState(prevState => ({
                  selectedMention: [...prevState.selectedMention, nMention]
                }))
                this.setState({ 
                  startSelected: -1,
                  endSelected: -1, 
                  current: -1,
                });
              }
            } else if (start >= 0) {
              this.createClusterFromMention(start, end, null);
              if (this.props.ui.includes("include") && this.props.labelOptions.length == 0) {
                Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, {start: start, end: end, label:""}, "keep");
              }
              const nMention = {start: start, end: end, label: ""};
              this.setState(prevState => ({
                selectedMention: [...prevState.selectedMention, nMention]
              }))
              this.setState({ 
                startSelected: -1,
                endSelected: -1, 
                current: -1,
              });
            }
          } else {
            const valid = validMention(start, end, this.props.clusters, this.props.ref_clusters, this.props.checks, this.props.ui != "mentions" && (!this.props.ui.startsWith("check")));
            if (valid) {
              if (this.state.selectedClusterId) {
                const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
                this.addMentionToCluster(selectedCluster, start, end)
              } else {
                const color = (
                  this.props.ui != "mentions" &&
                  this.props.ui != "conditionals" &&
                  this.props.ui != "actions"
                ) ? null : "blue";
                this.createClusterFromMention(start, end, color);
              }
            }
          }
        }
      }

      // Update the UI state to account for where the mouse now is
      if (this.props.ui.startsWith("check-mentions") &&
        this.props.ui.endsWith("editable")
      ) {
        //
      } else {
        this.setState({
          startSelected: token_id,
          endSelected: token_id, 
          current: token_id,
        });
      }
    }
  }
  handleUpMention(mention, shift, alt, token_id=-1) {
    if (this.props.readOnly) return;

    this.setState({ mouseIsDown: false, });
    const cluster = getClusterWithMention(this.props.clusters, mention);
    if (this.props.ui == 'mentions') {
      if (alt) {
        this.removeCluster(cluster);
      }
    } else if (this.props.ui.startsWith("check-mentions")) {
      if (this.props.ui.includes("exclude")) {
        const curCheck = this.props.checks.find(check => {
          return check.mention.start == mention.start && 
            check.mention.end == mention.end &&
            check.mention.label == mention.label;
        });
        if (curCheck != null) {
          Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, mention, "remove");
        } else {
          Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, mention, "remove");
        }
      } else {
        var change = "add";
        this.props.checks.forEach(check => {
          if (check.mention.start == mention.start && check.mention.end == mention.end) {
            if (check.decision == "keep") {
              change = "remove";
            } else {
              change = "update";
            }
          }
        });
        if (shift || alt) {
          this.removeMention(cluster, mention);
          if (change == "remove") {
            Meteor.call('mentions.uncheck', this.props.text._id, this.state.user, this.props.ann_id, mention, "keep");
          }
        } else {
          if (change == "add") {
            if (mention.label != "") {
              Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, mention, "keep");
            }
          } else if (change == "update") {
            Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, mention, "keep");
          } else {
            Meteor.call('mentions.uncheck', this.props.text._id, this.state.user, this.props.ann_id, mention, "keep");

            // Check if there are any that overlapped with this, as they will now be visible.
          }
        }
      }
    } else if (this.props.ui.startsWith("link-to-one")) {
      const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
      if (this.state.selectedClusterId != cluster._id) {
        this.mergeClusters(selectedCluster, cluster);
      } else if (
        mention.start == this.props.focusMention.start &&
        mention.end == this.props.focusMention.end
      ) {
        // Ignore clicks on the focus mention
      } else {
        this.splitMentionFromCluster(selectedCluster, mention, "#7570b3");
      }
    } else if (this.props.ui.startsWith("one-entity-at-a-time")) {
      if (shift) {
        if (this.state.selectedClusterId != null) {
          // Adding this mention to an existing cluster
          const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
          this.mergeClusters(selectedCluster, cluster);
        }
      } else if (alt) {
        Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, mention, "remove");
      } else {
        // Starting a new cluster
        if (this.state.selectedClusterId != null) {
          // Hide all mentions from the current cluster
          const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
          if (selectedCluster._id != cluster._id) {
            this.setState({ selectedClusterId: cluster._id, });
          }
        } else {
          this.setState({ selectedClusterId: cluster._id, });
        }
      }
    } else if (this.props.ui == "conditionals-from-actions") {
      if (alt) {
        if (
          mention.start == this.props.focusMention.start &&
          mention.end == this.props.focusMention.end
        ) {
          // Ignore attempts to remove the focus mention
        } else {
          this.removeMention(cluster, mention);
        }
      }
    } else {
      if (alt) {
        if (this.state.selectedClusterId == cluster._id) {
          this.splitMentionFromCluster(cluster, mention);
        } else {
          this.removeCluster(cluster);
        }
      } else if (shift) {
        if (! this.state.selectedClusterId) {
          // When nothing is selected, a shift-click selects the cluster
          this.setState({ selectedClusterId: cluster._id, });
        } else if (this.state.selectedClusterId == cluster._id) {
          // When the selected cluster is clicked, de-select it
          this.setState({ selectedClusterId: null, });
        } else {
          const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
          this.mergeClusters(selectedCluster, cluster);
        }
      }
    }
  }
  handleUpLabel(label, mention, shift, alt) {
    if (this.props.readOnly) return;

    this.setState({ mouseIsDown: false, });

    const clabel = mention.label;
    const tlabel = mention.label == label ? "" : label;
    const target = getMatchingMention(mention.start, mention.end, tlabel, this.props.clusters);
    const cluster = getClusterWithMention(this.props.clusters, mention);
    if (this.props.ui.includes("check")) {
      var curCheck = null;
      var targetCheck = null;
      this.props.checks.forEach(check => {
        if (check.mention.start == mention.start && check.mention.end == mention.end) {
          if (check.mention.label == tlabel) targetCheck = check;
          else if (check.mention.label == clabel) curCheck = check;
        }
      });

      // Mark current for removal
      if (curCheck != null) {
        Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, mention, "remove");
      } else {
        Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, mention, "remove");
      }

      // Create target
      if (target != null) {
        if (targetCheck != null) {
          if (tlabel != "") {
            Meteor.call('mentions.check.update', this.props.text._id, this.state.user, this.props.ann_id, target, "keep");
          } else {
            Meteor.call('mentions.uncheck', this.props.text._id, this.state.user, this.props.ann_id, target, "remove");
          }
        } else {
          if (tlabel != "") {
            Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, mention, "keep");
          }
        }
      } else {
        this.createClusterFromMention(mention.start, mention.end, "", tlabel);
        if (tlabel != "") {
          Meteor.call('mentions.check', this.props.text._id, this.state.user, this.props.ann_id, {start: mention.start, end: mention.end, label: tlabel}, "keep");
        }
      }
    } else {
      if (target == null) {
        this.createClusterFromMention(mention.start, mention.end, "", tlabel);
        this.removeMention(cluster, mention);
        if (this.props.ui == "freq-ents") {
          // Sort all mentions
          const mentions = this.props.clusters.flatMap(cluster => {
            return cluster.mentions;
          }).sort((a, b) => {
            return a.start < b.start ? -1 : 1;
          });

          if (this.state.warnings == 0) {
            // See if 8 out of the first 10 are labeled
            const labelCount = mentions.slice(0, 10).filter(mention => mention.label != "").length;
            if (labelCount >= 8) {
              Meteor.call('logError', this.state.user, "Warned: Labeled 8+ of the first 10");
              this.props.showModal("Please make sure you are assigning labels only when something is mentioned. Not every box should be labeled.");
              this.setState({
                warnings: 1
              });
            }
          } else if (this.state.warnings == 1) {
            // See if 80%
            const labelCount = mentions.filter(mention => mention.label != "").length;
            if (labelCount / mentions.length >= 0.8) {
              Meteor.call('logError', this.state.user, "Warned: Labeled 80%+");
              this.props.showModal("Please make sure you are assigning labels only when something is mentioned. Not every box should be labeled.");
              this.setState({
                warnings: 2
              });
            }
          }
        }
      }
    }
  }

  // These are tricky as we also get the handleUp event
  handleClick(token_id, shift, alt) { }
  handleClickMention(mention, shift, alt) { }
  handleDoubleClick(token_id, shift, alt) { }

  noPuncSpan(mention, isStart, text) {
    var pos = isStart ? mention.start : mention.end;
    while (/[\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,\-.\/:;<=>?@\[\]^_`{|}~]/.test(text.tokens[pos].text)) {
      if (isStart) {
        if (pos == mention.end) {
          pos = mention.start;
          break;
        }
        pos += 1;
      } else {
        if (pos == mention.start) {
          pos = mention.end;
          break;
        }
        pos -= 1;
      }
    }
    return pos;
  }

  render() {
    // Make a mapping from positions to mention sets. For each position there
    // is an object with the maximum length mention that covers it, and all
    // the mentions within that mention.
    const mentionMap = {};
    // First, mark incorrect annotations
    this.props.clusters.forEach(cluster => {
      cluster.mentions.forEach(mention => {
        if (ignoreMention(mention, this.props.checks)) return;
        
        const astart = this.noPuncSpan(mention, true, this.props.text);
        const aend = this.noPuncSpan(mention, false, this.props.text);
        var hint = "";
        if (this.props.ref_clusters != undefined) {
          var allButLabel = "";
          const exists = this.props.ref_clusters.some(cluster => {
            return cluster.mentions.some(other => {
              const gstart = this.noPuncSpan(other, true, this.props.text);
              const gend = this.noPuncSpan(other, false, this.props.text);
              if (astart == gstart && aend == gend && mention.label != other.label) {
                allButLabel = other.label;
              }
              return astart == gstart && aend == gend;
            });
          });
          if (! exists) {
            hint = "exclude";
          } else if (allButLabel != "") {
            hint = "wrong-label:" + allButLabel;
          } else {
            hint = "correct";
          }
        }

        var i;
        for (i = mention.start; i <= mention.end; i++) {
          if (i in mentionMap) {
            mentionMap[i].push({
              mention: mention,
              hint: hint,
              color: cluster.color,
            });
          } else {
            mentionMap[i] = [{
              mention: mention,
              hint: hint,
              color: cluster.color,
            }];
          }
        }
      });
    });
    // Second, show missing annotations
    if (this.props.ref_clusters != undefined) {
      this.props.ref_clusters.forEach(cluster => {
        cluster.mentions.forEach(mention => {
          // Check if it already exists
          const gstart = this.noPuncSpan(mention, true, this.props.text);
          const gend = this.noPuncSpan(mention, false, this.props.text);
          const exists = this.props.clusters.some(ocluster => {
            return ocluster.mentions.some(other => {
              const astart = this.noPuncSpan(other, true, this.props.text);
              const aend = this.noPuncSpan(other, false, this.props.text);
              return astart == gstart && aend == gend;
            });
          });
          // Check if it overlaps 
          const overlaps = this.props.clusters.some(ocluster => {
            return ocluster.mentions.some(other => {
              return (mention.start < other.start && 
                other.start <= mention.end &&
                mention.end < other.end) ||
                (other.start < mention.start && 
                mention.start <= other.end &&
                other.end < mention.end);
            });
          });
          if (! (exists || overlaps)) {
            var i;
            for (i = mention.start; i <= mention.end; i++) {
              if (i in mentionMap) {
                mentionMap[i].push({
                  mention: mention,
                  hint: "include",
                  color: "#1b9e77",
                });
              } else {
                mentionMap[i] = [{
                  mention: mention,
                  hint: "include",
                  color: "#1b9e77",
                }];
              }
            }
          }
        });
      });
    }
    var maxDepth = 1;
    for (var key in mentionMap) {
      if (mentionMap.hasOwnProperty(key)) {
        const value = mentionMap[key];
        maxDepth = Math.max(maxDepth, value.length);
      }
    }

    var numLabelOptions = 0
    if (this.props.labelOptions != null) numLabelOptions = this.props.labelOptions.length;
    const lineHeight = 50 + (Math.max(1, numLabelOptions) * 20) * maxDepth;

    // Now, go through tokens. If it is not in the mapping, render it. If it is
    // in the mapping, render a mention, passing the set of mentions in and the
    // relevant tokens.
    var next_token = -1;
    const selectedCluster = getClusterWithId(this.state.selectedClusterId, this.props.clusters);
    const text = this.props.text.tokens.map(token => {
      if (token._id < next_token) {
        // Skip, as we are covering it in a mention
      } else {
        const mention = getMentionFromMap(mentionMap, token._id, this.props.focusMention, this.props.ui == "link-to-one-before", this.props.ui == "link-to-one-after");
        if (mention != null) {
          next_token = mention.mention.end + 1;
          return (
            <Mention
              tokens={this.props.text.tokens}
              checks={this.props.checks}
              ui={this.props.ui}
              key={token._id}
              thisMention={mention}
              mentionMap={mentionMap}
              forceSelected={false}
              focusMention={this.props.focusMention}
              selectedCluster={selectedCluster}
              selectedMention={this.state.selectedMention[this.state.selectedMention.length - 1]}
              onMouseEnter={this.handleEnter}
              onMouseLeave={this.handleLeave}
              onMouseDown={this.handleDown}
              onMouseUp={this.handleUp}
              onClick={this.handleClick}
              onDoubleClick={this.handleDoubleClick}
              onMouseEnterMention={this.handleEnterMention}
              onMouseLeaveMention={this.handleLeaveMention}
              onMouseDownMention={this.handleDownMention}
              onMouseUpMention={this.handleUpMention}
              onClickMention={this.handleClickMention}
              onMouseDownLabel={this.handleDownLabel}
              onMouseUpLabel={this.handleUpLabel}
              currentToken={this.state.current}
              endSelected={this.state.endSelected}
              startSelected={this.state.startSelected}
              readOnly={this.props.readOnly}
              depth={0}
              maxDepth={maxDepth}
              lineHeight={lineHeight}
              labelOptions={this.props.labelOptions}
            />
          );
        } else {
          return (
            <Token 
              token={token} 
              key={token._id}
              ui={this.props.ui}
              onMouseEnter={this.handleEnter}
              onMouseLeave={this.handleLeave}
              onMouseDown={this.handleDown}
              onMouseUp={this.handleUp}
              onClick={this.handleClick}
              onDoubleClick={this.handleDoubleClick}
              currentToken={this.state.current}
              endSelected={this.state.endSelected}
              startSelected={this.state.startSelected}
              isSelected={'white'}
              parentSelected={false}
              depth={0}
              maxDepth={maxDepth}
              lineHeight={lineHeight}
              readOnly={this.props.readOnly}
            />
          );
        }
      }
      return null;
    }).filter(v => v != null);
    const className = (this.props.className != undefined) ? this.props.className : "textdoc";
    return (
      <div className={className}> {text} </div>
    );
  }
}

