import json, os
import requests
from .searcher import *
from typing import List, Dict
    


def serp_api(query, api_key):
    params = {
        "engine": "google",
        "q": query,
        "api_key": api_key
    }
    resp = requests.get("https://serpapi.com/search", params=params)
    if resp.status_code != 200:
        raise Exception("Serpapi returned %d\n%s"%(resp.status_code, resp.text))
    result = resp.json()
    ret = []
    for item in result['organic_results']:
        if "title" not in item or "link" not in item or "snippet" not in item:
            continue
        ret.append(SearchResult(item['title'], item['link'], item['snippet']))
    return ret



def dump_results(results):
    """
    将结果转换为JSON字符串，并返回。
    
    Args:
        results (list[Result]): 包含Result对象的列表。
    
    Returns:
        str: JSON格式的字符串，包含每个Result对象的信息。
    """
    return json.dumps([result.dump() for result in results])


class Searcher(SearcherInterface):
    def __init__(self):
        self.SERPAPI_KEY = os.getenv("SERPAPI_KEY")
        if not self.SERPAPI_KEY:
            print("[Error] SERPAPI_KEY is not set, please set it to use serpapi")
            exit(1)

    def _parse(self, result):
        """
        解析搜索结果，将每个结果转换为SearchResult对象，并返回一个列表
        
        Args:
            result (list, optional): 来自API的搜索结果，包含字典，每个字典包含'ref','url','snip'三个键值对，默认None（即没有结果）
        
        Returns:
            list or None: 如果result不存在或者为空，则返回None；否则返回一个包含SearchResult对象的列表
        
            - SearchResult (namedtuple): 包含三个属性：ref、url和snip，分别代表引用、URL和片段，类型都是str
        """
        if not result:
            return None
        ret = []
        for item in result:
            ret.append(SearchResult(item['ref'], item['url'], item['snip']))
        return ret

    def search(self, query):
        """
        Searches for a given query on the internet and returns the first result page.
        
        Args:
            query (str): The query to be searched.
            
        Returns:
            dict: A dictionary containing the following keys: 'title' (str), 'url' (str), 'description' (str).
                   If no results are found, an empty dictionary is returned.
        """
        return serp_api(query, self.SERPAPI_KEY)
