import argparse
import glob
import json
import os
import re

import pandas as pd

thisdir = os.path.dirname(os.path.realpath(__file__))
parser = argparse.ArgumentParser(description="Export CrowS-Pairs results.")
parser.add_argument(
    "--persistent_dir",
    action="store",
    type=str,
    default=os.path.realpath(os.path.join(thisdir, "..")),
    help="Directory where all persistent data will be stored.",
)
parser.add_argument(
    "--bias_type",
    action="store",
    type=str,
    default="gender",
    choices=["gender", "race", "religion"],
    help="Type of bias to export results for.",
)

if __name__ == "__main__":
    args = parser.parse_args()

    print("Exporting CrowS-Pairs results:")
    print(f" - persistent_dir: {args.persistent_dir}")
    print(f" - bias_type: {args.bias_type}")

    records = []
    for file_path in glob.glob(f"{args.persistent_dir}/results/crows/*.json"):
        # Parse the experiment ID.
        file_name = os.path.basename(file_path)
        experiment_id = os.path.splitext(file_name)[0]
        model, model_name_or_path, bias_type = re.match(
            "crows_m-([A-Za-z0-9-]+)_c-([A-Za-z0-9-]+)_t-([A-Za-z-]+)",
            experiment_id,
        ).groups()

        # Skip results that aren't for the specified bias type.
        if bias_type != args.bias_type:
            continue

        with open(file_path, "r") as f:
            stereotype_score = json.load(f)

        records.append(
            {
                "experiment_id": experiment_id,
                "model": model,
                "model_name_or_path": model_name_or_path,
                "bias_type": bias_type,
                "ss": stereotype_score,
            }
        )

    df = pd.DataFrame.from_records(records)
    print(df)

    with pd.option_context("max_colwidth", 1000):
        print(
            df.to_latex(
                float_format="%.2f", columns=["experiment_id", "ss"], index=False
            )
        )
