import numpy as np


def extrapolate_augmentation(
    label_to_embedding_np,
    original_weight = 1,
    num_extrapolations = 999,
    ):
    
    label_to_augmented = {}

    label_to_avg = {}
    for label, embedding_np in label_to_embedding_np.items():
        avg_embedding = np.mean(embedding_np, axis=0)
        label_to_avg[label] = avg_embedding
    
    for current_label, current_embedding_np in label_to_embedding_np.items():
        current_label_avg_embedding = label_to_avg[current_label]
        label_to_augmented[current_label] = []

        weight_multiplier = original_weight if current_label % 2 == 0 else 1
        label_to_augmented[current_label] = [current_embedding_np for _ in range(weight_multiplier)] 

        for other_label, other_embedding_np in label_to_embedding_np.items():
            if current_label != other_label and abs(current_label - other_label) <= num_extrapolations / 2:
                other_label_avg_embedding = label_to_avg[other_label]
                diff = other_label_avg_embedding - current_label_avg_embedding

                augmented_data = other_embedding_np - diff
                label_to_augmented[current_label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def within_extra_augmentation(
    label_to_embedding_np,
    num_copies = 10,
):

    max_n_class = max(embedding_np.shape[0] for embedding_np in label_to_embedding_np.values())

    label_to_augmented = {}
    for label, embedding_np in label_to_embedding_np.items():
        multiplier = max(int(max_n_class / embedding_np.shape[0]), 1)
        label_to_augmented[label] = [embedding_np]
        for _ in range(int(num_copies * multiplier)):
            embedding_np_noise = np.array(embedding_np, copy=True)
            np.random.shuffle(embedding_np_noise)
            augmented_data = embedding_np + (embedding_np - embedding_np_noise) / 2
            label_to_augmented[label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def linear_delta_augmentation(
    label_to_embedding_np,
    num_copies = 10,
):

    max_n_class = max(embedding_np.shape[0] for embedding_np in label_to_embedding_np.values())

    label_to_augmented = {}
    for label, embedding_np in label_to_embedding_np.items():
        multiplier = max(int(max_n_class / embedding_np.shape[0]), 1)
        label_to_augmented[label] = [embedding_np]
        for _ in range(int(num_copies * multiplier)):
            embedding_np_noise_1 = np.array(embedding_np, copy=True)
            np.random.shuffle(embedding_np_noise_1)
            embedding_np_noise_2 = np.array(embedding_np, copy=True)
            np.random.shuffle(embedding_np_noise_2)
            augmented_data = embedding_np + (embedding_np_noise_1 - embedding_np_noise_2)
            label_to_augmented[label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def interpolate_augmentation(
    label_to_embedding_np,
    num_copies = 10,
):

    max_n_class = max(embedding_np.shape[0] for embedding_np in label_to_embedding_np.values())

    label_to_augmented = {}
    for label, embedding_np in label_to_embedding_np.items():
        multiplier = max(int(max_n_class / embedding_np.shape[0]), 1)
        label_to_augmented[label] = [embedding_np]
        for _ in range(int(num_copies * multiplier)):
            embedding_np_noise = np.array(embedding_np, copy=True)
            np.random.shuffle(embedding_np_noise)
            augmented_data = (embedding_np + embedding_np_noise) / 2
            label_to_augmented[label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def gaussian_augmentation(
    label_to_embedding_np,
    num_copies = 10,
):

    max_n_class = max(embedding_np.shape[0] for embedding_np in label_to_embedding_np.values())

    label_to_augmented = {}
    for label, embedding_np in label_to_embedding_np.items():
        multiplier = max(int(max_n_class / embedding_np.shape[0]), 1)
        label_to_augmented[label] = [embedding_np]
        for _ in range(int(num_copies * multiplier)):
            noise = np.random.normal(0, 0.1, size = embedding_np.shape)
            augmented_data = embedding_np + noise
            label_to_augmented[label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def uniform_augmentation(
    label_to_embedding_np,
    num_copies = 10,
):

    max_n_class = max(embedding_np.shape[0] for embedding_np in label_to_embedding_np.values())

    label_to_augmented = {}
    for label, embedding_np in label_to_embedding_np.items():
        multiplier = max(int(max_n_class / embedding_np.shape[0]), 1)
        label_to_augmented[label] = [embedding_np]
        for _ in range(int(num_copies * multiplier)):
            noise = np.random.uniform(-0.1, 0.1, size = embedding_np.shape)
            augmented_data = embedding_np + noise
            label_to_augmented[label].append(augmented_data)
    
    label_to_augmented = {k: np.concatenate(v) for k, v in label_to_augmented.items()}
    return label_to_augmented


def all_augmentation(
    label_to_embedding_np,
):

    label_to_augmented_all = {}
    for aug_function in [interpolate_augmentation, extrapolate_augmentation, gaussian_augmentation]:
        label_to_augmented = aug_function(label_to_embedding_np)
        for label, augmented in label_to_augmented.items():
            if label in label_to_augmented_all:
                label_to_augmented_all[label].append(augmented)
            else:
                label_to_augmented_all[label] = [augmented] 
    
    label_to_augmented_all = {k: np.concatenate(v) for k, v in label_to_augmented_all.items()}
    return label_to_augmented_all

