﻿using System;
using System.Collections.Generic;

namespace SpreadsheetLLM.Heuristic
{
    internal class RegionGrowthDetector
    {
        private static int[] dx = { +1, -1, 0, 0, +1, -1, +1, -1 };
        private static int[] dy = { 0, 0, +1, -1, -1, -1, +1, +1 };
        private static int step = 4;
        private const int stRow = 1;
        private const int stCol = 1;

        /// <summary>
        /// Split tables from an Excel sheet
        /// </summary>
        public static List<Boundary> FindConnectedRanges(string[,] content, int[,] valueMapBorderOrNull, int thresholdHor, int thresholdVer, int direct = 1)
        {
            var ret = new List<(int, int, int, int)>();
            int rowNum = content.GetLength(0);
            int colNum = content.GetLength(1);

            int rangeUpRow = stRow - 1;
            int rangeLeftCol = stCol - 1;
            int rangeDownRow = stRow + rowNum - 2;
            int rangeRightCol = stCol + colNum - 2;
            bool[,] vis = new bool[rowNum, colNum];
            int[,] counterHor = new int[rowNum, colNum];
            int[,] counterVer = new int[rowNum, colNum];
            for (int i = 0; i < rowNum; ++i)
            {
                for (int j = 0; j < colNum; ++j)
                {
                    counterHor[i, j] = thresholdHor;
                    counterVer[i, j] = thresholdVer;
                }
            }

            var rangeRow = new int[rangeDownRow - rangeUpRow + 1];
            var rangeCol = new int[rangeRightCol - rangeLeftCol + 1];
            if (direct == 0)
            {
                rangeRow[0] = rangeUpRow;
                for (int row_i = 1; row_i <= rangeDownRow - rangeUpRow; ++row_i)
                {
                    rangeRow[row_i] = rangeRow[row_i - 1] + 1;
                }

                rangeCol[0] = rangeLeftCol;
                for (int col_i = 1; col_i <= rangeRightCol - rangeLeftCol; ++col_i)
                {
                    rangeCol[col_i] = rangeCol[col_i - 1] + 1;
                }
            }
            else
            {
                rangeRow[0] = rangeDownRow;
                for (int row_i = 1; row_i <= rangeDownRow - rangeUpRow; ++row_i)
                {
                    rangeRow[row_i] = rangeRow[row_i - 1] - 1;
                }

                rangeCol[0] = rangeLeftCol;
                for (int col_i = 1; col_i <= rangeRightCol - rangeLeftCol; ++col_i)
                {
                    rangeCol[col_i] = rangeCol[col_i - 1] + 1;
                }
            }

            for (int row_i = 0; row_i <= rangeDownRow - rangeUpRow; ++row_i)
            {
                for (int col_i = 0; col_i <= rangeRightCol - rangeLeftCol; ++col_i)
                {
                    int rowIdx = rangeRow[row_i];
                    int colIdx = rangeCol[col_i];
                    //if the cell has been visited during connected region search, just ignore
                    if (vis[rowIdx - stRow + 1, colIdx - stCol + 1])
                        continue;

                    //ignore empty or null cells, because we cannot start search from these cells
                    if (content[rowIdx, colIdx] == "")
                        continue;

                    //the following code will try to find a connected region containing the current cell with a breadth-first algorithm
                    var q = new Queue<(int row, int col)>();
                    q.Enqueue((rowIdx, colIdx));
                    vis[rowIdx - stRow + 1, colIdx - stCol + 1] = true;

                    int minRow = int.MaxValue;
                    int minCol = int.MaxValue;
                    int maxRow = int.MinValue;
                    int maxCol = int.MinValue;

                    while (q.Count > 0)
                    {
                        var cellCordinate = q.Dequeue();
                        int row = cellCordinate.row;
                        int col = cellCordinate.col;

                        //If the current cell does not have right/down neighbors 
                        if (counterHor[row - stRow + 1, col - stCol + 1] == 0 || counterVer[row - stRow + 1, col - stCol + 1] == 0)
                        {
                            continue;
                        }

                        minRow = Math.Min(row, minRow);
                        minCol = Math.Min(col, minCol);
                        maxRow = Math.Max(row, maxRow);
                        maxCol = Math.Max(col, maxCol);

                        for (int i = 0; i < step; ++i)
                        {
                            //Search the neighboring cell
                            int nextRow = row + dx[i];
                            int nextCol = col + dy[i];
                            if (nextRow >= rangeUpRow && nextRow <= rangeDownRow
                                && nextCol >= rangeLeftCol && nextCol <= rangeRightCol
                                && !vis[nextRow - stRow + 1, nextCol - stCol + 1])
                            {
                                vis[nextRow - stRow + 1, nextCol - stCol + 1] = true;
                                try
                                {
                                    if (content[nextRow, nextCol] == "")
                                    {
                                        if (dy[i] != 0)
                                        {
                                            counterHor[nextRow - stRow + 1, nextCol - stCol + 1] = counterHor[row - stRow + 1, col - stCol + 1] - 1;
                                        }
                                        if (dx[i] != 0)
                                        {
                                            counterVer[nextRow - stRow + 1, nextCol - stCol + 1] = counterVer[row - stRow + 1, col - stCol + 1] - 1;
                                        }
                                    }
                                    // If there exists vertical border, we can mark it as part of the current table.
                                    if (valueMapBorderOrNull != null && valueMapBorderOrNull[nextRow, nextCol] != 0)
                                    {
                                        if (dy[i] != 0)
                                            counterHor[nextRow - stRow + 1, nextCol - stCol + 1] = thresholdHor;
                                        if (dx[i] != 0)
                                            counterVer[nextRow - stRow + 1, nextCol - stCol + 1] = thresholdVer;
                                    }
                                }
                                catch
                                {
                                    if (dy[i] != 0)
                                    {
                                        counterHor[nextRow - stRow + 1, nextCol - stCol + 1] = counterHor[row - stRow + 1, col - stCol + 1] - 1;
                                    }
                                    if (dx[i] != 0)
                                    {
                                        counterVer[nextRow - stRow + 1, nextCol - stCol + 1] = counterVer[row - stRow + 1, col - stCol + 1] - 1;
                                    }

                                }
                                q.Enqueue((nextRow, nextCol));
                            }
                        }
                    }

                    if (minRow == int.MaxValue || minCol == int.MaxValue || maxRow == int.MinValue || maxCol == int.MinValue)
                        continue;

                    if (maxRow - minRow > 1)
                    {
                        ret.Add((minRow, minCol, maxRow, maxCol));
                        for (int ri = minRow; ri <= maxRow; ++ri)
                        {
                            for (int rj = minCol; rj <= maxCol; ++rj)
                            {
                                vis[ri - stRow + 1, rj - stCol + 1] = true;
                            }
                        }
                    }
                }
            }

            for (int i = 0; i < 3; i++)
            {
                ret = Trim(content, ret);
            }

            var boxes = ret;
            var boxesList = new List<Boundary>();
            foreach (var box in boxes)
            {
                if (box.Item1 >= box.Item3 || box.Item2 >= box.Item4)
                    continue;

                boxesList.Add(new Boundary(box.Item1 + 1, box.Item3 + 1, box.Item2 + 1, box.Item4 + 1));
            }

            return boxesList;
        }

        /// <summary>
        /// Remove leading or tailing empty rows or columns for each sub-tables
        /// </summary>
        private static List<(int, int, int, int)> Trim(string[,] content, List<(int, int, int, int)> list)
        {
            var ret = new List<(int, int, int, int)>();
            foreach (var cellRange in list)
            {
                int upRow = cellRange.Item1;
                int leftCol = cellRange.Item2;
                int downRow = cellRange.Item3;
                int rightCol = cellRange.Item4;

                //Remove the leading empty rows
                for (int i = upRow; i <= downRow; ++i)
                {
                    bool isEmpty = IsRowEmpty(content, leftCol, rightCol, i);
                    if (isEmpty)
                    {
                        ++upRow;
                    }
                    else
                    {
                        break;
                    }
                }

                //Remove the tailing empty rows
                for (int i = downRow; i >= upRow; --i)
                {
                    bool isEmpty = IsRowEmpty(content, leftCol, rightCol, i);
                    if (isEmpty)
                    {
                        --downRow;
                    }
                    else
                    {
                        break;
                    }
                }

                //Remove the leading empty columns
                for (int j = leftCol; j <= rightCol; ++j)
                {
                    bool isEmpty = IsColEmpty(content, upRow, downRow, j);
                    if (isEmpty)
                    {
                        ++leftCol;
                    }
                    else
                    {
                        break;
                    }
                }

                //Remove the tailing empty columns
                for (int j = rightCol; j >= leftCol; --j)
                {
                    bool isEmpty = IsColEmpty(content, upRow, downRow, j);
                    if (isEmpty)
                    {
                        --rightCol;
                    }
                    else
                    {
                        break;
                    }
                }

                ret.Add((upRow, leftCol, downRow, rightCol));
            }
            return ret;
        }

        /// <summary>
        /// Check whether a row is an empty row
        /// </summary>
        private static bool IsColEmpty(string[,] content, int upRow, int downRow, int j)
        {
            for (int i = upRow; i <= downRow; ++i)
            {
                if (content[i, j] != "")
                    return false;
            }

            return true;
        }

        /// <summary>
        /// Check whether a row is an empty row
        /// </summary>
        private static bool IsRowEmpty(string[,] content, int leftCol, int rightCol, int i)
        {
            for (int j = leftCol; j <= rightCol; ++j)
            {
                if (content[i, j] != "")
                    return false;
            }

            return true;
        }
    }
}
