///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "nl-iomacros.h"
#include "nl-cpt.h"
#include "nl-hmm.h"

#include "SyntacticLMModelFiles.h"

#include <string>

template <class MH, class MO, class X=typename MH::RandVarType, class B=NullBackDat<typename MH::RandVarType> >
class SyntacticLMState {
 public:

  // Initialize an empty LM state
  SyntacticLMState( SyntacticLMModelFiles<MH,MO>* modelData, int beamSize );

  // Get the next LM state from an existing LM state and the next word
  SyntacticLMState( SyntacticLMState* prev, std::string word );


 ~SyntacticLMState() {
   delete randomVariableStore;
 }

  // Get the LM score from this LM state
  double getScore();

 private:

 void setScore(double score);

 SafeArray1D<Id<int>,pair<X,LogProb> >* randomVariableStore;

 double score;
 int beamSize;
 SyntacticLMModelFiles<MH,MO>* modelData;

};


////////////////////////////////////////////////////////////////////////////////

  

// Initialize an empty LM state from grammar files
//
//    nArgs is the number of model files
//    argv is the list of model file names
//
template <class MH, class MO, class X, class B>
  SyntacticLMState<MH,MO,X,B>::SyntacticLMState( SyntacticLMModelFiles<MH,MO>* modelData, int beamSize ) {

  this->randomVariableStore = new SafeArray1D<Id<int>,pair<X,LogProb> >();
  this->modelData = modelData;
  this->beamSize = beamSize;

  // Initialize an empty random variable value
  X xBEG;
  StringInput(String(BEG_STATE).c_array())>>xBEG>>"\0";
  cerr<<xBEG<<"\n";

  // Initialize the random variable store
  this->randomVariableStore->init(1,pair<X,LogProb>(xBEG,0));

  // Get score of final frame in HHMM
  LogProb l(1.0);
  //score = l.toDouble();
  setScore(l.toDouble());
  MH::F_ROOT_OBS = true;

}


template <class MH, class MO, class X, class B>
  SyntacticLMState<MH,MO,X,B>::SyntacticLMState( SyntacticLMState* prev, std::string word ) {

  // Initialize member variables
  this->randomVariableStore = new SafeArray1D<Id<int>,pair<X,LogProb> >();
  this->modelData = prev->modelData;
  this->beamSize = prev->beamSize;

  // Get HHMM model files
  MH& mH = *(modelData->getHiddenModel());
  MO& mO = *(modelData->getObservedModel());
  
  // Initialize HHMM
  HMM<MH,MO,X,B> hmm(mH,mO);  
  int MAX_WORDS  = 2;
  hmm.init(MAX_WORDS,this->beamSize,prev->randomVariableStore);
  
  // Initialize observed variable
  typename MO::RandVarType ov;
  ov.set(word.c_str(),mO);
  MH::WORD = ov.getW();

  // Update HHMM based on observed variable
  hmm.updateRanked(ov);
  MH::F_ROOT_OBS = false;

  // Get the current score
  setScore(hmm.getCurrSum());

  // Get new hidden random variable store from HHMM
  hmm.gatherElementsInBeam(randomVariableStore);
  
}


template <class MH, class MO, class X, class B>
double SyntacticLMState<MH,MO,X,B>::getScore() {
  
  return score;
}


template <class MH, class MO, class X, class B>
  void SyntacticLMState<MH,MO,X,B>::setScore(double score) {

  // We want values to range from -100 to 0
  //
  // If the minimum positive value for a double is min=4.94065645841246544e-324
  //    then to scale, we want a logarithmic base such that log_b(min)=-100
  //
  // -100 = log(min) / log(b)
  //
  // log(b) = log(min) / -100
  //
  // b = exp( log(min) / -100 )
  //
  // b = 7.44440071921381

  this->score = log(score) / 7.44440071921381;

}
