#!/usr/bin/env python
# coding: utf-8

# In[1]:


# In[2]:


import pickle
import numpy as np
from datetime import datetime
import copy
import random
import math
from sklearn.metrics import confusion_matrix
import sys

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim

import sys
sys.path.append('./')
from query_handling import data_splitting_utils
from query_handling import graph_generation


# In[3]:


fpath = './data/composite_learner_data/'

with open(fpath + 'mod_dict.pkl', 'rb') as infile:
    mod_dict = pickle.load(infile)

ntype_to_dtype = {
    'author' : ['tweet', 'wiki', 'quote', 'statement'],
    'issue' : ['description'],
    'event' : ['news'],
    'ref' : ['tweet', 'wiki', 'quote', 'statement', 'news', 'description']
}


# In[4]:


def split_node_name(node_name):
    '''
        Input: string denoting node_name generated by query_handling.graph_generation
        Output: split of node name into node type, author, topic, event and index in data_dict
    '''
    nn = copy.deepcopy(node_name)
    for tname in mod_dict:
        if tname in node_name:
            nn = nn.replace(tname, mod_dict[tname])
    nsplit = nn.split('-')

    vals = [mod_dict[k] for k in mod_dict]
    for i, item in enumerate(nsplit):
        if item in vals:
            nsplit[i] = item.replace('_', '-')
    return nsplit


# In[5]:


def get_node_tensor(data_dict, G, node_name):
    '''
        Input: data_dict (generated by query_handling.query_processing); 
        graph G & node_name string (generated by query_handling.graph_generation)
        
        Output: A numpy doc tensor denoting BERT embedding for documents /
        a list of ints for other nodes that denoting documents to be used for initial node embedding computation
        
    '''
    G_n, G_i2n, G_a = G
    ntype = split_node_name(node_name)
    if ntype[0] == 'doc':
        if ntype[1] == 'description':
            emb = data_dict['description'][2][ntype[2]][int(ntype[3])]
        elif ntype[1] == 'wiki':
            emb = data_dict[ntype[2]]['wiki'][2][int(ntype[3])]
        elif ntype[1] == 'news':
            emb = data_dict['news'][2][ntype[2]][int(ntype[3])][int(ntype[4])]
        elif ntype[1] == 'quote':
            emb = data_dict[ntype[2]]['quotes'][2][ntype[3]][0]
        elif ntype[1] == 'tweet':
            emb = data_dict[ntype[2]]['tweets'][2][ntype[3]][int(ntype[4])][int(ntype[5])]
        elif ntype[1] == 'statement':
            emb = data_dict[ntype[2]]['statements'][2][ntype[3]][int(ntype[4])][int(ntype[5])]
        etensor = emb
        return etensor
    else:
        n_edges = G_a[G_n[node_name]]
        dtensors = []
        for n2 in n_edges:
            n2_type = split_node_name(G_i2n[n2])
            if n2_type[0] == ('doc') and n2_type[1] in ntype_to_dtype[ntype[0]]:
                dtensors.append(n2)
        return dtensors


# In[6]:


def get_node_tensors(data_dict, G):
    '''
        Input: Input: data_dict (generated by query_handling.query_processing); 
        graph G (generated by query_handling.graph_generation)
        
        Output: A dict of node_ID -> doc tensor if doc-node,
        else node_ID -> list of node_IDs whose docs should be used to compute the initial node embedding
    '''
    ret_tensor_dict = {}
    G_n, G_i2n, G_a = G
    for node_name in G_n:
        node_tensor = get_node_tensor(data_dict, G, node_name)
        ret_tensor_dict[G_n[node_name]] = node_tensor
    return ret_tensor_dict


# In[7]:


def in_idx_dict(cis_dict, idx_dict, node_name):
    '''
        Input: combined_index_dict and index_dict generated from query_handling.data_splitting_utils &
        a string node_name
        Output: boolean, True if the node_name documents is in index_dict
    '''
    try:
        ntype = split_node_name(node_name)
        if ntype[0] != 'doc':
            return False
        if ntype[1] == 'news':
            issue = ntype[2]
            event = int(ntype[3])
            didx = int(ntype[4])
            if cis_dict['news'][issue][event][didx] in idx_dict['news'][issue][event]:
                return True
            else:
                return False
        if ntype[1] == 'tweet':
            author = ntype[2]
            issue = ntype[3]
            event = int(ntype[4])
            didx = int(ntype[5])
            if cis_dict[author]['tweets'][issue][event][didx] in idx_dict[author]['tweets'][issue][event]:
                return True
            else:
                return False
        if ntype[1] == 'statement':
            author = ntype[2]
            issue = ntype[3]
            event = int(ntype[4])
            didx = int(ntype[5])
            if cis_dict[author]['statements'][issue][event][didx] in idx_dict[author]['statements'][issue][event]:
                return True
            else:
                return False
        return False
    except KeyError:
        return False


# In[8]:


def count_docs(idx_dict):
    '''
        Input: idx_dict generated from query_handling.data_splitting_utils
        Output: Number of news, tweet and statement documents in idx_dict
    '''
    count = 0
    if 'news' in idx_dict:
        for topic in idx_dict['news']:
            for event in idx_dict['news'][topic]:
                count += len(idx_dict['news'][topic][event])
    for ent in idx_dict:
        if ent != 'news':
            if 'tweets' in idx_dict[ent]:
                for topic in idx_dict[ent]['tweets']:
                    for event in idx_dict[ent]['tweets'][topic]:
                        count += len(idx_dict[ent]['tweets'][topic][event])
            if 'statements' in idx_dict[ent]:
                for topic in idx_dict[ent]['statements']:
                    for event in idx_dict[ent]['statements'][topic]:
                        count += len(idx_dict[ent]['statements'][topic][event])
    return count


# In[9]:


def get_node_lists(G, cis_dict, idx_dict):
    '''
        Input: Graph G generated in query_handling.graph_generation
        combined_index_dict & index_dict generated in query_handling.data_splitting_utils
        
        Output: Lists of various types of nodes in graph G which are present in index_dict
    '''
    G_n, G_i2n, G_a = G
    news_nodes = []
    tweet_nodes = []
    statement_nodes = []
    quote_nodes = []
    event_nodes = []
    author_nodes = []
    for node in G_n:
        if node.startswith('doc-news') and in_idx_dict(cis_dict, idx_dict, node):
            news_nodes.append(node)
        if node.startswith('doc-tweet') and in_idx_dict(cis_dict, idx_dict, node):
            tweet_nodes.append(node)
        if node.startswith('doc-statement') and in_idx_dict(cis_dict, idx_dict, node):
            statement_nodes.append(node)
        if node.startswith('doc-quote'):
            quote_nodes.append(node)
        if node.startswith('event-'):
            event_nodes.append(node)
        if node.startswith('author-'):
            author_nodes.append(node)
    return news_nodes, tweet_nodes, statement_nodes, quote_nodes, event_nodes, author_nodes


# In[10]:


def get_graph_tensors(data_dict, G):
    '''
        Input: Graph G generated by query_handling.graph_generation
        data_dict generated by query_handling.query_processing
        
        Output: doc_tensors -> List of tensors of documents in the graph (length num_docs in G)
        node_dids -> list of lists of document indices from doc_tensors for each node in G (length num_nodes in G)
        graph_amtx -> adjacency matrix of G
    '''
    ntensors = get_node_tensors(data_dict, G)
    G_n, G_i2n, G_a = G
    ret_tensors = []
    doc_tensors = []
    doc_to_idx = {}
    idx = 0
    for node in G_n:
        if node.startswith('doc-'):
            n_tensor = np.squeeze(ntensors[G_n[node]], axis=0)
            doc_tensors.append(n_tensor)
            doc_to_idx[G_n[node]] = idx
            ret_tensors.append((G_n[node], [idx]))
            idx += 1
    for node in G_n:
        if not node.startswith('doc-'):
            did_list = []
            for n2_id in ntensors[G_n[node]]:
                did_list.append(doc_to_idx[n2_id])
            ret_tensors.append((G_n[node], did_list))
    node_dids = [tup[1] for tup in sorted(ret_tensors, key = lambda x:int(x[0]))]
    graph_amtx = get_adj_matrix(G)
    return doc_tensors, node_dids, graph_amtx


# In[11]:


def get_adj_matrix(G):
    '''
        Input: graph G generated in query_handling.graph_generation
        Output: Adjacency matrix of the graph
    '''
    G_n, G_i2n, G_a = G
    num_nodes = len(G_n)
    adj_mtx = np.zeros((num_nodes, num_nodes), dtype=int)
    for node_name in G_n:
        src = G_n[node_name]
        adj_mtx[src, src] = 1 #Adding self-loops because adj_mtx is the atten_mask for GAT
        for dest in G_a[src]:
            adj_mtx[src, dest] = 1
    return adj_mtx


# In[12]:


def trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID, num_extra_nodes=30, trim_auth=False, ret_rem_ids=False):
    '''
        Input: Graph G generated by query_handling.graph_generation
            doc_tensors, node_dids, graph_amtx generated by get_graph_tensors
            n1_ID & n2_ID -> integers denoting node IDs of src_node and dest_node in example
            num_extra_nodes -> number of extraneous nodes to be retained
            trim_auth -> Whether auhtor's tweets and statements should be included in extraneous nodes
            (Use True when training, else example won't fit in 12 GB GPU)
            
        Output: Trimmed link prediction example with updated doc_tensors, node_dids, graph_amtx and n1_ID and n2_ID
            number of docs in trimmed example and number of nodes in trimmed graph
    '''
    G_n, G_i2n, G_a = G
    #Get document type of dest_node
    dest_name = G_i2n[n2_ID]
    dtype = split_node_name(dest_name)
    if dtype[1] == 'tweet' or dtype[1] == 'statement':
        dtopic = dtype[3]
        devent = int(dtype[4])
    elif dtype[1] == 'quote':
        dtopic = dtype[3]
        devent = -1
    elif dtype[1] == 'news':
        dtopic = dtype[2]
        devent = int(dtype[3])
        
    #Collect candidate nodes that can be dropped
    cand_drops = []
    for node_name in G_n:
        if node_name.startswith('doc-news'):
            ntype = split_node_name(node_name)
            ntopic = ntype[2]
            nevent = int(ntype[3])
            #All news documents that are not related to the event are drop candidates
            if ntopic != dtopic or nevent != devent:
                cand_drops.append(G_n[node_name])
            else:
                toss = random.random()
                #80% of the news articles from the same event are drop candidates
                if G_n[node_name] != n2_ID and toss <= 0.8:
                    cand_drops.append(G_n[node_name])
        #If trim_auth is set to True, collect author's tweets and statements as well
        elif trim_auth and node_name.startswith('doc-tweet') or node_name.startswith('doc-statement'):
            ntype = split_node_name(node_name)
            ntopic = ntype[3]
            nevent = int(ntype[4])
            #80% of Tweets and quotes that are not related to the event are drop candidates
            if ntopic != dtopic or nevent != devent:
                toss = random.random()
                if toss <= 0.8:
                    cand_drops.append(G_n[node_name])
                    
    num_drops = len(cand_drops) - num_extra_nodes
    if num_drops > 0:
        #Randomly sample nodes to be dropped
        drop_nodes = random.sample(cand_drops, num_drops)
        #Add all mentioned entity nodes that are 'only' referenced in the dropped docs to the drop_nodes list
        for node in G_n:
            if node.startswith('ref-'):
                drop = True
                for n2 in G_a[G_n[node]]:
                    if n2 not in drop_nodes:
                        drop = False
                        break
                if drop and G_n[node] != n1_ID:
                    drop_nodes.append(G_n[node])
        #Compute remaining node ID and doc ID lists
        rem_ids = list(G_i2n.keys())
        rem_dids = list(range(len(doc_tensors)))
        for drop_node in drop_nodes:
            #Drop node from node IDs
            rem_ids.remove(drop_node)
            #Drop the doc ID corresponding to the document that is being dropped
            if G_i2n[drop_node].startswith('doc-'):
                rem_dids.remove(node_dids[drop_node][0])
        #Extract trimmed doc list
        trimmed_doc_tensors = [doc_tensors[rem_did] for rem_did in rem_dids]
        #Extract trimmed ntensor lists
        trimmed_node_dids = []
        for i, rem_id in enumerate(rem_ids):
            trimmed_nlist = []
            nlist = node_dids[rem_id]
            for did in nlist:
                if did in rem_dids:
                    #Adding modified doc IDs to the ntensor list
                    trimmed_nlist.append(rem_dids.index(did))
            trimmed_node_dids.append(trimmed_nlist)
            if rem_id == n1_ID:
                trimmed_n1_ID = i #i is the new node index for n1
            if rem_id == n2_ID:
                trimmed_n2_ID = i #i is the new node index for n2
        rem_ids = np.array(rem_ids)
        trimmed_graph_amtx = graph_amtx[rem_ids[:, None], rem_ids[None, :]]
        if ret_rem_ids:
            return (trimmed_doc_tensors, trimmed_node_dids, trimmed_graph_amtx, trimmed_n1_ID, trimmed_n2_ID), (rem_dids, rem_ids)
        return (trimmed_doc_tensors, trimmed_node_dids, trimmed_graph_amtx, trimmed_n1_ID, trimmed_n2_ID), (len(rem_dids), len(rem_ids))
    else:
        if ret_rem_ids:
            return (doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID), (list(range(len(doc_tensors))), list(range(len(G_n))))
        return (doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID), (len(doc_tensors), len(G_n))


# In[13]:


def make_positive_examples(G, data_dict, cis_dict, idx_dict, prob=1., trim_size=40, trim_auth=False):
    '''
        Input: G - graph generated in query_handling.graph_generation
        combined_index_dict and index_dict generated in query_handling.data_splitting_utils
        prob -> probability of a node being picked for example data
        trim_auth -> for trim_example
        
        Output: List of link prediction example and max example size 
    '''
    G_n, G_i2n, G_a = G
    pos_egs = []
    max_docs = []
    max_nodes = []

    for node_name in G_n:
        #Generating author-document positive examples
        if node_name.startswith('author-'):
            n1_ID = G_n[node_name]
            for n2_ID in G_a[n1_ID]:
                if G_i2n[n2_ID].startswith('doc-tweet-') or G_i2n[n2_ID].startswith('doc-statement-') or G_i2n[n2_ID].startswith('doc-quote-'):
                    n2_type = split_node_name(G_i2n[n2_ID])
                    toss = random.random()
                    #If randomly selected and if the document is in index dict (to ensure train-dev-test splits)
                    if toss <= prob and (n2_type[1] == 'quote' or in_idx_dict(cis_dict, idx_dict, G_i2n[n2_ID])):
                        #Remove edges from author to the document and the other way
                        G[2][n1_ID].remove(n2_ID)
                        G[2][n2_ID].remove(n1_ID)
                        eg_doc_tensors, eg_node_dids, eg_graph_amtx = get_graph_tensors(data_dict, G)
                        eg, eg_size = trim_example(G, eg_doc_tensors, eg_node_dids, eg_graph_amtx, n1_ID, n2_ID,                                                 num_extra_nodes=trim_size, trim_auth=trim_auth)
                        max_docs.append(eg_size[0])
                        max_nodes.append(eg_size[1])
                        pos_egs.append(eg)
                        #Put the edges back in to the graph for future example generation
                        G[2][n1_ID].append(n2_ID)
                        G[2][n2_ID].append(n1_ID)
        #Generating event-news positive examples
        elif node_name.startswith('event-'):
            n1_ID = G_n[node_name]
            for n2_ID in G_a[n1_ID]:
                if G_i2n[n2_ID].startswith('doc-news-'):
                    toss = random.random()
                    if toss <= prob and in_idx_dict(cis_dict, idx_dict, G_i2n[n2_ID]):
                        #Remove edges from author to the document and the other way
                        G[2][n1_ID].remove(n2_ID)
                        G[2][n2_ID].remove(n1_ID)
                        eg_doc_tensors, eg_node_dids, eg_graph_amtx = get_graph_tensors(data_dict, G)
                        eg, eg_size = trim_example(G, eg_doc_tensors, eg_node_dids, eg_graph_amtx, n1_ID, n2_ID,                                                 num_extra_nodes=trim_size, trim_auth=trim_auth)
                        max_docs.append(eg_size[0])
                        max_nodes.append(eg_size[1])
                        pos_egs.append(eg)
                        #Put the edges back in to the graph for future example generation
                        G[2][n1_ID].append(n2_ID)
                        G[2][n2_ID].append(n1_ID)
    return pos_egs, max(max_docs), max(max_nodes)


# In[14]:


def get_positive_examples(query_dict, qd_split, num_train=50, trim_auth=False):
    '''
        Input: query_dict -> data_dict generated in query_handling.query_processing for this particular query
            qd_split -> index_dicts generated in query_handling.data_splitting_utils for training, development and testing
            num_train -> rough number of training examples to generate
            trim_auth -> For trim_example
            
        Output: Positive link prediction examples for the query and max example size
    '''
    t, d, e = qd_split
    tc = count_docs(t)
    tr_prob = ((num_train) / tc)
    dc = count_docs(d)
    de_prob = ((num_train) / dc)
    ec = count_docs(e)
    te_prob = ((num_train) / ec)
    
    train_dict, train_cis = data_splitting_utils.get_documents(query_dict, [t])
    train_graph = graph_generation.generate_graph(train_dict)
    tr_pos, tr_mdocs, tr_mnodes = make_positive_examples(train_graph, train_dict, train_cis, t, prob=tr_prob,                                                         trim_size=40, trim_auth=trim_auth)
    
    dev_dict, dev_cis = data_splitting_utils.get_documents(query_dict, [t, d])
    dev_graph = graph_generation.generate_graph(dev_dict)
    de_pos, de_mdocs, de_mnodes = make_positive_examples(dev_graph, dev_dict, dev_cis, d, prob=de_prob * 0.2,                                                         trim_size=40, trim_auth=trim_auth)
    
    test_dict, test_cis = data_splitting_utils.get_documents(query_dict, [t, d, e])
    test_graph = graph_generation.generate_graph(test_dict)
    te_pos, te_mdocs, te_mnodes = make_positive_examples(test_graph, test_dict, test_cis, e, prob=te_prob * 0.2,                                                         trim_size=40, trim_auth=trim_auth)
    
    
    return (tr_pos, tr_mdocs, tr_mnodes), (de_pos, de_mdocs, de_mnodes), (te_pos, te_mdocs, te_mnodes)


# In[15]:


def add_doc_to_graph(G, doc_tensors, node_dids, graph_amtx, doc_emb, doc_ments, doc_name, src_name):
    '''
        Input: Graph G (graph_generation) & doc_tensors, node_dids, graph_amtx (get_graph_tensors)
        doc_embs, doc_ments (from data_dict) & doc_name (name of doc's node) & src_name (name of issue/event node)
        
        Output: G, doc_tensors, node_dids, graph_amtx with the doc included
    '''
    G_n, G_i2n, G_a = G
    #Get node ID and add to graph
    doc_node_ID = len(G_n)
    G_n[doc_name] = doc_node_ID
    G_i2n[doc_node_ID] = doc_name
    #Add edges between new doc and event/issue node
    G_a[doc_node_ID] = []
    if src_name in G_n:
        G_a[doc_node_ID] = [G_n[src_name]]
        G_a[G_n[src_name]].append(doc_node_ID)
    #Add to doc_tensors
    doc_did = len(doc_tensors)
    doc_tensors.append(np.squeeze(doc_emb, axis=0))
    #Add to node doc-tensor IDs
    node_dids.append([doc_did])
    #Add to graph_amtx, first add a column and then add a row
    curr_nodes = graph_amtx.shape[0]
    amtx_col = np.zeros((curr_nodes, 1))
    graph_amtx = np.hstack((graph_amtx, amtx_col))
    amtx_row = np.zeros((1, curr_nodes + 1))
    graph_amtx = np.vstack((graph_amtx, amtx_row))
    #Place the edges from event/issue nodes in adjacenecy matrix
    if src_name in G_n:
        graph_amtx[G_n[src_name], doc_node_ID] = 1
        graph_amtx[doc_node_ID, G_n[src_name]] = 1
    #If any mentioned entity of the doc is in current graph, add edges from it to doc
    for sent in doc_ments:
        for ment in sent:
            ment_wname = graph_generation.get_wiki_name(ment)
            if ment_wname in G_n:
                G_a[G_n[ment_wname]].append(doc_node_ID)
                graph_amtx[G_n[ment_wname], doc_node_ID] = 1
    return G, doc_tensors, node_dids, graph_amtx


# In[16]:


def remove_doc_from_graph(G, doc_tensors, node_dids, graph_amtx, doc_node_ID, doc_ments):
    '''
        Input: G, doc_node_ID, doc_ments, doc_tensors, node_dids, graph_amtx
        Output: G, doc_tensors, node_dids, graph_amtx with the doc removed
    '''
    G_n, G_i2n, G_a = G
    #Remove doc ID from index_to_nodeID
    doc_name = G_i2n.pop(doc_node_ID)
    _ = G_n.pop(doc_name)
    src_name = G_a.pop(doc_node_ID)
    if len(src_name) > 0:
        src_name = src_name[0]
        #Remove event/issue -> doc edge
        G_a[src_name].remove(doc_node_ID)
    #Assuming doc is last node added to the graph tensors
    _ = doc_tensors.pop(-1)
    _ = node_dids.pop(-1)
    graph_amtx = graph_amtx[:-1, :-1]
    for sent in doc_ments:
        for ment in sent:
            ment_wname = graph_generation.get_wiki_name(ment)
            if ment_wname in G_n:
                G_a[G_n[ment_wname]].remove(doc_node_ID)
    return G, doc_tensors, node_dids, graph_amtx


# In[17]:


def make_negative_examples(G, data_dict, cis_dict, idx_dict, num_eg=100, trim_size=40,                           trim_auth=False, set_query_dicts=None, set_idx_dicts=None, only_hard=False):
    '''
         Input: 
         Output: 
    '''
    G_n, G_i2n, G_a = G
    doc_tensors, node_dids, graph_amtx = get_graph_tensors(data_dict, G)
    neg_egs = []
    max_docs = [0]
    max_nodes = [0]
    
    #Getting different types of nodes in the graph
    news_nodes, tweet_nodes, statement_nodes, quote_nodes, event_nodes, author_nodes = get_node_lists(G, cis_dict, idx_dict)
    author_node = author_nodes[0]
    author = author_node.split('-')[1]
    
    if not only_hard:
        #Adding negative examples for author-news nodes
        num_news_eg = min(len(news_nodes), math.ceil(0.2 * num_eg))
        sel_news_nodes = random.sample(news_nodes, num_news_eg)
        for n2 in sel_news_nodes:
            n1_ID = G_n[author_node]
            n2_ID = G_n[n2]
            eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                       num_extra_nodes=trim_size, trim_auth=trim_auth)
            max_docs.append(eg_size[0])
            max_nodes.append(eg_size[1])
            neg_egs.append(eg)

        #Adding negative examples for event nodes
        sel_event_nodes = random.sample(event_nodes, 5)
        for node in sel_event_nodes:
            #Adding negative examples for event-tweet nodes
            num_tweet_eg = min(math.ceil(0.05 * num_eg), len(tweet_nodes))
            sel_tweet_nodes = random.sample(tweet_nodes, num_tweet_eg)
            for n2 in sel_tweet_nodes:
                n1_ID = G_n[node]
                n2_ID = G_n[n2]
                eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                           num_extra_nodes=trim_size, trim_auth=trim_auth)
                max_docs.append(eg_size[0])
                max_nodes.append(eg_size[1])
                neg_egs.append(eg)
            #Adding negative examples for event-statement nodes
            num_statement_eg = min(math.ceil(0.05 * num_eg), len(statement_nodes))
            sel_statement_nodes = random.sample(statement_nodes, num_statement_eg)
            for n2 in sel_statement_nodes:
                n1_ID = G_n[node]
                n2_ID = G_n[n2]
                eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                           num_extra_nodes=trim_size, trim_auth=trim_auth)
                max_docs.append(eg_size[0])
                max_nodes.append(eg_size[1])
                neg_egs.append(eg)
            #Adding negative examples for event-quote nodes
            if len(quote_nodes) > 0:
                toss = random.random()
                if toss <= 0.8:
                    n1_ID = G_n[node]
                    n2_ID = G_n[quote_nodes[0]]
                    eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                           num_extra_nodes=trim_size, trim_auth=trim_auth)
                    max_docs.append(eg_size[0])
                    max_nodes.append(eg_size[1])
                    neg_egs.append(eg)
                
    #Adding negative examples from other authors
    if set_query_dicts:
        num_set_egs = (0.7 / len(set_query_dicts)) * num_eg
        for set_query_dict, set_idx_dict in zip(set_query_dicts, set_idx_dicts):
            set_doc_count = count_docs(set_idx_dict)
            set_prob = num_set_egs / set_doc_count
            if only_hard:
                set_prob = 1.
            key = [key for key in set_idx_dict if key != 'news'][0]
            #Adding tweet and statement doc neagtive examples from other authors
            for dtype in set_idx_dict[key]:
                for topic in set_idx_dict[key][dtype]:
                    for event in set_idx_dict[key][dtype][topic]:
                        for doc_index in set_idx_dict[key][dtype][topic][event]:
                            toss = random.random()
                            if toss <= set_prob:
                                doc_emb = set_query_dict[key][dtype][2][topic][event][doc_index]
                                doc_ments = set_query_dict[key][dtype][1][topic][event][doc_index]
                                doc_name = '-'.join(['doc', dtype[:-1], author, topic, str(event), 'no_num'])
                                src_name = 'event-' + topic + '-' + str(event)
                                G, doc_tensors, node_dids, graph_amtx = add_doc_to_graph(G, doc_tensors, node_dids,                                                                graph_amtx, doc_emb, doc_ments, doc_name, src_name)
                                n1_ID = G_n[author_node]
                                n2_ID = len(G_n) - 1
                                eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                                          num_extra_nodes=trim_size, trim_auth=trim_auth)
                                max_docs.append(eg_size[0])
                                max_nodes.append(eg_size[1])
                                neg_egs.append(eg)
                                G, doc_tensors, node_dids, graph_amtx = remove_doc_from_graph(G, doc_tensors,                                                                        node_dids, graph_amtx, n2_ID, doc_ments)
            #Adding set quotes doc to negative example
            for topic in set_query_dict[key]['quotes'][0]:
                for doc_index in range(len(set_query_dict[key]['quotes'][0][topic])):
                    doc_emb = set_query_dict[key]['quotes'][2][topic][doc_index]
                    doc_ments = set_query_dict[key]['quotes'][1][topic][doc_index]
                    doc_name = '-'.join(['doc-quote', author, topic, 'no_num'])
                    src_name = 'issue-' + topic
                    G, doc_tensors, node_dids, graph_amtx = add_doc_to_graph(G, doc_tensors, node_dids,                                                    graph_amtx, doc_emb, doc_ments, doc_name, src_name)
                    n1_ID = G_n[author_node]
                    n2_ID = len(G_n) - 1
                    eg, eg_size = trim_example(G, doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID,                                              num_extra_nodes=trim_size, trim_auth=trim_auth)
                    max_docs.append(eg_size[0])
                    max_nodes.append(eg_size[1])
                    neg_egs.append(eg)
                    G, doc_tensors, node_dids, graph_amtx = remove_doc_from_graph(G, doc_tensors,                                                        node_dids, graph_amtx, n2_ID, doc_ments)
                    
    return neg_egs, max(max_docs), max(max_nodes)


# In[18]:


def get_negative_examples(query_dicts, qd_splits, num_tr=40, trim_auth=False, log=sys.stdout, only_hard=False):
    '''
        Input: query_dicts, qd_splits -> generated in query_handling.query_processing and data_splitting_utils rly
        num_tr -> number fo training examples to be generated
        trim_auth -> for trim_example
        
        Output: negative link prediction examples and max examples sizes
    '''
    train = []
    dev = []
    test = []
    t_mdocs = []
    d_mdocs = []
    e_mdocs = []
    t_mnodes = []
    d_mnodes = []
    e_mnodes = []
    
    issue_sets = []
    for i in range(8):
        issue_set = []
        j = i
        while j < len(query_dicts):
            issue_set.append(j)
            j += 8
        issue_sets.append(issue_set)
    
    i = 0
    t1 = datetime.now()
    for query_dict, qd_split in zip(query_dicts, qd_splits):
        t, d, e, = qd_split
        
        train_dict, train_cis = data_splitting_utils.get_documents(query_dict, [t])
        train_graph = graph_generation.generate_graph(train_dict)
        set_query_dicts = [query_dicts[j] for j in issue_sets[i%8] if j != i]
        set_idx_dicts = [qd_splits[j][0] for j in issue_sets[i%8] if j != i]
        tr_neg, tr_mdocs, tr_mnodes = make_negative_examples(train_graph, train_dict, train_cis, t,                                                        num_eg=num_tr, trim_size=40, trim_auth=trim_auth,                                                        set_query_dicts=set_query_dicts, set_idx_dicts=set_idx_dicts, only_hard=only_hard)
        train += tr_neg
        t_mdocs.append(tr_mdocs)
        t_mnodes.append(tr_mnodes)
        

        dev_dict, dev_cis = data_splitting_utils.get_documents(query_dict, [t, d])
        dev_graph = graph_generation.generate_graph(dev_dict)
        set_query_dicts = [query_dicts[j] for j in issue_sets[i%8] if j != i]
        set_idx_dicts = [qd_splits[j][1] for j in issue_sets[i%8] if j != i]
        de_neg, de_mdocs, de_mnodes = make_negative_examples(dev_graph, dev_dict, dev_cis, d,                                                        num_eg=math.ceil(num_tr*0.25), trim_size=40, trim_auth=trim_auth,                                                        set_query_dicts=set_query_dicts, set_idx_dicts=set_idx_dicts, only_hard=only_hard)
        dev += de_neg
        d_mdocs.append(de_mdocs)
        d_mnodes.append(de_mnodes)

        test_dict, test_cis = data_splitting_utils.get_documents(query_dict, [t, d, e])
        test_graph = graph_generation.generate_graph(test_dict)
        set_query_dicts = [query_dicts[j] for j in issue_sets[i%8] if j != i]
        set_idx_dicts = [qd_splits[j][2] for j in issue_sets[i%8] if j != i]
        te_neg, te_mdocs, te_mnodes = make_negative_examples(test_graph, test_dict, test_cis, e,                                                        num_eg=math.ceil(num_tr*0.25), trim_size=40, trim_auth=trim_auth,                                                        set_query_dicts=set_query_dicts, set_idx_dicts=set_idx_dicts, only_hard=only_hard)
        test += te_neg
        e_mdocs.append(te_mdocs)
        e_mnodes.append(te_mnodes)
        i += 1
        t2 = datetime.now()
    
    print('Negative examples loaded - ', t2 - t1, file=log, flush=True)
    
    return (train, max(t_mdocs), max(t_mnodes)), (dev, max(d_mdocs), max(d_mnodes)), (test, max(e_mdocs), max(e_mnodes))


# In[19]:


def load_data_batch(qdb_path, log=sys.stdout, trim_auth=False, only_hard=False):
    with open(qdb_path, 'rb') as infile:
        query_dicts, qd_splits = pickle.load(infile)
    
    neg_data = get_negative_examples(query_dicts, qd_splits, log=log, trim_auth=trim_auth, only_hard=only_hard)
    pos_data = [([], -1, -1), ([], -1, -1), ([], -1, -1)]
    i = 0
    t1 = datetime.now()
    for query_dict, qd_split in zip(query_dicts, qd_splits):
        pos_egs =  get_positive_examples(query_dict, qd_split, trim_auth=trim_auth)
        pos_data[0] = (pos_data[0][0] + pos_egs[0][0], max(pos_data[0][1], pos_egs[0][1]), max(pos_data[0][2], pos_egs[0][2]))
        pos_data[1] = (pos_data[1][0] + pos_egs[1][0], max(pos_data[1][1], pos_egs[1][1]), max(pos_data[1][2], pos_egs[1][2]))
        pos_data[2] = (pos_data[2][0] + pos_egs[2][0], max(pos_data[2][1], pos_egs[2][1]), max(pos_data[2][2], pos_egs[2][2]))
        i += 1
        t2 = datetime.now()
    print('Positive examples loaded - ',  t2 - t1, file=log, flush=True)
    
    
    train_x = []
    train_y = []
    dev_x = []
    dev_y = []
    test_x = []
    test_y = []
    
    tr_pos, de_pos, te_pos = pos_data
    train_x += tr_pos[0]
    train_y += [1] * len(tr_pos[0])
    dev_x += de_pos[0]
    dev_y += [1] * len(de_pos[0])
    test_x += te_pos[0]
    test_y += [1] * len(te_pos[0])
    
    tr_neg, de_neg, te_neg = neg_data
    train_x += tr_neg[0]
    train_y += [0] * len(tr_neg[0])
    dev_x += de_neg[0]
    dev_y += [0] * len(de_neg[0])
    test_x += te_neg[0]
    test_y += [0] * len(te_neg[0])


    data = ((train_x, train_y), (dev_x, dev_y), (test_x, test_y))
    print('Train size', max(tr_pos[1], tr_neg[1]), max(tr_pos[2], tr_neg[2]), file=log, flush=True)
    print('Dev size', max(de_pos[1], de_neg[1]), max(de_pos[2], de_neg[2]), file=log, flush=True)
    print('Test size', max(te_pos[1], te_neg[1]), max(te_pos[2], te_neg[2]), file=log, flush=True)
    
    return data


# In[20]:


def batch_to_tensors(batch_x, batch_y):
    batch_size = len(batch_x)
    #batch_x is list of examples, each example is a tuple of doc_tensors, node_dids, graph_amtx, n1_ID, n2_ID
    max_num_nodes = max([ex[2].shape[0] for ex in batch_x])
    max_num_docs = max([len(ex[0]) for ex in batch_x])
    doc_size = batch_x[0][0][0].shape[-1]
    
    #Creating a batch adjacency matrix filled with zeros and set diagnol values to 1. All nodes have self-loops
    comb_amtx = torch.zeros(batch_size, max_num_nodes, max_num_nodes).type(torch.LongTensor)
    for i in range(batch_size):
        for j in range(max_num_nodes):
            comb_amtx[i, j, j] = 1
    
    #Creating a batch node_docs and batch node_mask tensors
    node_docs = torch.zeros(batch_size, max_num_nodes, max_num_docs, doc_size).type(torch.FloatTensor)
    node_mask = torch.zeros(batch_size, max_num_nodes, max_num_docs).type(torch.FloatTensor)
    #Creating batch src_ID, batch dest_ID & batch y tensors
    src_ids = np.ndarray((batch_size,), dtype=int)
    dest_ids = np.ndarray((batch_size,), dtype=int)
    batch_labels = np.ndarray((batch_size,), dtype=int)
    
    #For each example in the batch
    for i, ex in enumerate(batch_x):
        g_docs, g_node_dids, graph_amtx, src_idx, dest_idx = ex
        #Compute doc embeddings by sum-pooling sentence embeddings (can avoid if you have larger computational power/more time)
        g_doc_tensors = [torch.sum(torch.tensor(doc), dim=0) for doc in g_docs]
        
        #For every node in the current graph
        for j, node_dids in enumerate(g_node_dids):
            #If node is not associated with any documents (some event nodes / issue node with no background text)
            if len(node_dids) == 0:
                #Initialize node embedding randomly
                node_mask[i, j, 0] = 1
                node_docs[i, j, 0, :] = torch.rand(doc_size).type(torch.FloatTensor)
            else:
                #Else place all the doc tensors accociated with the node according to node_dids into batch node_docs tensor
                node_mask[i, j, :len(node_dids)] = 1
                node_docs[i, j, :len(node_dids), :] = torch.cat([g_doc_tensors[k].unsqueeze(0) for k in node_dids], dim=0)
                
        #Fill padding nodes with random initialization, leaving it 0 will cause NaN error in composer, doesn't come into play for b_sz=1
        for j in range(len(g_node_dids), max_num_nodes):
            node_mask[i, j, 0] = 1
            node_docs[i, j, 0, :] = torch.rand(doc_size).type(torch.FloatTensor)
        #Fill batch amtx, batch src_ID, batch dest_ID, batch y tensors
        num_nodes = graph_amtx.shape[0]
        comb_amtx[i, :num_nodes, :num_nodes] = torch.from_numpy(graph_amtx)
        src_ids[i] = src_idx
        dest_ids[i] = dest_idx
        batch_labels[i] = batch_y[i]
    
    #Create torch tensors from numpy arrays
    src_tensor = torch.from_numpy(src_ids)
    dest_tensor = torch.from_numpy(dest_ids)
    y_tensor = torch.from_numpy(batch_labels)
    
    #return batch tensors
    return (node_docs, node_mask, comb_amtx, src_tensor, dest_tensor, y_tensor)


# In[21]:


def batchify_data(data_x, data_y, batch_size=1, shuffle=False):
    dlen = len(data_x)
    if shuffle:
        indices = list(range(dlen))
        random.shuffle(indices)
        data_x = [data_x[i] for i in indices]
        data_y = [data_y[i] for i in indices]
    start = 0
    end = batch_size
    while start < dlen:
        batch_data = batch_to_tensors(data_x[start:end], data_y[start:end])
        yield batch_data
        start = end
        end += batch_size

