import time
import logging
import numpy as np
import h5py
import scipy.linalg
import scipy.sparse
import random
import sys
import os


# Visualizing on cortical surfaces using mapper files
def load_sparse_array(fname, varname):
    """Load a numpy sparse array from an hdf file

    Parameters
    ----------
    fname: string
        file name containing array to be loaded
    varname: string
        name of variable to be loaded

    Notes
    -----
    This function relies on variables being stored with specific naming
    conventions, so cannot be used to load arbitrary sparse arrays.

    """
    with h5py.File(fname, mode='r') as hf:
        data = (hf['%s_data' % varname], hf['%s_indices' % varname],
                hf['%s_indptr' % varname])
        sparsemat = scipy.sparse.csr_matrix(data,
                                            shape=hf['%s_shape' % varname])
    return sparsemat


def map_to_flat(voxels, mapper_file):
    """Generate flatmap image for an individual subject from voxel array

    This function maps a list of voxels into a flattened representation
    of an individual subject's brain.

    Parameters
    ----------
    voxels: array
        n x 1 array of voxel values to be mapped
    mapper_file: string
        file containing mapping arrays

    Returns
    -------
    image : array
        flatmap image, (n x 1024)

    """
    pixmap = load_sparse_array(mapper_file, 'pixmap')
    with h5py.File(mapper_file, mode='r') as hf:
        pixmask = hf['pixmask'].value
    badmask = np.array(pixmap.sum(1) > 0).ravel()
    img = (np.nan * np.ones(pixmask.shape)).astype(voxels.dtype)
    mimg = (np.nan * np.ones(badmask.shape)).astype(voxels.dtype)
    mimg[badmask] = (pixmap * voxels.ravel())[badmask].astype(mimg.dtype)
    img[pixmask] = mimg
    return img.T[::-1]

# Loading dictioneries
def recursively_save_dict_contents_to_group(h5file, path, filedict):
    """
    """
    for key, item in list(filedict.items()):
        if isinstance(item, (np.ndarray, np.int64, np.float64, str, bytes)):
            h5file[path + key] = item
        elif isinstance(item, dict):
            recursively_save_dict_contents_to_group(h5file, path + key + '/', item)
        else:
            raise ValueError('Cannot save %s type'%type(item))

def recursively_load_dict_contents_from_group(h5file, path):
    """
    """
    filedict = {}
    for key, item in list(h5file[path].items()):
        if isinstance(item, h5py._hl.dataset.Dataset):
            filedict[key] = item[()]
        elif isinstance(item, h5py._hl.group.Group):
            filedict[key] = recursively_load_dict_contents_from_group(h5file, path + key + '/')
    return filedict

def save_dict(filename, filedict):
    """Saves the variables in [filedict].
    """

    with h5py.File(filename, 'w') as h5file:
        recursively_save_dict_contents_to_group(h5file, '/', filedict)


def load_dict(filename):

    with h5py.File(filename, 'r') as h5file:
    	filedict = recursively_load_dict_contents_from_group(h5file, '/')
    return filedict


def make_delayed(stim, delays, circpad=False):
    """Creates non-interpolated concatenated delayed versions of [stim] with the given [delays] 
    (in samples).
    
    If [circpad], instead of being padded with zeros, [stim] will be circularly shifted.
    """
    nt,ndim = stim.shape
    dstims = []
    for di,d in enumerate(delays):
        dstim = np.zeros((nt, ndim))
        if d<0: ## negative delay
            dstim[:d,:] = stim[-d:,:]
            if circpad:
                dstim[d:,:] = stim[:-d,:]
        elif d>0:
            dstim[d:,:] = stim[:-d,:]
            if circpad:
                dstim[:d,:] = stim[-d:,:]
        else: ## d==0
            dstim = stim.copy()
        dstims.append(dstim)
    return np.hstack(dstims)


