#!/usr/bin/env python3
# Copyright (c) Facebook, Inc. and its affiliates.
# All rights reserved.
#
# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

"""
 Command line tool to get dense results and validate them
"""

import argparse
import os
import csv
import glob
import json
import gzip
import logging
import pickle
import time
import math
from typing import List, Tuple, Dict, Iterator

import numpy as np
import torch
from torch import Tensor as T
import torch.nn.functional as F
from torch import nn
from distutils.util import strtobool

from dpr.data.qa_validation import calculate_matches
from dpr.models import init_biencoder_components
from dpr.options import add_encoder_params, setup_args_gpu, print_args, set_encoder_params_from_state, \
    add_tokenizer_params, add_cuda_params
from dpr.utils.data_utils import Tensorizer
from dpr.utils.model_utils import setup_for_distributed_mode, get_model_obj, load_states_from_checkpoint
from dpr.indexer.faiss_indexers import DenseIndexer, DenseHNSWFlatIndexer, DenseFlatIndexer

logger = logging.getLogger()
logger.setLevel(logging.INFO)
if (logger.hasHandlers()):
    logger.handlers.clear()
console = logging.StreamHandler()
logger.addHandler(console)


class DenseRetriever(object):
    """
    Does passage retrieving over the provided index and question encoder
    """
    def __init__(self, args, question_encoder: nn.Module, batch_size: int, tensorizer: Tensorizer, index: DenseIndexer):
        self.question_encoder = question_encoder
        self.batch_size = batch_size
        self.tensorizer = tensorizer
        self.index = index
        self.args = args

    def generate_question_vectors(self, questions: List[str]) -> T:
        n = len(questions)
        bsz = self.batch_size
        query_vectors = []

        self.question_encoder.eval()

        with torch.no_grad():
            for j, batch_start in enumerate(range(0, n, bsz)):

                batch_token_tensors = [self.tensorizer.text_to_tensor(q) for q in
                                       questions[batch_start:batch_start + bsz]]

                q_ids_batch = torch.stack(batch_token_tensors, dim=0).cuda()
                q_seg_batch = torch.zeros_like(q_ids_batch).cuda()
                q_attn_mask = self.tensorizer.get_attn_mask(q_ids_batch)
                _q_seq, out, _ = self.question_encoder(q_ids_batch, q_seg_batch, q_attn_mask)

                out = _q_seq[:,0,:]
                query_vectors.extend(out.cpu().split(1, dim=0))

                if len(query_vectors) % 100 == 0:
                    logger.info('Encoded queries %d', len(query_vectors))

        query_tensor = torch.cat(query_vectors, dim=0)

        logger.info('Total encoded queries tensor %s', query_tensor.size())

        assert query_tensor.size(0) == len(questions)
        return query_tensor

    def get_top_docs(self, query_vectors: np.array, top_docs: int = 100) -> List[Tuple[List[object], List[float]]]:
        """
        Does the retrieval of the best matching passages given the query vectors batch
        :param query_vectors:
        :param top_docs:
        :return:
        """
        time0 = time.time()
        results = self.index.search_knn(query_vectors, top_docs)
        logger.info('index search time: %f sec.', time.time() - time0)
        return results


def parse_qa_csv_file(location) -> Iterator[Tuple[str, List[str]]]:
    with open(location) as ifile:
        reader = csv.reader(ifile, delimiter='\t')
        for row in reader:
            question = row[0]
            answers = eval(row[1])
            yield question, answers


def validate(passages: Dict[object, Tuple[str, str]], answers: List[List[str]],
             result_ctx_ids: List[Tuple[List[object], List[float]]],
             workers_num: int, match_type: str) -> List[List[bool]]:
    match_stats = calculate_matches(passages, answers, result_ctx_ids, workers_num, match_type)
    top_k_hits = match_stats.top_k_hits

    logger.info('Validation results: top k documents hits %s', top_k_hits)
    top_k_hits = [v / len(result_ctx_ids) for v in top_k_hits]
    for i in range(0, len(top_k_hits)):
        logger.info(f'Validation results: top {i+1:3d} document hits accuracy %s', top_k_hits[i])
    return match_stats.questions_doc_hits


def load_passages(ctx_file: str) -> Dict[object, Tuple[str, str]]:
    docs = {}
    logger.info('Reading data from: %s', ctx_file)
    if ctx_file.endswith(".gz"):
        with gzip.open(ctx_file, 'rt') as tsvfile:
            reader = csv.reader(tsvfile, delimiter='\t', )
            # file format: doc_id, doc_text, title
            for row in reader:
                if row[0] != 'id':
                    docs[row[0]] = (row[1], row[2])
    else:
        for file in glob.glob(ctx_file):
            with open(file) as tsvfile:
                reader = csv.reader(tsvfile, delimiter='\t', )
                # file format: doc_id, doc_text, title
                for row in reader:
                    if row[0] != 'id':
                        docs[row[0]] = (row[1], row[2])
    return docs


def save_results(passages: Dict[object, Tuple[str, str]], questions: List[str], answers: List[List[str]],
                 top_passages_and_scores: List[Tuple[List[object], List[float]]], per_question_hits: List[List[bool]],
                 out_file: str
                 ):
    # join passages text with the result ids, their questions and assigning has|no answer labels
    merged_data = []
    assert len(per_question_hits) == len(questions) == len(answers)
    for i, q in enumerate(questions):
        q_answers = answers[i]
        results_and_scores = top_passages_and_scores[i]
        hits = per_question_hits[i]
        docs = [passages[doc_id] for doc_id in results_and_scores[0]]
        scores = [str(score) for score in results_and_scores[1]]
        ctxs_num = len(hits)

        merged_data.append({
            'question': q,
            'answers': q_answers,
            'ctxs': [
                {
                    'id': results_and_scores[0][c],
                    'title': docs[c][1],
                    'text': docs[c][0],
                    'score': scores[c],
                    'has_answer': hits[c],
                } for c in range(ctxs_num)
            ]
        })

    with open(out_file, "w") as writer:
        writer.write(json.dumps(merged_data, indent=4) + "\n")
    logger.info('Saved results * scores  to %s', out_file)


def iterate_encoded_files(vector_files: list) -> Iterator[Tuple[object, np.array]]:
    for i, file in enumerate(vector_files):
        logger.info('Reading file %s', file)
        with open(file, "rb") as reader:
            doc_vectors = pickle.load(reader)
            for doc in doc_vectors:
                db_id, doc_vector = doc
                yield db_id, doc_vector


def main(args):
    saved_state = load_states_from_checkpoint(args.model_file)
    set_encoder_params_from_state(saved_state.encoder_params, args)

    tensorizer, encoder, _ = init_biencoder_components(args.encoder_model_type, args, inference_only=True)

    encoder = encoder.question_model

    encoder, _ = setup_for_distributed_mode(encoder, None, args.device, args.n_gpu,
                                            args.local_rank,
                                            args.fp16)
    encoder.eval()

    # load weights from the model file
    model_to_load = get_model_obj(encoder)
    logger.info('Loading saved model state ...')

    prefix_len = len('question_model.')
    question_encoder_state = {key[prefix_len:]: value for (key, value) in saved_state.model_dict.items() if
                              key.startswith('question_model.')}
    model_to_load.load_state_dict(question_encoder_state)
    vector_size = model_to_load.get_out_size()
    logger.info('Encoder vector_size=%d', vector_size)

    if args.hnsw_index:
        index = DenseHNSWFlatIndexer(vector_size, args.index_buffer)
    else:
        index = DenseFlatIndexer(vector_size, args.index_buffer)

    retriever = DenseRetriever(args, encoder, args.batch_size, tensorizer, index)


    # index all passages
    ctx_files_pattern = args.encoded_ctx_file
    input_paths = glob.glob(ctx_files_pattern)

    index_path = "_".join(input_paths[0].split("_")[:-1])
    if args.save_or_load_index and (os.path.exists(index_path) or os.path.exists(index_path + ".index.dpr")):
        retriever.index.deserialize_from(index_path)
    else:
        logger.info('Reading all passages data from files: %s', input_paths)
        retriever.index.index_data(input_paths)
        if args.save_or_load_index:
            retriever.index.serialize(index_path)
    # get questions & answers
    questions = []
    question_answers = []

    for ds_item in parse_qa_csv_file(args.qa_file):
        question, answers = ds_item
        questions.append(question)
        question_answers.append(answers)

    questions_tensor = retriever.generate_question_vectors(questions)

    # get top k results
    top_ids_and_scores = retriever.get_top_docs(questions_tensor.numpy(), args.n_docs)

    all_passages = load_passages(args.ctx_file)

    if len(all_passages) == 0:
        raise RuntimeError('No passages data found. Please specify ctx_file param properly.')

    questions_doc_hits = validate(all_passages, question_answers, top_ids_and_scores, args.validation_workers,
                                  args.match)

    if args.out_file:
        save_results(all_passages, questions, question_answers, top_ids_and_scores, questions_doc_hits, args.out_file)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()

    add_encoder_params(parser)
    add_tokenizer_params(parser)
    add_cuda_params(parser)

    parser.add_argument('--qa_file', required=True, type=str, default=None,
                        help="Question and answers file of the format: question \\t ['answer1','answer2', ...]")
    parser.add_argument('--ctx_file', required=True, type=str, default=None,
                        help="All passages file in the tsv format: id \\t passage_text \\t title")
    parser.add_argument('--encoded_ctx_file', type=str, default=None,
                        help='Glob path to encoded passages (from generate_dense_embeddings tool)')
    parser.add_argument('--out_file', type=str, default=None,
                        help='output .json file path to write results to ')
    parser.add_argument('--match', type=str, default='string', choices=['regex', 'string'],
                        help="Answer matching logic type")
    parser.add_argument('--n-docs', type=int, default=200, help="Amount of top docs to return")
    parser.add_argument('--validation_workers', type=int, default=64,
                        help="Number of parallel processes to validate results")
    parser.add_argument('--batch_size', type=int, default=32, help="Batch size for question encoder forward pass")
    parser.add_argument('--index_buffer', type=int, default=50000,
                        help="Temporal memory data buffer size (in samples) for indexer")
    parser.add_argument("--hnsw_index", action='store_true', help='If enabled, use inference time efficient HNSW index')
    parser.add_argument("--save_or_load_index", action='store_true', help='If enabled, save index')

    # add args
    parser.add_argument('--shared_encoder', type=lambda x: bool(strtobool(x)), default=True, help="whether used shared parameter for q/v encoder")
    parser.add_argument("--max_q_len", default=256, type=int, help="")
    parser.add_argument("--max_p_len", default=256, type=int, help="")

    # args = parser.parse_args()
    args, _ = parser.parse_known_args()

    assert args.model_file, 'Please specify --model_file checkpoint to init model weights'

    setup_args_gpu(args)
    print_args(args)


    # main function
    main(args)

    result = json.load(open(args.out_file, "r"))


    if "hotpot" in args.qa_file:
        comp_idx = [0, 3, 9, 13, 17, 19, 23, 27, 31, 33, 34, 38, 40, 48, 59, 62, 87, 89, 93, 97, 99, 113, 118, 119, 130, 131, 141, 143, 150, 154, 156, 166, 175, 188, 206, 208, 212, 214, 222, 229, 233, 239, 247, 248, 250, 253, 259, 271, 276, 283, 285, 289, 304, 308, 309, 311, 313, 318, 320, 328, 334, 337, 338, 343, 351, 352, 354, 355, 357, 370, 373, 380, 395, 401, 406, 407, 410, 415, 420, 421, 426, 431, 432, 433, 434, 441, 454, 456, 467, 472, 477, 478, 483, 486, 490, 499, 506, 513, 522, 524, 525, 528, 530, 531, 535, 552, 554, 556, 557, 558, 561, 565, 593, 595, 605, 610, 611, 612, 613, 622, 624, 649, 654, 657, 661, 666, 669, 679, 691, 696, 698, 699, 705, 708, 722, 726, 729, 735, 736, 739, 750, 752, 754, 759, 761, 766, 767, 769, 775, 783, 785, 798, 799, 810, 817, 818, 822, 825, 831, 835, 843, 847, 848, 853, 858, 868, 875, 881, 882, 890, 891, 892, 898, 899, 900, 906, 908, 909, 911, 917, 918, 923, 925, 944, 945, 958, 962, 970, 976, 979, 988, 989, 998, 1005, 1008, 1010, 1012, 1013, 1019, 1023, 1026, 1032, 1035, 1036, 1045, 1050, 1051, 1053, 1054, 1066, 1071, 1076, 1078, 1085, 1092, 1101, 1102, 1104, 1105, 1108, 1109, 1112, 1115, 1124, 1127, 1131, 1134, 1140, 1155, 1163, 1164, 1165, 1173, 1192, 1194, 1197, 1201, 1206, 1212, 1219, 1223, 1225, 1227, 1228, 1232, 1235, 1237, 1239, 1244, 1247, 1249, 1258, 1268, 1273, 1274, 1275, 1277, 1278, 1292, 1300, 1301, 1304, 1305, 1306, 1308, 1317, 1325, 1338, 1341, 1345, 1346, 1351, 1362, 1375, 1376, 1382, 1383, 1389, 1399, 1416, 1417, 1427, 1429, 1443, 1445, 1452, 1453, 1459, 1462, 1466, 1473, 1479, 1483, 1491, 1505, 1508, 1517, 1521, 1524, 1528, 1532, 1534, 1535, 1546, 1548, 1552, 1556, 1564, 1576, 1579, 1582, 1588, 1589, 1590, 1594, 1600, 1606, 1627, 1629, 1640, 1641, 1655, 1656, 1657, 1658, 1666, 1676, 1683, 1690, 1691, 1702, 1705, 1706, 1708, 1719, 1721, 1723, 1729, 1732, 1733, 1738, 1740, 1743, 1744, 1745, 1750, 1752, 1754, 1755, 1760, 1771, 1772, 1777, 1778, 1784, 1785, 1791, 1792, 1798, 1800, 1804, 1808, 1819, 1825, 1827, 1841, 1845, 1854, 1867, 1873, 1877, 1878, 1886, 1893, 1900, 1901, 1914, 1930, 1932, 1936, 1946, 1950, 1960, 1961, 1989, 1992, 1995, 1997, 2006, 2007, 2009, 2012, 2013, 2016, 2020, 2030, 2046, 2056, 2057, 2064, 2069, 2078, 2085, 2089, 2092, 2093, 2096, 2097, 2098, 2099, 2104, 2105, 2106, 2112, 2117, 2118, 2121, 2129, 2131, 2132, 2135, 2140, 2143, 2146, 2148, 2150, 2158, 2160, 2161, 2168, 2170, 2172, 2174, 2183, 2198, 2200, 2204, 2206, 2219, 2225, 2226, 2228, 2239, 2241, 2245, 2250, 2253, 2256, 2261, 2262, 2273, 2276, 2282, 2287, 2290, 2297, 2298, 2303, 2306, 2309, 2314, 2316, 2318, 2319, 2320, 2328, 2334, 2338, 2351, 2359, 2362, 2366, 2369, 2371, 2374, 2377, 2379, 2384, 2385, 2390, 2391, 2394, 2396, 2397, 2400, 2406, 2411, 2420, 2426, 2431, 2433, 2441, 2443, 2449, 2451, 2452, 2453, 2455, 2463, 2469, 2472, 2473, 2481, 2484, 2488, 2493, 2498, 2499, 2506, 2508, 2518, 2528, 2530, 2544, 2549, 2552, 2571, 2572, 2574, 2575, 2578, 2579, 2580, 2592, 2598, 2599, 2604, 2607, 2627, 2628, 2645, 2646, 2650, 2654, 2657, 2668, 2670, 2676, 2684, 2685, 2691, 2708, 2711, 2714, 2717, 2725, 2728, 2731, 2735, 2738, 2747, 2749, 2750, 2752, 2755, 2756, 2761, 2766, 2775, 2777, 2782, 2791, 2792, 2795, 2801, 2805, 2809, 2823, 2826, 2832, 2844, 2845, 2848, 2850, 2856, 2868, 2869, 2885, 2886, 2893, 2901, 2902, 2903, 2905, 2914, 2925, 2933, 2935, 2939, 2956, 2957, 2961, 2965, 2967, 2981, 2987, 2991, 2992, 2993, 2999, 3001, 3003, 3010, 3011, 3012, 3013, 3016, 3017, 3020, 3024, 3026, 3027, 3031, 3041, 3043, 3044, 3046, 3048, 3050, 3051, 3052, 3053, 3057, 3063, 3067, 3071, 3075, 3076, 3080, 3081, 3095, 3099, 3100, 3112, 3114, 3115, 3123, 3150, 3152, 3169, 3174, 3177, 3180, 3184, 3187, 3190, 3191, 3196, 3203, 3212, 3226, 3230, 3234, 3237, 3243, 3249, 3250, 3255, 3270, 3275, 3276, 3277, 3280, 3282, 3283, 3293, 3306, 3308, 3314, 3318, 3324, 3327, 3336, 3337, 3341, 3349, 3352, 3353, 3356, 3357, 3358, 3359, 3361, 3363, 3367, 3368, 3371, 3374, 3379, 3387, 3388, 3390, 3391, 3398, 3399, 3402, 3404, 3408, 3415, 3420, 3421, 3424, 3426, 3427, 3430, 3431, 3435, 3440, 3444, 3445, 3450, 3463, 3474, 3484, 3485, 3487, 3498, 3507, 3511, 3515, 3516, 3518, 3521, 3529, 3552, 3553, 3559, 3564, 3565, 3566, 3573, 3583, 3584, 3594, 3603, 3609, 3611, 3612, 3615, 3627, 3628, 3629, 3633, 3635, 3640, 3643, 3648, 3649, 3652, 3663, 3667, 3672, 3673, 3674, 3678, 3684, 3685, 3686, 3688, 3691, 3692, 3703, 3708, 3713, 3715, 3722, 3723, 3725, 3727, 3737, 3745, 3746, 3749, 3754, 3755, 3757, 3782, 3789, 3792, 3811, 3812, 3819, 3825, 3827, 3834, 3835, 3841, 3842, 3854, 3858, 3860, 3862, 3865, 3868, 3874, 3875, 3878, 3881, 3887, 3890, 3893, 3894, 3905, 3909, 3915, 3934, 3940, 3941, 3944, 3946, 3953, 3956, 3960, 3963, 3969, 3975, 3976, 3978, 3980, 3985, 3999, 4001, 4002, 4004, 4014, 4015, 4023, 4030, 4032, 4040, 4041, 4043, 4045, 4058, 4064, 4068, 4074, 4081, 4083, 4092, 4105, 4106, 4120, 4125, 4126, 4132, 4135, 4137, 4142, 4146, 4147, 4157, 4160, 4181, 4185, 4188, 4199, 4201, 4204, 4210, 4225, 4230, 4231, 4235, 4249, 4257, 4258, 4268, 4270, 4271, 4275, 4277, 4279, 4286, 4294, 4301, 4307, 4309, 4324, 4329, 4332, 4334, 4336, 4343, 4344, 4350, 4365, 4373, 4379, 4380, 4388, 4403, 4405, 4406, 4407, 4408, 4410, 4417, 4423, 4424, 4427, 4428, 4430, 4431, 4438, 4444, 4447, 4450, 4452, 4460, 4469, 4476, 4481, 4485, 4489, 4490, 4491, 4500, 4508, 4521, 4523, 4524, 4529, 4536, 4549, 4552, 4560, 4568, 4571, 4576, 4578, 4582, 4598, 4599, 4607, 4608, 4612, 4616, 4621, 4631, 4632, 4649, 4660, 4661, 4667, 4668, 4679, 4684, 4696, 4701, 4705, 4727, 4729, 4733, 4743, 4759, 4760, 4762, 4770, 4771, 4774, 4781, 4783, 4787, 4788, 4794, 4797, 4806, 4815, 4824, 4826, 4830, 4839, 4840, 4841, 4857, 4858, 4864, 4865, 4868, 4872, 4876, 4877, 4883, 4887, 4890, 4895, 4905, 4907, 4910, 4911, 4918, 4920, 4923, 4927, 4938, 4941, 4942, 4949, 4953, 4954, 4959, 4962, 4967, 4968, 4982, 4985, 4994, 5004, 5007, 5013, 5025, 5028, 5031, 5047, 5052, 5054, 5056, 5063, 5072, 5073, 5075, 5080, 5084, 5096, 5102, 5107, 5109, 5112, 5118, 5119, 5120, 5122, 5127, 5130, 5132, 5141, 5158, 5160, 5167, 5171, 5178, 5188, 5191, 5192, 5197, 5204, 5207, 5211, 5212, 5217, 5221, 5227, 5235, 5239, 5242, 5263, 5270, 5273, 5281, 5282, 5291, 5297, 5307, 5311, 5319, 5331, 5332, 5339, 5344, 5345, 5346, 5353, 5356, 5366, 5368, 5381, 5382, 5383, 5384, 5389, 5418, 5433, 5435, 5438, 5442, 5443, 5447, 5452, 5455, 5458, 5465, 5470, 5476, 5481, 5485, 5486, 5489, 5490, 5501, 5505, 5507, 5511, 5522, 5525, 5530, 5531, 5533, 5537, 5539, 5541, 5552, 5553, 5556, 5560, 5571, 5575, 5578, 5581, 5588, 5589, 5594, 5599, 5600, 5605, 5614, 5621, 5624, 5634, 5635, 5636, 5637, 5638, 5642, 5648, 5655, 5658, 5660, 5665, 5669, 5670, 5685, 5686, 5687, 5689, 5691, 5694, 5698, 5708, 5718, 5721, 5725, 5727, 5728, 5729, 5734, 5740, 5743, 5746, 5749, 5772, 5776, 5787, 5790, 5795, 5805, 5807, 5817, 5818, 5827, 5832, 5834, 5835, 5838, 5842, 5848, 5854, 5859, 5862, 5866, 5872, 5876, 5877, 5889, 5898, 5902, 5907, 5908, 5913, 5929, 5934, 5936, 5940, 5949, 5951, 5954, 5963, 5964, 5967, 5983, 5994, 5996, 6000, 6011, 6015, 6018, 6022, 6025, 6034, 6035, 6040, 6043, 6044, 6052, 6055, 6059, 6060, 6061, 6071, 6083, 6090, 6091, 6092, 6094, 6097, 6102, 6103, 6106, 6111, 6117, 6124, 6129, 6132, 6135, 6141, 6146, 6147, 6149, 6154, 6155, 6158, 6161, 6170, 6171, 6173, 6180, 6183, 6187, 6195, 6196, 6200, 6205, 6208, 6209, 6229, 6231, 6235, 6240, 6242, 6247, 6254, 6259, 6268, 6270, 6276, 6277, 6284, 6287, 6290, 6291, 6292, 6308, 6325, 6331, 6333, 6335, 6346, 6350, 6361, 6363, 6366, 6367, 6377, 6381, 6388, 6394, 6403, 6411, 6414, 6415, 6419, 6424, 6429, 6437, 6445, 6446, 6450, 6451, 6452, 6455, 6461, 6467, 6487, 6492, 6497, 6506, 6518, 6522, 6527, 6529, 6535, 6537, 6541, 6544, 6546, 6552, 6566, 6568, 6573, 6580, 6602, 6607, 6609, 6612, 6627, 6631, 6637, 6639, 6640, 6652, 6659, 6660, 6668, 6673, 6678, 6680, 6683, 6700, 6703, 6715, 6716, 6719, 6724, 6726, 6732, 6736, 6742, 6745, 6750, 6751, 6773, 6774, 6776, 6777, 6778, 6790, 6796, 6802, 6806, 6807, 6809, 6818, 6820, 6823, 6825, 6826, 6832, 6839, 6840, 6841, 6867, 6868, 6869, 6870, 6874, 6886, 6896, 6898, 6899, 6904, 6909, 6911, 6912, 6922, 6923, 6925, 6926, 6932, 6933, 6935, 6941, 6945, 6948, 6949, 6954, 6958, 6963, 6965, 6968, 6977, 6980, 6984, 6988, 6993, 6994, 6997, 7002, 7007, 7012, 7013, 7016, 7017, 7019, 7026, 7028, 7033, 7034, 7036, 7038, 7039, 7042, 7048, 7055, 7057, 7058, 7061, 7068, 7083, 7085, 7092, 7095, 7100, 7104, 7108, 7117, 7119, 7123, 7127, 7128, 7132, 7144, 7155, 7157, 7162, 7171, 7177, 7181, 7189, 7197, 7205, 7206, 7207, 7210, 7223, 7230, 7232, 7240, 7249, 7254, 7264, 7270, 7274, 7275, 7282, 7291, 7293, 7299, 7313, 7316, 7320, 7339, 7348, 7349, 7358, 7365, 7366, 7370, 7371, 7373, 7379, 7383, 7391, 7395, 7397, 7401, 7402, 7403]
        im_idx = [1, 2, 4, 6, 10, 11, 12, 14, 20, 29, 35, 37, 42, 45, 46, 47, 49, 50, 52, 53, 54, 55, 60, 61, 67, 68, 75, 81, 85, 92, 94, 100, 102, 106, 111, 112, 114, 116, 121, 124, 126, 129, 133, 134, 136, 139, 146, 149, 152, 153, 155, 158, 160, 162, 163, 164, 165, 168, 169, 170, 171, 176, 177, 178, 180, 183, 184, 185, 186, 187, 189, 190, 196, 197, 198, 199, 200, 201, 202, 204, 205, 207, 209, 210, 213, 215, 217, 219, 220, 221, 227, 228, 230, 231, 235, 236, 237, 240, 243, 244, 245, 247, 252, 256, 258, 261, 264, 265, 266, 272, 273, 274, 277, 280, 282, 286, 290, 291, 292, 295, 297, 298, 300, 305, 312, 314, 316, 317, 319, 322, 323, 324, 326, 327, 330, 331, 333, 342, 346, 347, 349, 350, 364, 366, 368, 375, 383, 392, 393, 394, 396, 398, 399, 405, 414, 418, 423, 428, 429, 435, 439, 443, 446, 448, 449, 450, 452, 453, 455, 457, 459, 462, 467, 469, 474, 475, 476, 479, 482, 484, 487, 488, 491, 494, 497, 501, 503, 505, 507, 508, 509, 510, 514, 527, 537, 538, 544, 546, 548, 551, 553, 559, 566, 567, 571, 576, 577, 581, 582, 583, 584, 585, 588, 589, 591, 596, 597, 598, 601, 603, 604, 606, 607, 614, 618, 619, 629, 631, 632, 633, 637, 638, 639, 640, 644, 647, 648, 650, 658, 664, 665, 668, 670, 672, 674, 675, 677, 678, 680, 684, 685, 686, 687, 688, 690, 697, 702, 706, 714, 720, 721, 724, 732, 734, 738, 742, 746, 747, 749, 755, 762, 764, 765, 768, 770, 772, 774, 776, 777, 778, 779, 784, 788, 789, 791, 794, 795, 797, 804, 805, 811, 812, 814, 815, 816, 819, 824, 826, 833, 834, 836, 838, 840, 844, 845, 852, 854, 856, 858, 859, 860, 865, 866, 869, 870, 879, 880, 883, 884, 886, 888, 889, 897, 903, 904, 905, 907, 908, 910, 912, 916, 919, 920, 922, 924, 926, 927, 932, 933, 936, 937, 940, 946, 951, 953, 958, 961, 966, 969, 971, 974, 975, 977, 978, 982, 983, 984, 987, 996, 1000, 1002, 1003, 1004, 1007, 1008, 1015, 1025, 1027, 1028, 1034, 1038, 1040, 1041, 1044, 1047, 1052, 1058, 1060, 1062, 1065, 1070, 1072, 1073, 1074, 1079, 1081, 1090, 1097, 1098, 1100, 1106, 1109, 1111, 1120, 1121, 1123, 1125, 1126, 1128, 1129, 1136, 1137, 1138, 1139, 1142, 1144, 1145, 1149, 1150, 1152, 1153, 1154, 1157, 1171, 1172, 1174, 1177, 1181, 1182, 1184, 1185, 1186, 1188, 1189, 1191, 1193, 1202, 1207, 1214, 1216, 1218, 1220, 1221, 1224, 1229, 1238, 1240, 1241, 1248, 1253, 1255, 1256, 1260, 1267, 1270, 1279, 1280, 1281, 1282, 1283, 1285, 1287, 1289, 1293, 1298, 1299, 1302, 1303, 1309, 1312, 1313, 1314, 1318, 1321, 1326, 1327, 1328, 1332, 1335, 1336, 1339, 1340, 1342, 1344, 1347, 1350, 1354, 1355, 1357, 1361, 1366, 1367, 1369, 1370, 1372, 1377, 1378, 1379, 1384, 1386, 1387, 1389, 1392, 1395, 1396, 1397, 1398, 1401, 1402, 1403, 1404, 1405, 1406, 1407, 1408, 1415, 1418, 1419, 1421, 1424, 1425, 1426, 1431, 1435, 1437, 1442, 1446, 1448, 1450, 1455, 1456, 1457, 1458, 1460, 1461, 1464, 1467, 1468, 1469, 1470, 1471, 1474, 1475, 1476, 1477, 1478, 1481, 1482, 1487, 1489, 1490, 1493, 1500, 1501, 1507, 1512, 1514, 1518, 1519, 1522, 1523, 1525, 1526, 1527, 1534, 1537, 1541, 1544, 1558, 1559, 1566, 1567, 1569, 1573, 1574, 1575, 1585, 1586, 1592, 1593, 1596, 1598, 1599, 1601, 1604, 1608, 1610, 1611, 1613, 1614, 1618, 1621, 1623, 1626, 1628, 1632, 1633, 1634, 1635, 1637, 1639, 1643, 1644, 1648, 1649, 1652, 1654, 1664, 1665, 1670, 1678, 1682, 1684, 1687, 1689, 1693, 1698, 1701, 1707, 1709, 1717, 1720, 1722, 1728, 1730, 1734, 1739, 1742, 1744, 1746, 1747, 1759, 1761, 1763, 1768, 1769, 1773, 1774, 1777, 1779, 1781, 1782, 1783, 1788, 1790, 1794, 1797, 1801, 1802, 1805, 1806, 1810, 1812, 1814, 1815, 1816, 1817, 1821, 1822, 1823, 1826, 1828, 1829, 1837, 1838, 1847, 1849, 1850, 1851, 1853, 1858, 1869, 1870, 1874, 1879, 1881, 1882, 1883, 1884, 1885, 1887, 1888, 1889, 1895, 1896, 1897, 1898, 1899, 1903, 1906, 1907, 1909, 1913, 1915, 1917, 1919, 1921, 1927, 1929, 1937, 1938, 1939, 1940, 1942, 1943, 1944, 1947, 1948, 1949, 1957, 1958, 1959, 1962, 1963, 1966, 1967, 1968, 1970, 1971, 1974, 1976, 1979, 1980, 1985, 1986, 1994, 1995, 1998, 2001, 2002, 2010, 2017, 2018, 2019, 2021, 2026, 2027, 2031, 2032, 2033, 2036, 2037, 2041, 2044, 2047, 2049, 2053, 2062, 2063, 2066, 2067, 2075, 2076, 2077, 2079, 2087, 2089, 2091, 2094, 2103, 2108, 2110, 2111, 2114, 2124, 2128, 2134, 2136, 2147, 2151, 2157, 2159, 2162, 2164, 2167, 2175, 2176, 2178, 2181, 2184, 2188, 2189, 2194, 2195, 2196, 2202, 2209, 2211, 2213, 2214, 2216, 2223, 2230, 2232, 2233, 2235, 2237, 2238, 2243, 2244, 2247, 2254, 2258, 2259, 2260, 2265, 2266, 2270, 2274, 2280, 2281, 2285, 2286, 2294, 2295, 2302, 2304, 2312, 2318, 2322, 2326, 2327, 2329, 2330, 2331, 2332, 2333, 2334, 2337, 2339, 2345, 2346, 2348, 2349, 2355, 2358, 2360, 2363, 2367, 2370, 2373, 2375, 2384, 2392, 2395, 2403, 2404, 2408, 2409, 2414, 2415, 2422, 2423, 2425, 2426, 2427, 2430, 2432, 2434, 2435, 2436, 2440, 2442, 2445, 2446, 2456, 2461, 2462, 2468, 2470, 2471, 2476, 2479, 2480, 2482, 2492, 2494, 2495, 2497, 2500, 2501, 2504, 2511, 2515, 2516, 2518, 2524, 2527, 2531, 2532, 2534, 2535, 2536, 2537, 2541, 2542, 2543, 2545, 2547, 2553, 2555, 2557, 2558, 2561, 2565, 2569, 2570, 2577, 2585, 2588, 2589, 2591, 2593, 2596, 2597, 2600, 2602, 2603, 2606, 2610, 2616, 2617, 2619, 2622, 2623, 2625, 2630, 2634, 2636, 2637, 2638, 2640, 2641, 2642, 2644, 2648, 2649, 2655, 2656, 2659, 2660, 2664, 2665, 2669, 2671, 2673, 2677, 2679, 2680, 2687, 2689, 2690, 2691, 2695, 2697, 2701, 2702, 2703, 2704, 2707, 2718, 2722, 2726, 2729, 2730, 2732, 2737, 2739, 2741, 2743, 2745, 2751, 2757, 2758, 2763, 2764, 2772, 2778, 2779, 2780, 2785, 2789, 2793, 2796, 2798, 2802, 2804, 2813, 2817, 2821, 2824, 2829, 2831, 2835, 2841, 2843, 2846, 2847, 2849, 2852, 2855, 2860, 2861, 2862, 2866, 2871, 2875, 2876, 2877, 2880, 2881, 2882, 2884, 2889, 2890, 2891, 2892, 2894, 2895, 2897, 2898, 2900, 2904, 2907, 2909, 2913, 2919, 2920, 2931, 2932, 2936, 2937, 2940, 2941, 2943, 2950, 2952, 2960, 2964, 2971, 2972, 2974, 2976, 2982, 2985, 2986, 2994, 2998, 3000, 3004, 3005, 3006, 3021, 3022, 3026, 3030, 3033, 3035, 3039, 3045, 3054, 3062, 3064, 3072, 3073, 3078, 3082, 3083, 3089, 3092, 3093, 3094, 3096, 3098, 3101, 3104, 3105, 3106, 3108, 3110, 3113, 3116, 3118, 3119, 3124, 3126, 3127, 3130, 3131, 3134, 3135, 3138, 3144, 3146, 3147, 3153, 3154, 3155, 3160, 3165, 3167, 3172, 3176, 3178, 3179, 3181, 3186, 3188, 3192, 3193, 3198, 3199, 3201, 3202, 3205, 3207, 3213, 3214, 3216, 3218, 3220, 3221, 3223, 3224, 3228, 3241, 3242, 3246, 3247, 3252, 3253, 3256, 3259, 3264, 3267, 3269, 3272, 3277, 3284, 3285, 3286, 3290, 3291, 3292, 3300, 3303, 3304, 3307, 3309, 3313, 3315, 3316, 3319, 3320, 3323, 3328, 3329, 3332, 3339, 3342, 3344, 3346, 3350, 3358, 3360, 3362, 3366, 3369, 3370, 3376, 3377, 3380, 3384, 3385, 3386, 3394, 3395, 3402, 3403, 3405, 3406, 3407, 3422, 3425, 3428, 3434, 3437, 3438, 3439, 3446, 3447, 3454, 3457, 3458, 3459, 3462, 3465, 3468, 3469, 3471, 3475, 3476, 3479, 3482, 3486, 3488, 3490, 3493, 3496, 3504, 3506, 3510, 3513, 3514, 3519, 3523, 3524, 3525, 3526, 3527, 3532, 3533, 3534, 3536, 3539, 3540, 3542, 3544, 3547, 3548, 3550, 3554, 3555, 3556, 3557, 3558, 3560, 3562, 3564, 3572, 3580, 3581, 3586, 3587, 3588, 3591, 3593, 3595, 3598, 3599, 3601, 3602, 3604, 3605, 3606, 3607, 3608, 3609, 3610, 3612, 3616, 3617, 3619, 3621, 3622, 3624, 3625, 3626, 3630, 3636, 3638, 3642, 3645, 3650, 3653, 3658, 3659, 3660, 3664, 3666, 3668, 3671, 3679, 3680, 3681, 3683, 3687, 3689, 3693, 3694, 3695, 3696, 3697, 3700, 3702, 3707, 3710, 3716, 3724, 3728, 3729, 3730, 3739, 3740, 3743, 3747, 3748, 3751, 3753, 3754, 3758, 3759, 3760, 3767, 3768, 3769, 3770, 3773, 3777, 3778, 3780, 3784, 3785, 3797, 3801, 3803, 3804, 3806, 3808, 3809, 3813, 3816, 3820, 3826, 3829, 3830, 3831, 3832, 3833, 3837, 3843, 3845, 3848, 3849, 3850, 3851, 3852, 3855, 3859, 3861, 3863, 3869, 3870, 3873, 3877, 3882, 3885, 3895, 3897, 3898, 3900, 3901, 3902, 3904, 3907, 3910, 3913, 3919, 3920, 3925, 3927, 3930, 3931, 3936, 3942, 3943, 3948, 3949, 3954, 3955, 3957, 3966, 3967, 3968, 3977, 3979, 3982, 3984, 3986, 3991, 3996, 4000, 4003, 4007, 4009, 4010, 4011, 4013, 4020, 4024, 4025, 4027, 4031, 4048, 4054, 4056, 4057, 4066, 4071, 4073, 4075, 4077, 4086, 4087, 4088, 4089, 4091, 4094, 4096, 4098, 4102, 4103, 4107, 4112, 4113, 4115, 4116, 4128, 4131, 4134, 4136, 4140, 4143, 4144, 4149, 4152, 4153, 4154, 4155, 4156, 4158, 4161, 4162, 4165, 4166, 4167, 4171, 4173, 4174, 4175, 4179, 4184, 4191, 4195, 4196, 4197, 4202, 4205, 4207, 4211, 4221, 4222, 4223, 4228, 4229, 4236, 4237, 4238, 4239, 4245, 4247, 4251, 4253, 4255, 4260, 4261, 4262, 4264, 4265, 4267, 4269, 4273, 4274, 4278, 4280, 4281, 4285, 4299, 4300, 4305, 4308, 4311, 4315, 4316, 4323, 4325, 4326, 4328, 4330, 4333, 4335, 4337, 4338, 4341, 4346, 4348, 4349, 4350, 4351, 4353, 4355, 4357, 4358, 4359, 4360, 4368, 4374, 4375, 4381, 4382, 4383, 4385, 4392, 4394, 4395, 4396, 4400, 4401, 4411, 4415, 4416, 4420, 4422, 4426, 4431, 4432, 4434, 4435, 4439, 4440, 4441, 4443, 4444, 4449, 4453, 4454, 4456, 4465, 4468, 4471, 4474, 4475, 4478, 4482, 4486, 4489, 4492, 4495, 4496, 4499, 4504, 4507, 4512, 4515, 4519, 4525, 4526, 4531, 4532, 4533, 4534, 4535, 4537, 4539, 4540, 4545, 4546, 4548, 4550, 4553, 4554, 4558, 4559, 4577, 4579, 4581, 4583, 4589, 4592, 4614, 4619, 4620, 4624, 4626, 4629, 4630, 4634, 4635, 4639, 4641, 4643, 4645, 4650, 4652, 4653, 4656, 4657, 4663, 4664, 4666, 4669, 4671, 4673, 4674, 4675, 4676, 4677, 4678, 4680, 4681, 4687, 4688, 4689, 4692, 4700, 4703, 4709, 4711, 4716, 4720, 4722, 4723, 4725, 4731, 4732, 4735, 4737, 4742, 4745, 4746, 4749, 4751, 4753, 4754, 4758, 4763, 4764, 4765, 4766, 4778, 4782, 4784, 4785, 4786, 4791, 4801, 4809, 4811, 4813, 4820, 4822, 4825, 4828, 4829, 4831, 4832, 4835, 4837, 4842, 4843, 4845, 4846, 4847, 4848, 4849, 4850, 4853, 4856, 4861, 4862, 4863, 4869, 4870, 4871, 4873, 4875, 4879, 4880, 4881, 4893, 4896, 4897, 4898, 4899, 4902, 4903, 4909, 4913, 4914, 4916, 4917, 4921, 4922, 4924, 4926, 4927, 4929, 4930, 4932, 4935, 4936, 4937, 4944, 4945, 4950, 4951, 4952, 4958, 4960, 4965, 4968, 4973, 4988, 4989, 4991, 4992, 4995, 4996, 4997, 4998, 5000, 5001, 5003, 5008, 5014, 5015, 5024, 5026, 5030, 5032, 5036, 5043, 5044, 5048, 5049, 5050, 5051, 5058, 5059, 5062, 5068, 5070, 5071, 5074, 5076, 5078, 5079, 5081, 5083, 5084, 5085, 5088, 5090, 5093, 5101, 5105, 5106, 5108, 5110, 5111, 5113, 5124, 5126, 5128, 5131, 5134, 5135, 5138, 5144, 5145, 5148, 5159, 5161, 5165, 5168, 5172, 5173, 5175, 5176, 5179, 5181, 5196, 5199, 5201, 5206, 5212, 5213, 5214, 5216, 5219, 5220, 5223, 5226, 5229, 5231, 5232, 5240, 5244, 5246, 5248, 5251, 5257, 5259, 5260, 5262, 5267, 5271, 5272, 5275, 5278, 5279, 5280, 5283, 5284, 5285, 5287, 5294, 5296, 5298, 5302, 5304, 5305, 5310, 5321, 5322, 5323, 5324, 5327, 5329, 5330, 5335, 5342, 5343, 5347, 5348, 5352, 5353, 5354, 5357, 5362, 5366, 5367, 5369, 5373, 5374, 5378, 5379, 5380, 5385, 5392, 5393, 5394, 5396, 5398, 5400, 5401, 5402, 5403, 5406, 5410, 5411, 5415, 5416, 5421, 5423, 5426, 5428, 5431, 5432, 5437, 5440, 5441, 5444, 5446, 5449, 5450, 5451, 5452, 5453, 5454, 5457, 5459, 5461, 5463, 5467, 5471, 5472, 5473, 5475, 5477, 5478, 5479, 5482, 5487, 5488, 5492, 5498, 5499, 5502, 5504, 5507, 5508, 5515, 5516, 5517, 5523, 5524, 5525, 5528, 5532, 5534, 5536, 5542, 5543, 5544, 5545, 5546, 5547, 5548, 5551, 5554, 5561, 5563, 5567, 5570, 5572, 5573, 5574, 5576, 5577, 5579, 5582, 5583, 5584, 5592, 5593, 5595, 5597, 5600, 5601, 5606, 5607, 5609, 5613, 5616, 5618, 5619, 5620, 5627, 5628, 5629, 5630, 5632, 5639, 5645, 5646, 5661, 5662, 5663, 5664, 5666, 5671, 5672, 5673, 5675, 5677, 5680, 5681, 5688, 5689, 5701, 5702, 5704, 5706, 5711, 5713, 5719, 5720, 5721, 5731, 5733, 5735, 5737, 5741, 5747, 5750, 5751, 5752, 5753, 5754, 5765, 5768, 5769, 5773, 5775, 5777, 5780, 5784, 5786, 5791, 5792, 5793, 5794, 5796, 5798, 5800, 5806, 5811, 5815, 5822, 5823, 5824, 5830, 5839, 5840, 5845, 5846, 5851, 5852, 5853, 5855, 5864, 5865, 5867, 5869, 5871, 5878, 5879, 5880, 5883, 5884, 5885, 5886, 5890, 5891, 5893, 5894, 5895, 5896, 5901, 5903, 5905, 5911, 5912, 5915, 5916, 5917, 5923, 5927, 5930, 5932, 5935, 5937, 5938, 5941, 5942, 5944, 5947, 5948, 5952, 5956, 5960, 5962, 5966, 5971, 5975, 5984, 5985, 5987, 5988, 5990, 5995, 5997, 5998, 6002, 6003, 6005, 6007, 6008, 6010, 6014, 6016, 6017, 6020, 6024, 6025, 6030, 6036, 6039, 6042, 6044, 6045, 6046, 6048, 6050, 6051, 6054, 6058, 6062, 6064, 6065, 6068, 6075, 6077, 6078, 6079, 6080, 6081, 6082, 6089, 6095, 6098, 6099, 6101, 6104, 6112, 6113, 6114, 6115, 6116, 6118, 6119, 6123, 6130, 6134, 6136, 6137, 6138, 6139, 6140, 6148, 6150, 6153, 6157, 6164, 6165, 6166, 6168, 6169, 6172, 6174, 6175, 6176, 6182, 6184, 6185, 6192, 6193, 6194, 6197, 6201, 6202, 6207, 6210, 6212, 6213, 6214, 6219, 6220, 6226, 6227, 6232, 6238, 6243, 6244, 6246, 6250, 6253, 6257, 6260, 6262, 6264, 6266, 6271, 6274, 6275, 6279, 6283, 6285, 6289, 6293, 6294, 6296, 6303, 6309, 6310, 6314, 6319, 6322, 6328, 6332, 6336, 6344, 6345, 6352, 6358, 6359, 6360, 6362, 6365, 6368, 6369, 6371, 6372, 6375, 6379, 6382, 6390, 6391, 6396, 6398, 6401, 6402, 6404, 6407, 6408, 6412, 6416, 6420, 6422, 6423, 6425, 6426, 6427, 6428, 6430, 6432, 6435, 6438, 6443, 6453, 6456, 6460, 6462, 6469, 6472, 6473, 6474, 6475, 6478, 6483, 6485, 6491, 6494, 6499, 6500, 6501, 6502, 6503, 6505, 6513, 6514, 6516, 6517, 6519, 6520, 6523, 6524, 6525, 6528, 6532, 6533, 6536, 6543, 6547, 6549, 6556, 6558, 6561, 6563, 6564, 6565, 6567, 6569, 6574, 6578, 6581, 6582, 6584, 6586, 6588, 6589, 6591, 6594, 6595, 6596, 6597, 6599, 6604, 6605, 6606, 6608, 6616, 6617, 6618, 6620, 6624, 6625, 6629, 6630, 6634, 6638, 6647, 6648, 6650, 6651, 6653, 6654, 6655, 6664, 6665, 6671, 6672, 6675, 6679, 6681, 6682, 6686, 6689, 6690, 6696, 6699, 6701, 6702, 6707, 6708, 6709, 6710, 6711, 6712, 6713, 6717, 6719, 6721, 6722, 6723, 6725, 6729, 6733, 6734, 6735, 6737, 6744, 6747, 6758, 6760, 6761, 6762, 6763, 6764, 6765, 6769, 6780, 6781, 6783, 6785, 6786, 6789, 6791, 6792, 6794, 6795, 6797, 6798, 6804, 6811, 6814, 6817, 6819, 6821, 6822, 6833, 6836, 6846, 6849, 6852, 6854, 6856, 6858, 6859, 6864, 6866, 6871, 6873, 6878, 6879, 6880, 6881, 6884, 6888, 6889, 6890, 6892, 6894, 6900, 6901, 6903, 6910, 6914, 6915, 6917, 6920, 6924, 6927, 6928, 6930, 6939, 6942, 6943, 6946, 6952, 6959, 6960, 6967, 6971, 6972, 6973, 6974, 6978, 6982, 6985, 6986, 6992, 6993, 6995, 6998, 7000, 7001, 7003, 7004, 7020, 7024, 7027, 7029, 7031, 7035, 7037, 7041, 7044, 7046, 7053, 7059, 7060, 7063, 7064, 7066, 7069, 7071, 7072, 7073, 7074, 7075, 7077, 7080, 7082, 7084, 7089, 7090, 7091, 7094, 7097, 7105, 7109, 7110, 7111, 7112, 7114, 7115, 7118, 7126, 7130, 7131, 7137, 7140, 7141, 7150, 7151, 7152, 7153, 7158, 7159, 7164, 7165, 7168, 7169, 7170, 7172, 7173, 7176, 7178, 7179, 7180, 7184, 7185, 7186, 7190, 7191, 7194, 7198, 7199, 7200, 7201, 7202, 7203, 7208, 7209, 7212, 7216, 7220, 7221, 7227, 7231, 7233, 7234, 7238, 7241, 7242, 7245, 7247, 7248, 7251, 7252, 7253, 7255, 7257, 7260, 7261, 7271, 7280, 7283, 7290, 7303, 7306, 7307, 7309, 7311, 7314, 7315, 7317, 7324, 7326, 7328, 7329, 7331, 7332, 7341, 7342, 7343, 7344, 7349, 7355, 7356, 7357, 7359, 7360, 7362, 7363, 7368, 7375, 7377, 7378, 7381, 7384, 7385, 7386, 7388, 7389, 7393, 7398]
        hard_imp_idx = [1,2,4,6,10,11,12,14,20,29,35,37,42,45,46,47,49,50,52,53,54,55,60,61,67,75,85,92,94,100,102,106,111,112,116,124,126,129,133,134,136,139,146,149,152,153,155,158,160,162,163,164,165,168,169,170,171,176,178,180,183,184,185,186,187,189,196,198,199,202,205,207,209,210,213,215,217,219,220,221,227,228,230,231,235,237,240,243,244,252,256,258,264,265,266,272,273,274,277,282,286,290,291,292,295,297,298,300,305,312,314,316,317,324,326,327,330,331,333,342,346,347,349,350,364,366,368,383,392,393,394,396,399,405,414,418,423,439,443,446,448,449,450,452,453,455,457,459,462,469,474,475,476,479,482,484,487,488,491,494,501,503,505,507,508,509,510,514,527,537,538,544,546,548,551,553,559,566,567,576,577,581,583,584,585,588,589,596,598,601,603,604,606,607,614,618,619,629,631,632,633,637,638,639,640,644,647,658,664,665,668,670,672,674,675,677,678,684,685,686,687,688,690,697,702,706,714,720,721,724,738,742,746,747,749,755,762,764,765,768,770,774,776,777,778,779,784,788,789,791,794,795,797,805,811,812,814,815,816,819,824,826,833,836,838,840,844,852,854,856,859,860,865,866,879,880,886,888,889,897,903,904,905,907,910,919,920,922,926,927,932,936,937,940,946,951,953,961,969,971,974,975,977,982,984,996,1003,1004,1007,1015,1025,1027,1028,1034,1038,1041,1044,1047,1052,1058,1060,1062,1065,1072,1073,1074,1081,1098,1106,1111,1120,1121,1123,1126,1128,1129,1138,1145,1149,1150,1152,1153,1154,1157,1172,1174,1177,1182,1184,1185,1189,1191,1193,1202,1207,1218,1220,1221,1224,1229,1238,1248,1253,1255,1256,1260,1267,1279,1280,1281,1282,1283,1285,1287,1289,1293,1298,1299,1302,1303,1309,1312,1313,1314,1318,1321,1326,1327,1332,1335,1336,1339,1340,1342,1344,1347,1350,1355,1357,1361,1366,1367,1369,1370,1372,1377,1378,1379,1384,1387,1392,1395,1396,1397,1398,1403,1404,1407,1408,1415,1418,1421,1426,1431,1442,1446,1448,1450,1457,1460,1461,1464,1467,1468,1469,1470,1471,1474,1475,1476,1477,1478,1481,1482,1487,1489,1493,1500,1501,1507,1512,1518,1519,1522,1523,1526,1527,1537,1541,1544,1559,1566,1567,1569,1573,1574,1575,1585,1586,1592,1593,1598,1599,1601,1604,1608,1610,1611,1613,1614,1618,1621,1623,1626,1628,1632,1635,1639,1643,1644,1648,1649,1652,1654,1664,1665,1670,1682,1684,1689,1693,1698,1701,1717,1720,1722,1728,1730,1734,1739,1742,1747,1763,1768,1769,1774,1779,1781,1788,1790,1797,1801,1805,1810,1812,1814,1815,1817,1821,1822,1823,1826,1828,1829,1837,1838,1847,1849,1850,1851,1853,1858,1869,1870,1874,1879,1881,1882,1883,1884,1885,1887,1888,1889,1896,1897,1898,1899,1903,1906,1907,1909,1915,1917,1919,1921,1929,1937,1938,1939,1940,1942,1943,1944,1947,1948,1949,1957,1958,1959,1962,1963,1966,1967,1968,1970,1971,1974,1976,1979,1980,1985,1986,1994,1998,2002,2010,2017,2018,2019,2031,2032,2033,2036,2044,2047,2049,2063,2066,2067,2075,2076,2077,2079,2087,2091,2094,2103,2110,2111,2114,2124,2128,2136,2147,2157,2159,2162,2164,2167,2175,2176,2178,2181,2188,2196,2202,2209,2213,2214,2216,2223,2230,2232,2237,2238,2244,2254,2265,2266,2270,2274,2280,2281,2285,2286,2294,2295,2304,2312,2322,2326,2327,2329,2330,2331,2333,2337,2345,2348,2349,2355,2358,2360,2363,2367,2370,2373,2375,2392,2395,2404,2408,2409,2414,2415,2422,2423,2427,2432,2434,2435,2440,2442,2445,2446,2456,2461,2476,2479,2480,2482,2492,2494,2495,2497,2500,2515,2516,2524,2527,2531,2532,2535,2536,2537,2542,2543,2545,2547,2553,2555,2557,2558,2561,2565,2569,2570,2577,2585,2588,2589,2591,2596,2600,2602,2603,2606,2610,2616,2617,2619,2622,2623,2625,2630,2634,2636,2638,2640,2641,2642,2644,2648,2649,2655,2656,2660,2664,2673,2677,2679,2680,2687,2690,2695,2697,2702,2703,2704,2707,2718,2722,2726,2729,2730,2732,2737,2739,2741,2743,2745,2751,2757,2758,2763,2764,2772,2778,2779,2785,2789,2793,2796,2798,2802,2804,2817,2821,2824,2831,2835,2841,2843,2846,2847,2849,2852,2855,2860,2861,2862,2866,2871,2875,2876,2880,2881,2882,2884,2889,2890,2891,2892,2894,2895,2897,2900,2904,2907,2909,2913,2919,2920,2932,2937,2941,2943,2952,2960,2964,2971,2972,2976,2985,2986,2994,2998,3000,3004,3006,3021,3022,3030,3033,3035,3039,3045,3062,3073,3078,3082,3083,3089,3092,3093,3096,3098,3101,3104,3105,3106,3108,3110,3113,3116,3118,3119,3124,3126,3127,3130,3131,3135,3138,3144,3146,3147,3153,3154,3155,3160,3165,3167,3172,3178,3179,3181,3186,3188,3192,3193,3199,3201,3202,3205,3207,3213,3214,3218,3220,3221,3223,3224,3228,3241,3242,3246,3253,3256,3259,3264,3267,3269,3272,3284,3285,3286,3290,3291,3292,3300,3303,3304,3309,3313,3315,3316,3319,3328,3329,3332,3339,3342,3346,3350,3360,3362,3366,3370,3377,3380,3384,3385,3386,3394,3395,3403,3405,3406,3407,3422,3425,3428,3434,3437,3438,3439,3446,3457,3458,3459,3462,3465,3468,3469,3471,3475,3479,3482,3486,3488,3490,3493,3504,3506,3510,3513,3514,3519,3523,3524,3525,3526,3527,3533,3534,3536,3539,3540,3542,3544,3547,3548,3554,3555,3556,3557,3558,3560,3562,3572,3580,3581,3586,3591,3593,3595,3599,3601,3602,3604,3605,3606,3607,3608,3610,3617,3619,3621,3622,3624,3625,3630,3636,3638,3642,3650,3653,3658,3659,3660,3664,3671,3680,3681,3683,3687,3693,3695,3697,3702,3707,3710,3716,3724,3729,3730,3740,3743,3747,3748,3751,3753,3758,3759,3767,3769,3770,3777,3778,3780,3785,3801,3803,3804,3806,3808,3809,3813,3816,3820,3829,3830,3831,3832,3833,3837,3843,3848,3849,3850,3851,3852,3855,3859,3861,3863,3869,3870,3877,3882,3885,3895,3897,3898,3900,3901,3902,3904,3907,3913,3919,3925,3927,3931,3936,3942,3948,3954,3955,3957,3966,3967,3977,3979,3982,3984,3986,3991,3996,4000,4003,4007,4010,4011,4020,4025,4027,4031,4048,4054,4056,4057,4066,4071,4073,4075,4086,4087,4088,4089,4091,4096,4098,4102,4103,4107,4112,4113,4116,4128,4131,4134,4136,4144,4149,4152,4153,4155,4156,4158,4161,4165,4166,4167,4173,4174,4175,4179,4184,4191,4195,4196,4202,4207,4211,4221,4222,4223,4228,4229,4236,4237,4245,4247,4251,4253,4255,4260,4262,4264,4265,4267,4269,4273,4274,4278,4280,4281,4285,4299,4300,4305,4311,4323,4325,4326,4328,4330,4335,4346,4348,4349,4351,4353,4355,4357,4358,4359,4360,4368,4375,4381,4382,4383,4385,4394,4395,4396,4400,4411,4415,4416,4420,4422,4426,4431,4432,4439,4440,4441,4443,4453,4454,4456,4465,4471,4474,4475,4478,4482,4492,4495,4496,4499,4504,4507,4512,4515,4525,4526,4531,4532,4533,4534,4535,4537,4539,4540,4546,4548,4550,4553,4554,4558,4559,4577,4579,4581,4589,4592,4614,4619,4620,4626,4629,4630,4635,4639,4641,4643,4645,4650,4652,4663,4664,4666,4669,4671,4673,4674,4675,4676,4677,4678,4680,4681,4688,4689,4692,4700,4703,4709,4711,4720,4722,4723,4725,4731,4732,4735,4737,4742,4745,4751,4754,4758,4763,4764,4765,4766,4778,4782,4784,4786,4809,4811,4820,4822,4825,4828,4829,4831,4835,4837,4842,4843,4845,4847,4848,4849,4850,4853,4856,4861,4863,4869,4870,4871,4873,4875,4879,4881,4893,4896,4898,4899,4902,4903,4909,4913,4914,4916,4917,4921,4922,4924,4926,4929,4930,4932,4936,4945,4950,4951,4952,4958,4960,4988,4989,4991,4995,4996,4997,4998,5000,5001,5008,5014,5015,5024,5026,5030,5032,5036,5043,5044,5048,5049,5050,5058,5068,5071,5074,5076,5078,5079,5081,5090,5093,5101,5105,5106,5108,5110,5111,5113,5124,5126,5128,5131,5134,5135,5138,5144,5145,5148,5161,5165,5168,5172,5173,5176,5179,5196,5199,5206,5213,5216,5219,5220,5223,5226,5231,5232,5240,5246,5248,5251,5257,5259,5260,5262,5267,5275,5278,5279,5280,5284,5285,5287,5294,5298,5302,5304,5305,5310,5321,5322,5323,5324,5327,5330,5335,5343,5348,5352,5354,5357,5362,5367,5373,5374,5378,5379,5385,5392,5393,5394,5396,5398,5400,5401,5402,5410,5411,5415,5416,5426,5428,5431,5432,5437,5440,5444,5446,5449,5450,5451,5453,5454,5457,5459,5461,5463,5467,5471,5472,5473,5475,5477,5478,5479,5482,5487,5492,5498,5499,5502,5508,5515,5516,5517,5523,5524,5528,5532,5534,5536,5542,5544,5547,5548,5551,5554,5563,5567,5570,5572,5573,5574,5576,5577,5579,5582,5583,5584,5592,5593,5595,5597,5601,5606,5609,5613,5616,5618,5620,5627,5629,5630,5632,5639,5645,5646,5661,5662,5663,5664,5666,5672,5673,5675,5680,5681,5688,5701,5702,5706,5711,5719,5731,5733,5735,5741,5750,5752,5753,5754,5768,5769,5775,5777,5780,5786,5791,5792,5796,5798,5811,5815,5822,5823,5824,5839,5840,5845,5846,5851,5853,5855,5864,5865,5867,5869,5871,5879,5880,5883,5884,5885,5886,5895,5896,5901,5905,5912,5915,5916,5917,5923,5930,5932,5935,5937,5938,5941,5942,5944,5947,5948,5956,5960,5962,5966,5971,5975,5984,5985,5987,5990,5995,5998,6002,6003,6007,6008,6010,6014,6016,6020,6024,6030,6036,6039,6042,6045,6046,6050,6051,6054,6058,6062,6064,6065,6068,6075,6077,6079,6089,6095,6098,6099,6101,6104,6113,6114,6115,6116,6123,6130,6134,6137,6138,6139,6140,6150,6153,6157,6164,6165,6166,6168,6169,6172,6174,6175,6176,6182,6184,6185,6192,6193,6194,6201,6210,6212,6213,6219,6220,6226,6227,6232,6238,6243,6244,6246,6253,6260,6264,6266,6271,6274,6275,6283,6285,6289,6293,6294,6296,6303,6309,6310,6314,6319,6322,6332,6336,6344,6345,6352,6358,6359,6360,6362,6365,6368,6371,6372,6375,6379,6382,6390,6391,6396,6398,6401,6402,6404,6407,6408,6412,6416,6420,6422,6423,6425,6426,6428,6430,6435,6453,6456,6460,6462,6469,6472,6473,6474,6475,6478,6483,6485,6491,6500,6501,6503,6505,6514,6516,6517,6519,6520,6523,6525,6528,6532,6533,6536,6543,6549,6556,6558,6561,6563,6567,6569,6574,6578,6581,6582,6584,6586,6588,6589,6591,6594,6595,6596,6597,6599,6604,6605,6608,6616,6618,6620,6624,6625,6634,6638,6647,6648,6651,6654,6664,6665,6671,6672,6675,6679,6682,6686,6689,6690,6696,6699,6707,6709,6710,6712,6717,6721,6723,6725,6729,6733,6735,6744,6747,6758,6761,6762,6763,6764,6765,6769,6780,6781,6783,6785,6786,6789,6791,6794,6795,6797,6798,6804,6811,6814,6817,6819,6821,6833,6836,6846,6849,6852,6854,6856,6858,6859,6864,6866,6871,6879,6880,6881,6884,6888,6889,6890,6892,6894,6900,6901,6903,6910,6914,6917,6920,6928,6942,6943,6946,6952,6960,6967,6971,6973,6974,6978,6982,6985,6992,6995,6998,7001,7003,7004,7020,7024,7027,7031,7035,7037,7041,7044,7046,7053,7059,7060,7063,7064,7069,7071,7072,7073,7074,7075,7077,7080,7082,7084,7089,7090,7091,7094,7097,7105,7109,7110,7111,7114,7115,7118,7126,7130,7137,7140,7150,7151,7152,7153,7158,7159,7164,7165,7168,7169,7170,7172,7176,7178,7179,7184,7185,7186,7191,7194,7198,7199,7200,7201,7202,7208,7209,7212,7216,7221,7227,7231,7233,7234,7238,7241,7242,7247,7251,7252,7253,7255,7257,7260,7261,7280,7303,7306,7307,7309,7311,7314,7315,7317,7324,7326,7329,7331,7332,7341,7342,7344,7356,7359,7360,7362,7363,7375,7378,7381,7384,7386,7388,7389,7393,7398]


        K = 20

        top8_either = 0
        top1 = 0
        top8_both = 0
        top8_1 = 0
        top8_2 = 0
        top2 = 0
        total = 0
        false_example = []
        excat_match_d1, excat_match_d2 = 0, 0
        top1_excat_match_d1, top1_excat_match_d2 = 0, 0
        for i, x in enumerate(result):
            x[f"top{K}_ctxs"] = x["ctxs"][:K]
            if i not in comp_idx:  # BRIDGE
                top8_title = [t["title"] for t in x[f"top{K}_ctxs"]]
                ans = x["answers"]
                top8_match = [t for t in top8_title if t in x["answers"]]

                if ans[1] in top8_title:
                    excat_match_d2 += 1
                    if ans[1] == top8_title[0]:
                        top1_excat_match_d2 += 1

                if ans[0] in top8_title:
                    excat_match_d1 += 1
                    if ans[0] == top8_title[0]:
                        top1_excat_match_d1 += 1

                if len(top8_match) > 0:
                    top8_either += 1
                if len(top8_match) == 2:
                    top8_both += 1
                if len(top8_match) == 0:
                    false_example.append(x)
                if any([t["title"] in x["answers"] for t in x[f"top{K}_ctxs"][:1]]):
                    top1 += 1
                total += 1

        print(f"\
        TYPE: BRIDGE \n\
        total sample : {total} \n\
        top{K} either match acc: {top8_either}/{total} = {top8_either / total} \n\
        top{K} both match acc: {top8_both}/{total} = {top8_both / total} \n\
        top1 either match acc: {top1}/{total} = {top1 / total} \n\
        ")


        top8_either = 0
        top1 = 0
        top8_both = 0
        top8_1 = 0
        top8_2 = 0
        top2 = 0
        total = 0
        false_example = []
        excat_match_d1, excat_match_d2 = 0, 0
        top1_excat_match_d1, top1_excat_match_d2 = 0, 0
        for i, x in enumerate(result):
            x[f"top{K}_ctxs"] = x["ctxs"][:K]
            if i in comp_idx:  # COMPARISON
                top8_title = [t["title"] for t in x[f"top{K}_ctxs"]]
                ans = x["answers"]
                top8_match = [t for t in top8_title if t in x["answers"]]

                if ans[1] in top8_title:
                    excat_match_d2 += 1
                    if ans[1] == top8_title[0]:
                        top1_excat_match_d2 += 1

                if ans[0] in top8_title:
                    excat_match_d1 += 1
                    if ans[0] == top8_title[0]:
                        top1_excat_match_d1 += 1

                if len(top8_match) > 0:
                    top8_either += 1
                if len(top8_match) == 2:
                    top8_both += 1
                if len(top8_match) == 0:
                    false_example.append(x)
                if any([t["title"] in x["answers"] for t in x[f"top{K}_ctxs"][:1]]):
                    top1 += 1
                total += 1

        print(f"\
        TYPE: COMPARISON \n\
        total sample : {total} \n\
        top{K} either match acc: {top8_either}/{total} = {top8_either / total} \n\
        top{K} both match acc: {top8_both}/{total} = {top8_both / total} \n\
        top1 either match acc: {top1}/{total} = {top1 / total} \n\
        ")


        b_result = {}
        c_result = {}
        for K in [5, 20, 100]:
            match_either, match_both = 0, 0
            for i, x in enumerate(result):
                x[f"top{K}_ctxs"] = x["ctxs"][:K]
                if i not in comp_idx: # bridge
                    topk_title = [t["title"] for t in x[f"top{K}_ctxs"]]
                    ans = x["answers"]
                    topk_match = [t for t in topk_title if t in x["answers"]]
                    if len(topk_match) > 0:
                        b_result[f"top{K}_either"] = b_result.get(f"top{K}_either", 0) + 1
                    if len(topk_match) == 2:
                        b_result[f"top{K}_both"] = b_result.get(f"top{K}_both", 0) + 1

                elif i in comp_idx:
                    topk_title = [t["title"] for t in x[f"top{K}_ctxs"]]
                    ans = x["answers"]
                    topk_match = [t for t in topk_title if t in x["answers"]]
                    if len(topk_match) > 0:
                        c_result[f"top{K}_either"] = c_result.get(f"top{K}_either", 0) + 1
                    if len(topk_match) == 2:
                        c_result[f"top{K}_both"] = c_result.get(f"top{K}_both", 0) + 1

        print("#### Print Bridge Results ####")
        for k,v in b_result.items():
            print(k,v,v/5918)
        print("#### Print Comparson Results ####")
        for k,v in c_result.items():
            print(k,v,v/1487)

    elif "nq-test.csv" in args.qa_file:
        no_answer_overlap_ids = [3, 4, 6, 10, 14, 15, 16, 18, 19, 20, 21, 27, 32, 33, 34, 38, 39, 41, 44, 46, 47, 48, 50, 51, 53, 56, 57, 62, 67, 68, 69, 70, 71, 73, 76, 77, 87, 90, 91, 93, 94, 97, 102, 103, 104, 105, 106, 109, 110, 113, 114, 119, 122, 124, 125, 130, 131, 134, 135, 138, 142, 146, 148, 154, 155, 158, 160, 161, 162, 164, 165, 166, 168, 169, 177, 178, 180, 181, 184, 186, 187, 189, 192, 193, 194, 196, 199, 208, 210, 213, 217, 219, 223, 224, 225, 226, 230, 232, 233, 239, 240, 248, 249, 250, 251, 261, 262, 263, 266, 273, 274, 275, 279, 283, 286, 287, 290, 291, 295, 296, 297, 299, 302, 310, 311, 312, 318, 319, 322, 323, 326, 329, 339, 340, 342, 343, 347, 349, 351, 355, 356, 357, 364, 366, 372, 374, 376, 378, 380, 381, 386, 390, 393, 400, 401, 402, 403, 408, 409, 411, 417, 419, 420, 421, 423, 424, 428, 430, 432, 435, 438, 439, 441, 442, 443, 444, 447, 452, 454, 455, 456, 460, 464, 465, 476, 479, 481, 482, 484, 486, 490, 494, 501, 503, 506, 513, 514, 515, 517, 518, 519, 520, 522, 524, 525, 528, 533, 534, 537, 542, 546, 551, 552, 556, 557, 558, 559, 560, 562, 567, 569, 572, 576, 578, 580, 588, 589, 592, 594, 595, 598, 599, 600, 609, 613, 614, 615, 618, 620, 624, 625, 627, 628, 634, 636, 641, 642, 649, 652, 654, 657, 662, 663, 666, 668, 669, 670, 671, 674, 676, 677, 678, 679, 682, 683, 684, 686, 695, 697, 698, 700, 701, 702, 703, 710, 711, 714, 719, 720, 723, 727, 728, 731, 734, 735, 738, 740, 743, 745, 746, 750, 751, 752, 756, 757, 760, 763, 766, 767, 772, 774, 777, 779, 780, 781, 786, 788, 793, 796, 800, 803, 804, 805, 806, 807, 809, 811, 813, 814, 817, 822, 823, 828, 833, 835, 837, 839, 840, 842, 844, 846, 849, 852, 854, 856, 858, 865, 867, 869, 870, 871, 872, 877, 878, 884, 885, 890, 891, 895, 896, 897, 899, 901, 904, 907, 909, 913, 917, 918, 919, 920, 921, 926, 931, 933, 935, 938, 945, 946, 947, 948, 949, 950, 951, 953, 954, 957, 958, 961, 965, 966, 967, 969, 970, 974, 979, 980, 990, 996, 997, 998, 999, 1003, 1006, 1008, 1009, 1011, 1015, 1018, 1021, 1024, 1026, 1029, 1030, 1032, 1033, 1034, 1037, 1040, 1042, 1043, 1048, 1052, 1053, 1057, 1068, 1070, 1074, 1078, 1082, 1083, 1084, 1085, 1089, 1090, 1093, 1094, 1095, 1098, 1103, 1108, 1112, 1116, 1121, 1122, 1124, 1128, 1132, 1135, 1136, 1139, 1141, 1143, 1146, 1152, 1154, 1161, 1163, 1166, 1171, 1173, 1182, 1185, 1186, 1187, 1190, 1192, 1196, 1197, 1202, 1203, 1208, 1209, 1211, 1212, 1215, 1218, 1220, 1222, 1224, 1229, 1230, 1234, 1236, 1237, 1244, 1245, 1254, 1255, 1257, 1258, 1259, 1260, 1261, 1263, 1266, 1274, 1275, 1280, 1281, 1283, 1288, 1291, 1298, 1299, 1300, 1302, 1303, 1309, 1312, 1313, 1316, 1317, 1318, 1319, 1321, 1324, 1326, 1329, 1333, 1334, 1337, 1340, 1341, 1342, 1343, 1345, 1349, 1354, 1355, 1358, 1360, 1365, 1372, 1373, 1375, 1381, 1383, 1384, 1392, 1393, 1395, 1398, 1399, 1400, 1402, 1407, 1418, 1419, 1426, 1428, 1430, 1432, 1440, 1441, 1444, 1445, 1446, 1447, 1448, 1450, 1454, 1460, 1462, 1463, 1465, 1466, 1467, 1468, 1476, 1477, 1480, 1481, 1482, 1486, 1493, 1494, 1495, 1499, 1502, 1503, 1506, 1509, 1511, 1512, 1515, 1516, 1517, 1520, 1521, 1522, 1524, 1525, 1528, 1530, 1532, 1533, 1543, 1545, 1546, 1548, 1549, 1550, 1553, 1556, 1558, 1564, 1565, 1566, 1574, 1575, 1576, 1577, 1584, 1590, 1591, 1592, 1593, 1594, 1597, 1602, 1604, 1605, 1606, 1608, 1609, 1610, 1621, 1622, 1626, 1631, 1636, 1637, 1640, 1641, 1643, 1648, 1654, 1655, 1660, 1661, 1662, 1663, 1664, 1667, 1670, 1672, 1675, 1676, 1682, 1683, 1685, 1693, 1699, 1700, 1701, 1702, 1704, 1705, 1708, 1714, 1715, 1716, 1717, 1718, 1720, 1722, 1724, 1726, 1727, 1729, 1730, 1731, 1736, 1745, 1749, 1750, 1751, 1754, 1755, 1758, 1767, 1770, 1773, 1776, 1778, 1779, 1780, 1782, 1786, 1788, 1796, 1800, 1801, 1802, 1803, 1805, 1808, 1814, 1815, 1816, 1818, 1826, 1827, 1828, 1830, 1837, 1846, 1849, 1854, 1857, 1859, 1861, 1864, 1868, 1872, 1878, 1880, 1881, 1885, 1886, 1887, 1888, 1890, 1900, 1901, 1903, 1913, 1915, 1919, 1924, 1925, 1926, 1930, 1933, 1941, 1942, 1950, 1954, 1959, 1971, 1972, 1977, 1978, 1983, 1985, 1993, 1995, 1996, 1998, 1999, 2000, 2001, 2004, 2005, 2008, 2015, 2017, 2018, 2024, 2030, 2036, 2038, 2039, 2041, 2042, 2043, 2045, 2047, 2048, 2050, 2055, 2063, 2067, 2068, 2070, 2073, 2074, 2077, 2081, 2082, 2088, 2091, 2093, 2094, 2096, 2099, 2101, 2104, 2111, 2112, 2114, 2116, 2126, 2129, 2134, 2139, 2141, 2142, 2143, 2144, 2145, 2146, 2147, 2148, 2153, 2159, 2165, 2166, 2168, 2172, 2177, 2179, 2190, 2192, 2194, 2197, 2200, 2201, 2204, 2206, 2212, 2215, 2219, 2221, 2227, 2230, 2232, 2233, 2236, 2238, 2241, 2245, 2248, 2250, 2261, 2262, 2263, 2265, 2271, 2274, 2275, 2276, 2280, 2282, 2284, 2286, 2292, 2301, 2302, 2304, 2305, 2307, 2309, 2312, 2315, 2316, 2322, 2323, 2324, 2328, 2330, 2338, 2339, 2340, 2342, 2352, 2356, 2361, 2362, 2363, 2366, 2371, 2372, 2373, 2377, 2378, 2384, 2386, 2388, 2390, 2393, 2395, 2401, 2403, 2405, 2407, 2411, 2414, 2415, 2417, 2420, 2423, 2426, 2433, 2441, 2442, 2446, 2449, 2451, 2452, 2454, 2458, 2460, 2464, 2470, 2471, 2476, 2480, 2482, 2485, 2489, 2491, 2492, 2493, 2496, 2498, 2504, 2511, 2513, 2514, 2520, 2523, 2524, 2526, 2529, 2530, 2532, 2533, 2537, 2538, 2540, 2542, 2543, 2546, 2550, 2551, 2558, 2559, 2561, 2562, 2563, 2565, 2566, 2568, 2574, 2575, 2577, 2578, 2582, 2584, 2586, 2590, 2593, 2594, 2595, 2604, 2606, 2609, 2613, 2616, 2617, 2621, 2624, 2627, 2629, 2631, 2633, 2637, 2643, 2651, 2652, 2656, 2660, 2664, 2669, 2670, 2672, 2675, 2678, 2680, 2686, 2687, 2688, 2690, 2693, 2694, 2696, 2698, 2699, 2700, 2703, 2706, 2709, 2712, 2717, 2721, 2722, 2724, 2727, 2729, 2731, 2741, 2747, 2748, 2750, 2751, 2754, 2756, 2763, 2764, 2765, 2774, 2775, 2785, 2786, 2787, 2788, 2791, 2792, 2795, 2797, 2806, 2807, 2809, 2812, 2813, 2814, 2816, 2821, 2828, 2831, 2837, 2841, 2842, 2849, 2851, 2853, 2857, 2859, 2860, 2861, 2868, 2869, 2870, 2881, 2883, 2885, 2886, 2889, 2890, 2899, 2902, 2904, 2905, 2911, 2912, 2913, 2916, 2919, 2920, 2921, 2923, 2924, 2925, 2928, 2930, 2932, 2934, 2936, 2940, 2945, 2948, 2954, 2957, 2958, 2960, 2965, 2966, 2970, 2971, 2974, 2975, 2977, 2978, 2980, 2982, 2985, 2987, 2988, 2992, 2993, 2994, 3010, 3015, 3022, 3032, 3033, 3034, 3035, 3037, 3039, 3041, 3043, 3046, 3053, 3054, 3058, 3063, 3066, 3067, 3068, 3070, 3072, 3074, 3076, 3077, 3078, 3079, 3080, 3083, 3084, 3087, 3088, 3089, 3090, 3092, 3096, 3097, 3099, 3103, 3107, 3111, 3113, 3114, 3115, 3117, 3118, 3119, 3120, 3121, 3124, 3133, 3135, 3139, 3141, 3143, 3144, 3145, 3146, 3150, 3151, 3152, 3153, 3154, 3156, 3157, 3159, 3160, 3162, 3163, 3165, 3166, 3167, 3168, 3174, 3175, 3179, 3180, 3183, 3184, 3185, 3186, 3187, 3189, 3190, 3191, 3192, 3194, 3197, 3205, 3206, 3215, 3222, 3224, 3228, 3234, 3237, 3242, 3246, 3247, 3255, 3256, 3261, 3262, 3267, 3269, 3270, 3271, 3275, 3277, 3278, 3287, 3289, 3292, 3295, 3299, 3300, 3301, 3303, 3304, 3307, 3308, 3313, 3318, 3320, 3324, 3325, 3327, 3328, 3330, 3332, 3336, 3337, 3339, 3345, 3351, 3352, 3353, 3354, 3360, 3361, 3363, 3369, 3370, 3375, 3376, 3382, 3392, 3395, 3399, 3400, 3401, 3402, 3405, 3409, 3411, 3412, 3417, 3418, 3419, 3420, 3428, 3429, 3431, 3435, 3437, 3440, 3447, 3448, 3450, 3455, 3457, 3459, 3462, 3470, 3472, 3474, 3479, 3480, 3483, 3486, 3490, 3491, 3495, 3496, 3497, 3499, 3501, 3507, 3509, 3510, 3514, 3518, 3521, 3522, 3524, 3527, 3532, 3533, 3536, 3538, 3539, 3549, 3551, 3552, 3553, 3557, 3558, 3564, 3565, 3568, 3569, 3571, 3573, 3575, 3579, 3580, 3581, 3583, 3587, 3588, 3589, 3590, 3593, 3594, 3595, 3596, 3598, 3603, 3605, 3607, 3608]
        no_question_overlap_ids = [11,14,16,20,28,31,48,50,54,57,66,70,77,90,94,95,102,110,113,116,130,132,133,134,136,141,144,145,149,169,171,172,178,181,182,189,194,196,205,207,212,219,225,226,231,239,241,243,244,266,276,277,285,286,287,288,295,302,306,311,318,326,328,333,339,343,353,361,364,379,386,390,393,395,397,404,408,421,428,432,439,474,476,484,489,493,494,504,513,518,522,524,533,534,548,556,569,576,578,580,587,596,597,600,611,613,618,622,624,627,633,634,635,640,647,656,657,666,672,674,677,679,681,682,688,689,695,698,699,700,701,702,709,714,715,717,719,723,727,728,729,732,733,748,752,760,763,767,772,775,781,786,788,793,796,801,818,824,825,829,842,844,850,854,858,867,876,892,893,900,904,907,913,916,918,931,937,947,960,976,979,980,983,1003,1009,1011,1015,1016,1023,1037,1043,1047,1051,1064,1074,1079,1086,1089,1093,1112,1122,1125,1136,1138,1156,1157,1161,1176,1178,1184,1185,1187,1190,1201,1213,1223,1233,1236,1240,1241,1242,1254,1259,1260,1263,1273,1275,1286,1291,1299,1308,1311,1317,1319,1329,1333,1338,1341,1342,1343,1344,1345,1358,1370,1375,1387,1395,1398,1399,1405,1406,1414,1424,1425,1428,1429,1430,1432,1435,1441,1447,1449,1454,1456,1459,1462,1468,1473,1475,1476,1485,1493,1498,1499,1503,1508,1530,1531,1532,1540,1554,1556,1563,1564,1565,1574,1580,1588,1590,1592,1594,1599,1605,1607,1612,1613,1621,1622,1624,1626,1627,1633,1637,1641,1643,1649,1652,1657,1658,1659,1661,1666,1691,1694,1696,1700,1702,1715,1718,1720,1721,1722,1725,1727,1731,1736,1737,1746,1751,1759,1763,1773,1774,1777,1780,1781,1782,1783,1785,1796,1800,1807,1808,1818,1835,1844,1846,1849,1859,1860,1864,1866,1867,1878,1879,1880,1881,1883,1887,1890,1897,1900,1901,1912,1915,1923,1925,1928,1942,1946,1950,1952,1953,1956,1961,1971,1975,1978,1984,1986,2000,2002,2004,2006,2009,2013,2019,2026,2030,2045,2047,2052,2054,2055,2058,2074,2078,2087,2093,2111,2114,2116,2128,2129,2130,2139,2141,2143,2149,2185,2192,2198,2204,2206,2213,2214,2230,2232,2248,2249,2251,2255,2257,2259,2261,2275,2277,2279,2286,2293,2301,2305,2310,2312,2313,2323,2332,2334,2340,2342,2352,2356,2358,2363,2367,2368,2375,2379,2381,2387,2395,2405,2410,2418,2422,2423,2430,2437,2446,2447,2449,2462,2469,2470,2471,2474,2479,2482,2489,2491,2493,2497,2498,2511,2528,2532,2535,2538,2549,2556,2561,2563,2564,2568,2569,2574,2575,2584,2585,2588,2592,2601,2605,2609,2614,2616,2621,2626,2632,2647,2651,2654,2660,2668,2675,2676,2677,2678,2688,2690,2693,2694,2695,2700,2705,2717,2724,2727,2730,2740,2741,2748,2758,2759,2764,2785,2788,2803,2808,2809,2810,2812,2814,2821,2824,2825,2831,2838,2839,2849,2859,2864,2869,2874,2876,2878,2889,2905,2918,2920,2943,2944,2948,2949,2951,2956,2957,2959,2963,2974,2975,2977,2993,3003,3015,3016,3032,3034,3041,3045,3046,3049,3053,3054,3055,3069,3077,3079,3081,3084,3102,3111,3113,3124,3126,3134,3137,3140,3145,3152,3153,3155,3157,3162,3163,3165,3166,3174,3177,3184,3197,3206,3215,3221,3222,3229,3234,3237,3255,3261,3265,3266,3272,3287,3290,3303,3307,3318,3320,3324,3332,3339,3346,3348,3350,3351,3354,3363,3373,3378,3381,3393,3395,3402,3405,3407,3409,3411,3413,3423,3427,3431,3435,3439,3442,3445,3448,3449,3458,3459,3470,3479,3486,3493,3498,3509,3510,3511,3514,3518,3522,3523,3524,3552,3555,3570,3584,3591,3598,3600,3606,3607]
        no_answer_overlap_result = [x for i, x in enumerate(result) if i in no_answer_overlap_ids]
        no_question_overlap_result = [x for i, x in enumerate(result) if i in no_question_overlap_ids]

        print("#### no_answer_overlap_result ####")
        for K in [1, 5, 10, 20, 50, 100]:
            cnt = 0
            for i, x in enumerate(no_answer_overlap_result):
                context = x["ctxs"][:K]
                if any([c for c in context if c["has_answer"]]):
                    cnt += 1
            print(f"top{K} retrieval acc = {cnt}/{len(no_answer_overlap_result)} = {cnt/len(no_answer_overlap_result)}")

        print("#### no_question_overlap_result ####")
        for K in [1, 5, 10, 20, 50, 100]:
            cnt = 0
            for i, x in enumerate(no_question_overlap_result):
                context = x["ctxs"][:K]
                if any([c for c in context if c["has_answer"]]):
                    cnt += 1
            print(f"top{K} retrieval acc = {cnt}/{len(no_question_overlap_result)} = {cnt/len(no_question_overlap_result)}")


