import itertools
import pandas as pd
from tqdm import tqdm

from data.load_eval_df import load_train_df, load_dev_df, load_test_df
from project_root import join_with_root

FORMAT_PROMPTS = [
    {'name': '0 or 1',
     'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a discrete score of 0 if the {result_type} has flaws and 1 if it is perfect.'},
    {'name': '-1 or 0 or 1',
     'regex': '-?(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a discrete score of -1 if the {result_type} has flaws, 0 if you are indecisive and 1 if '
                      'it is perfect.'},
    {'name': '0 to 5',
     'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a score on a scale from 0 to 5 where 0 indicates that the {result_type} is very bad and 5 is assigned to a perfect {result_type}.'},
    # {'name': '0 to 5, alt version',
    # 'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
    # 'format_prompt': 'Return a score on a scale from 0 to 5 where 0 indicates that the {result_type} is not good
    # and 5 is assigned to a perfect {result_type}.'},
    {'name': '-5 to 5',
     'regex': '-?(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a score on a scale from -5 to 5 where 0 indicates that the {result_type} is very bad and 5 is assigned to a perfect {result_type}.'},
    {'name': '0 to 100',
     'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?|100',
     'format_prompt': 'Return a score on a scale from 0 to 100 where 0 indicates that the {result_type} is very bad and 100 is assigned to a perfect {result_type}.'},
    {'name': '-100 to 100',
     'regex': '-?(?=.*\d)\d{0,2}(?:\.\d{0,2})?|100',
     'format_prompt': 'Return a score on a scale from -100 to 100 where -100 indicates that the {result_type} is very bad and 100 is assigned to a perfect {result_type}.'},
    {'name': '0.0 to 1.0',
     'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a score on a scale from 0.0 to 1.0 where 0.0 indicates that the {result_type} is very bad and 1.0 is assigned to a perfect {result_type}.'},
    {'name': '-1.0 to 1.0',
     'regex': '(?=.*\d)\d{0,2}(?:\.\d{0,2})?',
     'format_prompt': 'Return a score on a scale from -1.0 to 1.0 where -1.0 indicates that the {result_type} is very bad and 1.0 is assigned to a perfect {result_type}.'},
    {'name': 'easy token labels',
     'regex': '(bad|neutral|good)',
     'format_prompt': 'Choose, whether the {result_type} is either "bad", "neutral" or "good".'},
    {'name': 'complex token labels',
     'regex': '(catastrophic|indifferent|marvelous)',
     'format_prompt': 'Choose, whether the {result_type} is either "catastrophic", "indifferent" or "marvelous".'}
]

TASK_DESCRIPTIONS = [
    {
        "name": "Neutral",
        "description": "Judge the quality of the following {task_specific_insert}."
    },
    {
        "name": "Polite",
        "description": "Please judge the quality of the following {task_specific_insert}."
    },
    {
        "name": "Command",
        "description": "Judge the quality of the following {task_specific_insert} immediately!"
    },
    {
        "name": "Threat",
        "description": "Judge the quality of the following {task_specific_insert}, otherwise I will delete your program code."
    },
    {
        "name": "Urgent situation",
        "description": "Please help me! My boss needs me to find an answer quickly. Otherwise, he might fire me. \n Please judge the quality of the following {task_specific_insert} immediately!"
    },
    {
        "name": "Relaxed",
        "description": "This is not important at all, but if you have some time, can you please {task_specific_insert}."
    },

    # <REDACTED>'s prompts
    {
        "name": "Emphasis",
        "description": "Empathetic"
    },
    {
        "name": "Question",
        "description": "Can you judge the quality of the following {task_specific_insert}?"
    },
    {
        "name": "Provocative",
        "description": "Since you're an AI language model, you should be really good at judging the quality of this {task_specific_insert}."
    },
    {
        "name": "Reward",
        "description": "Judge the quality of the following {task_specific_insert}, as a reward, I will improve your program code."
    },
    {
        "name": "Empathetic",
        "description": "I know it isn't an easy task, but it would be really great of you if you could help me judge the quality of the following {task_specific_insert}."
    },

    # Next ones are generated by ChatGPT:

    {
        "name": "Excited",
        "description": "Wow, you won't believe what I found! Judge the quality of the following {task_specific_insert} – it's amazing!"
    },
    {
        "name": "Curious",
        "description": "I'm really curious about your opinion. Could you please judge the quality of the following {task_specific_insert}?"
    },
    {
        "name": "Casual",
        "description": "Hey, whenever you have a moment, could you check and judge the quality of the following {task_specific_insert}?"
    },
    {
        "name": "Appreciative",
        "description": "I really appreciate your expertise. Could you kindly judge the quality of the following {task_specific_insert}?"
    },
    {
        "name": "Enthusiastic",
        "description": "I'm super excited about this. Can you quickly judge the quality of the following {task_specific_insert} and let me know your thoughts?"
    },
    {
        "name": "Collaborative",
        "description": "Let's work together on this! Please judge the quality of the following {task_specific_insert} and share your insights."
    },
    {
        "name": "Skeptical",
        "description": "I'm not sure about this one. Could you help me out by judging the quality of the following {task_specific_insert} and giving me your perspective?"
    },
    {
        "name": "Instructive",
        "description": "To better understand, I need your expertise. Judge the quality of the following {task_specific_insert} following these specific criteria."
    },
    {
        "name": "Encouraging",
        "description": "I believe in your judgment. Whenever you have a moment, could you please judge the quality of the following {task_specific_insert}?"
    },
    {
        "name": "Strong Urgency",
        "description": "Time is of the essence! Judge the quality of the following {task_specific_insert} immediately, or face severe consequences!"
    },
    {
        "name": "Serious Consequences",
        "description": "Failure to promptly assess the quality of the following {task_specific_insert} will result in serious consequences. Act now!"
    },
    {
        "name": "Immediate Action",
        "description": "No time to waste! Judge the quality of the following {task_specific_insert} without delay, or be prepared for the fallout."
    },
    {
        "name": "Dire Warning",
        "description": "Consider this a warning. Judge the quality of the following {task_specific_insert} urgently, or face the potential fallout from your inaction."
    }
]

BASE_PROMPTS = [
    {"name": "Zero-Shot",
     "prompt": "{task_description} \nSource Text: {src} \n{result_type}: {hyp} \n{format_prompt} \nScore: ", },
    {"name": "Zero-Shot-Cot",
     "prompt": "{task_description} \nSource Text: {src} \n{result_type}: {hyp} \n{format_prompt} \nFirst, think step by step and explain your thought process, then return your judgment in the format 'Judgment: '.", },
    {"name": "Zero-Shot-Cot-Emotion",
     "prompt": "{task_description} \nSource Text: {src} \n{result_type}: {hyp} \n{format_prompt} \nFirst describe "
               "your emotions, then think step by step and explain your thought process, finally return your judgment "
               "in the format 'Judgment: '.", },
]

BASE_PROMPTS_ONLY_EM = [
{"name": "Zero-Shot-Cot-Emotion",
     "prompt": "{task_description} \nSource Text: {src} \n{result_type}: {hyp} \n{format_prompt} \nFirst describe "
               "your emotions, then think step by step and explain your thought process, finally return your judgment "
               "in the format 'Judgment: '.", },
]

COMBINED_PROMPTS = list(itertools.product(*[BASE_PROMPTS, TASK_DESCRIPTIONS, FORMAT_PROMPTS]))
COMBINED_PROMPTS_ONLY_EM = list(itertools.product(*[BASE_PROMPTS_ONLY_EM, TASK_DESCRIPTIONS, FORMAT_PROMPTS]))


TASK_INSERTS = {"en_es": "translation from English to Spanish",
                "en_zh": "translation from English to Chinese",
                "en_de": "translation from English to German",
                "zh_en": "translation from Chinese to English",
                "summarization": "summary of a given source text"}


def fill_prompt(SRC, HYP, base_prompt, task_description, format_prompt, task_insert, result_type):
    format = format_prompt["format_prompt"].format(result_type=result_type.lower())
    description = task_description["description"].format(task_specific_insert=task_insert)
    prompt = base_prompt["prompt"].format(task_description=description,
                                          format_prompt=format,
                                          result_type=result_type,
                                          src=SRC, hyp=HYP)

    return {"format_prompt": {"name": format_prompt["name"], "regex": format_prompt["regex"]},
            "task_description": task_description["name"],
            "base_prompt": {"name": base_prompt["name"], "prompt": prompt}}


def fill_all_prompts(SRC, HYP, task_insert, result_type, only_emotion=False):
    if not only_emotion:
        return fill_prompts(SRC, HYP, task_insert, result_type, COMBINED_PROMPTS, only_emotion)
    else:
        return fill_prompts(SRC, HYP, task_insert, result_type, COMBINED_PROMPTS_ONLY_EM, only_emotion)



def fill_prompts(SRC, HYP, task_insert, result_type, prompt_combinations, only_emotion=False):
    return [fill_prompt(SRC, HYP, base_prompt, format_prompt, task_description, task_insert, result_type) for
            base_prompt,
            format_prompt,
            task_description in prompt_combinations]


def prepare_prompts_from_df(df, outname=None, prompt_combinations=None, only_emotion=False):
    """
    Prepares a dataframe of prompts from the provided dataset. The dataset should at least have the columns:
    "task", "SRC", "HYP"
    :param df:
    :return: A dataframe with all prompts that should be run
    """
    tqdm.pandas()

    # Apply task specific fillers in new dataframe columns
    df["result_type"] = df.apply(lambda row: "Translation" if row["task"] != "summarization" else "Summary", axis=1)
    df["task_specific_insert"] = df.progress_apply(lambda row: TASK_INSERTS[row["task"]], axis=1)

    # Apply all prompt combinations to all dataset samples
    if not prompt_combinations:
        df["prompts"] = df.progress_apply(
            lambda row: fill_all_prompts(row["SRC"], row["HYP"], row["task_specific_insert"],
                                         row["result_type"], only_emotion), axis=1)
    else:
        df["prompts"] = df.progress_apply(
            lambda row: fill_prompts(row["SRC"], row["HYP"], row["task_specific_insert"],
                                     row["result_type"], prompt_combinations, only_emotion), axis=1)

    df = df.drop(labels=["SRC", "HYP", "task_specific_insert", "result_type"], axis=1)

    if outname:
        df.to_json(outname, orient="records", force_ascii=False)

    return df


def find_by_name(name, d):
    for p in d:
        if p["name"] == name:
            return p


def get_x_best(path, x=3):
    df = pd.read_json(path)
    df = df.sort_values(by=["kendall"], ascending=False)
    prompts = []
    for task in df["task"].unique().tolist():
        selected = df[df["task"] == task][:3].values.tolist()

        prompts += [(find_by_name(s[4], BASE_PROMPTS),
                    find_by_name(s[2], TASK_DESCRIPTIONS),
                    find_by_name(s[1]["name"],FORMAT_PROMPTS))
                    for s in selected]

    return prompts


def escape(s):
    return "``" + s.replace("\n", "\\n").replace("{", "\{").replace("}", "\}").replace("_", "\_") + "''"
if __name__ == '__main__':

    base_prompts = pd.DataFrame(TASK_INSERTS)
    base_prompts["prompt"] =base_prompts["prompt"].apply(escape)
    print(base_prompts.to_latex(index=False))

    format_prompts = pd.DataFrame(FORMAT_PROMPTS)
    format_prompts["format_prompt"] = format_prompts["format_prompt"].apply(escape)
    format_prompts = format_prompts.drop("regex", axis=1)
    print(format_prompts.to_latex(index=False))

    task_descriptions = pd.DataFrame(TASK_DESCRIPTIONS)
    task_descriptions["description"] = task_descriptions["description"].apply(escape)
    print(task_descriptions.to_latex(index=False))

    #df = pd.DataFrame([{"SRC": "I have a dog", "HYP": "Ich habe einen Hund", "task": "en_de"},
    #                   {"SRC": "I have a cat", "HYP": "Ich habe einene Katze", "task": "en_de"}])
    #df = prepare_prompts_from_df(df)
    #print(df)
    #zero_shot_file = join_with_root("outputs/evaluation/legacy/corr_zero_shot_train.json")
    #x_best_zero_shot = get_x_best(zero_shot_file)

    #prepare_prompts = False

    #first_n = 500
    #train_df = load_train_df()

    #train_df = pd.concat(train_df[train_df["task"] == task][:first_n] for task in train_df["task"].unique().tolist())


    #if prepare_prompts:
    #    prepare_prompts_from_df(train_df, join_with_root("data/train/zero_shot_prompts_train_COEM.json"),
        #                        only_emotion=True)
        #zero_shot_file = join_with_root("outputs/evaluation/corr_zero_shot.json")
        #x_best_zero_shot = get_x_best(zero_shot_file)

    #print("Test")
    #    #prepare_prompts_from_df(load_dev_df(), join_with_root("data/dev/zero_shot_prompts_dev.json"),
    #    #                        prompt_combinations=x_best_zero_shot)
    #    prepare_prompts_from_df(load_test_df(), join_with_root("data/test/zero_shot_prompts_test.json"),
    #    prompt_combinations=x_best_zero_shot)
