#include <cmath>
#include <stack>
#include <algorithm>
#include <ctime>
#include <unordered_map>

#include "graphcjj2gc_prune_depparser.h"
#include "common/token/word.h"
#include "common/token/pos.h"

#define NEG(x) ( (x) == -1?-1:(x)^1 ) 
#define RIGHT_ARC(x) ( (x) << 1 )
#define LEFT_ARC(x) ( ((x) << 1) + 1 )
#define IS_RIGHT_ARC(x) ( ((x) & 1) == 0)
#define IS_LEFT_ARC(x) ( ((x) & 1) == 1)
#define ARC(x) ( (x) >> 1 )
typedef QuarGram<int> LsLlRsRl;

namespace graphcjj2gc_prune {
	WordPOSTag DepParser::empty_taggedword = WordPOSTag();
	WordPOSTag DepParser::start_taggedword = WordPOSTag();
	WordPOSTag DepParser::end_taggedword = WordPOSTag();
	double time, ftime;
	int fsum;
	int debug;
	int totalerr, total;
	int totaledgecor;
	DepParser::DepParser(const std::string & sFeatureInput, const std::string & sFeatureOut, int nState) :
		DepParserBase(sFeatureInput, nState) {
		//std::cout << "start loading feature1" << std::endl;
		m_nSentenceLength = 0;
		time = ftime = 0;
		fsum = 0;
		DepLabel.init();
		m_Weight = new Weight(sFeatureInput, sFeatureOut, DepLabel);
		totalmy = total = totalcor = totaledgecor = 0;
		for (int i = 0; i < MAX_SENTENCE_SIZE; ++i) {
			m_lItems[i][i].init(i, i);
		}

		DepParser::empty_taggedword.refer(TWord::code(EMPTY_WORD), TPOSTag::code(EMPTY_POSTAG));
		DepParser::start_taggedword.refer(TWord::code(START_WORD), TPOSTag::code(START_POSTAG));
		DepParser::end_taggedword.refer(TWord::code(END_WORD), TPOSTag::code(END_POSTAG));
	}

	DepParser::~DepParser() {
		delete m_Weight;
	}

	void DepParser::train(const DependencyGraph & correct, const int & round) {
		// initialize
		int idx = 0;
		time_t b, e;
		b = clock();
		m_vecCorrectArcs.clear();
		m_nTrainingRound = round;
		m_nSentenceLength = correct.getSentence().size();
		m_vecDependencyTree = correct.getTree();
		m_lSentence[0].refer(TWord::code(ROOT_WORD), TPOSTag::code(ROOT_POSTAG));
		//if (correct.getSentence().size() >= 40) return;
		for (const auto & node : correct.getSentence()) {
			m_lSentence[++idx].refer(TWord::code(std::get<0>(node)), TPOSTag::code(std::get<1>(node)));
		}
		for (const auto arc : correct.getGraph())
		{
			//m_vecCorrectArcs.push_back(PCLArc(EDGE_X(arc), EDGE_Y(arc), DepLabel.code("NULL")));
			m_vecCorrectArcs.push_back(PCLArc(EDGE_X(arc), EDGE_Y(arc), DepLabel.code(EDGE_LABEL(arc))));
		}

		for (const auto & arc : m_vecCorrectArcs)
		{
			int p = arc.first();
			int c = arc.second();
			int l = arc.third();
			DepLabel.addPPL(TriGram<int>(m_lSentence[p].second(), m_lSentence[c].second(), l));
		}
		debug = 0;
		if (debug) {
			m_setFirstGoldScore.clear();
			for (const auto & arc : m_vecCorrectArcs) {
				m_setFirstGoldScore.insert(PCLArc(arc.first(), arc.second(),arc.third()));
			}
			/*for (auto arc : m_setFirstGoldScore)
				std::cout << "single " << arc << std::endl;*/
			std::vector<PCLArc> corarc;
			corarc.clear();

			for (auto arc : m_vecCorrectArcs)
			{
				int l = arc.first();
				int r = arc.second();
				int dir = RIGHT_ARC(arc.third());
				if (l > r) std::swap(l, r), dir = NEG(dir);
				corarc.push_back(PCLArc(l, r, dir));
			}

			m_setSecondGoldScore.clear();
			for (auto arc1 : corarc)
			{
				int rsp = arc1.second();
				int rla = -1;
				int lsp = arc1.first();
				int lla = -1;
				for (auto arc2 : corarc)
				{
					if (arc2.first() == arc1.first() && arc2.second() < arc1.second() && lsp < arc2.second())
					{
						lsp = arc2.second();
						lla = arc2.third();
					}
					if (arc2.second() == arc1.second() && arc2.first() > arc1.first() && rsp > arc2.first())
					{
						rsp = arc2.first();
						rla = NEG(arc2.third());
					}
				}

				m_setSecondGoldScore.insert(PC1L1C2L2Arc(arc1.first(), arc1.second(), arc1.third(), lsp, lla));
				m_setSecondGoldScore.insert(PC1L1C2L2Arc(arc1.second(), arc1.first(), NEG(arc1.third()), rsp, rla));
			}
			/*for (auto arc : m_setSecondGoldScore)
				std::cout << "2rd score " << arc << std::endl;*/

			m_setGCGoldScore.clear();
			for (int i = 1; i <= m_nSentenceLength; i++)
			{
				int l = i;
				int lla = -1;
				int r = i;
				int rla = -1;
				for (auto arc : corarc)
				{
					if (arc.first() == i && arc.second() > r)
					{
						r = arc.second();
						rla = arc.third();
					}
					if (arc.second() == i && arc.first() < l)
					{
						l = arc.first();
						lla = NEG(arc.third());
					}

				}
				if (l != i && r != i)
					m_setGCGoldScore.insert(PC1L1C2L2Arc(i, l, lla, r, rla));
			}
			//for (auto itr : m_setGCGoldScore)
			//	std::cout << "gc " << itr << std::endl;
		}
		//correct.print();
		//system("pause");
		// train
		work(nullptr, correct);
		time += clock() - b;
		if (m_nTrainingRound % OUTPUT_STEP == 0 || m_nTrainingRound == 20) {
			//std::cout<<m_nSentenceLength << std::endl;
			std::cout << "total time " << 0.001 * time << ' ' << ' ' << 0.001*time / m_nTrainingRound << std::endl;
			std::cout << "sentence wrong " << m_nTotalErrors << " / " << m_nTrainingRound << std::endl;
			std::cout << "unlabel precision " << totaledgecor << " / " << total << ' ' << 1.0*totaledgecor / total << std::endl;
			std::cout << "unlabel recall " << totaledgecor << " / " << totalmy << ' ' << 1.0*totaledgecor / totalmy << std::endl;
			std::cout << "unlabel F-Score  " << 2.0*totaledgecor / (total + totalmy) << std::endl;
			std::cout << "label precision " << totalcor << " / " << total << ' ' << 1.0*totalcor / total << std::endl;
			std::cout << "label recall " << totalcor << " / " << totalmy << ' ' << 1.0*totalcor / totalmy << std::endl;
			std::cout << "label F-Score  " << 2.0*totalcor / (total + totalmy) << std::endl;
			//totaledgecor = totalcor = total = totalmy = 0;
		}
	}

	void DepParser::parse(const DependencyGraph & graph, DependencyGraph * retval) {
		int idx = 0;
		m_nTrainingRound = 0;
		//std::cout << "xx" << std::endl;
		m_nSentenceLength = graph.getSentence().size();
		m_vecDependencyTree = graph.getTree();
		retval->clear();
		retval->m_lTree = m_vecDependencyTree;
		for (const auto & token : graph.getSentence()) {
			m_lSentence[++idx].refer(TWord::code(SENT_WORD(token)), TPOSTag::code(SENT_POSTAG(token)));
			retval->m_lSentence.push_back(token);
		}
		work(retval, graph);
	}

	void DepParser::work(DependencyGraph * retval, const DependencyGraph & correct) {

		decode();
		decodeArcs();
		/*for (auto arc : m_vecCorrectArcs)
			std::cout << "cor  " << arc << std::endl;
		for (auto arc : m_vecTrainArcs)
			std::cout << "my  " << arc << std::endl;
		std::cout << "---------------" << std::endl;*/
		switch (m_nState) {
		case ParserState::TRAIN:
			update();
			break;
		case ParserState::PARSE:
			generate(retval, correct);
			break;
		case ParserState::GOLDTEST:
			update();
			break;
		default:
			break;
		}
	}


	void DepParser::work_after_scoring(DependencyGraph * retval, const DependencyGraph & correct) {
		decode_after_scoring();
		decodeArcs();
		generate(retval, correct);
	}

	void DepParser::outans()
	{
		std::cout << "total time " << 0.001 * time << ' ' << ' ' << 0.001*time / m_nTrainingRound << std::endl;
		std::cout << "sentence wrong " << m_nTotalErrors << " / " << m_nTrainingRound << std::endl;
		std::cout << "unlabel precision " << totaledgecor << " / " << total << ' ' << 1.0*totaledgecor / total << std::endl;
		std::cout << "unlabel recall " << totaledgecor << " / " << totalmy << ' ' << 1.0*totaledgecor / totalmy << std::endl;
		std::cout << "unlabel F-Score  " << 2.0*totaledgecor / (total + totalmy) << std::endl;
		std::cout << "label precision " << totalcor << " / " << total << ' ' << 1.0*totalcor / total << std::endl;
		std::cout << "label recall " << totalcor << " / " << totalmy << ' ' << 1.0*totalcor / totalmy << std::endl;
		std::cout << "label F-Score  " << 2.0*totalcor / (total + totalmy) << std::endl;
	}
	void DepParser::decode_after_scoring()
	{
		int sumlen = m_nSentenceLength;
		int labelsize = DepLabel.count();
		m_lcaTree.loadPath(m_vecDependencyTree);
//		InitScore_single();
		//InitScore_sibling();
		//InitScore_gc();
		Score_sibling.clear();
		Score_gc.clear();
		std::unordered_map<LsLlRsRl, tscore> map_LsLlRsRlScore;

		for (int d = 1; d < sumlen; d++)
		{
			for (int l = 1; l <= sumlen - d; l++)
			{
				int r = l + d;

				tscore score;
				StateItem  & item = m_lItems[l][r];
				item.init(l, r);
				LabelScoreAgenda l2r_label_beam;
				l2r_label_beam.clear();
				for (int l2r_la = RIGHT_ARC(1); l2r_la <= LEFT_ARC(labelsize); l2r_la++)
					if (Score_single[l][r][l2r_la] > HALF_MIN_SCORE)
						l2r_label_beam.insertItem(LabelScore(l2r_la, Score_single[l][r][l2r_la]));

				for ( auto l2r_label_iterator : l2r_label_beam)
				{
					int l2r_la = l2r_label_iterator->first();
					int r2l_la = NEG(l2r_la);
					tscore labelscore = l2r_label_iterator->second();
					tscore lrl_score = arcScore_sibling(l, r, l2r_la, l, -1);
					if (d == 1)
					{
						item.updateL2R_LABEL(l, r, labelscore + lrl_score + arcScore_sibling(r, l, NEG(l2r_la),r,-1),
							l2r_la ,-1, -1);
					}
					else
					{
						ScoreWithBiSplit maxswbs;
						maxswbs.refer(l, r, MIN_SCORE);

						for (auto right_nolabel : m_lItems[l+1][r].l2r_nolabel)
						{
							int lsplit = l;
							int lsplit_label = -1;
							int rsplit = right_nolabel->getInnerSplit();
							int rsplit_label = right_nolabel->getInnerSplit_Label();

							score = labelscore
								+ right_nolabel->getScore()
								+ lrl_score
								+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label));
							if (maxswbs < score)
								maxswbs = ScoreWithBiSplit(lsplit, rsplit, score, l2r_la, lsplit_label, rsplit_label);
						}
						for (int split = l + 1; split < r; split++)
							for (auto left_label : m_lItems[l][split].l2r_label)
								for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
								{
									int lsplit = split;
									int lsplit_label = left_label->getLabel();
									int rsplit = right_nolabel->getInnerSplit();
									int rsplit_label = right_nolabel->getInnerSplit_Label();
									int gclsplit = l;
									int gclsplit_label = left_label->getLabel();
									int gcrsplit = right_nolabel->getSplit();
									int gcrsplit_label = right_nolabel->getSplit_Label();

									score = labelscore
										+ left_label->getScore()
										+ right_nolabel->getScore()
										+ arcScore_sibling(l, r, l2r_la, lsplit, lsplit_label)
										+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label))
										+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);
									if (maxswbs < score)
										maxswbs = ScoreWithBiSplit(lsplit, rsplit, score, l2r_la, lsplit_label, rsplit_label);
								}
						item.updateL2R_LABEL(maxswbs);
					}

				}

				//dinghu
				for (auto l2r_label : item.l2r_label)
				item.updateL2R_NOLABEL(r,l,l2r_label->getScore(),-1,l2r_label->getLabel(), l2r_label->getLabel());
				map_LsLlRsRlScore.clear();
				if (d == 1)
				{
					item.updateL2R_NOLABEL(l, r, 0, -1, -1, -1);
				}
				else
				{
					for (auto right_nolabel : m_lItems[l + 1][r].l2r_nolabel)
					{
						int lsplit = l;
						int lsplit_label = -1;
						int rsplit = right_nolabel->getInnerSplit();
						int rsplit_label = right_nolabel->getInnerSplit_Label();

						score = right_nolabel->getScore();

						LsLlRsRl tmp = LsLlRsRl(lsplit, lsplit_label, rsplit, rsplit_label);
						if (map_LsLlRsRlScore.find(tmp) == map_LsLlRsRlScore.end())
							map_LsLlRsRlScore[tmp] = score;
						else map_LsLlRsRlScore[tmp] = std::max(score, map_LsLlRsRlScore[tmp]);
					}

					for (int split = l + 1; split < r; split++)
						for (auto left_label : m_lItems[l][split].l2r_label)
							for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
							{
								int lsplit = split;
								int lsplit_label = left_label->getLabel();
								int rsplit = right_nolabel->getInnerSplit();
								int rsplit_label = right_nolabel->getInnerSplit_Label();
								int gclsplit = l;
								int gclsplit_label = left_label->getLabel();
								int gcrsplit = right_nolabel->getSplit();
								int gcrsplit_label = right_nolabel->getSplit_Label();

								score =  left_label->getScore()
									+ right_nolabel->getScore()
									+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);

								LsLlRsRl tmp = LsLlRsRl(lsplit, lsplit_label, rsplit, rsplit_label);
								if (map_LsLlRsRlScore.find(tmp) == map_LsLlRsRlScore.end())
									map_LsLlRsRlScore[tmp] = score;
								else map_LsLlRsRlScore[tmp] = std::max(score, map_LsLlRsRlScore[tmp]);

							}
					for (auto lsllrsrl_itr : map_LsLlRsRlScore)
					{
						LsLlRsRl lsllrsrl = lsllrsrl_itr.first;
						item.updateL2R_NOLABEL(lsllrsrl.first(), lsllrsrl.third(), lsllrsrl_itr.second, -1, lsllrsrl.second(), lsllrsrl.forth());
					}
				}
			}
		}
	}

	void DepParser::decode() {
		int sumlen = m_nSentenceLength;
		int labelsize = DepLabel.count();
		m_lcaTree.loadPath(m_vecDependencyTree);
		InitScore_single();
		//InitScore_sibling();
		//InitScore_gc();
		Score_sibling.clear();
		Score_gc.clear();
		std::unordered_map<LsLlRsRl, tscore> map_LsLlRsRlScore;

		for (int d = 1; d < sumlen; d++)
		{
			for (int l = 1; l <= sumlen - d; l++)
			{
				int r = l + d;

				tscore score;
				StateItem  & item = m_lItems[l][r];
				item.init(l, r);
				LabelScoreAgenda l2r_label_beam;
				l2r_label_beam.clear();
				for (int l2r_la = RIGHT_ARC(1); l2r_la <= LEFT_ARC(labelsize); l2r_la++)
					if (Score_single[l][r][l2r_la] > HALF_MIN_SCORE)
						l2r_label_beam.insertItem(LabelScore(l2r_la, Score_single[l][r][l2r_la]));

				for ( auto l2r_label_iterator : l2r_label_beam)
				{
					int l2r_la = l2r_label_iterator->first();
					int r2l_la = NEG(l2r_la);
					tscore labelscore = l2r_label_iterator->second();
					tscore lrl_score = arcScore_sibling(l, r, l2r_la, l, -1);
					if (d == 1)
					{
						item.updateL2R_LABEL(l, r, labelscore + lrl_score + arcScore_sibling(r, l, NEG(l2r_la),r,-1),
							l2r_la ,-1, -1);
					}
					else
					{
						ScoreWithBiSplit maxswbs;
						maxswbs.refer(l, r, MIN_SCORE);

						for (auto right_nolabel : m_lItems[l+1][r].l2r_nolabel)
						{
							int lsplit = l;
							int lsplit_label = -1;
							int rsplit = right_nolabel->getInnerSplit();
							int rsplit_label = right_nolabel->getInnerSplit_Label();

							score = labelscore
								+ right_nolabel->getScore()
								+ lrl_score
								+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label));
							if (maxswbs < score)
								maxswbs = ScoreWithBiSplit(lsplit, rsplit, score, l2r_la, lsplit_label, rsplit_label);
						}
						for (int split = l + 1; split < r; split++)
							for (auto left_label : m_lItems[l][split].l2r_label)
								for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
								{
									int lsplit = split;
									int lsplit_label = left_label->getLabel();
									int rsplit = right_nolabel->getInnerSplit();
									int rsplit_label = right_nolabel->getInnerSplit_Label();
									int gclsplit = l;
									int gclsplit_label = left_label->getLabel();
									int gcrsplit = right_nolabel->getSplit();
									int gcrsplit_label = right_nolabel->getSplit_Label();

									score = labelscore
										+ left_label->getScore()
										+ right_nolabel->getScore()
										+ arcScore_sibling(l, r, l2r_la, lsplit, lsplit_label)
										+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label))
										+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);
									if (maxswbs < score)
										maxswbs = ScoreWithBiSplit(lsplit, rsplit, score, l2r_la, lsplit_label, rsplit_label);
								}
						item.updateL2R_LABEL(maxswbs);
					}

				}

				//dinghu
				for (auto l2r_label : item.l2r_label)
				item.updateL2R_NOLABEL(r,l,l2r_label->getScore(),-1,l2r_label->getLabel(), l2r_label->getLabel());
				map_LsLlRsRlScore.clear();
				if (d == 1)
				{
					item.updateL2R_NOLABEL(l, r, 0, -1, -1, -1);
				}
				else
				{
					for (auto right_nolabel : m_lItems[l + 1][r].l2r_nolabel)
					{
						int lsplit = l;
						int lsplit_label = -1;
						int rsplit = right_nolabel->getInnerSplit();
						int rsplit_label = right_nolabel->getInnerSplit_Label();

						score = right_nolabel->getScore();

						LsLlRsRl tmp = LsLlRsRl(lsplit, lsplit_label, rsplit, rsplit_label);
						if (map_LsLlRsRlScore.find(tmp) == map_LsLlRsRlScore.end())
							map_LsLlRsRlScore[tmp] = score;
						else map_LsLlRsRlScore[tmp] = std::max(score, map_LsLlRsRlScore[tmp]);
					}

					for (int split = l + 1; split < r; split++)
						for (auto left_label : m_lItems[l][split].l2r_label)
							for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
							{
								int lsplit = split;
								int lsplit_label = left_label->getLabel();
								int rsplit = right_nolabel->getInnerSplit();
								int rsplit_label = right_nolabel->getInnerSplit_Label();
								int gclsplit = l;
								int gclsplit_label = left_label->getLabel();
								int gcrsplit = right_nolabel->getSplit();
								int gcrsplit_label = right_nolabel->getSplit_Label();

								score =  left_label->getScore()
									+ right_nolabel->getScore()
									+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);

								LsLlRsRl tmp = LsLlRsRl(lsplit, lsplit_label, rsplit, rsplit_label);
								if (map_LsLlRsRlScore.find(tmp) == map_LsLlRsRlScore.end())
									map_LsLlRsRlScore[tmp] = score;
								else map_LsLlRsRlScore[tmp] = std::max(score, map_LsLlRsRlScore[tmp]);

							}
					for (auto lsllrsrl_itr : map_LsLlRsRlScore)
					{
						LsLlRsRl lsllrsrl = lsllrsrl_itr.first;
						item.updateL2R_NOLABEL(lsllrsrl.first(), lsllrsrl.third(), lsllrsrl_itr.second, -1, lsllrsrl.second(), lsllrsrl.forth());
					}
				}
			}
		}
	}

	void DepParser::decodeArcs()
	{
		m_vecTrainArcs.clear();
		m_lItems[1][m_nSentenceLength].type = L2R_NOLABEL;
		m_lItems[1][m_nSentenceLength].exact_l2r_nolabel = m_lItems[1][m_nSentenceLength].l2r_nolabel.bestItem();
		//std::cout <<"EDGE "<< 1.0 * next_l2r_nolabel[1][m_nSentenceLength].getScore() / GOLD_POS_SCORE << std::endl;
		getArc(1, m_nSentenceLength);
	}

	void DepParser::getArc(int l, int r)
	{
		if (l >= r) return;
		int label;
		tscore score;
		ScoreWithBiSplit l2r_nolabel, l2r_label;
		StateItem &item = m_lItems[l][r];
		//system("pause");
		switch (item.type) {
		case L2R_NOLABEL:
			l2r_nolabel = item.exact_l2r_nolabel;
			//std::cout << "l2r_nolabel  "<<l << ' '<< r <<' '<< l2r_nolabel.getSplit() <<' '<< l2r_nolabel.getInnerSplit() << std::endl;

			if (l2r_nolabel.getSplit() == r && l2r_nolabel.getInnerSplit() == l )
			{
				m_lItems[l][r].type = L2R_LABEL;
				for (auto exact_l2r_label : item.l2r_label)
					if (exact_l2r_label->getScore() == l2r_nolabel.getScore())
					{
						m_lItems[l][r].exact_l2r_label = item.l2r_label.bestItem();
						getArc(l, r);
						return;
					}
			}
			else
			{
				if (l2r_nolabel.getSplit() == l)
				{
					for (auto right_nolabel : m_lItems[l + 1][r].l2r_nolabel)
					{
						score = right_nolabel->getScore();

						if (score == l2r_nolabel.getScore())
						{
							m_lItems[l + 1][r].type = L2R_NOLABEL;
							m_lItems[l + 1][r].exact_l2r_nolabel = *right_nolabel;
							getArc(l + 1, r);
							return;
						}
					}
				}
				else
				{
					int split = l2r_nolabel.getSplit();
					for (auto left_label : m_lItems[l][split].l2r_label)
						for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
						{
							int lsplit = split;
							int lsplit_label = left_label->getLabel();
							int rsplit = right_nolabel->getInnerSplit();
							int rsplit_label = right_nolabel->getInnerSplit_Label();
							int gclsplit = l;
							int gclsplit_label = left_label->getLabel();
							int gcrsplit = right_nolabel->getSplit();
							int gcrsplit_label = right_nolabel->getSplit_Label();

							score = left_label->getScore()
								+ right_nolabel->getScore()
								+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);

							if (score == l2r_nolabel.getScore())
							{
								m_lItems[l][split].type = L2R_LABEL;
								m_lItems[l][split].exact_l2r_label = *left_label;
								getArc(l, split);

								m_lItems[split][r].type = L2R_NOLABEL;
								m_lItems[split][r].exact_l2r_nolabel = *right_nolabel;
								getArc(split, r);
								return;
							}
						}
				}
			}
			break;
		case L2R_LABEL:
			if (l >= r) return;
			l2r_label = m_lItems[l][r].exact_l2r_label;
			int label = l2r_label.getLabel();
			//std::cout << "l2r_label  " << l << ' ' << r << ' ' <<label<<' '<< l2r_label.getSplit() << ' ' << l2r_label.getInnerSplit() << std::endl;
			//std::cout << "push arc!" << std::endl;
			if ( IS_RIGHT_ARC(label) )
			{
				m_vecTrainArcs.push_back(PCLArc(l, r, ARC(label)));
				//std::cout << " push arc " << l << ' ' << r << ' ' << ARC(label) << std::endl;
			}
			else if ( IS_LEFT_ARC(label) )
			{
				m_vecTrainArcs.push_back(PCLArc(r, l, ARC(label)));
				//std::cout << " push arc " << r << ' ' << l << ' ' << ARC(label) << std::endl;
			}

			//std::cout << "l2r_label  " << l << ' ' << lsp<< ' '<<rsp<<' ' << r << ' '<<l2r_label<<' '<<lla<<' '<<rla<<std::endl;

			//std::cout << "push  " << r << ' ' << sp << std::endl;

			int l2r_la = label;
			int r2l_la = NEG(l2r_la);
			tscore labelscore = Score_single[l][r][l2r_la];

			if (l2r_label.getSplit() == l)
			{
				tscore lrl_score = arcScore_sibling(l, r, l2r_la, l, -1);
				for (auto right_nolabel : m_lItems[l + 1][r].l2r_nolabel)
				{
					int lsplit = l;
					int lsplit_label = -1;
					int rsplit = right_nolabel->getInnerSplit();
					int rsplit_label = right_nolabel->getInnerSplit_Label();

					score = labelscore
						+ right_nolabel->getScore()
						+ lrl_score
						+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label));

					if (score == l2r_label.getScore())
					{
						m_lItems[l+1][r].type = L2R_NOLABEL;
						m_lItems[l+1][r].exact_l2r_nolabel = *right_nolabel;
						getArc(l+1, r);
						return;
					}
				}
			}
			else
			{
				for (int split = l + 1; split < r; split++)
					for (auto left_label : m_lItems[l][split].l2r_label)
						for (auto right_nolabel : m_lItems[split][r].l2r_nolabel)
						{
							int lsplit = split;
							int lsplit_label = left_label->getLabel();
							int rsplit = right_nolabel->getInnerSplit();
							int rsplit_label = right_nolabel->getInnerSplit_Label();
							int gclsplit = l;
							int gclsplit_label = left_label->getLabel();
							int gcrsplit = right_nolabel->getSplit();
							int gcrsplit_label = right_nolabel->getSplit_Label();

							score = labelscore
								+ left_label->getScore()
								+ right_nolabel->getScore()
								+ arcScore_sibling(l, r, l2r_la, lsplit, lsplit_label)
								+ arcScore_sibling(r, l, r2l_la, rsplit, NEG(rsplit_label))
								+ arcScore_gc(split, gclsplit, NEG(gclsplit_label), gcrsplit, gcrsplit_label);
							if (score == l2r_label.getScore())
							{
								m_lItems[l][split].type = L2R_LABEL;
								m_lItems[l][split].exact_l2r_label = *left_label;
								getArc(l, split);

								m_lItems[split][r].type = L2R_NOLABEL;
								m_lItems[split][r].exact_l2r_nolabel = *right_nolabel;
								getArc(split, r);
								return;
							}
						}
			}


			break;
		}
	}

	void DepParser::update_sibling()
	{
		std::vector<PCLArc> corarc1, myarc1;
		std::vector<PC1L1C2L2Arc> corarc2, myarc2;
		corarc1.clear();
		myarc1.clear();
		corarc2.clear();
		myarc2.clear();
		for (auto arc : m_vecCorrectArcs)
		{
			int l = arc.first();
			int r = arc.second();
			int label = RIGHT_ARC(arc.third());
			if (l > r) std::swap(l, r), label = NEG(label);
			corarc1.push_back(PCLArc(l, r, label));
		}
		for (auto arc : m_vecTrainArcs)
		{
			int l = arc.first();
			int r = arc.second();
			int label = RIGHT_ARC(arc.third());
			if (l > r) std::swap(l, r), label = NEG(label);
			myarc1.push_back(PCLArc(l, r, label));
		}

		for (auto arc1 : corarc1)
		{
			int rsp = arc1.second();
			int rla = -1;
			int lsp = arc1.first();
			int lla = -1;
			for (auto arc2 : corarc1)
			{
				if (arc2.first() == arc1.first() && arc2.second() < arc1.second() && lsp < arc2.second())
				{
					lsp = arc2.second();
					lla = arc2.third();
				}
				if (arc2.second() == arc1.second() && arc2.first() > arc1.first() && rsp > arc2.first())
				{
					rsp = arc2.first();
					rla = NEG(arc2.third());
				}

			}

			corarc2.push_back(PC1L1C2L2Arc(arc1.first(), arc1.second(), arc1.third(), lsp, lla));
			corarc2.push_back(PC1L1C2L2Arc(arc1.second(), arc1.first(), NEG(arc1.third()), rsp, rla));
		}

		for (auto arc1 : myarc1)
		{
			int rsp = arc1.second();
			int rla = -1;
			int lsp = arc1.first();
			int lla = -1;
			for (auto arc2 : myarc1)
			{
				if (arc2.first() == arc1.first() && arc2.second() < arc1.second() && lsp < arc2.second())
				{
					lsp = arc2.second();
					lla = arc2.third();
				}
				if (arc2.second() == arc1.second() && arc2.first() > arc1.first() && rsp > arc2.first())
				{
					rsp = arc2.first();
					rla = NEG(arc2.third());
				}
			}
			myarc2.push_back(PC1L1C2L2Arc(arc1.first(), arc1.second(), arc1.third(), lsp, lla));
			myarc2.push_back(PC1L1C2L2Arc(arc1.second(), arc1.first(), NEG(arc1.third()), rsp, rla));
		}

		std::unordered_set<PC1L1C2L2Arc> positivearc2set, negativearc2set;
		positivearc2set.clear();
		negativearc2set.clear();
		positivearc2set.insert(corarc2.begin(), corarc2.end());
		negativearc2set.insert(myarc2.begin(), myarc2.end());

		for (const auto & arc : myarc2)
			positivearc2set.erase(arc);
		for (const auto & arc : corarc2)
			negativearc2set.erase(arc);

		int p, c1, c2, l1, l2;
		for (const auto & arc : positivearc2set) {
			p = arc.first();
			c1 = arc.second();
			l1 = arc.third();
			c2 = arc.forth();
			l2 = arc.fifth();
			//if (c1 >= p) std::swap(c1, c2), std::swap(l1, l2);
			getOrUpdateStackScore_sibling(p, c1, l1, c2, l2, 1);
		}
		for (const auto & arc : negativearc2set) {
			p = arc.first();
			c1 = arc.second();
			l1 = arc.third();
			c2 = arc.forth();
			l2 = arc.fifth();
			//if (c1 >= p) std::swap(c1, c2), std::swap(l1, l2);
			getOrUpdateStackScore_sibling(p, c1, l1, c2, l2, -1);
		}
	}

	void DepParser::update_gc()
	{
		std::vector<PCLArc> corarc1, myarc1;
		std::vector<PC1L1C2L2Arc> corarc2, myarc2;
		corarc1.clear();
		myarc1.clear();
		corarc2.clear();
		myarc2.clear();
		for (auto arc : m_vecCorrectArcs)
		{
			int l = arc.first();
			int r = arc.second();
			int label = RIGHT_ARC(arc.third());
			if (l > r) std::swap(l, r), label = NEG(label);
			corarc1.push_back(PCLArc(l, r, label));
		}
		for (auto arc : m_vecTrainArcs)
		{
			int l = arc.first();
			int r = arc.second();
			int label = RIGHT_ARC(arc.third());
			if (l > r) std::swap(l, r), label = NEG(label);
			myarc1.push_back(PCLArc(l, r, label));
		}

		int n = m_nSentenceLength;
		for (int i = 1; i <= n; i++)
		{
			int l = i;
			int lla = -1;
			int r = i;
			int rla = -1;
			for (auto arc : corarc1)
			{
				if (arc.first() == i && arc.second() > r )
				{
					r = arc.second();
					rla = arc.third();
				}
				if (arc.second() == i && arc.first() < l)
				{
					l = arc.first();
					lla = NEG(arc.third());
				}

			}
			if (l != i && r != i)
			corarc2.push_back(PC1L1C2L2Arc(i, l, lla, r, rla));
		}

		for (int i = 1; i <= n; i++)
		{
			int l = i;
			int lla = -1;
			int r = i;
			int rla = -1;
			for (auto arc : myarc1)
			{
				if (arc.first() == i && arc.second() > r)
				{
					r = arc.second();
					rla = arc.third();
				}
				if (arc.second() == i && arc.first() < l)
				{
					l = arc.first();
					lla = NEG(arc.third());
				}

			}
			if (l != i && r != i)
				myarc2.push_back(PC1L1C2L2Arc(i, l, lla, r, rla));
		}

		std::unordered_set<PC1L1C2L2Arc> positivearc2set, negativearc2set;
		positivearc2set.clear();
		negativearc2set.clear();
		positivearc2set.insert(corarc2.begin(), corarc2.end());
		negativearc2set.insert(myarc2.begin(), myarc2.end());

		for (const auto & arc : myarc2)
			positivearc2set.erase(arc);
		for (const auto & arc : corarc2)
			negativearc2set.erase(arc);

		int p, c1, c2, l1, l2;
		for (const auto & arc : positivearc2set) {
			p = arc.first();
			c1 = arc.second();
			l1 = arc.third();
			c2 = arc.forth();
			l2 = arc.fifth();
			//if (c1 >= p) std::swap(c1, c2), std::swap(l1, l2);
			getOrUpdateStackScore_gc(p, c1, l1, c2, l2, 1);
		}
		for (const auto & arc : negativearc2set) {
			p = arc.first();
			c1 = arc.second();
			l1 = arc.third();
			c2 = arc.forth();
			l2 = arc.fifth();
			//if (c1 >= p) std::swap(c1, c2), std::swap(l1, l2);
			getOrUpdateStackScore_gc(p, c1, l1, c2, l2, -1);
		}

	}

	void DepParser::update() {
		std::unordered_set<PCLArc> positiveArcs;
		std::unordered_set<PCLArc> negativeArcs;

		positiveArcs.clear();
		for (const auto arc : m_vecCorrectArcs)
			positiveArcs.insert(PCLArc(arc.first(), arc.second(), 0));
		for (const auto arc : m_vecTrainArcs)
		{
			if (positiveArcs.find(PCLArc(arc.first(), arc.second(), 0)) != positiveArcs.end())
				totaledgecor++;
			positiveArcs.erase(PCLArc(arc.first(), arc.second(), 0));
		}
		positiveArcs.clear();
		negativeArcs.clear();
		positiveArcs.insert(m_vecCorrectArcs.begin(), m_vecCorrectArcs.end());
		total += positiveArcs.size();
		for (const auto & arc : m_vecTrainArcs) {
			if (positiveArcs.find(arc) != positiveArcs.end()) totalcor++;
			//std::cout << arc.first() << ' ' << arc.second() << std::endl;
			positiveArcs.erase(arc);
		}
		negativeArcs.insert(m_vecTrainArcs.begin(), m_vecTrainArcs.end());
		totalmy += negativeArcs.size();
		for (const auto & arc : m_vecCorrectArcs) {
			negativeArcs.erase(arc);
		}
		if (!positiveArcs.empty() || !negativeArcs.empty()) {
			++m_nTotalErrors;
		}
		if (m_nState == ParserState::TRAIN)
		{
			//std::cout << m_nSentenceLength <<' '<<DepLabel.count() << std::endl;
			for (const auto & arc : positiveArcs) {
				//std::cout << "pos " << arc << std::endl;
				getOrUpdateStackScore_single(arc.first() , arc.second(), arc.third(),1);
			}
			for (const auto & arc : negativeArcs) {
				//std::cout << "neg " << arc << std::endl;
				getOrUpdateStackScore_single(arc.first() , arc.second(), arc.third(),-1);
			}
			//std::cout << "------------------" << std::endl;

			update_sibling();
			update_gc();
		}
	}

	void DepParser::generate(DependencyGraph * retval, const DependencyGraph & correct) {
		for (const auto & arc : m_vecTrainArcs) {
			retval->m_lGraph.push_back(GraphEdge(arc.first(), arc.second(), DepLabel.key(arc.third())));
		}
	}

	void DepParser::goldCheck() {
		/*if (m_vecCorrectArcs.size() != m_vecTrainArcs.size() || m_lItems[m_nSentenceLength + 1][0].l2r_nolabel.bestItem().getScore() / GOLD_POS_SCORE != m_vecTrainArcs.size()) {
			std::cout << "gold parse len error at " << m_nTrainingRound << std::endl;
			std::cout << "score is " << m_lItems[0][m_nSentenceLength].l2r_nolabel.bestItem().getScore() << std::endl;
			std::cout << "len is " << m_vecTrainArcs.size() << std::endl;
			++m_nTotalErrors;
		}
		else {
			int i = 0;
			std::sort(m_vecCorrectArcs.begin(), m_vecCorrectArcs.end(), [](const Arc & arc1, const Arc & arc2){ return arc1 < arc2; });
			std::sort(m_vecTrainArcs.begin(), m_vecTrainArcs.end(), [](const Arc & arc1, const Arc & arc2){ return arc1 < arc2; });
			for (int n = m_vecCorrectArcs.size(); i < n; ++i) {
				if (m_vecCorrectArcs[i].first() != m_vecTrainArcs[i].first() || m_vecCorrectArcs[i].second() != m_vecTrainArcs[i].second()) {
					break;
				}
			}
			if (i != m_vecCorrectArcs.size()) {
				std::cout << "gold parse tree error at " << m_nTrainingRound << std::endl;
				++m_nTotalErrors;
			}
		}*/
	}

	tscore DepParser::arcScore_single(const int & p, const int & c , const int & label) {

		//return 0;
		if (debug) {
			m_nRetval = m_setFirstGoldScore.find(PCLArc(p, c , label)) == m_setFirstGoldScore.end() ? GOLD_NEG_SCORE : GOLD_POS_SCORE;
			return m_nRetval;
		}
		m_nRetval = 0;
		getOrUpdateStackScore_single(p, c , label);
		return m_nRetval;
	}
	tscore DepParser::arcScore_sibling(const int & p, const int & c, const int & la1, const int& mid, const int & la2) {

			//return 0;
			if (debug) {
				m_nRetval = m_setSecondGoldScore.find(PC1L1C2L2Arc(p,c,la1,mid,la2)) == m_setSecondGoldScore.end() ? GOLD_NEG_SCORE : GOLD_POS_SCORE;
				return m_nRetval;
			}
			int cla1, cla2;
			cla1 = la1 == -1 ? 0 : la1;
			cla2 = la2 == -1 ? 0 : la2;
			PC1L1C2L2Arc key(p,c,cla1,mid,cla2);
			if (Score_sibling.find(key) != Score_sibling.end())
				return Score_sibling[key];
			//return 0;
			m_nRetval = 0;
			getOrUpdateStackScore_sibling(p, c, la1, mid, la2);
			return  Score_sibling[key] = m_nRetval;
			//return m_nRetval;
		}
	tscore DepParser::arcScore_gc(const int & mid, const int & left, const int & lla, const int& right, const int & rla) {

		//return 0;
		if (debug) {
			m_nRetval = m_setGCGoldScore.find(PC1L1C2L2Arc(mid, left, lla, right, rla)) == m_setGCGoldScore.end() ? GOLD_NEG_SCORE : GOLD_POS_SCORE;
			return m_nRetval;
		}
		if (lla == -1 || rla == -1) return 0;
		PC1L1C2L2Arc key(mid, left, lla, right, rla);
		if (Score_gc.find(key) != Score_gc.end())
			return Score_gc[key];
		//return 0;
		m_nRetval = 0;
		getOrUpdateStackScore_gc(mid, left, lla, right, rla);
		return Score_gc[key] = m_nRetval;
		//return m_nRetval;
	}

	void DepParser::InitScore_single() {
		int sumlen = m_nSentenceLength;
		int labelsize = DepLabel.count();
		for (int i = 1; i <= sumlen; i++)
			for (int j = 1; j <= sumlen; j++)
			{
				int p = m_lSentence[i].second();
				int c = m_lSentence[j].second();
				for (int label = 1; label <= labelsize; label++)
					if (i != j && DepLabel.findPPL(TriGram<int>(p, c, label)))
						Score_single[i][j][RIGHT_ARC(label)] = arcScore_single(i, j, label);
					else Score_single[i][j][RIGHT_ARC(label)] = MIN_SCORE;
			}

		for (int i = 1; i <= sumlen; i++)
			for (int j = 1; j <= sumlen; j++)
				for (int label = 1; label <= labelsize; label++)
				Score_single[i][j][LEFT_ARC(label)] = Score_single[j][i][RIGHT_ARC(label)];

		//for (int i = 1; i <= sumlen; i++)
		//	for (int j = 1; j <= sumlen; j++)
			//	for (int label = 1; label <= labelsize; label++)
			//		std::cout << i << ' ' << j << ' ' << label << ' ' << Score_single[i][j][RIGHT_ARC(label)] << std::endl;
	}
	/*
	void DepParser::InitScore_sibling() {
			int labelsize = DepLabel.count();
			int sumlen = m_nSentenceLength;
			for (int i = 1; i <= sumlen; i++)
				for (int j = 1; j <= sumlen; j++)
					for (int ij = 0; ij <= LEFT_ARC(labelsize); ij++)
						for (int k = 1; k <= sumlen; k++)
							for (int ik = 0; ik <= LEFT_ARC(labelsize); ik++)
								Score_sibling[i][j][ij][k][ik] = MIN_SCORE;
		}

		void DepParser::InitScore_gc() {
			int sumlen = m_nSentenceLength;
			int labelsize = DepLabel.count();
			for (int i = 1; i <= sumlen; i++)
				for (int j = 1; j <= sumlen; j++)
					for (int ij = 0; ij <= LEFT_ARC(labelsize); ij++)
						for (int k = 1; k <= sumlen; k++)
							for (int ik = 0; ik <= LEFT_ARC(labelsize); ik++)
								Score_gc[i][j][ij][k][ik] = MIN_SCORE;
		}
*/
	void DepParser::getOrUpdateStackScore_single(const int & p, const int & c,const int & label,const int & amount) {
		fsum++;
		Weight * cweight = (Weight*)m_Weight;

		p_1_tag = p - 1 >= 1 ? m_lSentence[p - 1].second() : start_taggedword.second();
		p1_tag = p + 1 <= m_nSentenceLength ? m_lSentence[p + 1].second() : end_taggedword.second();
		c_1_tag = c - 1 >= 1 ? m_lSentence[c - 1].second() : start_taggedword.second();
		c1_tag = c + 1 <= m_nSentenceLength ? m_lSentence[c + 1].second() : end_taggedword.second();

		p_1_word = p - 1 >= 1 ? m_lSentence[p - 1].first() : start_taggedword.first();
		p1_word = p + 1 <= m_nSentenceLength ? m_lSentence[p + 1].first() : end_taggedword.first();
		c_1_word = c - 1 >= 1 ? m_lSentence[c - 1].first() : start_taggedword.first();
		c1_word = c + 1 <= m_nSentenceLength ? m_lSentence[c + 1].first() : end_taggedword.first();

		p_word = m_lSentence[p].first();
		p_tag = m_lSentence[p].second();

		c_word = m_lSentence[c].first();
		c_tag = m_lSentence[c].second();

		int m_nLabel = label * 1000;
		m_nDis = m_nLabel + encodeLinkDistanceOrDirection(p, c, false);
		m_nDir = m_nLabel + encodeLinkDistanceOrDirection(p, c, true);


		//P
		word_int.refer(p_word, m_nLabel);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p_tag, m_nLabel);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, p_tag, m_nLabel);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C
		word_int.refer(c_word, m_nLabel);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c_tag, m_nLabel);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c_word, c_tag, m_nLabel);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1
		word_int.refer(p_1_word, m_nLabel);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p_1_tag, m_nLabel);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_1_word, p_1_tag, m_nLabel);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P1
		word_int.refer(p1_word, m_nLabel);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p1_tag, m_nLabel);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p1_word, p1_tag, m_nLabel);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C_1
		word_int.refer(c_1_word, m_nLabel);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c_1_tag, m_nLabel);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c_1_word, c_1_tag, m_nLabel);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C1
		word_int.refer(c1_word, m_nLabel);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c1_tag, m_nLabel);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c1_tag, m_nLabel);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_tag_int.refer(p_word, c_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_tag_int.refer(p_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_tag_int.refer(c_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_int.refer(p_word, c_word, p_tag, m_nLabel);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_int.refer(p_word, c_word, c_tag, m_nLabel);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDis);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDir);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(p_word, c_word, m_nLabel);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(p_tag, c_tag, m_nLabel);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//window_feat

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_1_tag, c_tag, m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, empty_taggedword.second(), c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, empty_taggedword.second(), c_tag, c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_1_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_1_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, empty_taggedword.second(), c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, empty_taggedword.second(), c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, empty_taggedword.second(), c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1C
		flag_tag_tag_int.refer(0,p_1_tag,c_tag,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_1_word,c_tag,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_1_tag,c_word,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_1_word,c_word,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);


		//P1C
		flag_tag_tag_int.refer(0,p1_tag,c_tag,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p1_word,c_tag,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p1_tag,c_word,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p1_word,c_word,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC_1
		flag_tag_tag_int.refer(0,p_tag,c_1_tag,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_word,c_1_tag,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_tag,c_1_word,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_word,c_1_word,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC1
		flag_tag_tag_int.refer(0,p_tag,c1_tag,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_word,c1_tag,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_tag,c1_word,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_word,c1_word,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1PC
		flag_word_tag_tag_int.refer(0,p_1_word,p_tag,c_tag,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_1_tag,p_word,c_tag,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_1_tag,p_tag,c_word,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PP1C
		flag_word_tag_tag_int.refer(0,p_word,p1_tag,c_tag,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,p1_word,c_tag,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis+0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir+0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,p1_tag,c_word,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis+2*0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir+2*0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC_1C
		flag_word_tag_tag_int.refer(0,p_word,c_1_tag,c_tag,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,c_1_word,c_tag,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,c_1_tag,c_word,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PCC1
		flag_word_tag_tag_int.refer(0,p_word,c_tag,c1_tag,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,c_word,c1_tag,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,c_tag,c1_word,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		for (int b = (int)std::fmin(p, c) + 1, e = (int)std::fmax(p, c); b < e; ++b) {
			b_tag = m_lSentence[b].second();
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, 0);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, m_nDis);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, m_nDir);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		}

		std::string path = "PATH=" + m_lcaTree.POSPath[p][c];
		path_int.refer(TPOSTag::code(path), 0);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDir);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDis);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string fpath = "FPATH=" + m_lcaTree.FPOSPath[p][c];
		path_int.refer(TPOSTag::code(fpath), 0);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDir);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDis);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);

		fpath = "UpFPATH=" + m_lcaTree.UpFPOSPath[p][c];
		path_tag_int.refer(TPOSTag::code(fpath),c_word, 0);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDir);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDis);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		fpath = "DownFPATH=" + m_lcaTree.DownFPOSPath[p][c];
		path_tag_int.refer(TPOSTag::code(fpath),p_word, 0);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDir);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDis);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string lpath = "LPATH=" + m_lcaTree.LABELPath[p][c];
		path_int.refer(TPOSTag::code(lpath), m_nLabel);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast(m_nDir);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast(m_nDis);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string subcatp = "subcat=" + TREENODE_POSTAG(m_vecDependencyTree[p]);
		for (auto & child : m_lcaTree.Children[p])
			subcatp+=TREENODE_POSTAG(m_vecDependencyTree[child]);
		subcat_int.refer(TPOSTag::code(subcatp), m_nLabel);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDir);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDis);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.refer(TPOSTag::code(subcatp),c_tag,m_nLabel);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDir);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDis);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string subcatc = "subcat=" + TREENODE_POSTAG(m_vecDependencyTree[c]);
		for (auto & child : m_lcaTree.Children[c])
			subcatc+=TREENODE_POSTAG(m_vecDependencyTree[child]);
		subcat_int.refer(TPOSTag::code(subcatc), m_nLabel);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDir);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDis);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.refer(TPOSTag::code(subcatc),p_tag,m_nLabel);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDir);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDis);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		int edgeflag = TREENODE_HEAD(m_vecDependencyTree[c]) == p ? 1 :
                               (TREENODE_HEAD(m_vecDependencyTree[p]) == c ? -1 : 0);
		if (edgeflag != 0){
			path_int.refer(edgeflag, m_nLabel);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
			path_int.referLast(m_nDir);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
			path_int.referLast(m_nDis);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		}


		int clabel = TPOSTag::code("CLABEL=" + TREENODE_LABEL(m_vecDependencyTree[c]));
		label_int.refer(clabel, m_nLabel);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);
		label_int.referLast(m_nDir);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);
		label_int.referLast(m_nDis);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);


		int f = TREENODE_HEAD(m_vecDependencyTree[c]);
		int lc = m_lcaTree.Children[c].empty() ? -1 : m_lcaTree.Children[c].front();
		int rc = m_lcaTree.Children[c].empty() ? -1 : m_lcaTree.Children[c].back();
		int cf = 0;
		for (int i = 0; i < m_lcaTree.Children[f].size(); i++)
			if (m_lcaTree.Children[f][i] == c)
			{
				cf = i;
				break;
			}
		int ls = cf == 0 ? -1 : m_lcaTree.Children[f][cf - 1];
		int rs = cf + 1 == m_lcaTree.Children[f].size() ? -1 : m_lcaTree.Children[f][cf + 1];
		int lw=m_lcaTree.Range[c].first(),rw = m_lcaTree.Range[c].second();

		int lw_tag = m_lSentence[lw].second(),rw_tag = m_lSentence[rw].second();
		int lw_word = m_lSentence[lw].first(),rw_word = m_lSentence[rw].first();
		int lw_1_tag = lw -1 >= 1? m_lSentence[lw-1].second() : start_taggedword.second();
		int rw1_tag = lw + 1 <= m_nSentenceLength ? m_lSentence[rw+1].second() : end_taggedword.second();
		int lw_1_word = lw -1 >= 1? m_lSentence[lw-1].first() : start_taggedword.first();
		int rw1_word = lw + 1 <= m_nSentenceLength ? m_lSentence[rw+1].first() : end_taggedword.first();

		int lc_tag = lc == -1 ? start_taggedword.second() : m_lSentence[lc].second();
		int lc_word = lc == -1 ? start_taggedword.first() : m_lSentence[lc].first();
		int rc_tag = rc == -1 ? end_taggedword.second() : m_lSentence[rc].second();
		int rc_word = rc == -1 ? end_taggedword.first() : m_lSentence[rc].first();

		int ls_tag = ls == -1 ? start_taggedword.second() : m_lSentence[ls].second();
		int ls_word = ls == -1 ? start_taggedword.first() : m_lSentence[ls].first();
		int rs_tag = rs == -1 ? end_taggedword.second() : m_lSentence[rs].second();
		int rs_word = rs == -1 ? end_taggedword.first() : m_lSentence[rs].first();

		//lw
		word_tag_int.refer(lw_word,0,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,lw_tag,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(lw_word,lw_tag,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//rw
		word_tag_int.refer(rw_word,0,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,rw_tag,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(rw_word,rw_tag,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//lw_1
		word_tag_int.refer(lw_1_word,0,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,lw_1_tag,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(lw_1_word,lw_1_tag,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//rw1
		word_tag_int.refer(rw1_word,0,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,rw1_tag,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(rw1_word,rw1_tag,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);



		//lc-p
		tag_tag_int.refer(lc_tag, p_tag,m_nLabel);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(lc_word, p_tag, m_nLabel);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, lc_tag, m_nLabel);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(lc_word, p_word, m_nLabel);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

		//rc-p
		tag_tag_int.refer(rc_tag, p_tag, m_nLabel);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(rc_word, p_tag, m_nLabel);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, rc_tag, m_nLabel);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(rc_word, p_word, m_nLabel);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);


		//ls-p
		tag_tag_int.refer(ls_tag, p_tag, m_nLabel);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(ls_word, p_tag, m_nLabel);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//rs-p
		tag_tag_int.refer(rs_tag, p_tag, m_nLabel);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(rs_word, p_tag, m_nLabel);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

	}

	void DepParser::getOrUpdateStackScore_sibling(const int & p, const int & c1, const int &la1, const int & c2, const int &la2, const int & amount) {
		//return;
		Weight * cweight = (Weight*)m_Weight;

		p_word = m_lSentence[p].first();
		p_tag = m_lSentence[p].second();

		Word c1_word = m_lSentence[c1].first();
		POSTag c1_tag = m_lSentence[c1].second();

		Word c2_word = m_lSentence[c2].first();
		POSTag c2_tag = m_lSentence[c2].second();

		if (c2 == p) c2_word = TWord::code(EMPTY_WORD), c2_tag = TWord::code(EMPTY_POSTAG);
		if (c1 == p) c1_word = TWord::code(EMPTY_WORD), c1_tag = TWord::code(EMPTY_POSTAG);
		int m_nLabel = (la1 * 1000 + la2 + 10) * 20;
		m_nDis = m_nLabel + encodeLinkDistanceOrDirection(c2, c1, false);
		m_nDir = m_nLabel + encodeLinkDistanceOrDirection(c2, c1, true);
		int m_nLabelDir = (la1%2 * 2 + la2%2) * 20;
		int m_nDisDir = m_nLabelDir + encodeLinkDistanceOrDirection(c2, c1, false);
		int m_nDirDir = m_nLabelDir + encodeLinkDistanceOrDirection(c2, c1, true);

		tag_tag_tag_int.refer(p_tag, c1_tag, c2_tag, m_nLabel);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(c1_tag, c2_tag, m_nLabel);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c2_tag, m_nLabel);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c2_word, c1_tag, m_nLabel);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(c1_word, c2_word, m_nLabel);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

	//adding

		/*tag_tag_tag_int.refer(p_tag, c1_tag, c2_tag, m_nLabelDir);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDisDir);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDirDir);
		cweight->m_mapPpC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(c1_tag, c2_tag, m_nLabelDir);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDirDir);
		cweight->m_mapC1pC2p.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c2_tag, m_nLabelDir);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDirDir);
		cweight->m_mapC1wC2p.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c2_word, c1_tag, m_nLabelDir);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDirDir);
		cweight->m_mapC1pC2w.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(c1_word, c2_word, m_nLabelDir);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDisDir);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDirDir);
		cweight->m_mapC1wC2w.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
*/
	}

	void DepParser::getOrUpdateStackScore_gc(const int & mid, const int &left , const int &lla, const int & right, const int &rla, const int & amount) {
		//return;
		Weight * cweight = (Weight*)m_Weight;

		p_word = m_lSentence[mid].first();
		p_tag = m_lSentence[mid].second();

		Word c1_word = m_lSentence[left].first();
		POSTag c1_tag = m_lSentence[left].second();

		Word c2_word = m_lSentence[right].first();
		POSTag c2_tag = m_lSentence[right].second();

		int m_nLabel = (lla * 1000 + rla + 10) * 20;
		m_nDis = m_nLabel + encodeLinkDistanceOrDirection(left, right, false);
		int m_nLabelDir = (lla%2 * 2+ rla%2) * 20;
		int m_nDisDir = m_nLabel + encodeLinkDistanceOrDirection(left, right, false);

		tag_tag_tag_int.refer(p_tag, c1_tag, c2_tag, m_nLabel);
		cweight->m_mapPpC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(c1_tag, c2_tag, m_nLabel);
		cweight->m_mapC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c2_tag, m_nLabel);
		cweight->m_mapC1wC2p_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1wC2p_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c2_word, c1_tag, m_nLabel);
		cweight->m_mapC1pC2w_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1pC2w_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(c1_word, c2_word, m_nLabel);
		cweight->m_mapC1wC2w_gc.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapC1wC2w_gc.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);


		//adding
		tag_tag_tag_int.refer(p_tag, c1_tag, c2_tag, m_nLabelDir);
		cweight->m_mapPpC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_int.referLast(m_nDisDir);
		cweight->m_mapPpC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(c1_tag, c2_tag, m_nLabelDir);
		cweight->m_mapC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1pC2p_gc.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c2_tag, m_nLabelDir);
		cweight->m_mapC1wC2p_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1wC2p_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c2_word, c1_tag, m_nLabelDir);
		cweight->m_mapC1pC2w_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDisDir);
		cweight->m_mapC1pC2w_gc.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(c1_word, c2_word, m_nLabelDir);
		cweight->m_mapC1wC2w_gc.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDisDir);
		cweight->m_mapC1wC2w_gc.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

	}
}
