from datetime import datetime
import shutil
import yaml
import os


def create_experiment_dir(path: str) -> str:
    """
    Creates a new directory in a provided base_path. The folder name of the directory is
    the current data and current time in order to avoid directory collision.

    Parameters
    ----------
    path: Path, in which the directory should be created.

    Returns
    -------
    exp_dir: Complete path (base_path + folder_name) to the newly created directory.
    """

    # get the current date and time
    now = datetime.now()

    # convert it into a suitable format
    dt_str = now.strftime("%m_%d_%Y__%H_%M_%S")
    exp_dir = path + '/' + dt_str + '/'

    # create the directory, if it doesn't already exist (it shouldn't)
    if not os.path.exists(exp_dir):
        os.makedirs(exp_dir)

    # tesnorboard directory
    if not os.path.exists(exp_dir + 'tb/'):
        os.makedirs(exp_dir + 'tb/')
    return exp_dir, dt_str


def init_experiment(parameter_file: str) -> tuple:
    """

    Parameters
    ----------
    parameter_file

    Returns
    -------
    parameters
    exp_dir
    writer
    """

    with open(parameter_file, 'r') as f:
        parameters = yaml.safe_load(f)

    # extract the base path, in which the current experiment directory is created
    base_path = parameters['experiment']['base_path']
    model_dir = parameters['experiment']['model_path']

    # create the experiment directory structure
    exp_dir, dt_str = create_experiment_dir(base_path)
    
    # create a model checkpoint dir
    model_dir = model_dir + '/' + dt_str + '/'
    if not os.path.exists(model_dir):
        os.makedirs(model_dir)

    # copy the parameters into the current directory
    shutil.copy(parameter_file, exp_dir)
    return parameters, exp_dir, model_dir
