"""Complete linked data cells with answer and/or parsed formula."""


from preprocess.operation.argument import complete_cells



# %% link the raw annotation

def find_corecell(cell_matrix, irow, icol):
    """Find the top-left core-cell of the current cell span."""
    curr_cell = cell_matrix[irow][icol]

    srow = curr_cell.span['start_row']
    scol = curr_cell.span['start_col']
    core_cell = cell_matrix[srow][scol]
    assert core_cell.iscorecell

    return core_cell


def update_linked_dict(input_dict, linked_cells, cell_matrix, hdr_nrows, hdr_ncols):
    """Update 'linked_cells' with the 'input_dict'."""

    nrows = len(cell_matrix)
    ncols = len(cell_matrix[0])

    for (irow,icol), item in input_dict.items():
        if item is None: continue
        if (irow >= nrows) or (icol >= ncols): continue

        if (irow < hdr_nrows) and (icol < hdr_ncols):  # corner
            corner_cell = find_corecell(cell_matrix, irow, icol)
            # if corner_cell.cell_string != str(item):
            #     print(f"got unmatching CORNER [{corner_cell.cell_string}] and ITEM [{item}]")
            crow = corner_cell.span['start_row']
            ccol = corner_cell.span['start_col']
            linked_cells['corner'].update({
                (crow, ccol): corner_cell
            })
        elif (irow < hdr_nrows) and (icol >= hdr_ncols):  # top
            top_cell = find_corecell(cell_matrix, irow, icol)
            # if top_cell.cell_string != str(item):
            #     print(f"got unmatching TOP [{top_cell.cell_string}] and ITEM [{item}]")
            trow = top_cell.span['start_row']
            tcol = top_cell.span['start_col']
            linked_cells['top'].update({
                (trow, tcol): top_cell
            })
        elif (irow >= hdr_nrows) and (icol < hdr_ncols):  # left
            left_cell = find_corecell(cell_matrix, irow, icol)
            # if left_cell.cell_string != str(item):
            #     print(f"got unmatching LEFT [{left_cell.cell_string}] and ITEM [{item}]")
            lrow = left_cell.span['start_row']
            lcol = left_cell.span['start_col']
            linked_cells['left'].update({
                (lrow, lcol): left_cell
            })
        elif (irow >= hdr_nrows) and (icol >= hdr_ncols):  # data
            data_cell = find_corecell(cell_matrix, irow, icol)
            # if data_cell.cell_string != str(item):
            #     print(f"got unmatching DATA [{data_cell.cell_string}] and ITEM [{item}]")
            drow = data_cell.span['start_row']
            dcol = data_cell.span['start_col']
            linked_cells['data'].update({
                (drow, dcol): data_cell
            })
    return linked_cells


_LINK_KEYS = {'top', 'left', 'corner', 'data'}   # type == set

def link_schema_cells(cell_matrix, schema_dict, linked_cells, hdr_nrows, hdr_ncols):
    """Read schema linking from subdict, locate corresponding cells in structure. """

    for phrase, phrase_dict in schema_dict.items():
        linked_cells = update_linked_dict(phrase_dict, linked_cells, 
            cell_matrix, hdr_nrows, hdr_ncols)

    return linked_cells


def link_answer_cells(cell_matrix, answer_cells, linked_cells, hdr_nrows, hdr_ncols):
    """Parse formula from subdict, extract out data regions mentioned.
    @USER TODO: fix 'top' and 'left' positions, from (irow, icol) to (ilevel, index).
    """
    linked_cells = update_linked_dict(
        input_dict=answer_cells, 
        linked_cells=linked_cells, 
        cell_matrix=cell_matrix, 
        hdr_nrows=hdr_nrows, 
        hdr_ncols=hdr_ncols
    )
    return linked_cells


def link_annotated(structure, subdict):
    """Link the raw annotated cells."""

    # read annotation and merge multiple results
    hdr_nrows = structure['num_top_header_rows']
    hdr_ncols = structure['num_left_header_cols']

    linked_cells = {k: {} for k in _LINK_KEYS}

    linked_cells = link_schema_cells(
        cell_matrix=structure['cell_matrix'], 
        schema_dict=subdict['schema_link'], 
        linked_cells=linked_cells, 
        hdr_nrows=hdr_nrows, 
        hdr_ncols=hdr_ncols
    )
    linked_cells = link_answer_cells(
        cell_matrix=structure['cell_matrix'], 
        answer_cells=subdict['answer_cells'], 
        linked_cells=linked_cells, 
        hdr_nrows=hdr_nrows, 
        hdr_ncols=hdr_ncols
    )
    return linked_cells

# %% complete data cells with formula
# especially the ARGS type

def complete_linking(structure, subdict):
    linked_cells = link_annotated(structure, subdict)
    if any([('arg' in op) for op in subdict['aggregation']]):
        print(f"[complete-link] got ops: {subdict['aggregation']}")
        linked_cells = complete_cells(structure, subdict, linked_cells)
    return linked_cells
