#include <iostream>
#include <fstream>

#include "include/basic.h"
#include "io/utility.h"
#include "command_line.h"

#include <boost/filesystem.hpp>

namespace po = boost::program_options;
namespace fs = boost::filesystem;

po::variables_map vm;

void printOptions() {
    LOG_INFO(<< "CommandLineOptions:");
    for (auto& option : vm) {
        auto& value_holder = option.second;
        std::cout << "  " << option.first << " = ";
        if (value_holder.value().empty())
            std::cout << "(unset)";

        if (value_holder.value().type() == typeid(int))
            std::cout << value_holder.as< int >();
        else if (value_holder.value().type() == typeid(std::string))
            std::cout << value_holder.as< std::string >();
        else if (value_holder.value().type() == typeid(float))
            std::cout << value_holder.as< float >();
        else
            std::cout << "(unknown)";
        std::cout << '\n';
    }
}

void setupOptons() {
    static std::vector< const char* > file_options{
        "config", "literals", "states",
        "rules", "input",
        "output", "feature", "dynamic_feature",
        "rules_extended", "prototype_indeices"};
    for (auto key : file_options) {
        if (vm.count(key))
            vm.at(key).value() =
                fs::absolute(vm[key].as< std::string >()).string();
    }
}

void initializeCommandLineVariables(int argc, char* argv[]) {
    po::options_description general_options{"General"};
    po::options_description file_options{"File"};
    general_options.add_options()                                                                          //
        ("help,h", "useage")                                                                               //
        ("config,c", po::value< std::string >()->default_value("transducer.config"), "configuration file") //
        ("model,m", po::value< std::string >()->default_value("baseline"), "model name for train or transduce");

    file_options.add_options()                                                    //
        ("literals", po::value< std::string >()->required(), "literals filename") //
        ("states", po::value< std::string >()->required(), "states filename")     //

        ("rules", po::value< std::string >()->required(), "rules filename")                      //
        ("rules_extended", po::value< std::string >(), "extended rules filename")                //
        ("prototype_indeices", po::value< std::string >(), "prototype indices filename")         //

        ("method", po::value< std::string >()->required(), "evaluation method")                                    //
        ("iterations", po::value< int >(), "iterations for training")                                              //
        ("feature", po::value< std::string >()->default_value("train.feat"), "features filename")                  //
        ("dynamic_feature", po::value< std::string >()->default_value("dynamic.feat"), "dynamic feature filename") //

        ("input", po::value< std::string >()->required(), "edsgraph level 0 input filename") //

        ("output", po::value< std::string >(), "output filename")  //

        ("beam_size", po::value< int >(), "beam size")                                               //
        ("add_gold_answer_when_beam_is_empty", po::value< int >(), "transducer option")              //
        ("generator_k_best", po::value< int >(), "transducr option")                                 //
        ("generator_use_transition", po::value< int >(), "transducer option")                        //
        ("predict_rule_head", po::value< int >(), "transducer option")                               //
        ("special_nodes_order", po::value< int >(), "transducer option");

    try {
        po::store(po::parse_command_line(argc, argv, general_options), vm);
        if (vm.count("help")) {
            std::cout << general_options << '\n'
                      << file_options << '\n';
            std::exit(0);
        }
        std::ifstream is(vm["config"].as< std::string >());

        if (!is) {
            LOG_ERROR(<< "Can't not find configuration file: "
                      << vm["config"].as< std::string >());
            std::exit(1);
        }

        po::store(po::parse_config_file(is, file_options), vm);
        po::notify(vm);
    } catch (const po::error& err) {
        LOG_ERROR(<< err.what());
        std::exit(1);
    }

    setupOptons();
    printOptions();
}

const po::variables_map& commandLineVariables() {
    return vm;
}
