#ifndef _SCORE_H
#define _SCORE_H

#include <string>
#include <iostream>
#include <unordered_map>

#include "common/token/token.h"

typedef int cscore;
typedef long long tscore;

enum ScoreType {
    eNonAverage,
    eAverage,
};

class Score {

private:
    cscore m_nCurrent;
    tscore m_nTotal;

    int m_nLastUpdate;

public:
    Score(const cscore& c = 0, const tscore& t = 0);
    Score(const Score& s);
    ~Score();

    void reset();
    bool empty() const;
    bool zero() const;
    void updateCurrent(const int& added = 0, const int& round = 0);
    void updateAverage(const int& round);
    void updateRetval(tscore& retval, const int& which);
    void updateRetvalCurrent(tscore& retval);
    void updateRetvalTotal(tscore& retval);

    friend std::istream& operator>>(std::istream& is, Score& s) {
        ttoken token;
        is >> s.m_nCurrent >> token >> s.m_nTotal;
        return is;
    }

    friend std::ostream& operator<<(std::ostream& os, const Score& s) {
        os << s.m_nCurrent << " / " << s.m_nTotal;
        return os;
    }
};

inline Score::Score(const cscore& c, const tscore& t)
: m_nCurrent(c), m_nTotal(t), m_nLastUpdate(0) {}

inline Score::Score(const Score& s)
: m_nCurrent(s.m_nCurrent),
  m_nTotal(s.m_nTotal), m_nLastUpdate(s.m_nLastUpdate) {}

inline Score::~Score() = default;

inline bool Score::empty() const {
    return m_nCurrent == 0 && m_nTotal == 0 && m_nLastUpdate == 0;
}

inline bool Score::zero() const {
    return m_nCurrent == 0 && m_nTotal == 0;
}

inline void Score::reset() {
    m_nCurrent = 0;
    m_nTotal = ( tscore )0;
    m_nLastUpdate = 0;
}

inline void Score::updateCurrent(const int& added, const int& round) {
    if (round > m_nLastUpdate) {
        updateAverage(round);
        m_nLastUpdate = round;
    }
    m_nCurrent += added;
    m_nTotal += added;
}

inline void Score::updateAverage(const int& round) {
    if (round > m_nLastUpdate) {
        m_nTotal += (tscore)(m_nCurrent * (round - m_nLastUpdate));
    }
}

inline void Score::updateRetval(tscore& retval, const int& which) {
    switch (which) {
        case eNonAverage:
            retval += ( tscore )m_nCurrent;
            break;
        case eAverage:
            retval += m_nTotal;
            break;
        default:
            break;
    }
}

inline void Score::updateRetvalCurrent(tscore& retval) {
    retval += ( tscore )m_nCurrent;
}

inline void Score::updateRetvalTotal(tscore& retval) {
    retval += m_nTotal;
}

#endif

// Local Variables:
// mode: c++
// End:
