import json
import os
import sys
from dataclasses import field, dataclass
from typing import Optional

import torch
from transformers import AutoConfig, HfArgumentParser, AutoTokenizer

from gector import GectorForTokenClassification
from helpers_new import get_target_sent_by_edits, START_TOKEN, UNK, PAD, normalize
from gector_utils import gector_tokenize
from utils.inference_tweak_param_json_utils import get_inf_tweaks_dict
from transformers.utils import EntryNotFoundError

USE_CUDA = torch.cuda.is_available()
if USE_CUDA:
    device = 'cuda'
else:
    device = 'cpu'


@dataclass
class LegacyGectorArguments:
    input_file: str = field(
        metadata={"help": "An input file containing the sentences to be corrected."},
    )
    output_file: str = field(
        metadata={"help": "Where corrected sentences will be saved."},
    )
    max_len: int = field(
        default=50,
        metadata={"help": "The maximum sentence length (all longer?? will be returned w/o changes)"},
    )
    min_len: int = field(
        default=3,
        metadata={"help": "The minimum sentence length (all longer?? will be returned w/o changes)"},
    )
    batch_size: int = field(
        default=128,
        metadata={"help": "The size of hidden unit cell."},
    )
    lowercase_tokens: int = field(
        default=0,
        metadata={"help": "Whether to lowercase tokens."},
    )
    iteration_count: int = field(
        default=5,
        metadata={"help": "The number of iterations of the model."},
    )
    additional_confidence: float = field(
        default=None,
        metadata={"help": "How many probability to add to $KEEP token."},
    )
    min_error_probability: float = field(
        default=None,
        metadata={"help": 'Minimum probability for each action to apply. '
                          'Also, minimum error probability, as described in the paper.'},
    )
    normalize: bool = field(
        default=False,
        metadata={"help": "Use for text simplification."},
    )


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    model_revision: str = field(
        default="main",
        metadata={"help": "The specific model version to use (can be a branch name, tag name or commit id)."},
    )
    use_auth_token: bool = field(
        default=False,
        metadata={
            "help": "Will use the token generated when running `transformers-cli login` (necessary to use this script "
                    "with private models)."
        },
    )


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.
    """

    task_name: Optional[str] = field(default="gec", metadata={"help": "The name of the task (ner, pos...)."})
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached training and evaluation sets"}
    )
    preprocessing_num_workers: Optional[int] = field(
        default=None,
        metadata={"help": "The number of processes to use for the preprocessing."},
    )
    max_seq_length: int = field(
        default=None,
        metadata={
            "help": "The maximum total input sequence length after tokenization. If set, sequences longer "
                    "than this will be truncated, sequences shorter will be padded."
        },
    )
    pad_to_max_length: bool = field(
        default=False,
        metadata={
            "help": "Whether to pad all samples to model maximum sentence length. "
                    "If False, will pad the samples dynamically when batching to the maximum length in the batch. More "
                    "efficient on GPU but very bad for TPU."
        },
    )
    label_all_tokens: bool = field(
        default=False,
        metadata={
            "help": "Whether to put the label for one word on all tokens of generated by that word or just on the "
                    "one (in which case the other tokens will have a padding index)."
        },
    )

    def __post_init__(self):
        self.task_name = self.task_name.lower()


def read_lines(fn, skip_strip=False):
    if not os.path.exists(fn):
        return []
    with open(fn, 'r', encoding='utf-8') as f:
        lines = f.readlines()
    return [s.strip() for s in lines if s.strip() or skip_strip]


def get_token_action(gector_args, index, prob, sugg_token):
    """Get lost of suggested actions for token."""

    if int(os.environ.get('SPELL_ON_OOV', 0)) > 0:
        if sugg_token == UNK:
            sugg_token = '$SPELL_2'

    # cases when we don't need to do anything
    if prob < gector_args.min_error_probability or sugg_token in [UNK, PAD, '$KEEP']:
        return None

    if sugg_token.startswith('$REPLACE') or sugg_token.startswith('$INFLECT_') or sugg_token.startswith('$TRANSFORM_') or sugg_token in ['$DELETE', '$SPELL', '$SPELL_2']:
        start_pos = index
        end_pos = index + 1
    elif sugg_token.startswith("$APPEND") or sugg_token.startswith("$MERGE_"):
        start_pos = index + 1
        end_pos = index + 1

    return start_pos - 1, end_pos - 1, sugg_token, prob


def postprocess_batch(gector_args, config, batch, all_probabilities, all_idxs,
                      error_probs):
    all_results = []
    noop_index = config.label2id['$KEEP']
    for tokens, probabilities, idxs, error_prob in zip(batch,
                                                       all_probabilities,
                                                       all_idxs,
                                                       error_probs):
        length = min(len(tokens), gector_args.max_len)
        edits = []

        # skip whole sentences if there no errors
        if max(idxs) == 0:
            all_results.append(tokens)
            continue

        # skip whole sentence if probability of correctness is not high
        if error_prob < gector_args.min_error_probability:
            all_results.append(tokens)
            continue

        for i in range(length + 1):
            # because of START token
            if i == 0:
                token = START_TOKEN
            else:
                token = tokens[i - 1]
            # skip if there is no error
            if idxs[i] == noop_index:
                continue

            sugg_token = config.id2label[idxs[i]]
            action = get_token_action(gector_args, i, probabilities[i],
                                           sugg_token)
            if not action:
                continue

            edits.append(action)
        all_results.append(get_target_sent_by_edits(tokens, edits, USE_CUDA))
    return all_results


def preprocess(token_batch, max_len):
    token_batch = [['$START'] + token_sequence[:max_len] for token_sequence in token_batch]
    return token_batch


def update_final_batch(final_batch, pred_ids, pred_batch,
                       prev_preds_dict):
    new_pred_ids = []
    total_updated = 0
    for i, orig_id in enumerate(pred_ids):
        orig = final_batch[orig_id]
        pred = pred_batch[i]
        prev_preds = prev_preds_dict[orig_id]
        if orig != pred and pred not in prev_preds:
            final_batch[orig_id] = pred
            new_pred_ids.append(orig_id)
            prev_preds_dict[orig_id].append(pred)
            total_updated += 1
        elif orig != pred and pred in prev_preds:
            # update final batch, but stop iterations
            final_batch[orig_id] = pred
            total_updated += 1
        else:
            continue
    return final_batch, new_pred_ids, total_updated



def handle_batch(model, config, tokenizer, full_batch, gector_args):
    """
    Handle batch of requests.
    """
    final_batch = full_batch[:]
    batch_size = len(full_batch)
    prev_preds_dict = {i: [final_batch[i]] for i in range(len(final_batch))}
    short_ids = [i for i in range(len(full_batch))
                 if len(full_batch[i]) < gector_args.min_len]
    pred_ids = [i for i in range(len(full_batch)) if i not in short_ids]
    total_updates = 0

    for n_iter in range(gector_args.iteration_count):
        orig_batch = [final_batch[i] for i in pred_ids]

        sequences = preprocess(orig_batch, gector_args.max_len)

        if not sequences:
            break

        with torch.no_grad():
            tokenized_sequences = gector_tokenize(tokenizer, sequences, return_word_start_indices=True, return_tensors='pt', device=device)
            word_start_indices = tokenized_sequences['word_start_indices']

            model.eval()
            outputs = model(input_ids=tokenized_sequences['input_ids'], attention_mask=tokenized_sequences['attention_mask'])

            label_probs = outputs['logits'].softmax(dim=-1)
            error_detection_probs = outputs['logits_d'].softmax(dim=-1)
            label_probs[:, :, 0] += gector_args.additional_confidence
            probabilities, idxs = label_probs.max(axis=-1)

            # reduce to word start tokens
            probabilities = [r[ws].tolist() for r, ws in zip(probabilities, word_start_indices)]
            idxs = [r[ws].tolist() for r, ws in zip(idxs, word_start_indices)]
            error_probs = [r[ws][:, 1].max().item() for r, ws in zip(error_detection_probs, word_start_indices)]

        pred_batch = postprocess_batch(gector_args, config, orig_batch, probabilities,
                                             idxs, error_probs)
        # if model.log:
        print(f"Iteration {n_iter + 1}. Predicted {round(100 * len(pred_ids) / batch_size, 1)}% of sentences.")

        final_batch, pred_ids, cnt = update_final_batch(final_batch, pred_ids, pred_batch, prev_preds_dict)
        total_updates += cnt

        if not pred_ids:
            break

    return final_batch, total_updates


def predict_for_file(input_file, output_file, model, config, tokenizer, gector_args, batch_size=32, to_normalize=False):
    test_data = read_lines(input_file)
    predictions = []
    cnt_corrections = 0
    batch = []
    for sent in test_data:
        batch.append(sent.split())
        if len(batch) == batch_size:
            preds, cnt = handle_batch(model, config, tokenizer, batch, gector_args)
            predictions.extend(preds)
            cnt_corrections += cnt
            batch = []
    if batch:
        preds, cnt = handle_batch(model, config, tokenizer, batch, gector_args)
        predictions.extend(preds)
        cnt_corrections += cnt

    result_lines = [" ".join(x) for x in predictions]
    if to_normalize:
        result_lines = [normalize(line) for line in result_lines]

    with open(output_file, 'w') as f:
        f.write("\n".join(result_lines) + '\n')
    return cnt_corrections


def main():
    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, LegacyGectorArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, gector_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, gector_args = parser.parse_args_into_dataclasses()

    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    tokenizer_name_or_path = model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path
    tokenizer = AutoTokenizer.from_pretrained(
        tokenizer_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=True,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
        add_prefix_space=True,
        add_special_tokens=False,
    )

    # get all paths
    model = GectorForTokenClassification.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    try:
        d = get_inf_tweaks_dict(model_args.model_name_or_path)

        if gector_args.min_error_probability is not None:
            print('warning: model save has inference_tweak_params.json, but min_error_probability was also provided. Ignoring value in model save.')
        else:
            gector_args.min_error_probability = d['min_error_probability']

        if gector_args.additional_confidence is not None:
            print('warning: model save has inference_tweak_params.json, but additional_confidence was also provided. Ignoring value in model save.')
        else:
            gector_args.additional_confidence = d['additional_confidence']

    except EntryNotFoundError:  # json file not found in model save dir
        pass

    assert gector_args.min_error_probability is not None, 'please supply --min_error_probability argument'
    assert gector_args.additional_confidence is not None, 'please supply --additional_confidence argument'

    if not USE_CUDA:
        print('warning: CUDA not available, using CPU')

    if USE_CUDA:
        model.to('cuda')

    cnt_corrections = predict_for_file(gector_args.input_file, gector_args.output_file, model, config, tokenizer, gector_args,
                                       batch_size=gector_args.batch_size,
                                       to_normalize=gector_args.normalize)
    # evaluate with m2 or ERRANT
    print(f"Produced overall corrections: {cnt_corrections}")


if __name__ == '__main__':
    main()
