from sklearn.model_selection import train_test_split
from sklearn.metrics import accuracy_score, precision_recall_fscore_support
import pandas as pd
from typing import List
from sentence_transformers import SentenceTransformer
from lightgbm import LGBMClassifier

# Code for baseline model SBERT+LGBM from Klie et al., 2020

class SbertSentenceClassifier():
    def __init__(self, model_name: str):
        self.encoder_model = SentenceTransformer(model_name)
        self.classifier = None

    def fit(self, sentences: List[str], targets: List[str]):
        # encode sentences
        featurized_sentences = self.encoder_model.encode(sentences)
        # training
        self.classifier = LGBMClassifier().fit(featurized_sentences, targets)

    def predict(self, sentences: List[str]):

        featurized_sentences = self.encoder_model.encode(sentences)
        predictions = self.classifier.predict(featurized_sentences)
        return predictions

label_column = "gold"
inputpath = "../data/expertdata_resolved.tsv"
df = pd.read_csv(inputpath, sep="\t")
num_labels = len(df[label_column].unique())

seeds = [2020, 2021, 2022, 2023, 2024, 2025, 2026, 2027, 2028, 2029]
#model_name = "distilbert-multilingual-nli-stsb-quora-ranking"
model_name = "distilbert-multilingual-nli-stsb-quora-ranking"
#model_name = "distiluse-base-multilingual-cased-v2"

final = {}
for seed in seeds:
    df_train, df_val = train_test_split(df, test_size=0.20, train_size=0.80, shuffle=True, stratify=df[[label_column]],
                                        random_state=seed)
    model = SbertSentenceClassifier(model_name=model_name)
    train_sentences = []
    train_targets = []
    for i in df_train.itertuples():
        train_sentences.append(i.sentence)
        train_targets.append(i.gold)
    model.fit(train_sentences, train_targets)

    eval_sentences = []
    eval_targets = []
    for i in df_val.itertuples():
        eval_sentences.append(i.sentence)
        eval_targets.append(i.gold)
    eval_predictions = model.predict(eval_sentences)
    precision, recall, f1, _ = precision_recall_fscore_support(eval_targets, eval_predictions, average='macro')
    acc = accuracy_score(eval_targets, eval_predictions)
    final[str(seed)] = {
        "eval_accuracy": acc,
        "eval_f1": f1,
        "eval_precision": precision,
        "eval_recall": recall
    }

import json
with open('expert_results_train_test_80_20_sbert_multilingual2_jck.json', 'w') as of:
    json.dump(final, of)