#include <cmath>
#include <stack>
#include <algorithm>
#include <ctime>
#include <unordered_set>
#include <thread>
#include "graphcjj1L_depparser.h"
#include "common/token/word.h"
#include "common/token/pos.h"

namespace graphcjj1L {
	WordPOSTag DepParser::empty_taggedword = WordPOSTag();
	WordPOSTag DepParser::start_taggedword = WordPOSTag();
	WordPOSTag DepParser::end_taggedword = WordPOSTag();
	double time, ftime;
	int fsum;
	int totalerr, total;
	int totaledgecor;
	DepParser::DepParser(const std::string & sFeatureInput, const std::string & sFeatureOut, int nState) :
		DepParserBase(sFeatureInput, nState) {

		m_nSentenceLength = 0;
		time = ftime = 0;
		fsum = 0;
		DepLabel.init();
		m_Weight = new Weight(sFeatureInput, sFeatureOut, DepLabel);
		totalmy = total = totalcor = totaledgecor = 0;
		for (int i = 0; i < MAX_SENTENCE_SIZE; ++i) {
			m_lItems[i][i].init(i, i);
		}

		DepParser::empty_taggedword.refer(TWord::code(EMPTY_WORD), TPOSTag::code(EMPTY_POSTAG));
		DepParser::start_taggedword.refer(TWord::code(START_WORD), TPOSTag::code(START_POSTAG));
		DepParser::end_taggedword.refer(TWord::code(END_WORD), TPOSTag::code(END_POSTAG));
	}

	DepParser::~DepParser() {
		delete m_Weight;
	}
	void DepParser::outans()
	{
		std::cout << "total time " << 0.001 * time << ' ' << ' ' << 0.001*time / m_nTrainingRound << std::endl;
		std::cout << "sentence wrong " << m_nTotalErrors << " / " << m_nTrainingRound << std::endl;
		std::cout << "unlabel precision " << totaledgecor << " / " << total << ' ' << 1.0*totaledgecor / total << std::endl;
		std::cout << "unlabel recall " << totaledgecor << " / " << totalmy << ' ' << 1.0*totaledgecor / totalmy << std::endl;
		std::cout << "unlabel F-Score  " << 2.0*totaledgecor / (total + totalmy) << std::endl;
		std::cout << "label precision " << totalcor << " / " << total << ' ' << 1.0*totalcor / total << std::endl;
		std::cout << "label recall " << totalcor << " / " << totalmy << ' ' << 1.0*totalcor / totalmy << std::endl;
		std::cout << "label F-Score  " << 2.0*totalcor / (total + totalmy) << std::endl;
	}
	void DepParser::train(const DependencyGraph & correct, const int & round) {
		// initialize
		int idx = 0;
		time_t b, e;
		b = clock();
		m_vecCorrectArcs.clear();
		m_nTrainingRound = round;
		m_nSentenceLength = correct.getSentence().size();
		m_vecDependencyTree = correct.getTree();
		m_lSentence[0].refer(TWord::code(ROOT_WORD), TPOSTag::code(ROOT_POSTAG));
		for (const auto & node : correct.getSentence()) {
			m_lSentence[++idx].refer(TWord::code(std::get<0>(node)), TPOSTag::code(std::get<1>(node)));
		}
		for (const auto arc : correct.getGraph())
		{
			m_vecCorrectArcs.push_back(PCLArc(EDGE_X(arc), EDGE_Y(arc),DepLabel.code(EDGE_LABEL(arc))));
		}
		
		for (const auto & arc : m_vecCorrectArcs)
		{
			int p = arc.first();
			int c = arc.second();
			int l = arc.third();
			DepLabel.addPPL(TriGram<int>(m_lSentence[p].second(), m_lSentence[c].second(), l));
		}

		/*if (m_nState == ParserState::GOLDTEST) {
			m_setFirstGoldScore.clear();
			for (const auto & arc : m_vecCorrectArcs) {
				m_setFirstGoldScore.insert(BiGram<int>(arc.first() == -1 ? m_nSentenceLength : arc.first(), arc.second()));
			}
		}*/
		//correct.print();
		//system("pause");
		// train
		work(nullptr, correct);
		time += clock() - b;
		if (m_nTrainingRound % OUTPUT_STEP == 0) {
			std::cout << "total time "<<0.001 * time << ' ' << ' ' << 0.001*time / m_nTrainingRound << std::endl;
			std::cout << "sentence wrong "<<m_nTotalErrors << " / " << m_nTrainingRound << std::endl;
			std::cout << "unlabel precision " <<totaledgecor << " / " << total << ' ' <<1.0*totaledgecor / total << std::endl;
			std::cout << "unlabel recall "<< totaledgecor << " / " << totalmy << ' ' << 1.0*totaledgecor / totalmy << std::endl;
			std::cout << "unlabel F-Score  " << 2.0*totaledgecor / (total + totalmy) << std::endl;
			std::cout << "label precision " << totalcor << " / " << total << ' ' << 1.0*totalcor / total << std::endl;
			std::cout << "label recall " << totalcor << " / " << totalmy << ' ' << 1.0*totalcor / totalmy << std::endl;
			std::cout << "label F-Score  " << 2.0*totalcor / (total + totalmy) << std::endl;

		}
	}

	void DepParser::parse(const Sentence & sentence, DependencyGraph * retval) {
		int idx = 0;
		m_nTrainingRound = 0;
		DependencyGraph correct;
		//std::cout << "xx" << std::endl;
		m_nSentenceLength = sentence.size();
		retval->m_lSentence.clear();
		retval->m_lTree = m_vecDependencyTree;
		for (const auto & token : sentence) {
			m_lSentence[++idx].refer(TWord::code(SENT_WORD(token)), TPOSTag::code(SENT_POSTAG(token)));
			retval->m_lSentence.push_back(token);
		}
		work(retval, correct);
	}
	void DepParser::debug_arc()
	{
		for (auto arc : m_vecCorrectArcs)
		{
			std::cout << "COR " << arc.first() << ' ' << arc.second() << ' ' << DepLabel.key(arc.third()) << std::endl;
		}

		for (auto arc : m_vecTrainArcs)
		{
			std::cout << "MY " << arc.first() << ' ' << arc.second() << ' ' << DepLabel.key(arc.third()) << std::endl;
		}
		system("pause");
	}
	void DepParser::work(DependencyGraph * retval, const DependencyGraph & correct) {
				
		decode();
		decodeArcs();		
		
		//debug_arc();

		switch (m_nState) {
		case ParserState::TRAIN:
			update();
			break;
		case ParserState::PARSE:
			generate(retval, correct);
			break;
		case ParserState::GOLDTEST:
			update();
			break;
		default:
			break;
		}
	}

	void DepParser::work_after_scoring(const Sentence & sentence, DependencyGraph * retval) {
		DependencyGraph correct;
//		for (const auto & token : sentence) {
//			correct.push_back(DependencyTreeNode(token, -1, NULL_LABEL));
//		}
		//std::cout << "decode_after_scoring" << std::endl;
		decode_after_scoring();
		//std::cout << "decodeArcs" << std::endl;
		decodeArcs();
		//std::cout << "generate" << std::endl;
		generate(retval, correct);
	}

	void DepParser::decode_after_scoring() {
		int sumlen = m_nSentenceLength;
		int labelsize = DepLabel.count();
		//std::cout << "begin load tree" << std::endl;
		//std::cout << m_vecDependencyTree.size() << std::endl;
		//for (auto treenode : m_vecDependencyTree)
			//std::cout << TREENODE_WORD(treenode) <<TREENODE_HEAD(treenode) <<  TREENODE_LABEL(treenode) << std::endl;
		m_lcaTree.loadPath(m_vecDependencyTree);
//		for (int i = 1; i <= sumlen; i++)
//			for (int j = 1; j <= sumlen; j++)
//			{
//				int p = m_lSentence[i].second();
//				int c = m_lSentence[j].second();
//				for (int label = 1; label <= labelsize; label++)
//					if (i != j && DepLabel.findPPL(TriGram<int>(p, c, label)))
//						Score[i][j][label][0] = arcScore(i, j, label);
//					else Score[i][j][label][0] = MIN_SCORE;
//			}
//
//		for (int i = 1; i <= sumlen; i++)
//			for (int j = 1; j <= sumlen; j++)
//				for (int la = 1; la <= labelsize; la++)
//				Score[i][j][la][1] = Score[j][i][la][0];
//std::cout << sumlen << ' ' << labelsize<<std::endl;
		for (int d = 1; d <= sumlen; d++)
		{
			for (int l = 1; l <= sumlen - d; l++)
			{
				int r = l + d;

				StateItem  & item = m_lItems[l][r];
				item.init(l, r);

				item.updateL2R_LABEL(l, m_lItems[l + 1][r].l2r_nolabel.getScore() );
				for (int sp = l + 1; sp < r; sp++)
					for (int la = 1; la <= labelsize; la++)
						for (int dir = 0, fdir = 1; dir <= 1; dir++, fdir = -1)
							if (Score[l][sp][la][dir] > HALF_MIN_SCORE)
								item.updateL2R_LABEL(sp,
								m_lItems[l][sp].l2r_label.getScore()
								+ m_lItems[sp][r].l2r_nolabel.getScore()
								+ Score[l][sp][la][dir],
								la*fdir);

				item.updateL2R_NOLABEL(l, m_lItems[l + 1][r].l2r_nolabel.getScore());
				for (int sp = l + 1; sp <= r; sp++)
					for (int la = 0; la <= labelsize; la++)
						for (int dir = 0, fdir = 1; dir <= 1; dir++, fdir = -1)
							if (Score[l][sp][la][dir] > HALF_MIN_SCORE)
								item.updateL2R_NOLABEL(sp,
								m_lItems[l][sp].l2r_label.getScore()
								+ m_lItems[sp][r].l2r_nolabel.getScore()
								+ Score[l][sp][la][dir],
								la*fdir);
			}
		}
	}

	void DepParser::decode() {
		int sumlen = m_nSentenceLength;
		int labelsize = DepLabel.count();
		m_lcaTree.loadPath(m_vecDependencyTree);
		for (int i = 1; i <= sumlen; i++)
			for (int j = 1; j <= sumlen; j++)
			{
				int p = m_lSentence[i].second();
				int c = m_lSentence[j].second();
				for (int label = 1; label <= labelsize; label++)
					if (i != j && DepLabel.findPPL(TriGram<int>(p, c, label)))
						Score[i][j][label][0] = arcScore(i, j, label);
					else Score[i][j][label][0] = MIN_SCORE;
			}

		for (int i = 1; i <= sumlen; i++)
			for (int j = 1; j <= sumlen; j++)
				for (int la = 1; la <= labelsize; la++)
				Score[i][j][la][1] = Score[j][i][la][0];

		for (int d = 1; d <= sumlen; d++)
		{
			for (int l = 1; l <= sumlen - d; l++)
			{
				int r = l + d;

				StateItem  & item = m_lItems[l][r];
				item.init(l, r);

				item.updateL2R_LABEL(l, m_lItems[l + 1][r].l2r_nolabel.getScore() );
				for (int sp = l + 1; sp < r; sp++)
					for (int la = 1; la <= labelsize; la++)
						for (int dir = 0, fdir = 1; dir <= 1; dir++, fdir = -1)
							if (Score[l][sp][la][dir] > HALF_MIN_SCORE)
								item.updateL2R_LABEL(sp, 
								m_lItems[l][sp].l2r_label.getScore()
								+ m_lItems[sp][r].l2r_nolabel.getScore()
								+ Score[l][sp][la][dir],
								la*fdir);

				item.updateL2R_NOLABEL(l, m_lItems[l + 1][r].l2r_nolabel.getScore());
				for (int sp = l + 1; sp <= r; sp++)
					for (int la = 0; la <= labelsize; la++)
						for (int dir = 0, fdir = 1; dir <= 1; dir++, fdir = -1)
							if (Score[l][sp][la][dir] > HALF_MIN_SCORE)
								item.updateL2R_NOLABEL(sp, 
								m_lItems[l][sp].l2r_label.getScore()
								+ m_lItems[sp][r].l2r_nolabel.getScore()
								+ Score[l][sp][la][dir],
								la*fdir);				
			}			
		}
	}

	void DepParser::decodeArcs()
	{
		m_vecTrainArcs.clear();
		m_lItems[1][m_nSentenceLength].type = L2R_NOLABEL;
		getArc(1, m_nSentenceLength);
	}

	void DepParser::getArc(int l, int r)
	{
		//std::cout << "edge " << l << ' ' << r << std::endl;
		//system("pause");
		int sp , la;
		StateItem &item = m_lItems[l][r];
		switch (item.type) {
		case L2R_NOLABEL:
			sp = item.l2r_nolabel.getSplit();
			la = item.l2r_nolabel.getLabel();
			//std::cout << "l2r_nolabel  "<<l << ' ' << sp << ' '<<r<<' '<<la<<std::endl;

			if (l >= r) return;
			//std::cout << "push  " << l << ' ' << sp << std::endl;
			if (sp != l)
			{
				if (la > 0)
					m_vecTrainArcs.push_back(PCLArc(l, sp, la));
				else
					m_vecTrainArcs.push_back(PCLArc(sp, l, -la));

				m_lItems[l][sp].type = L2R_LABEL;
				m_lItems[sp][r].type = L2R_NOLABEL;

				getArc(l, sp);
				getArc(sp, r);
			}
			else
			{
				m_lItems[sp+1][r].type = L2R_NOLABEL;
				getArc(sp + 1, r);
			}
			break;
		case L2R_LABEL:
			sp = item.l2r_label.getSplit();
			la = item.l2r_label.getLabel();
			//std::cout << "l2r_label  " << l << ' ' << sp<< ' ' << r << ' '<<la<<std::endl;

			if (l >= r) return;
			//std::cout << "push  " << r << ' ' << sp << std::endl;
			if (sp != l)
			{
				if (la > 0)
					m_vecTrainArcs.push_back(PCLArc(l, sp,la));
				else
					m_vecTrainArcs.push_back(PCLArc(sp, l , -la));

				m_lItems[l][sp].type = L2R_LABEL;
				m_lItems[sp][r].type = L2R_NOLABEL;

				getArc(l, sp);
				getArc(sp, r);
			}
			else
			{
				m_lItems[sp + 1][r].type = L2R_NOLABEL;
				getArc(sp + 1, r);
			}

			break;
		}
	}

	void DepParser::update() {
		std::unordered_set<PCLArc> positiveArcs;
		std::unordered_set<PCLArc> negativeArcs;

		positiveArcs.clear();
		for (const auto arc : m_vecCorrectArcs)
			positiveArcs.insert(PCLArc(arc.first(), arc.second(), 0));
		for (const auto arc : m_vecTrainArcs)
		{
			if (positiveArcs.find(PCLArc(arc.first(), arc.second(), 0)) != positiveArcs.end())
				totaledgecor++;
			positiveArcs.erase(PCLArc(arc.first(), arc.second(), 0));
		}

		positiveArcs.clear();
		negativeArcs.clear();
		positiveArcs.insert(m_vecCorrectArcs.begin(), m_vecCorrectArcs.end());		
		total += positiveArcs.size();		
		for (const auto & arc : m_vecTrainArcs) {
			if (positiveArcs.find(arc) != positiveArcs.end()) totalcor++;
			//std::cout << arc.first() << ' ' << arc.second() << std::endl;
			positiveArcs.erase(arc);
		}
		negativeArcs.insert(m_vecTrainArcs.begin(), m_vecTrainArcs.end());
		totalmy += negativeArcs.size();
		for (const auto & arc : m_vecCorrectArcs) {
			negativeArcs.erase(arc);
		}
		if (!positiveArcs.empty() || !negativeArcs.empty()) {
			++m_nTotalErrors;
		}
		if (m_nState == ParserState::TRAIN)
		{
			for (const auto & arc : positiveArcs) {
				getOrUpdateStackScore(arc.first(), arc.second(), arc.third(), 1);
			}
			for (const auto & arc : negativeArcs) {
				getOrUpdateStackScore(arc.first(), arc.second(), arc.third(), -1);
			}
		}
	}

	void DepParser::generate(DependencyGraph * retval, const DependencyGraph & correct) {
		retval->m_lGraph.clear();
		for (const auto & arc : m_vecTrainArcs) {
			retval->m_lGraph.push_back(GraphEdge(arc.first(), arc.second(), DepLabel.key(arc.third())));
		}
	}

	void DepParser::goldCheck() {
		/*if (m_vecCorrectArcs.size() != m_vecTrainArcs.size() || m_lItems[m_nSentenceLength + 1][0].l2r_nolabel.getScore() / GOLD_POS_SCORE != m_vecTrainArcs.size()) {
			std::cout << "gold parse len error at " << m_nTrainingRound << std::endl;
			std::cout << "score is " << m_lItems[1][m_nSentenceLength].l2r_nolabel.getScore() << std::endl;
			std::cout << "len is " << m_vecTrainArcs.size() << std::endl;
			++m_nTotalErrors;
		}
		else {
			int i = 0;
			std::sort(m_vecCorrectArcs.begin(), m_vecCorrectArcs.end(), [](const Arc & arc1, const Arc & arc2){ return arc1 < arc2; });
			std::sort(m_vecTrainArcs.begin(), m_vecTrainArcs.end(), [](const Arc & arc1, const Arc & arc2){ return arc1 < arc2; });
			for (int n = m_vecCorrectArcs.size(); i < n; ++i) {
				if (m_vecCorrectArcs[i].first() != m_vecTrainArcs[i].first() || m_vecCorrectArcs[i].second() != m_vecTrainArcs[i].second()) {
					break;
				}
			}
			if (i != m_vecCorrectArcs.size()) {
				std::cout << "gold parse tree error at " << m_nTrainingRound << std::endl;
				++m_nTotalErrors;
			}
		}*/
	}

	tscore DepParser::arcScore(const int & p, const int & c, const int & label) {
		/*if (m_nState == ParserState::GOLDTEST) {
		m_nRetval = m_setFirstGoldScore.find(TriGram<int>(p, c,label)) == m_setFirstGoldScore.end() ? GOLD_NEG_SCORE : GOLD_POS_SCORE;
		return m_nRetval;
		}*/
		m_nRetval = 0;
		int a = 0;
		//std::thread T(&DepParser::getOrUpdateStackScore,p,c,label);
		//std::thread T(&DepParser::getOrUpdateStackScore,this, std::ref(p), std::ref(c), std::ref(label));
		getOrUpdateStackScore(p, c, label);
		//T.join();
		return m_nRetval;
	}

	void DepParser::getOrUpdateStackScore(const int & p, const int & c, const int &label, const int & amount) {
		
		fsum++;
		Weight * cweight = (Weight*)m_Weight;

		p_1_tag = p - 1 >= 1 ? m_lSentence[p - 1].second() : start_taggedword.second();
		p1_tag = p + 1 <= m_nSentenceLength ? m_lSentence[p + 1].second() : end_taggedword.second();
		c_1_tag = c - 1 >= 1 ? m_lSentence[c - 1].second() : start_taggedword.second();
		c1_tag = c + 1 <= m_nSentenceLength ? m_lSentence[c + 1].second() : end_taggedword.second();

		p_1_word = p - 1 >= 1 ? m_lSentence[p - 1].first() : start_taggedword.first();
		p1_word = p + 1 <= m_nSentenceLength ? m_lSentence[p + 1].first() : end_taggedword.first();
		c_1_word = c - 1 >= 1 ? m_lSentence[c - 1].first() : start_taggedword.first();
		c1_word = c + 1 <= m_nSentenceLength ? m_lSentence[c + 1].first() : end_taggedword.first();

		p_word = m_lSentence[p].first();
		p_tag = m_lSentence[p].second();

		c_word = m_lSentence[c].first();
		c_tag = m_lSentence[c].second();

		m_nLabel = label * 1000;
		m_nDis = m_nLabel + encodeLinkDistanceOrDirection(p, c, false);
		m_nDir = m_nLabel + encodeLinkDistanceOrDirection(p, c, true);


		//P
		word_int.refer(p_word, m_nLabel);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapPw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p_tag, m_nLabel);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapPp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, p_tag, m_nLabel);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapPwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C
		word_int.refer(c_word, m_nLabel);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapCw.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c_tag, m_nLabel);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapCp.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c_word, c_tag, m_nLabel);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapCwp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1
		word_int.refer(p_1_word, m_nLabel);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapP_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p_1_tag, m_nLabel);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapP_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_1_word, p_1_tag, m_nLabel);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapP_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P1
		word_int.refer(p1_word, m_nLabel);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapP1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(p1_tag, m_nLabel);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapP1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p1_word, p1_tag, m_nLabel);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapP1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C_1
		word_int.refer(c_1_word, m_nLabel);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapC_1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c_1_tag, m_nLabel);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapC_1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c_1_word, c_1_tag, m_nLabel);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC_1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//C1
		word_int.refer(c1_word, m_nLabel);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDis);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_int.referLast(m_nDir);
		cweight->m_mapC1w.getOrUpdateScore(m_nRetval, word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_int.refer(c1_tag, m_nLabel);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDis);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_int.referLast(m_nDir);
		cweight->m_mapC1p.getOrUpdateScore(m_nRetval, tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(c1_word, c1_tag, m_nLabel);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapC1wp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_tag_int.refer(p_word, c_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCwp.getOrUpdateScore(m_nRetval, word_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_tag_int.refer(p_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_tag_int.refer(c_word, p_tag, c_tag, m_nLabel);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpCwp.getOrUpdateScore(m_nRetval, word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_int.refer(p_word, c_word, p_tag, m_nLabel);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDis);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDir);
		cweight->m_mapPwpCw.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_tag_int.refer(p_word, c_word, c_tag, m_nLabel);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDis);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_tag_int.referLast(m_nDir);
		cweight->m_mapPwCwp.getOrUpdateScore(m_nRetval, word_word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(p_word, c_word, m_nLabel);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapPwCw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_int.refer(p_tag, c_tag, m_nLabel);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpCp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//window_feat

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_1_tag, c_tag, m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, empty_taggedword.second(), c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p1_tag, c_tag, c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, empty_taggedword.second(), c_tag, c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_1_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_1_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, empty_taggedword.second(), c1_tag, m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, empty_taggedword.second(), c1_tag, m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, empty_taggedword.second(), c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1Cp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(empty_taggedword.second(), p_tag, c_1_tag, c_tag, m_nLabel);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCp_1Cp.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_tag, p1_tag, c_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPpPp1CpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		tag_tag_tag_tag_int.refer(p_1_tag, p_tag, c_tag, empty_taggedword.second(), m_nLabel);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPp_1PpCpCp1.getOrUpdateScore(m_nRetval, tag_tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1C
		flag_tag_tag_int.refer(0,p_1_tag,c_tag,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_1_word,c_tag,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_1_tag,c_word,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_1_word,c_word,m_nLabel);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);


		//P1C
		flag_tag_tag_int.refer(0,p1_tag,c_tag,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p1_word,c_tag,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p1_tag,c_word,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p1_word,c_word,m_nLabel);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP1C.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC_1
		flag_tag_tag_int.refer(0,p_tag,c_1_tag,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_word,c_1_tag,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_tag,c_1_word,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_word,c_1_word,m_nLabel);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC1
		flag_tag_tag_int.refer(0,p_tag,c1_tag,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(1,p_word,c1_tag,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(2,p_tag,c1_word,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_tag_tag_int.refer(3,p_word,c1_word,m_nLabel);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC1.getOrUpdateScore(m_nRetval, flag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//P_1PC
		flag_word_tag_tag_int.refer(0,p_1_word,p_tag,c_tag,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_1_tag,p_word,c_tag,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_1_tag,p_tag,c_word,m_nLabel);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapP_1PC.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PP1C
		flag_word_tag_tag_int.refer(0,p_word,p1_tag,c_tag,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,p1_word,c_tag,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis+0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir+0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,p1_tag,c_word,m_nLabel);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis+2*0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir+2*0);
		cweight->m_mapPP1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PC_1C
		flag_word_tag_tag_int.refer(0,p_word,c_1_tag,c_tag,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,c_1_word,c_tag,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,c_1_tag,c_word,m_nLabel);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPC_1C.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//PCC1
		flag_word_tag_tag_int.refer(0,p_word,c_tag,c1_tag,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(1,p_tag,c_word,c1_tag,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		flag_word_tag_tag_int.refer(2,p_tag,c_tag,c1_word,m_nLabel);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDis);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		flag_word_tag_tag_int.referLast(m_nDir);
		cweight->m_mapPCC1.getOrUpdateScore(m_nRetval, flag_word_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		for (int b = (int)std::fmin(p, c) + 1, e = (int)std::fmax(p, c); b < e; ++b) {
			b_tag = m_lSentence[b].second();
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, 0);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, m_nDis);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
			tag_tag_tag_int.refer(p_tag, b_tag, c_tag, m_nDir);
			cweight->m_mapPpBpCp.getOrUpdateScore(m_nRetval, tag_tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		}

		std::string path = "PATH=" + m_lcaTree.POSPath[p][c];
		path_int.refer(TPOSTag::code(path), 0);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDir);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDis);
		cweight->m_mapPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		
		std::string fpath = "FPATH=" + m_lcaTree.FPOSPath[p][c];
		path_int.refer(TPOSTag::code(fpath), 0);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDir);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast( m_nDis);
		cweight->m_mapFPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);

		fpath = "UpFPATH=" + m_lcaTree.UpFPOSPath[p][c];
		path_tag_int.refer(TPOSTag::code(fpath),c_word, 0);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDir);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDis);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		fpath = "DownFPATH=" + m_lcaTree.DownFPOSPath[p][c];
		path_tag_int.refer(TPOSTag::code(fpath),p_word, 0);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDir);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_tag_int.referLast( m_nDis);
		cweight->m_mapPFPathCtag.getOrUpdateScore(m_nRetval, path_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string lpath = "LPATH=" + m_lcaTree.LABELPath[p][c];
		path_int.refer(TPOSTag::code(lpath), m_nLabel);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast(m_nDir);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		path_int.referLast(m_nDis);
		cweight->m_mapLPath.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string subcatp = "subcat=" + TREENODE_POSTAG(m_vecDependencyTree[p]);
		for (auto & child : m_lcaTree.Children[p])
			subcatp+=TREENODE_POSTAG(m_vecDependencyTree[child]);
		subcat_int.refer(TPOSTag::code(subcatp), m_nLabel);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDir);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDis);
		cweight->m_mapPsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.refer(TPOSTag::code(subcatp),c_tag,m_nLabel);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDir);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDis);
		cweight->m_mapPsubcatCtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		std::string subcatc = "subcat=" + TREENODE_POSTAG(m_vecDependencyTree[c]);
		for (auto & child : m_lcaTree.Children[c])
			subcatc+=TREENODE_POSTAG(m_vecDependencyTree[child]);
		subcat_int.refer(TPOSTag::code(subcatc), m_nLabel);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDir);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_int.referLast(m_nDis);
		cweight->m_mapCsubcat.getOrUpdateScore(m_nRetval, subcat_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.refer(TPOSTag::code(subcatc),p_tag,m_nLabel);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDir);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		subcat_tag_int.referLast(m_nDis);
		cweight->m_mapCsubcatPtag.getOrUpdateScore(m_nRetval, subcat_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		int edgeflag = TREENODE_HEAD(m_vecDependencyTree[c]) == p ? 1 :
							   (TREENODE_HEAD(m_vecDependencyTree[p]) == c ? -1 : 0);
		if (edgeflag != 0){
			path_int.refer(edgeflag, m_nLabel);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
			path_int.referLast(m_nDir);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
			path_int.referLast(m_nDis);
			cweight->m_mapPC.getOrUpdateScore(m_nRetval, path_int, m_nScoreIndex, amount, m_nTrainingRound);
		}


		int clabel = TPOSTag::code("CLABEL=" + TREENODE_LABEL(m_vecDependencyTree[c]));
		label_int.refer(clabel, m_nLabel);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);
		label_int.referLast(m_nDir);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);
		label_int.referLast(m_nDis);
		cweight->m_mapCLabel.getOrUpdateScore(m_nRetval, label_int, m_nScoreIndex, amount, m_nTrainingRound);


		int f = TREENODE_HEAD(m_vecDependencyTree[c]);
		int lc = m_lcaTree.Children[c].empty() ? -1 : m_lcaTree.Children[c].front();
		int rc = m_lcaTree.Children[c].empty() ? -1 : m_lcaTree.Children[c].back();
		int cf = 0;
		for (int i = 0; i < m_lcaTree.Children[f].size(); i++)
			if (m_lcaTree.Children[f][i] == c)
			{
				cf = i;
				break;
			}
		int ls = cf == 0 ? -1 : m_lcaTree.Children[f][cf - 1];
		int rs = cf + 1 == m_lcaTree.Children[f].size() ? -1 : m_lcaTree.Children[f][cf + 1];
		int lw=m_lcaTree.Range[c].first(),rw = m_lcaTree.Range[c].second();

		int lw_tag = m_lSentence[lw].second(),rw_tag = m_lSentence[rw].second();
		int lw_word = m_lSentence[lw].first(),rw_word = m_lSentence[rw].first();
		int lw_1_tag = lw -1 >= 1? m_lSentence[lw-1].second() : start_taggedword.second();
		int rw1_tag = lw + 1 <= m_nSentenceLength ? m_lSentence[rw+1].second() : end_taggedword.second();
		int lw_1_word = lw -1 >= 1? m_lSentence[lw-1].first() : start_taggedword.first();
		int rw1_word = lw + 1 <= m_nSentenceLength ? m_lSentence[rw+1].first() : end_taggedword.first();

		int lc_tag = lc == -1 ? start_taggedword.second() : m_lSentence[lc].second();
		int lc_word = lc == -1 ? start_taggedword.first() : m_lSentence[lc].first();
		int rc_tag = rc == -1 ? end_taggedword.second() : m_lSentence[rc].second();
		int rc_word = rc == -1 ? end_taggedword.first() : m_lSentence[rc].first();

		int ls_tag = ls == -1 ? start_taggedword.second() : m_lSentence[ls].second();
		int ls_word = ls == -1 ? start_taggedword.first() : m_lSentence[ls].first();
		int rs_tag = rs == -1 ? end_taggedword.second() : m_lSentence[rs].second();
		int rs_word = rs == -1 ? end_taggedword.first() : m_lSentence[rs].first();

		//lw
		word_tag_int.refer(lw_word,0,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,lw_tag,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(lw_word,lw_tag,m_nLabel);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//rw
		word_tag_int.refer(rw_word,0,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,rw_tag,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(rw_word,rw_tag,m_nLabel);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//lw_1
		word_tag_int.refer(lw_1_word,0,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,lw_1_tag,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(lw_1_word,lw_1_tag,m_nLabel);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLW_1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		//rw1
		word_tag_int.refer(rw1_word,0,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(0,rw1_tag,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.refer(rw1_word,rw1_tag,m_nLabel);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRW1.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);



		//lc-p
		tag_tag_int.refer(lc_tag, p_tag,m_nLabel);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapLCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(lc_word, p_tag, m_nLabel);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, lc_tag, m_nLabel);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(lc_word, p_word, m_nLabel);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapLCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);

		//rc-p
		tag_tag_int.refer(rc_tag, p_tag, m_nLabel);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapRCpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(rc_word, p_tag, m_nLabel);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRCwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(p_word, rc_tag, m_nLabel);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRCpPw.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_word_int.refer(rc_word, p_word, m_nLabel);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDis);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_word_int.referLast(m_nDir);
		cweight->m_mapRCwPw.getOrUpdateScore(m_nRetval, word_word_int, m_nScoreIndex, amount, m_nTrainingRound);


		//ls-p
		tag_tag_int.refer(ls_tag, p_tag, m_nLabel);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapLSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(ls_word, p_tag, m_nLabel);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapLSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		//rs-p
		tag_tag_int.refer(rs_tag, p_tag, m_nLabel);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDis);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		tag_tag_int.referLast(m_nDir);
		cweight->m_mapRSpPp.getOrUpdateScore(m_nRetval, tag_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

		word_tag_int.refer(rs_word, p_tag, m_nLabel);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDis);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);
		word_tag_int.referLast(m_nDir);
		cweight->m_mapRSwPp.getOrUpdateScore(m_nRetval, word_tag_int, m_nScoreIndex, amount, m_nTrainingRound);

	}
}
