#!/usr/bin/env python

import argparse
import csv
import os
import re
import sys

from collections import defaultdict
import nltk

# compute kappa against majority expert or crowd ratings
def kappa(data, syslist, use_crowd=False):
    
    
    # initialize the list that will hold the contingency table in the order 
    # [(1,1), (1,3), (3,1), (3,3)] where the first number represents the majority 
    # rating and the second number represents the system prediction
    contable = {}
    for sys in syslist:
        contable[sys] = [0]*4

    # iterate over each rown in the data
    for row in data:        
        
        # Get the various fields in the data row        
        unitid, prep, sentence, preplocation, sys1pred, sys2pred, internal, crowd = row

        # Use an NLTK FreqDist to count the ratings
        fd = nltk.FreqDist()
        
        # use expert/crowd ratings depending on what's asked for
        ratings = crowd if use_crowd else internal
        
        # make sure to ignore any 'Ungram' (indicated by '2') ratings
        ratings = [r for r in ratings.split('|') if r != '2']
        
        # ignore instances where the experts might have all assigned 
        # 'Ungram' rating
        if not ratings:
            continue
            
        # count the ratings and get the majority rating
        fd.update(ratings)
        majority = fd.max()
            
        # compute the contingency table entries for Sys1 and Sys2
        for sys in syslist:
            pred = eval(sys + 'pred')
            if pred == majority == '1':
                contable[sys][0] += 1
            elif pred == majority == '3':
                contable[sys][3] += 1
            elif pred == '1' != majority:
                contable[sys][2] += 1
            elif pred == '3' != majority:
                contable[sys][1] += 1
                                        
    # return the contingency tables 
    return contable
    

if __name__ == '__main__':

    parser = argparse.ArgumentParser(description="Compute kappa for two given systems.", prog='compute_kappa.py')
    parser.add_argument('inputfile', help='the input file containing the records')
    parser.add_argument('-s', '--system', dest='systems', default="both", choices=["1", "2", "both"], help='compute for system 1, 2 or both (default:BOTH)')
    parser.add_argument('-c', '--crowd', dest='crowd', default=False, action="store_true", help='compute against crowd judgments (default: FALSE)')

    # Parse the command line arguments
    args = parser.parse_args()
    kappas = {}

    # what is the list of systems that we need to compute kappa for
    if args.systems == "1":
        syslist = ['sys1']
    elif args.systems == "2":
        syslist = ['sys2']
    if args.systems == "both":
        syslist = ['sys1', 'sys2']

    # compute kappa against expert/crowd judgments depending on the command line switches
    r = csv.reader(file(args.inputfile))
    rows = list(r)[1:]
    contable = kappa(rows, syslist, args.crowd)

    # print out the results
    for sys in syslist:
        total = sum(contable[sys])
        P_a = float(contable[sys][0] + contable[sys][3])/total
        P_a1 = float(contable[sys][0] + contable[sys][1])/total
        P_a2 = 1.0 - P_a1
        P_b1 = float(contable[sys][0] + contable[sys][2])/total
        P_b2 = 1.0 - P_b1
        P_e = P_a1 * P_b1 + P_a2 * P_b2
        kappas[sys] = (P_a - P_e)/(1.0 - P_e)
        print ' %s => kappa: %.4f' % (sys, kappas[sys])
