///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <getopt.h>

#include "nl-iomacros.h"
#include "nl-cpt.h"


////////////////////////////////////////////////////////////////////////////////
//
//  Given Variables
//
////////////////////////////////////////////////////////////////////////////////

char psX[]   = "";
char psSpc[] = " ";

//// G: grammar rule symbol
DiscreteDomain<int> domG;
typedef DiscreteDomainRV<int,domG> G;
const G G_BOT ("-");
const G G_TOP ("REST");

//// GG: pair of grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,G,psSpc,G,psX> GG;
const GG GG_BOT ( G_BOT, G_BOT );

//// P: pos symbol
DiscreteDomain<int> domP;
typedef DiscreteDomainRV<int,domP> P;
const P P_BOT ("-");

//// W: pos symbol
DiscreteDomain<int> domW;
typedef DiscreteDomainRV<int,domW> W;
const W W_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Given Models
//
////////////////////////////////////////////////////////////////////////////////

//// Model of G (prior)
typedef CPT1DModel<G,Prob> GModel;
//// Model of GG given G (binary branching expansion)
typedef CPT2DModel<GG,G,Prob> GGgModel;

//// Model of term P given G
typedef CPT2DModel<P,G,Prob> PcModel;
//// Model of term P given W
typedef CPT2DModel<P,W,Prob> PwModel;
//// Model of term P
typedef CPT1DModel<P,Prob> PModel;
//// Model of word W
typedef CPT1DModel<W,Prob> WModel;


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Variables
//
////////////////////////////////////////////////////////////////////////////////

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
typedef DiscreteDomainRV<char,domD> D;

//// K: length (input only, to HHMM models)...
DiscreteDomain<char> domK;
typedef DiscreteDomainRV<char,domK> K;

//// Q: incomplete category state
DiscreteDomain<int> domQ;
typedef DiscreteDomainRV<int,domQ> Q;
const Q Q_BOT ("-/-");

//// B: boolean
DiscreteDomain<int> domB;
typedef DiscreteDomainRV<int,domB> B;
const B B_0 ("0");
const B B_1 ("1");

//// F: final state
DiscreteDomain<int> domF;
typedef DiscreteDomainRV<int,domF> F;
const F F_0 ("0");
const F F_1 ("1");
const F F_BOT ("-");


////////////////////////////////////////////////////////////////////////////////
//
//  Derived Models
//
////////////////////////////////////////////////////////////////////////////////

//// Expansion model of G given D and G (from above)
typedef CPT3DModel<G,D,G,Prob> GeModel;
//// Transition model of G given D and G (awaited cat from previous) and G (from reduction)
typedef CPT4DModel<G,D,G,G,Prob> GtmModel;
//// Transition model of G (active cat) given D and G (active cat from previous) and G (awaited cat from above)
typedef CPT4DModel<G,D,G,G,Prob> GtpModel;
//// Transition model of G (awaited cat) given D and G (active cat from current) and G (active cat from previous)
typedef CPT4DModel<G,D,G,G,Prob> GtqModel;
//// Reduction model of F given D and G (active cat from previous) and G (awaited cat from above) (only check if previous awaited = reduction)
typedef CPT4DModel<F,D,G,G,Prob> FrModel;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
///
///  Main function
///
////////////////////////////////////////////////////////////////////////////////

static struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {0, 0, 0, 0}};


void printUsage(char* progname) {
  //fprintf(stderr,"Usage: cat <RAW_FILE> | %s [OPTIONS]... [MODEL_FILE1] [MODEL_FILE2] ...\n",progname);
  //fprintf(stderr,"Runs the speech recognition system on RAW_FILE using the models in MODEL_FILE1, 2, etc.\n");
  fprintf(stderr," System variable Settings:\n");
  fprintf(stderr,"  -h, --help\t\tPrint this message\n");
}


int main (int nArgs, char* argv[]) {

  // Given models...
  GModel   mGr;
  GGgModel mGGg;
  PcModel  mPc;
  PwModel  mPw;
  PModel   mP;
  WModel   mW;

  // Derived variable constants...
  const int D_MAX=4;
  const int K_MAX=98;
  // Init D values (strings)...
  for ( int d=0; d<=D_MAX+2; d++ ) { String s; s<<d; cerr<<D(s.c_array())<<"\n"; }
  // Init K values (strings)...
  for ( int k=0; k<=K_MAX+1; k++ ) { String s; if(k<10)s<<"0"; s<<k;; cerr<<K(s.c_array())<<"\n"; }


  //////////////////// PART 0: COMMAND-LINE OPTIONS

  // Parse command-line options
  char* progname = strdup(argv[0]);
  int option_index = 0;
  char opt;
  while( (opt = getopt_long(nArgs, argv, "hp:q:Vv", long_options, &option_index)) != -1 ) {
    switch(opt) {
    case 'h': printUsage(progname);
              exit(0);
    //case 'p': UPDATE_PORT_NUM  = atoi(optarg); break;
    //case 'q': QUERY_PORT_NUM = atoi(optarg); break;
    //case 'V': WorldModel::OUTPUT_VERY_VERBOSE = true; break; // do break.
    //case 'v': WorldModel::OUTPUT_VERBOSE = true; break;
    default: break;
    }
  }
  // Complain if too few args...
  if (optind > nArgs ) {
    printUsage(progname);
    exit(0);
  }


  //////////////////// PART I. LOAD GIVEN MODELS (FROM STDIN)

  FILE* pf = stdin;

  int c=' '; int i=0; int line=1; String sBuff(1000);                          // Lookahead/ctrs/buffers
  CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                   // Get to first record
  while ( c!=-1 && c!='\0' && c!='\5' ) {                                      // For each record
    CONSUME_STR ( pf, c, (c!='\n' && c!='\0' && c!='\5'), sBuff, i, line );    //   Consume line
    StringInput si(sBuff.c_array());
    if ( !( sBuff[0]=='#' ||                                                   //   Accept comments/fields
	    (si>>"Gr ">>mGr >>"\0"!=NULL) ||
	    (si>>"GG ">>mGGg>>"\0"!=NULL) ||
            (si>>"Pc ">>mPc >>"\0"!=NULL) ||
	    (si>>"Pw ">>mPw >>"\0"!=NULL) ||
	    (si>>"P " >>mP  >>"\0"!=NULL) ||
	    (si>>"W " >>mW  >>"\0"!=NULL) ) )
      cerr<<"\nERROR: unable to process line "<<line<<": "<<sBuff<<"\n";
    CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                 //   Consume whitespace
  }

  //mPc.dump(cout,"Pc");
  mPw.dump(cout,"Pw");
  mP.dump(cout,"P");
  mW.dump(cout,"W");

  //mPc.clear();
  mPw.clear();
  mP.clear();
  mW.clear();

  //////////////////// PART II. BUILD DERIVED MODELS

  ////////////////////////////////////////////////////////////
  ////////// 1. Derive Helper Model Probs...

  ///// 1a. Add REST -> START REST for all START symbols...

  // For each Ae expanded from zero level...
  GModel::IterVal gAe; for ( bool bAe=mGr.setFirst(gAe); bAe; bAe=mGr.setNext(gAe) ) {
    GG ggAe_TOP ( gAe, G_TOP );
    mGGg.setProb ( ggAe_TOP, G_TOP ) += mGr.getProb(gAe);
  }

  //mGGg.dump(cout,"GGg");

  ///// 1b. Derive depth-bounded subtree probs...

  typedef CPT4DModel<B,K,D,G,Prob> BkdgModel;
  BkdgModel mBboundedLC;
  BkdgModel mBboundedRC;
  // For chains of increasing length k...
  for ( int k=0; k<=K_MAX; k++ ) {
    // For decreasing depths d...
    for ( int d=D_MAX+2; d>=1 && d>=D_MAX+1-k; d-- ) {

      cerr<<"1b,"<<k<<","<<d<<"...\n";

      // Ground bottom values...
      if ( d<=D_MAX+1 ) mBboundedLC.setProb ( B_1, k, d, G_BOT ) = 1.0;
      if ( d<=D_MAX+1 ) mBboundedRC.setProb ( B_1, k, d, G_BOT ) = 1.0;

      // For each given Ah...
      for ( GGgModel::const_iterator iGGg=mGGg.begin(); iGGg!=mGGg.end(); iGGg++ ) {
        G gAh = iGGg->first.getX1();

        // For each Ah0, Ah1 by rule from Ah...
        GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
                                           bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
          G gAh0 = ggAh0_Ah1.first;
          G gAh1 = ggAh0_Ah1.second;

          if (  mBboundedLC.getProb(B_1,k-1,d,gAh0)!=Prob() && mBboundedRC.getProb(B_1,k-1,d,gAh1)!=Prob() ) {
            // Define prob...
            mBboundedLC.setProb ( B_1, k, d, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,k-1,d,gAh0) *          mBboundedRC.getProb(B_1,k-1,d,gAh1);
          }
          if (  mBboundedLC.getProb(B_1,k-1,d+1,gAh0)!=Prob() && mBboundedRC.getProb(B_1,k-1,d,gAh1)!=Prob() ) {
            // Define prob...
            mBboundedRC.setProb ( B_1, k, d, gAh ) += mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,k-1,d+1,gAh0) *          mBboundedRC.getProb(B_1,k-1,d,gAh1);
          }
        }
      }
    }
  }
  mBboundedRC.setProb ( B_1, K_MAX, 0, G_TOP ) = 1.0;

  //mBboundedLC.dump(cout,"BbLC");
  //mBboundedRC.dump(cout,"BbRC");


  ///// 1c. Derive depth-bounded PCFG probs...

  typedef CPT3DModel<GG,D,G,Prob> GGdgModel;
  GGdgModel mGGdgLC;
  GGdgModel mGGdgRC;
  // For decreasing depths d...
  for ( int d=D_MAX+1; d>=0; d-- ) {
    cerr<<"1c,"<<d<<"...\n";
    // Base case...
    if ( 0==d ) {
      // For each Ae expanded from zero level...
      GModel::IterVal gAe; for ( bool bAe=mGr.setFirst(gAe); bAe; bAe=mGr.setNext(gAe) ) {
        GG ggAe_TOP ( gAe, G_TOP );
        mGGdgRC.setProb ( ggAe_TOP, d, G_TOP ) += mGr.getProb(gAe) * mBboundedLC.getProb(B_1,K_MAX,d+1,gAe);
      }
    }
    // Induction step...
    else {
      // For each given Ah...
      for ( GGgModel::const_iterator iGGg=mGGg.begin(); iGGg!=mGGg.end(); iGGg++ ) {
        G gAh = iGGg->first.getX1();
        // For each Ah0, Ah1 by rule from Ah...
        GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
                                           bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
          G gAh0 = ggAh0_Ah1.first;
          G gAh1 = ggAh0_Ah1.second;
          if (  mBboundedLC.getProb(B_1,K_MAX,d,gAh0)!=Prob() && mBboundedRC.getProb(B_1,K_MAX,d,gAh1)!=Prob() ) {
            // Define prob...
            mGGdgLC.setProb ( ggAh0_Ah1, d, gAh ) = mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,K_MAX,d,gAh0) *          mBboundedRC.getProb(B_1,K_MAX,d,gAh1);
          }
          if (  mBboundedLC.getProb(B_1,K_MAX,d+1,gAh0)!=Prob() && mBboundedRC.getProb(B_1,K_MAX,d,gAh1)!=Prob() ) {
            // Define prob...
            mGGdgRC.setProb ( ggAh0_Ah1, d, gAh ) = mGGg.getProb(ggAh0_Ah1,gAh)
              * mBboundedLC.getProb(B_1,K_MAX,d+1,gAh0) *          mBboundedRC.getProb(B_1,K_MAX,d,gAh1);
          }
        }
      }
    }
  }
  mGGdgLC.normalize();
  mGGdgRC.normalize();

  //mGGdgLC.dump(cout,"GGdgLC");
  //mGGdgRC.dump(cout,"GGdgRC");



  ////////// 2. DUMP

  // For each given Ae...
  for ( bool bAe=mGr.setFirst(gAe);
        bAe; bAe=mGr.setNext (gAe) ) {
    cout<<"Gr : "<<gAe<<"-l1 = "<<mGr.getProb(gAe)<<"\n";
  }

  // For each depth d...
  for ( int d=1; d<=D_MAX+1; d++ ) {
    // For each given Ah...
    for ( GGgModel::const_iterator iGGg=mGGg.begin(); iGGg!=mGGg.end(); iGGg++ ) {
      G gAh = iGGg->first.getX1();
      // For each Ah0, Ah1 by rule from Ah...
      GGgModel::IterVal ggAh0_Ah1; for ( bool      bAh0_Ah1=mGGg.setFirst(ggAh0_Ah1,gAh);
                                         bAh0_Ah1; bAh0_Ah1=mGGg.setNext (ggAh0_Ah1,gAh) ) {
        G gAh0 = ggAh0_Ah1.first;
        G gAh1 = ggAh0_Ah1.second;
        #define CONSISTENT // comment this out for inconsistent model used in naacl09
        #ifdef CONSISTENT ////////////////////////////////////////////////////// consistent model (slightly worse for bpcfg)
        if ( ggAh0_Ah1 == GG_BOT ) {
          cout<<"GG " <<  gAh<<"-"<<"l"<<d  <<" : "<<  gAh0  <<" "<<  gAh1  <<" = "<<mGGdgLC.getProb(GG(ggAh0_Ah1),d,gAh)<<"\n";
          cout<<"GG " <<  gAh<<"-"<<"r"<<d  <<" : "<<  gAh0  <<" "<<  gAh1  <<" = "<<mGGdgRC.getProb(GG(ggAh0_Ah1),d,gAh)<<"\n";
        } else if ( d<=D_MAX ) {
          cout<<"GG " <<  gAh<<"-"<<"l"<<d  <<" : "<<  gAh0<<"-"<<"l"<<d      <<" "<<  gAh1<<"-"<<"r"<<d  <<" = "<<mGGdgLC.getProb(GG(ggAh0_Ah1),d,gAh)<<"\n";
          cout<<"GG " <<  gAh<<"-"<<"r"<<d  <<" : "<<  gAh0<<"-"<<"l"<<(d+1)  <<" "<<  gAh1<<"-"<<"r"<<d  <<" = "<<mGGdgRC.getProb(GG(ggAh0_Ah1),d,gAh)<<"\n";
        }
        #else ////////////////////////////////////////////////////////////////// inconsistent model (leaks probability mass, but slightly better)
        if ( ggAh0_Ah1 == GG_BOT ) {
          cout<<"GG " <<  gAh<<"-"<<"l"<<d  <<" : "<<  gAh0  <<" "<<  gAh1  <<" = "<<mGGg.getProb(GG(ggAh0_Ah1),gAh)<<"\n";
          cout<<"GG " <<  gAh<<"-"<<"r"<<d  <<" : "<<  gAh0  <<" "<<  gAh1  <<" = "<<mGGg.getProb(GG(ggAh0_Ah1),gAh)<<"\n";
        } else if ( d<=D_MAX ) {
          cout<<"GG " <<  gAh<<"-"<<"l"<<d  <<" : "<<  gAh0<<"-"<<"l"<<d      <<" "<<  gAh1<<"-"<<"r"<<d  <<" = "<<mGGg.getProb(GG(ggAh0_Ah1),gAh)<<"\n";
          cout<<"GG " <<  gAh<<"-"<<"r"<<d  <<" : "<<  gAh0<<"-"<<"l"<<(d+1)  <<" "<<  gAh1<<"-"<<"r"<<d  <<" = "<<mGGg.getProb(GG(ggAh0_Ah1),gAh)<<"\n";
        }
        #endif /////////////////////////////////////////////////////////////////
      }
    }
  }

  // For each depth d...
  for ( int d=1; d<=D_MAX+1; d++ ) {
    // For each given G...
    for ( PcModel::const_iterator iPc=mPc.begin(); iPc!=mPc.end(); iPc++ ) {
      G g = iPc->first.getX1();
      // For each P given G...
      PcModel::IterVal p; for ( bool bP=mPc.setFirst(p,g);
                                bP;  bP=mPc.setNext (p,g) ) {
        cout<<"Pc "<<g<<"-"<<"l"<<d<<" : "<<p<<" = "<<mPc.getProb(p,g)<<"\n";
        cout<<"Pc "<<g<<"-"<<"r"<<d<<" : "<<p<<" = "<<mPc.getProb(p,g)<<"\n";
      }
    }
  }

}

