///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "nl-cpt.h"

char psX[]="";
char psComma[]=",";
char psTilde[]="~";
char psSemi[]=";";
char psSemiSemi[]=";;";
char psDashDiamondDash[]="-<>-";
char psLangle[]="<";
char psRangle[]=">";
char psLbrack[]="[";
char psRbrack[]="]";


////////////////////////////////////////////////////////////////////////////////
//
//  Random Variables
//
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////// Simple Variables

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
typedef DiscreteDomainRV<char,domD> D;
const D D_0("0");
const D D_1("1");
const D D_2("2");
const D D_3("3");
const D D_4("4");
const D D_5("5");

//// F: final-state...
DiscreteDomain<char> domF;
typedef DiscreteDomainRV<char,domF> F;
const F F_0("0");
const F F_1("1");
typedef DiscreteDomainRV<char,domF> FO;
typedef DiscreteDomainRV<char,domF> FP;

//// E: (reified) entity referent...
DiscreteDomain<int> domE;
typedef DiscreteDomainRV<int,domE> E;
const E E_NONE("-");
const E E_TOP("eT");

//// L: relation label...
DiscreteDomain<int> domL;
typedef DiscreteDomainRV<int,domL> L;
const L L_NONE("-");
const L L_IDENT("IDENT");

//// Csymb: next value after slash (for `shift' expansions, to ignore material to left and right of expanded symbol)
DiscreteDomain<int> domCsymb;
class Csymb : public DiscreteDomainRV<int,domCsymb> {
 private:
  static SimpleHash<Csymb,bool> hIsTerminal;
  void calcDetModels ( string s ) {
    if (!hIsTerminal.contains(*this)) {
      hIsTerminal.set(*this) = ( 'a' <= s.c_str()[0] && s.c_str()[0] <= 'z' );
      //cerr<<"hIsTerminal "<<*this<<" : "<<hIsTerminal.get(*this)<<"\n";
    }
  }
 public:
  Csymb ( )                : DiscreteDomainRV<int,domCsymb> ( )    { }
  Csymb ( const char* ps ) : DiscreteDomainRV<int,domCsymb> ( ps ) { calcDetModels(ps); }
  //Csymb ( string s ) : DiscreteDomainRV<int,domCsymb> ( s )  { calcDetModels(s); }
  bool isTerminal ( ) const { return hIsTerminal.get(*this); }
  friend pair<StringInput,Csymb*> operator>> ( StringInput si, Csymb& m ) { return pair<StringInput,Csymb*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,Csymb*> delimbuff, const char* psD ) {
    if ( delimbuff.first == NULL ) return NULL;
    StringInput si=delimbuff.first>>(DiscreteDomainRV<int,domCsymb>&)*delimbuff.second>>psD;
    delimbuff.second->calcDetModels(delimbuff.second->getString()); return si; }
};
SimpleHash<Csymb,bool> Csymb::hIsTerminal;
#define C_SYMB_NONE Csymb("-")
#ifdef TEXTOBSMODEL
#define C_SYMB_TOP  Csymb("start")
#else
#define C_SYMB_TOP  Csymb("START")
#endif
//const Csymb C_SYMB_NONE("-");
//const Csymb C_SYMB_TOP("START");

//// C: syntactic category...
DiscreteDomain<int> domC;
class C : public DiscreteDomainRV<int,domC> {
 private:
  static SimpleHash<C,Csymb> hToCsymb;
  static SimpleHash<C,Csymb> hToCfol;
  static SimpleHash<C,C>     hToCnext;
  static SimpleHash<C,bool>  hIsTwoEndNonterm;
  void calcDetModels ( string s ) {
    if (!hToCsymb.contains(*this)) {
      size_t i=s.find('/');     assert(i!=string::npos);
      size_t j=s.find("_",i+1); if (j==string::npos) j=s.length();
      if ( j-i>1 ) {
        hToCsymb.set(*this) = Csymb(s.substr(i+1,j-i-1).c_str());
        //cerr<<"hToCsymb "<<*this<<" : "<<hToCsymb.get(*this)<<"\n";
      }
    }
    if (!hToCfol.contains(*this)) {
      size_t i=s.find('_');
      if (i!=string::npos) {
        size_t j=s.find("_",i+1); if (j==string::npos) j=s.length();
        hToCfol.set(*this) = Csymb(s.substr(i+1,j-i-1).c_str());
        //cerr<<"hToCfol "<<*this<<" : "<<hToCfol.get(*this)<<"\n";
      }
    }
    if (!hToCnext.contains(*this)) {
      size_t i = s.find('/');      assert (i!=string::npos);
      size_t j = s.find('_',i+1);  //assert (j!=string::npos);
      if ( s.substr(i)!=string("/") ) {
        hToCnext.set(*this) = C ( (s.substr(0,i+1)+((string::npos==j)?string(""):s.substr(j+1))).c_str() );
        //cerr<<"hToCnext "<<*this<<" : "<<hToCnext.get(*this)<<"\n";
      }
    }
    if (!hIsTwoEndNonterm.contains(*this)) {
      size_t i = s.find('_');
      size_t j = (i==string::npos) ? string::npos : s.find_first_of("_,",i+1);
      hIsTwoEndNonterm.set(*this) = ( i!=string::npos && j==string::npos && !Csymb(s.substr(i+1).c_str()).isTerminal() );
      //cerr<<"hIsTwoEndNonterm "<<*this<<" : "<<hIsTwoEndNonterm.get(*this)<<"\n";
    }
  }
 public:
  C ( )                : DiscreteDomainRV<int,domC> ( )    { }
  C ( const char* ps ) : DiscreteDomainRV<int,domC> ( ps ) { calcDetModels(ps); }
  //C ( string s ) : DiscreteDomainRV<int,domC> ( s )  { calcDetModels(s); }
  Csymb getCsymb ( ) const { return hToCsymb.get(*this); }
  Csymb getCfol  ( ) const { return hToCfol.get(*this); }
  C     getCnext ( ) const { return hToCnext.get(*this); }
  bool  hasCnext ( ) const { return hToCnext.contains(*this); }
  bool  isTwoEndNonterm ( ) const { return hIsTwoEndNonterm.get(*this); }
  friend pair<StringInput,C*> operator>> ( StringInput si, C& m ) { return pair<StringInput,C*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,C*> delimbuff, const char* psD ) {
    if ( delimbuff.first == NULL ) return NULL;
    StringInput si=delimbuff.first>>(DiscreteDomainRV<int,domC>&)*delimbuff.second>>psD;
    delimbuff.second->calcDetModels(delimbuff.second->getString()); return si; }
};
SimpleHash<C,Csymb> C::hToCsymb;
SimpleHash<C,Csymb> C::hToCfol;
SimpleHash<C,C>     C::hToCnext;
SimpleHash<C,bool>  C::hIsTwoEndNonterm;
#define C_NONE C("-/-")
#define C_TOP  C("ROOT/START_RIGHTBRANCHPADDING")
//const C C_NONE; //("-/-");
//const C C_TOP; //("ROOT/START_RIGHTBRANCHPADDING");

//// LL: pair of relation labels associated with rule expansion
typedef DelimitedJoint2DRV<psX,L,psTilde,L,psX> LL;
const LL LL_NONE   ( L_NONE, L_NONE );

//// Psymb: a simple phone symbol, e.g. IY, K, ...
//typedef InterVar Psymb;
DiscreteDomain<int> domPsymb;
typedef DiscreteDomainRV<int,domPsymb> Psymb;
const Psymb P_SYMB_NONE("-");

//// P: a sequence of phone symbols yet to be recognized...
DiscreteDomain<int> domP;
class P : public DiscreteDomainRV<int,domP> {
 private:
  static SimpleHash<P,Psymb> hToPsymb;
  static SimpleHash<P,P>     hToPnext;
  void calcDetModels ( string s ) {
    if (!hToPsymb.contains(*this)) {
      size_t i=s.find('/'); assert(i!=string::npos);
      size_t j=s.find("_",i+1); if (j==string::npos) j=s.length();
      hToPsymb.set(*this) = Psymb(s.substr(i+1,j-i-1).c_str());
      //cerr<<"hToPsymb "<<*this<<" : "<<hToPsymb.get(*this)<<"\n";
    }
    if (!hToPnext.contains(*this)) {
      size_t i=s.find('/');
      string sPrefix = s.substr(0,i+1);
      size_t j=s.find('_');
      if ( j!=string::npos ) {
        string sSuffixNew = s.substr(j+1);
        hToPnext.set(*this) = P((sPrefix+sSuffixNew).c_str());
        //cerr<<"hToPnext "<<*this<<" : "<<hToPnext.get(*this)<<"\n";
      }
    }
  }
 public:
  P ( )                : DiscreteDomainRV<int,domP> ( )   { }
  P ( const char* ps ) : DiscreteDomainRV<int,domP> ( ps ) { calcDetModels(ps); }
  //P ( string s ) : DiscreteDomainRV<int,domP> ( s )  { calcDetModels(s); }
  Psymb getPsymb ( ) const { return hToPsymb.get(*this); }
  P     getPnext ( ) const { return hToPnext.get(*this); }
  bool  hasPnext ( ) const { return hToPnext.contains(*this); }
  friend pair<StringInput,P*> operator>> ( StringInput si, P& m ) { return pair<StringInput,P*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,P*> delimbuff, const char* psD ) {
    if ( delimbuff.first == NULL ) return NULL;
    StringInput si=delimbuff.first>>(DiscreteDomainRV<int,domP>&)*delimbuff.second>>psD;
    delimbuff.second->calcDetModels(delimbuff.second->getString()); return si; }
};
SimpleHash<P,Psymb> P::hToPsymb;
SimpleHash<P,P>     P::hToPnext;


//////////////////////////////////////// Joint Variables

//// EF: the referential semantic language model instantiation of the HHMM `reduce' variable at each depth
typedef DelimitedJoint2DRV<psLbrack,E,psRbrack,F,psX> EF;
//// Rrs: collection of syntactic/referential variables at all depths in each `reduce' phase...
typedef DelimitedStaticSafeArray<4,psSemi,EF> Rrs;
//// R: collection of all variables in each `reduce' phase...
typedef DelimitedJoint2DRV<psX,Rrs,psSemiSemi,DelimitedJoint2DRV<psX,F,psSemi,F,psX>,psX> R;

//// LLE: the result of a referential transition -- a label pair and a destination referent
typedef DelimitedJoint2DRV<psX,LL,psComma,E,psX> LLE;
const LLE LLE_NONE(LL_NONE,E_NONE);

//// LE: the stored result of a referential transition -- a label and a destination referent
typedef DelimitedJoint2DRV<psX,L,psComma,E,psX> LE;
const LE LE_NONE(L_NONE,E_NONE);
//// LEC: the referential semantic language model instantiation of the HHMM `shift' variable at each depth
typedef DelimitedJoint2DRV<psLbrack,LE,psRbrack,C,psX> LEC;
//// Srs: collection of syntactic/referential variables at all depths in each `shift' phase...
class Srs : public DelimitedStaticSafeArray<4,psSemi,LEC> {
 public:
  operator C() const {return ((get(3).getSub2()!=C_NONE) ? get(3) : (get(2).getSub2()!=C_NONE) ? get(2) : (get(1).getSub2()!=C_NONE) ? get(1) : get(0)).getSub2();}
  bool compareFinal ( const Srs& srs ) const {
    return ( get(0).getSub2()==srs.get(0).getSub2() &&
             get(1).getSub2()==srs.get(1).getSub2() );
  }
};
//// S: collection of all variables in each `shift' phase...
class S : public DelimitedJoint2DRV<psX,Srs,psSemiSemi,P,psX> {
 public:
  operator Psymb() const {return getSub2().getPsymb();} //mPexpdetP.getDeterm(getSub2());} 
  LEC& set ( int i ) { return setSub1().set(i); }
  bool compareFinal ( const S& s ) const {
    return ( getSub1().get(0).getSub2()==s.getSub1().get(0).getSub2() &&
             getSub1().get(1).getSub2()==s.getSub1().get(1).getSub2() );
  }
};

//// H: the set of all (marginalized) reduce and (modeled) shift variables in the HHMM...
class H : public DelimitedJoint2DRV<psX,R,psDashDiamondDash,S,psX>
{ public:
  operator S() const {return getSub2();}
  operator D() const {return ( (getSub1().getSub2().getSub1()       ==F_0) ? D_5 :
                               (getSub1().getSub1().get(3).getSub2()==F_0) ? D_4 :
                               (getSub1().getSub1().get(2).getSub2()==F_0) ? D_3 :
                               (getSub1().getSub1().get(1).getSub2()==F_0) ? D_2 :
                               (getSub1().getSub1().get(0).getSub2()==F_0) ? D_1 : D_0 );}
};


////////////////////////////////////////////////////////////////////////////////
//
//  Models
//
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////// "Wrapper" models for individual RVs...

//// Model of F^O given Psymb
typedef CPT2DModel<FO,Psymb,LogProb> FOcptModel;

//// Model of F^O given P
class FOModel {
 private:
  FOcptModel mFOcpt;
 public:
  typedef FOcptModel::IterVal IterVal;
  bool setFirst ( FOModel::IterVal& fo, const P& pP ) const {
    if(!mFOcpt.setFirst(fo,pP.getPsymb())) cerr<<"ERROR no condition: FO "<<pP<<endl; return true;
  }
  bool setNext ( FOModel::IterVal& fo, const P& pP ) const {
    return mFOcpt.setNext(fo,pP.getPsymb());
  }
  LogProb getProb ( const FOModel::IterVal& fo, const P& pP ) const {
    return mFOcpt.getProb(fo,pP.getPsymb());
  }
  void clear      ( )                  { mFOcpt.clear(); }
  void subsume    ( FOModel& m )       { mFOcpt.subsume(m.mFOcpt); }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"FO") && mFOcpt.readFields(aps) ); }
  friend pair<StringInput,FOModel*> operator>> ( StringInput si, FOModel& m ) { return pair<StringInput,FOModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,FOModel*> delimbuff, const char* psD ) { return delimbuff.first>>"FO ">>delimbuff.second->mFOcpt>>psD; }
};

//// Model of F^P given F^O and P
class FPModel {
 public:
  typedef FP IterVal;
  bool setFirst ( FPModel::IterVal& fp, const F& fo, const P& pP ) const {
    if (fo==F_1) {fp=(pP.hasPnext())?F_0:F_1; return true;}
    fp=F_0; return true; }
  bool setNext  ( FPModel::IterVal& fp, const F& fo, const P& pP ) const {
    if (fo==F_1 && pP.hasPnext() && pP.getPnext()==P("/SIL?") && fp==F_0) {fp=F_1; return true;}
    return false; }
  LogProb getProb  ( const FPModel::IterVal& fp, const F& fo, const P& pP ) const {
    if (fo==F_1 && pP.hasPnext() && pP.getPnext()==P("/SIL?")) {return (fp==F_0) ? LogProb(0.1) : LogProb(0.9);}
    if (fo==F_1) return (fp==((pP.hasPnext())?F_0:F_1)) ? LogProb(1.0) : LogProb();
    return (fp==F_0) ? LogProb(1.0) : LogProb(); }
  bool readFields ( Array<char*>& aps ) { return false; }
  friend pair<StringInput,FPModel*> operator>> ( StringInput si, FPModel& m ) { return pair<StringInput,FPModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,FPModel*> delimbuff, const char* psD ) { return NULL; }
};

//// Model of E given F and L and E and E
typedef CPT4DModel<E,L,E,E,LogProb> EcptModel;

//// Model of F given D and C  ****NOTE: 09/2008 SWU, no longer includes E dependency
typedef CPT4DModel<F,D,C,Csymb,LogProb> FcptModel;

//// Model of LE given E and C
typedef CPT6DModel<LLE,D,F,Csymb,Csymb,E,LogProb> LLEcptModel;

//// Model of C given D and F and C and Csymb and L
typedef CPT6DModel<C,D,F,C,Csymb,LL,LogProb> CcptModel;

//// Model of C given D and F and C and Csymb and L, making indep assumptions depending on F
class CModel {
 private:
  CcptModel mCcpt;
 public:
  typedef CcptModel::IterVal IterVal;
  bool setFirst ( CModel::IterVal& c, const D& d, const F& f, const C& cP, const Csymb& cU, const LL& ll ) const {
    if ( cU==C_SYMB_NONE || cU.isTerminal() ) { /*cerr<<"spec case\n";*/ c=C_NONE; return true; }                          // SPECIAL: stack bottom
    if (! mCcpt.setFirst ( c, d, f, (f==F_1)?C_NONE:cP.getCnext(), cU, ll ) )
      cerr<<"ERROR no condition: C "<<d<<" "<<f<<" "<<((f==F_1)?C_NONE:cP.getCnext())<<" "<<cU<<" "<<ll<<endl;
    return true;
  }
  bool setNext ( CModel::IterVal& c, const D& d, const F& f, const C& cP, const Csymb& cU, const LL& ll ) const {
    if ( cU==C_SYMB_NONE || cU.isTerminal() ) { return false; }                                                            // SPECIAL: stack bottom
    return mCcpt.setNext  ( c, d, f, (f==F_1)?C_NONE:cP.getCnext(), cU, ll );
  }
  LogProb getProb  ( const CModel::IterVal& c, const D& d, const F& f, const C& cP, const Csymb& cU, const LL& ll ) const {
    if ( cU==C_SYMB_NONE || cU.isTerminal() ) { return (c==C_NONE) ? LogProb(1.0) : LogProb(); }                           // SPECIAL: stack bottom
    LogProb lgpr = mCcpt.getProb  ( c, d, f, (f==F_1)?C_NONE:cP.getCnext(), cU, ll );
    ////cerr << "    ----    finding C "<<d<<" "<<f<<" "<<((f==F_1)?C_NONE:mCnextdetC.getDeterm(cP))<<" "<<cU<<" "<<ll<<" : "<<c<<" = "<<lgpr.toInt()<<"\n";
    return lgpr;
  }
  void clear      ( )                  { mCcpt.clear(); }
  void subsume    ( CModel& m )        { mCcpt.subsume(m.mCcpt); }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"C") && mCcpt.readFields(aps) ); }
  friend pair<StringInput,CModel*> operator>> ( StringInput si, CModel& m ) { return pair<StringInput,CModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,CModel*> delimbuff, const char* psD ) { return delimbuff.first>>"C ">>delimbuff.second->mCcpt>>psD; }
};

//// Model of P given C
typedef CPT2DModel<P,Csymb,LogProb> PcptModel;

//// Model of P given F^O and F^P and C and P
class PModel {
 private:
  PcptModel mPcpt;
 public:
  typedef PcptModel::IterVal IterVal;
  typedef PcptModel::const_iterator const_iterator;
  const PcptModel* getCPT ( ) const { return &mPcpt; }
  bool setFirst ( PModel::IterVal& p, const F& fo, const F& fp, const C& c, const P& pP ) const {
    if (fo==F_1 && fp==F_1) {if(!mPcpt.setFirst(p,c.getCsymb())) cerr<<"ERROR no condition: P "<<c.getCsymb()<<endl; return true;}
    if (fo==F_1) {p=pP.getPnext(); if(p==P("/SIL?"))p=P("/SIL"); return true;}
    p=pP; return true; }
  bool setNext  ( PModel::IterVal& p, const F& fo, const F& fp, const C& c, const P& pP ) const {
    if (fo==F_1 && fp==F_1) return mPcpt.setNext(p,c.getCsymb());
    return false; }
  LogProb getProb  ( const PModel::IterVal& p, const F& fo, const F& fp, const C& c, const P& pP ) const {
    if (fo==F_1 && fp==F_1) return mPcpt.getProb(p,c.getCsymb());
    if (fo==F_1) return (p==pP.getPnext() || pP.getPnext()==P("/SIL?") && p==P("/SIL")) ? LogProb(1.0) : LogProb();
    return (p==pP) ? LogProb(1.0) : LogProb(); }
  bool setFirst ( PModel::IterVal& p, const Csymb& csymb ) const {
    if(!mPcpt.setFirst(p,csymb)) cerr<<"ERROR no condition: P "<<csymb<<endl; return true;
    //    return mPcpt.setFirst(p,csymb);
  }
  bool setNext  ( PModel::IterVal& p, const Csymb& csymb ) const {
    return mPcpt.setNext(p,csymb);
  } 
  LogProb getProb ( const PModel::IterVal& p, const Csymb& csymb ) const {
    return mPcpt.getProb(p,csymb);
  }
  void clear      ( )                  { mPcpt.clear(); }
  void subsume    ( PModel& m )        { mPcpt.subsume(m.mPcpt); }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"P") && mPcpt.readFields(aps) ); }
  friend pair<StringInput,PModel*> operator>> ( StringInput si, PModel& m ) { return pair<StringInput,PModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,PModel*> delimbuff, const char* psD ) { return delimbuff.first>>"P ">>delimbuff.second->mPcpt>>psD; }
};


//////////////////////////////////////// Local models...

//// Local Model of Er given F and F and L and E and E and E
class LocalErModel {
 private:
  boost::mutex mutexWorldModel;
  EcptModel mEcpt;
 public:
  typedef E RVType;
  typedef EcptModel::IterVal IterVal;

  boost::mutex& setMutex ( )                 { return mutexWorldModel; }
  void          clear    ( )                 { boost::mutex::scoped_lock lock(mutexWorldModel); mEcpt.clear(); }
  void          subsume  ( LocalErModel& m ) { boost::mutex::scoped_lock lock(mutexWorldModel); mEcpt.subsume(m.mEcpt); }

  bool setFirst ( LocalErModel::IterVal& e, const F& fD, const L& l, const E& eD, const E& eP, const E& eU ) const {
    if (fD!=F_0 && eU==E_TOP )                 { e=eP; return true; }  // SPECIAL CASE: at top of stack
    if (fD!=F_0 && l==L_IDENT && eD!=E_NONE )  { e=eD; return true; }  // SPECIAL CASE: ident just returns lower ref
    if (fD==F_0) { e=eP; return true; }
    if ( PORT_NUM>0 ) {
      bool b;
      { ////cerr<<"waiting for E lock...\n";
        boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
        ////cerr<<"done waiting for E lock.\n";
        b=mEcpt.setFirst(e,l,(eD!=E_NONE)?eD:eP,eU);
      }
      // Block here if model not yet caught up...
      if ( !b ) {
        cerr<<"Waiting on E world model fetch from: "<<l<<" "<<((eD!=E_NONE)?eD:eP)<<" "<<eU<<endl;
        while ( !b ) {
          boost::thread::yield();
          { ////cerr<<"waiting for E lock...\n";
            boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
            ////cerr<<"done waiting for E lock.\n";
            b=mEcpt.setFirst(e,l,(eD!=E_NONE)?eD:eP,eU);
          }
        }
        cerr<<"Received E world model fetch from: "<<l<<" "<<((eD!=E_NONE)?eD:eP)<<" "<<eU<<endl;
      }
      return b;
    }
    else return mEcpt.setFirst(e,l,(eD!=E_NONE)?eD:eP,eU);
  }
  bool setNext ( LocalErModel::IterVal& e, const F& fD, const L& l, const E& eD, const E& eP, const E& eU ) const {
    if (fD!=F_0 && eU==E_TOP )                 { return false; }  // SPECIAL CASE: at top of stack
    if (fD!=F_0 && l==L_IDENT && eD!=E_NONE )  { return false; }  // SPECIAL CASE: ident just returns lower ref
    if (fD==F_0) { return false; }
    ////cerr<<"waiting for E lock...\n";
    boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
    ////cerr<<"done waiting for E lock.\n";
    return mEcpt.setNext(e,l,(eD!=E_NONE)?eD:eP,eU);
  }
  LogProb getProb ( const LocalErModel::IterVal& e, const F& fD, const L& l, const E& eD, const E& eP, const E& eU ) const {
    if (fD!=F_0 && eU==E_TOP )                 { return (e==eP) ? LogProb(1.0) : LogProb(); }  // SPECIAL CASE: at top of stack
    if (fD!=F_0 && l==L_IDENT && eD!=E_NONE )  { return (e==eD) ? LogProb(1.0) : LogProb(); }  // SPECIAL CASE: ident just returns lower ref
    if (fD==F_0) { return (e==eP) ? LogProb(1.0) : LogProb(); }
    ////cerr<<"waiting for E lock...\n";
    boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
    ////cerr<<"done waiting for E lock.\n";
    return mEcpt.getProb(e,l,(eD!=E_NONE)?eD:eP,eU);
  }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"E") && mEcpt.readFields(aps) ); }
  friend pair<StringInput,LocalErModel*> operator>> ( StringInput si, LocalErModel& m ) { return pair<StringInput,LocalErModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,LocalErModel*> delimbuff, const char* psD ) { return delimbuff.first>>"E ">>delimbuff.second->mEcpt>>psD; }
};


//// Local Model of F given D and F and E and C and C
class LocalFrModel {
 private:
  boost::mutex mutexWorldModel;
  FcptModel mFcpt;
 public:
  typedef FcptModel::IterVal IterVal;

  boost::mutex& setMutex ( )                 { return mutexWorldModel; }
  void          clear    ( )                 { boost::mutex::scoped_lock lock(mutexWorldModel); mFcpt.clear(); }
  void          subsume  ( LocalFrModel& m ) { boost::mutex::scoped_lock lock(mutexWorldModel); mFcpt.subsume(m.mFcpt); }

  bool setFirst ( LocalFrModel::IterVal& f, const D& d, const F& fD, const C& cP, const Csymb& cU ) const {
    if (d==D_1) { f=F_0; return true; }
    if (fD==F_1 && cP==C_NONE )  { f=F_1; /*cerr<<"f=1 default at "<<d<<endl;*/ return true; }  // SPECIAL CASE: unused bottom of stack
    if (fD==F_0) { f=F_0; return true; }
    if ( PORT_NUM>0 ) {
      for ( bool b=false; true; b=true ) {
        { boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
          if ( mFcpt.setFirst(f,d,cP.getCnext(),cU) ) break;
        }
        if ( !b ) cerr<<"Lag: forced to wait on F "<<d<<" "<<cP.getCnext()<<" "<<cU<<" : ... !\n";
        boost::thread::yield();
      }
    }
    return true;
  }
  bool setNext  ( LocalFrModel::IterVal& f, const D& d, const F& fD, const C& cP, const Csymb& cU ) const {
    if (d==D_1) { return false; }
    if (fD==F_1 && cP==C_NONE )  { return false; }                                              // SPECIAL CASE: unused bottom of stack
    if (fD==F_0) { return false; }
    { boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
      return mFcpt.setNext(f,d,cP.getCnext(),cU);
    }
  }
  LogProb getProb  ( const LocalFrModel::IterVal& f, const D& d, const F& fD, const C& cP, const Csymb& cU ) const {
    if (d==D_1) { return (f==F_0) ? LogProb(1.0) : LogProb(); }
    if (fD==F_1 && cP==C_NONE )  { return (f==F_1) ? LogProb(1.0) : LogProb(); }                // SPECIAL CASE: unused bottom of stack
    if (fD==F_0) { return (f==F_0) ? LogProb(1.0) : LogProb(); }
    LogProb lgpr;
    { boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
      lgpr = mFcpt.getProb(f,d,cP.getCnext(),cU);
    }
    ////cerr << "    ----    finding F "<<d<<" "<<fD<<" "<<eD<<" "<<mCnextdetC.getDeterm(cP)<<" "<<cU<<" : "<<f<<" = "<<lgpr.toInt()<<"\n";
    return lgpr;
  }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"F") && mFcpt.readFields(aps) ); }
  friend pair<StringInput,LocalFrModel*> operator>> ( StringInput si, LocalFrModel& m ) { return pair<StringInput,LocalFrModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,LocalFrModel*> delimbuff, const char* psD ) { return delimbuff.first>>"F ">>delimbuff.second->mFcpt>>psD; }
};


//// Local Model of LLE given F and Csymb and Csymb and and E
class LocalLLEModel {
 private:
  boost::mutex mutexWorldModel;
  LLEcptModel mLLEcpt;
 public:
  typedef LLE RVType;
  typedef LLEcptModel::IterVal IterVal;

  boost::mutex& setMutex ( )                  { return mutexWorldModel; }
  void          clear    ( )                  { boost::mutex::scoped_lock lock(mutexWorldModel); mLLEcpt.clear(); }
  void          subsume  ( LocalLLEModel& m ) { boost::mutex::scoped_lock lock(mutexWorldModel); mLLEcpt.subsume(m.mLLEcpt); }

  bool setFirst ( LocalLLEModel::IterVal& lle, const D& d, const F& f, const Csymb& cP, const Csymb& cU, const E& eRD, const E& eU ) const {
    if ( f==F_1 && (cU==C_SYMB_NONE || cU.isTerminal()) ) { lle=LLE_NONE; return true; }                          // SPECIAL CASE: stack bottom
    bool b;
    if ( PORT_NUM>0 ) {
      { ////cerr<<"waiting for LLE lock...\n";
        boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
        ////cerr<<"done waiting for LLE lock.\n";
        b=mLLEcpt.setFirst ( lle, d, f, (f==F_1)?C_SYMB_NONE:cP, cU, (f==F_1)?eU:eRD );
      }
      // Block here if model not yet caught up...
      if ( !b ) {
        cerr<<"Waiting on LLE world model fetch from: "<<d<<" "<<f<<" "<<((f==F_1)?C_SYMB_NONE:cP)<<" "<<cU<<" "<<((f==F_1)?eU:eRD)<<endl;
        while ( !b ) {
          boost::thread::yield();
          { ////cerr<<"waiting for LLE lock...\n";
            boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
            ////cerr<<"done waiting for LLE lock.\n";
            b=mLLEcpt.setFirst ( lle, d, f, (f==F_1)?C_SYMB_NONE:cP, cU, (f==F_1)?eU:eRD );
          }
        }
        cerr<<"Received LLE world model fetch from: "<<d<<" "<<f<<" "<<((f==F_1)?C_SYMB_NONE:cP)<<" "<<cU<<" "<<((f==F_1)?eU:eRD)<<endl;
      }
    }
    else {
      b = mLLEcpt.setFirst ( lle, d, f, (f==F_1)?C_SYMB_NONE:cP, cU, (f==F_1)?eU:eRD );
    }
    return b;
  }
  bool setNext ( LocalLLEModel::IterVal& lle, const D& d, const F& f, const Csymb& cP, const Csymb& cU, const E& eRD, const E& eU ) const {
    if ( f==F_1 && (cU==C_SYMB_NONE || cU.isTerminal()) ) { return false; }                                       // SPECIAL CASE: stack bottom
    bool b;
    { ////cerr<<"waiting for LLE lock...\n";
      boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
      ////cerr<<"done waiting for LLE lock.\n";
      b=mLLEcpt.setNext ( lle, d, f, (f==F_1)?C_SYMB_NONE:cP, cU, (f==F_1)?eU:eRD );
    }
    return b;
  }
  LogProb getProb ( const LocalLLEModel::IterVal& lle, const D& d, const F& f, const Csymb& cP, const Csymb& cU, const E& eRD, const E& eU ) const {
    if ( f==F_1 && (cU==C_SYMB_NONE || cU.isTerminal()) ) { return (lle==LLE_NONE) ? LogProb(1.0) : LogProb(); }  // SPECIAL CASE: stack bottom
    LogProb lgpr;
    { ////cerr<<"waiting for LLE lock...\n";
      boost::mutex::scoped_lock lock(const_cast<boost::mutex&>(mutexWorldModel));
      ////cerr<<"done waiting for LLE lock.\n";
      lgpr = mLLEcpt.getProb ( lle, d, f, (f==F_1)?C_SYMB_NONE:cP, cU, (f==F_1)?eU:eRD );
      ////cerr << "    ----    finding LLE "<<d<<" "<<f<<" "<<((f==F_1)?C_SYMB_NONE:cP)<<" "<<cU<<" "<<((f==F_1)?eU:eRD)<<" : "<<lle<<" = "<<lgpr.toInt()<<"\n";
    }
    return lgpr;
  }
  bool readFields ( Array<char*> aps ) { return ( 0==strcmp(aps[0],"LLE") && mLLEcpt.readFields(aps) ); }
  friend pair<StringInput,LocalLLEModel*> operator>> ( StringInput si, LocalLLEModel& m ) { return pair<StringInput,LocalLLEModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,LocalLLEModel*> delimbuff, const char* psD ) { return delimbuff.first>>"LLE ">>delimbuff.second->mLLEcpt>>psD; }
};


//////////////////////////////////////// Joint models...

//////////////////// Reduce phase...

//// Model of EF=E,F given EF and LEC and LEC
class EFModel : public DoubleFactoredModel<LocalErModel,LocalFrModel> {
 public:
  typedef EF RVType;
  typedef ComplexDoubleIteratedModeledRV<psLbrack,LocalErModel::IterVal,psRbrack,LocalFrModel::IterVal,psX> IterVal;
  bool setFirst ( EFModel::IterVal& ef, const D& d, const EFModel::IterVal& efRD, const LEC& lecP, const LEC& lecU ) const {
    const LocalErModel& mE = getM1();
    const LocalFrModel& mF = getM2();
    const LocalErModel::IterVal&  eRD = efRD.iter_first;
    const LocalFrModel::IterVal&  fRD = efRD.iter_second;
    const L    lP  = lecP.first.first; //DetModeledL(cP);
    const E&   eP  = lecP.first.second;
    const C&   cP  = lecP.second;
    const E&   eU  = lecU.first.second;
    const C&   cU  = lecU.second;
    LocalErModel::IterVal& eR  = ef.iter_first;
    LocalFrModel::IterVal& fR  = ef.iter_second;
    if (fRD==F_1) {
      //cerr<<"deciding ef^d_t="<<ef<<" given d="<<d<<" ef^d+1_t="<<efRD<<" ef^d_t="<<ef<<" lec^d_t-1="<<lecP<<" lec^d-1_t="<<lecU<<"\n";
      return ( mE.setFirst(eR,fRD,lP,eRD,eP,eU) && mF.setFirst(fR,d,fRD,cP,cU.getCsymb()) );
    }
    fR=F_0; eR=eU; return true;
  }
  bool setNext ( EFModel::IterVal& ef, const D& d, const EFModel::IterVal& efRD, const LEC& lecP, const LEC& lecU ) const {
    const LocalErModel& mE = getM1();
    const LocalFrModel& mF = getM2();
    const LocalErModel::IterVal&  eRD = efRD.iter_first;
    const LocalFrModel::IterVal&  fRD = efRD.iter_second;
    const L    lP  = lecP.first.first; //DetModeledL(cP);
    const E&   eP  = lecP.first.second;
    const C&   cP  = lecP.second;
    const E&   eU  = lecU.first.second;
    const C&   cU  = lecU.second;
    LocalErModel::IterVal& eR  = ef.iter_first;
    LocalFrModel::IterVal& fR  = ef.iter_second;
    if (fRD==F_1) {
      //cerr<<"deciding lec="<<"??"<<" given d="<<d<<" ef^d+1_t="<<efRD<<" ef^d_t="<<ef<<" lec^d_t-1="<<lecP<<" lec^d-1_t="<<lecU<<"\n";
      return ( (                                     mF.setNext(fR,d,fRD,cP,cU.getCsymb()) ) ||
               ( mE.setNext(eR,fRD,lP,eRD,eP,eU) && mF.setFirst(fR,d,fRD,cP,cU.getCsymb()) ) );
    }
    return false;
  }
  LogProb getProb ( const EFModel::IterVal& ef, const D& d, const EFModel::IterVal& efRD, const LEC& lecP, const LEC& lecU ) const {
    const LocalErModel& mE = getM1();
    const LocalFrModel& mF = getM2();
    const LocalErModel::IterVal&  eRD = efRD.iter_first;
    const LocalFrModel::IterVal&  fRD = efRD.iter_second;
    const L    lP  = lecP.first.first; //DetModeledL(cP);
    const E&   eP  = lecP.first.second;
    const C&   cP  = lecP.second;
    const E&   eU  = lecU.first.second;
    const C&   cU  = lecU.second;
    const LocalErModel::IterVal& eR  = ef.iter_first;
    const LocalFrModel::IterVal& fR  = ef.iter_second;
    if (fRD==F_1) {
      return ( mE.getProb(eR,fRD,lP,eRD,eP,eU) * mF.getProb(fR,d,fRD,cP,cU.getCsymb()) );
    }
    return (fR==F_0 && eR==eU) ? LogProb(1.0) : LogProb();
  }
};


//// Model of Rrs=EF*4 given F and Srs
class RrsModel : public SingleFactoredModel<EFModel> {
 public:
  typedef Rrs RVType;
  typedef ComplexArrayIteratedModeledRV<EFModel::IterVal,psSemi,4> IterVal;
  bool setFirst ( RrsModel::IterVal& rrs, const FPModel::IterVal& fp, const Srs& srsP ) const {
    const EFModel& mEF = getM1();
    const LEC  lecP0 ( LE(L_NONE,E_TOP), C_TOP );
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    EFModel::IterVal  ef5   ( lecP4.first.second, fp );
    EFModel::IterVal& ef4 = rrs.iter_array.set(3);
    EFModel::IterVal& ef3 = rrs.iter_array.set(2);
    EFModel::IterVal& ef2 = rrs.iter_array.set(1);
    EFModel::IterVal& ef1 = rrs.iter_array.set(0);
    return ( mEF.setFirst(ef4,4,ef5,lecP4,lecP3) && mEF.setFirst(ef3,3,ef4,lecP3,lecP2) && mEF.setFirst(ef2,2,ef3,lecP2,lecP1) && mEF.setFirst(ef1,1,ef2,lecP1,lecP0) );
  }
  bool setNext ( RrsModel::IterVal& rrs, const FPModel::IterVal& fp, const Srs& srsP ) const {
    const EFModel& mEF = getM1();
    const LEC  lecP0 ( LE(L_NONE,E_TOP), C_TOP );
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    EFModel::IterVal  ef5   ( lecP4.first.second, fp );
    EFModel::IterVal& ef4 = rrs.iter_array.set(3);
    EFModel::IterVal& ef3 = rrs.iter_array.set(2);
    EFModel::IterVal& ef2 = rrs.iter_array.set(1);
    EFModel::IterVal& ef1 = rrs.iter_array.set(0);
    return ( (                                                                                                                      mEF.setNext(ef1,1,ef2,lecP1,lecP0) ) ||
             (                                                                               mEF.setNext(ef2,2,ef3,lecP2,lecP1) && mEF.setFirst(ef1,1,ef2,lecP1,lecP0) ) ||
             (                                        mEF.setNext(ef3,3,ef4,lecP3,lecP2) && mEF.setFirst(ef2,2,ef3,lecP2,lecP1) && mEF.setFirst(ef1,1,ef2,lecP1,lecP0) ) ||
             ( mEF.setNext(ef4,4,ef5,lecP4,lecP3) && mEF.setFirst(ef3,3,ef4,lecP3,lecP2) && mEF.setFirst(ef2,2,ef3,lecP2,lecP1) && mEF.setFirst(ef1,1,ef2,lecP1,lecP0) ) );
  }
  LogProb getProb ( const RrsModel::IterVal& rrs, const FPModel::IterVal& fp, const Srs& srsP ) const {
    const EFModel& mEF = getM1();
    const LEC  lecP0 ( LE(L_NONE,E_TOP), C_TOP );
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    const EFModel::IterVal  ef5   ( lecP4.first.second, fp );
    const EFModel::IterVal& ef4 = rrs.iter_array.get(3);
    const EFModel::IterVal& ef3 = rrs.iter_array.get(2);
    const EFModel::IterVal& ef2 = rrs.iter_array.get(1);
    const EFModel::IterVal& ef1 = rrs.iter_array.get(0);
    return ( mEF.getProb(ef4,4,ef5,lecP4,lecP3) * mEF.getProb(ef3,3,ef4,lecP3,lecP2) * mEF.getProb(ef2,2,ef3,lecP2,lecP1) * mEF.getProb(ef1,1,ef2,lecP1,lecP0) );
  }
};


//// Model of R=Rrs,FP,FO given S
class RModel : public TripleFactoredModel<RrsModel,FPModel,FOModel> {
 public:
  typedef R RVType;
  typedef ComplexTripleIteratedModeledRV<psX,RrsModel::IterVal,psSemi,FPModel::IterVal,psSemi,FOModel::IterVal,psX> IterVal;
  bool setFirst ( RModel::IterVal& r, const S& sP ) const {
    const RrsModel& mRrs = getM1();
    const FPModel&  mFP  = getM2();
    const FOModel&  mFO  = getM3();
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second;
    RrsModel::IterVal& rrs = r.iter_first;
    FPModel::IterVal&  fp  = r.iter_second;
    FOModel::IterVal&  fo  = r.iter_third;
    return ( mFO.setFirst(fo,pP) && mFP.setFirst(fp,fo,pP) && mRrs.setFirst(rrs,fp,srsP) );
  }
  bool setNext ( RModel::IterVal& r, const S& sP ) const {
    const RrsModel& mRrs = getM1();
    const FPModel&  mFP  = getM2();
    const FOModel&  mFO  = getM3();
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second;
    RrsModel::IterVal& rrs = r.iter_first;
    FPModel::IterVal&  fp  = r.iter_second;
    FOModel::IterVal&  fo  = r.iter_third;
    return ( (                                                  mRrs.setNext(rrs,fp,srsP) ) ||
             (                        mFP.setNext(fp,fo,pP) && mRrs.setFirst(rrs,fp,srsP) ) ||
             ( mFO.setNext(fo,pP) && mFP.setFirst(fp,fo,pP) && mRrs.setFirst(rrs,fp,srsP) ) );
  }
  LogProb getProb ( const RModel::IterVal& r, const S& sP ) const {
    const RrsModel& mRrs = getM1();
    const FPModel&  mFP  = getM2();
    const FOModel&  mFO  = getM3();
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second;
    const RrsModel::IterVal& rrs = r.iter_first;
    const FPModel::IterVal&  fp  = r.iter_second;
    const FOModel::IterVal&  fo  = r.iter_third;
    return ( mFO.getProb(fo,pP) * mFP.getProb(fp,fo,pP) * mRrs.getProb(rrs,fp,srsP) );
  }
};


//////////////////// Shift phase...

//// Model of LEC given D and EF and F and LEC
class LECModel  : public DoubleFactoredModel<LocalLLEModel,CModel> {
 public:
  typedef LEC RVType;
  typedef ComplexDoubleIteratedModeledRV<psLbrack,LocalLLEModel::IterVal,psRbrack,CModel::IterVal,psX> IterVal;
  bool setFirst ( LECModel::IterVal& lec, const D& d, const EFModel::IterVal& efRD, const EFModel::IterVal& ef, const LEC& lecP, const LECModel::IterVal& lecU ) const {
    const LocalLLEModel& mLLE = getM1();
    const CModel&        mC   = getM2();
    const LocalErModel::IterVal&   eRD = efRD.iter_first;
    const LocalFrModel::IterVal&   fRD = efRD.iter_second;
    //const LocalErModel::IterVal&   eR  = ef.iter_first;
    const LocalFrModel::IterVal&   fR  = ef.iter_second;
    const L&   lP  = lecP.first.first; //DetModeledL(cP);
    const E&   eP  = lecP.first.second;
    const C&   cP  = lecP.second;
    const E&   eU  = lecU.iter_first.second;
    const CModel::IterVal&   cU  = lecU.iter_second;
    LL& ll = lec.iter_first.first;
    E&  e  = lec.iter_first.second;
    LocalLLEModel::IterVal& lle = lec.iter_first;
    CModel::IterVal&        c   = lec.iter_second;
    if (fRD==F_1) {
      if (fR==F_0 && !cP.isTwoEndNonterm()) {ll.first=L_IDENT; ll.second=lP; e=eRD; return ( mC.setFirst(c,d,fR,cP,cU.getCsymb(),LL(L_IDENT,L_IDENT)) ); }
      ////cerr<<"  deciding lec="<<"??"<<" given d="<<d<<" ef^d+1_t="<<efRD<<" ef^d_t="<<ef<<" lec^d_t-1="<<lecP<<" lec^d-1_t="<<lecU<<"\n";
      Csymb csP = (fR==F_0 && cP.isTwoEndNonterm()) ? cP.getCfol() : C_SYMB_NONE ;
      bool b = ( mLLE.setFirst(lle,d,fR,csP,cU.getCsymb(),eRD,eU) && mC.setFirst(c,d,fR,cP,cU.getCsymb(),ll) );
      if (F_0==fR && L_NONE!=lP && L_IDENT==ll.second) ll.second=lP;   // LL-RELATED HACK: don't wipe out sems like "GO"
      ////cerr<<"  deciding lec="<<lec<<" given d="<<d<<" ef^d+1_t="<<efRD<<" ef^d_t="<<ef<<" lec^d_t-1="<<lecP<<" lec^d-1_t="<<lecU<<"\n";
      return b;
    }
    ll.second=lP; e=eP; c=cP; return true;
  }
  bool setNext ( LECModel::IterVal& lec, const D& d, const EFModel::IterVal& efRD, const EFModel::IterVal& ef, const LEC& lecP, const LECModel::IterVal& lecU ) const {
    const LocalLLEModel& mLLE = getM1();
    const CModel&        mC   = getM2();
    const LocalErModel::IterVal&   eRD = efRD.iter_first;
    const LocalFrModel::IterVal&   fRD = efRD.iter_second;
    //const LocalErModel::IterVal&   eR  = ef.iter_first;
    const LocalFrModel::IterVal&   fR  = ef.iter_second;
    const L&   lP  = lecP.first.first; //DetModeledL(cP);
    const C&   cP  = lecP.second;
    const E&   eU  = lecU.iter_first.second;
    const CModel::IterVal&   cU  = lecU.iter_second;
    LL& ll = lec.iter_first.first;
    //E&  e  = lec.iter_first.second;
    LocalLLEModel::IterVal& lle = lec.iter_first;
    CModel::IterVal&        c   = lec.iter_second;
    if (fRD==F_1) {
      if (fR==F_0 && !cP.isTwoEndNonterm()) return ( mC.setNext(c,d,fR,cP,cU.getCsymb(),LL(L_IDENT,L_IDENT)) );
      Csymb csP = (fR==F_0 && cP.isTwoEndNonterm()) ? cP.getCfol() : C_SYMB_NONE ;
      bool b = ( (                                                     mC.setNext(c,d,fR,cP,cU.getCsymb(),ll) ) ||
                 ( mLLE.setNext(lle,d,fR,csP,cU.getCsymb(),eRD,eU) && mC.setFirst(c,d,fR,cP,cU.getCsymb(),ll) ) );
      if (F_0==fR && L_NONE!=lP && L_IDENT==ll.second) ll.second=lP;   // LL-RELATED HACK: don't wipe out sems like "GO"
      return b;
    }
    return false;
  }
  LogProb getProb ( const LECModel::IterVal& lec, const D& d, const EFModel::IterVal& efRD, const EFModel::IterVal& ef, const LEC& lecP, const LECModel::IterVal& lecU ) const {
    const LocalLLEModel& mLLE = getM1();
    const CModel&        mC   = getM2();
    const LocalErModel::IterVal&  eRD = efRD.iter_first;
    const LocalFrModel::IterVal&  fRD = efRD.iter_second;
    //const LocalErModel::IterVal&  eR  = ef.iter_first;
    const LocalFrModel::IterVal&  fR  = ef.iter_second;
    const L&  lP  = lecP.first.first; //DetModeledL(cP);
    const E&  eP  = lecP.first.second;
    const C&  cP  = lecP.second;
    const E&  eU  = lecU.iter_first.second;
    const CModel::IterVal&  cU  = lecU.iter_second;
    const LL& ll  = lec.iter_first.first;
    const E&  e   = lec.iter_first.second;
    const LocalLLEModel::IterVal& lle = lec.iter_first;
    const CModel::IterVal&        c   = lec.iter_second;
    LogProb pLE,pC;
    if (fRD==F_1) {
      if (fR==F_0 && !cP.isTwoEndNonterm()) return (ll.second==lP && e==eRD) ? LogProb(1.0)*mC.getProb(c,d,fR,cP,cU.getCsymb(),ll) : LogProb();
      Csymb csP = (fR==F_0 && cP.isTwoEndNonterm()) ? cP.getCfol() : C_SYMB_NONE ;
      return ( (pLE=mLLE.getProb(lle,d,fR,csP,cU.getCsymb(),eRD,eU)) * (pC=mC.getProb(c,d,fR,cP,cU.getCsymb(),ll)) );
    }
    return (ll.second==lP && e==eP && c==cP) ? LogProb(1.0) : LogProb();
  }
};


//// Model of Srs=LEC*4 given F and Rrs and Srs
class SrsModel : public SingleFactoredModel<LECModel> {
 public:
  typedef Srs RVType;
  typedef ComplexArrayIteratedModeledRV<LECModel::IterVal,psSemi,4> IterVal;
  bool setFirst ( SrsModel::IterVal& srs, const FPModel::IterVal& fp, const RrsModel::IterVal& rrs, const Srs& srsP ) const {
    const LECModel& mLEC = getM1();
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    const EFModel::IterVal   ef5   ( lecP4.first.second, fp );
    const EFModel::IterVal&  ef4   = rrs.iter_array.get(3);
    const EFModel::IterVal&  ef3   = rrs.iter_array.get(2);
    const EFModel::IterVal&  ef2   = rrs.iter_array.get(1);
    const EFModel::IterVal&  ef1   = rrs.iter_array.get(0);
    LECModel::IterVal  lec0 ( LLE(LL_NONE,E_TOP), C_TOP );
    LECModel::IterVal& lec1 = srs.iter_array.set(0);
    LECModel::IterVal& lec2 = srs.iter_array.set(1);
    LECModel::IterVal& lec3 = srs.iter_array.set(2);
    LECModel::IterVal& lec4 = srs.iter_array.set(3);
    return ( mLEC.setFirst(lec1,1,ef2,ef1,lecP1,lec0) && mLEC.setFirst(lec2,2,ef3,ef2,lecP2,lec1) && mLEC.setFirst(lec3,3,ef4,ef3,lecP3,lec2) && mLEC.setFirst(lec4,4,ef5,ef4,lecP4,lec3) );
  }
  bool setNext ( SrsModel::IterVal& srs, const FPModel::IterVal& fp, const RrsModel::IterVal& rrs, const Srs& srsP ) const {
    const LECModel& mLEC = getM1();
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    const EFModel::IterVal   ef5   ( lecP4.first.second, fp );
    const EFModel::IterVal&  ef4   = rrs.iter_array.get(3);
    const EFModel::IterVal&  ef3   = rrs.iter_array.get(2);
    const EFModel::IterVal&  ef2   = rrs.iter_array.get(1);
    const EFModel::IterVal&  ef1   = rrs.iter_array.get(0);
    LECModel::IterVal  lec0 ( LLE(LL_NONE,E_TOP), C_TOP );
    LECModel::IterVal& lec1 = srs.iter_array.set(0);
    LECModel::IterVal& lec2 = srs.iter_array.set(1);
    LECModel::IterVal& lec3 = srs.iter_array.set(2);
    LECModel::IterVal& lec4 = srs.iter_array.set(3);
    return ( (                                                                                                                                     mLEC.setNext(lec4,4,ef5,ef4,lecP4,lec3) ) ||
             (                                                                                         mLEC.setNext(lec3,3,ef4,ef3,lecP3,lec2) && mLEC.setFirst(lec4,4,ef5,ef4,lecP4,lec3) ) ||
             (                                             mLEC.setNext(lec2,2,ef3,ef2,lecP2,lec1) && mLEC.setFirst(lec3,3,ef4,ef3,lecP3,lec2) && mLEC.setFirst(lec4,4,ef5,ef4,lecP4,lec3) ) ||
             ( mLEC.setNext(lec1,1,ef2,ef1,lecP1,lec0) && mLEC.setFirst(lec2,2,ef3,ef2,lecP2,lec1) && mLEC.setFirst(lec3,3,ef4,ef3,lecP3,lec2) && mLEC.setFirst(lec4,4,ef5,ef4,lecP4,lec3) ) );
  }
  LogProb getProb ( const SrsModel::IterVal& srs, const FPModel::IterVal& fp, const RrsModel::IterVal& rrs, const Srs& srsP ) const {
    const LECModel& mLEC = getM1();
    const LEC& lecP1 = srsP.get(0);
    const LEC& lecP2 = srsP.get(1);
    const LEC& lecP3 = srsP.get(2);
    const LEC& lecP4 = srsP.get(3);
    const EFModel::IterVal   ef5   ( lecP4.first.second, fp );
    const EFModel::IterVal&  ef4   = rrs.iter_array.get(3);
    const EFModel::IterVal&  ef3   = rrs.iter_array.get(2);
    const EFModel::IterVal&  ef2   = rrs.iter_array.get(1);
    const EFModel::IterVal&  ef1   = rrs.iter_array.get(0);
    const LECModel::IterVal  lec0 ( LLE(LL_NONE,E_TOP), C_TOP );
    const LECModel::IterVal& lec1 = srs.iter_array.get(0);
    const LECModel::IterVal& lec2 = srs.iter_array.get(1);
    const LECModel::IterVal& lec3 = srs.iter_array.get(2);
    const LECModel::IterVal& lec4 = srs.iter_array.get(3);
    return ( mLEC.getProb(lec1,1,ef2,ef1,lecP1,lec0) * mLEC.getProb(lec2,2,ef3,ef2,lecP2,lec1) * mLEC.getProb(lec3,3,ef4,ef3,lecP3,lec2) * mLEC.getProb(lec4,4,ef5,ef4,lecP4,lec3) );
  }
};


//// Clex: lowest non-null value on stack...
const CModel::IterVal& cLex ( const SrsModel::IterVal& srs ) {
  return ( (C_NONE!=srs.iter_array.get(3).iter_second) ? srs.iter_array.get(3).iter_second :
           (C_NONE!=srs.iter_array.get(2).iter_second) ? srs.iter_array.get(2).iter_second :
           (C_NONE!=srs.iter_array.get(1).iter_second) ? srs.iter_array.get(1).iter_second :
                                                         srs.iter_array.get(0).iter_second ) ;
}


//// Model of S=Srs,P,Q given R and S
class SModel : public DoubleFactoredModel<SrsModel,PModel> {
 public:
  typedef S RVType;
  typedef ComplexDoubleIteratedModeledRV<psX,SrsModel::IterVal,psSemi,PModel::IterVal,psX> IterVal;
  bool setFirst ( SModel::IterVal& s, const RModel::IterVal& r, const S& sP ) const {
    const SrsModel& mSrs = getM1();
    const PModel&   mP   = getM2();
    const RrsModel::IterVal& rrs  = r.iter_first;
    const FPModel::IterVal&  fp   = r.iter_second;
    const FOModel::IterVal&  fo   = r.iter_third;
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second; 
    SrsModel::IterVal& srs = s.iter_first;
    PModel::IterVal&   p   = s.iter_second;
    return ( mSrs.setFirst(srs,fp,rrs,srsP) && mP.setFirst(p,fo,fp,cLex(srs),pP) );
  }
  bool setNext ( SModel::IterVal& s, const RModel::IterVal& r, const S& sP ) const {
    const SrsModel& mSrs = getM1();
    const PModel&   mP   = getM2();
    const RrsModel::IterVal& rrs  = r.iter_first;
    const FPModel::IterVal&  fp   = r.iter_second;
    const FOModel::IterVal&  fo   = r.iter_third;
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second;
    SrsModel::IterVal& srs = s.iter_first;
    PModel::IterVal&   p   = s.iter_second;
    return ( (                                   mP.setNext(p,fo,fp,cLex(srs),pP) ) ||
             ( mSrs.setNext(srs,fp,rrs,srsP) && mP.setFirst(p,fo,fp,cLex(srs),pP) ) );
  }
  LogProb getProb ( const SModel::IterVal& s, const RModel::IterVal& r, const S& sP ) const {
    const SrsModel& mSrs = getM1();
    const PModel&   mP   = getM2();
    const RrsModel::IterVal& rrs  = r.iter_first;
    const FPModel::IterVal&  fp   = r.iter_second;
    const FOModel::IterVal&  fo   = r.iter_third;
    const Srs& srsP = sP.first;
    const P&   pP   = sP.second;
    const SrsModel::IterVal& srs = s.iter_first;
    const PModel::IterVal&   p   = s.iter_second;
    return ( mSrs.getProb(srs,fp,rrs,srsP) * mP.getProb(p,fo,fp,cLex(srs),pP) );
  }
};


//////////////////// Overall...

//// Model of H=R,S given S
class HModel : public DoubleFactoredModel<RModel,SModel> {
 public:
  typedef H RVType;
  typedef ComplexDoubleIteratedModeledRV<psX,RModel::IterVal,psDashDiamondDash,SModel::IterVal,psX> IterVal;
  S& setTrellDat ( S& s, const HModel::IterVal& h ) const {
    for(int i=0;i<4;i++){
      s.setSub1().set(i).setSub1().setSub1()=h.iter_second.iter_first.iter_array.get(i).iter_first.first.second;    // L
      s.setSub1().set(i).setSub1().setSub2()=h.iter_second.iter_first.iter_array.get(i).iter_first.second;    // E
      s.setSub1().set(i).setSub2()=h.iter_second.iter_first.iter_array.get(i).iter_second;   // C
    }
    s.setSub2()=h.iter_second.iter_second;
    return s;
  }
  D setBackDat ( const HModel::IterVal& h ) const {
    return ( (h.iter_first.iter_second                             ==F_0) ? D_5 :
             (h.iter_first.iter_first.iter_array.get(3).iter_second==F_0) ? D_4 :
             (h.iter_first.iter_first.iter_array.get(2).iter_second==F_0) ? D_3 :
             (h.iter_first.iter_first.iter_array.get(1).iter_second==F_0) ? D_2 :
             (h.iter_first.iter_first.iter_array.get(0).iter_second==F_0) ? D_1 : D_0 );
  }
  bool setFirst ( HModel::IterVal& h, const S& sP ) const {
    const RModel& mR = getM1();
    const SModel& mS = getM2();
    RModel::IterVal& r = h.iter_first;
    SModel::IterVal& s = h.iter_second;
    return ( mR.setFirst(r,sP) && mS.setFirst(s,r,sP) );
  }
  bool setNext ( HModel::IterVal& h, const S& sP ) const {
    const RModel& mR = getM1();
    const SModel& mS = getM2();
    RModel::IterVal& r = h.iter_first;
    SModel::IterVal& s = h.iter_second;
    return ( (                      mS.setNext(s,r,sP) ) ||
             ( mR.setNext(r,sP) && mS.setFirst(s,r,sP) ) );
  }
  LogProb getProb ( const HModel::IterVal& h, const S& sP ) const {
    const RModel& mR = getM1();
    const SModel& mS = getM2();
    const RModel::IterVal& r = h.iter_first;
    const SModel::IterVal& s = h.iter_second;
    return ( mR.getProb(r,sP) * mS.getProb(s,r,sP) );
  }
};


////////////////////////////////////////////////////////////////////////////////
//
//  Reader
//
////////////////////////////////////////////////////////////////////////////////

/*
bool readHiddenFields ( Array<char*>& aps ) {
//  if ( 0==strcmp(aps[0],"C") ) {
//    mCexpdetC.readField(aps[6]);
//    mZdetC.readField(aps[6]);
//    mCnextdetC.readField(aps[6]);
//    if ( mZdetC.getDeterm(aps[6])==Z_1 )
//      mCfoldetC.readField(aps[6]);
//  }
  return ( //(0==strcmp(aps[0],"P") && mPexpdetP.readField(aps[2])) ||
           //(0==strcmp(aps[0],"P") && mPdetP.readField(aps[2]))    ||
           (0==strcmp(aps[0],"FO")  && mFOcpt.readFields(aps)) ||
           (0==strcmp(aps[0],"F")   && mFcpt.readFields(aps))  ||
           (0==strcmp(aps[0],"E")   && mEcpt.readFields(aps))  ||
           (0==strcmp(aps[0],"LLE") && mLLEcpt.readFields(aps)) ||
           (0==strcmp(aps[0],"C")   && mCcpt.readFields(aps))  ||
           (0==strcmp(aps[0],"P")   && mPcpt.readFields(aps))  );
}
*/

