import torch
import torch.nn as nn
from collections import OrderedDict
from functools import partial


# https://github1s.com/huggingface/pytorch-image-models/blob/main/timm/models/vision_transformer.py

class Attention(nn.Module):
    def __init__(self,
                 dim,  # 输入token的dim
                 num_heads=8,  # head的个数
                 qkv_bias=False,  # 生成qkv时是否使用偏置
                 qk_scale=None,
                 attn_drop_ratio=0.,  # 两个dropout ratio
                 proj_drop_ratio=0.):
        super(Attention, self).__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads  # 每个head的dim
        self.scale = qk_scale or head_dim ** -0.5  # 不去传入qkscale，也就是1/√dim_k
        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)  # 使用一个全连接层，一次得到qkv
        self.attn_drop_ratio = attn_drop_ratio
        self.attn_drop = nn.Dropout(attn_drop_ratio)
        self.proj = nn.Linear(dim, dim)  # 把多个head进行Concat操作，然后通过Wo映射，这里用全连接层代替
        self.proj_drop = nn.Dropout(proj_drop_ratio)

    def forward(self, x):
        # [batch_size, num_patches + 1, total_embed_dim] 加1代表类别，针对ViT-B/16，dim是768
        B, N, C = x.shape

        # qkv(): -> [batch_size, num_patches + 1, 3 * total_embed_dim]
        # reshape: -> [batch_size, num_patches + 1, 3（代表qkv）, num_heads（代表head数）, embed_dim_per_head（每个head的qkv维度）]
        # permute: -> [3, batch_size, num_heads, num_patches + 1, embed_dim_per_head]
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        # [batch_size, num_heads, num_patches + 1, embed_dim_per_head]
        q, k, v = qkv[0], qkv[1], qkv[2]  # make torchscript happy (cannot use tensor as tuple)

        # transpose: -> [batch_size, num_heads, embed_dim_per_head, num_patches + 1]
        # @: multiply -> [batch_size, num_heads, num_patches + 1, num_patches + 1]
        attn = (q @ k.transpose(-2, -1)) * self.scale  # 每个header的q和k相乘，除以√dim_k（相当于norm处理）
        # using DropKey 
        #m_r = torch.ones_like(attn) * self.attn_drop_ratio
        #attn = attn + torch.bernoulli(m_r) * -1e12
        
        attn = attn.softmax(dim=-1)  # 通过softmax处理（相当于对每一行的数据softmax）
        attn = self.attn_drop(attn)  # dropOut层

        # @: multiply -> [batch_size, num_heads, num_patches + 1, embed_dim_per_head]
        # transpose: -> [batch_size, num_patches + 1, num_heads, embed_dim_per_head]
        # reshape: -> [batch_size, num_patches + 1, total_embed_dim]
        x = (attn @ v).transpose(1, 2).reshape(B, N, C)  # 得到的结果和V矩阵相乘（加权求和），reshape相当于把head拼接
        x = self.proj(x)  # 通过全连接进行映射（相当于乘论文中的Wo）
        x = self.proj_drop(x)
        return x


class Mlp(nn.Module):
    """
    MLP as used in Vision Transformer, MLP-Mixer and related networks
    """
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features  # 如果没有传入out features，就默认是in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()  # 默认是GELU激活函数
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


def drop_path(x, drop_prob: float = 0., training: bool = False, scale_by_keep: bool = True):
    """Drop paths (Stochastic Depth) per sample (when applied in main path of residual blocks).
    This is the same as the DropConnect impl I created for EfficientNet, etc networks, however,
    the original name is misleading as 'Drop Connect' is a different form of dropout in a separate paper...
    See discussion: https://github.com/tensorflow/tpu/issues/494#issuecomment-532968956 ... I've opted for
    changing the layer and argument names to 'drop path' rather than mix DropConnect as a layer name and use
    'survival rate' as the argument.
    """
    if drop_prob == 0. or not training:
        return x
    keep_prob = 1 - drop_prob
    shape = (x.shape[0],) + (1,) * (x.ndim - 1)  # work with diff dim tensors, not just 2D ConvNets
    random_tensor = x.new_empty(shape).bernoulli_(keep_prob)
    if keep_prob > 0.0 and scale_by_keep:
        random_tensor.div_(keep_prob)
    return x * random_tensor


class DropPath(nn.Module):
    """Drop paths (Stochastic Depth) per sample  (when applied in main path of residual blocks).
    """
    def __init__(self, drop_prob: float = 0., scale_by_keep: bool = True):
        super(DropPath, self).__init__()
        self.drop_prob = drop_prob
        self.scale_by_keep = scale_by_keep

    def forward(self, x):
        return drop_path(x, self.drop_prob, self.training, self.scale_by_keep)

    def extra_repr(self):
        return f'drop_prob={round(self.drop_prob, 3):0.3f}'


class Block(nn.Module):
    def __init__(self,
                 dim,  # 每个token的维度
                 num_heads,  # head个数
                 mlp_ratio=4.,  # 第一个结点个数是输入节点的四倍
                 qkv_bias=False,  # 是否使用bias
                 qk_scale=None,
                 drop_ratio=0.,  # Attention模块中最后一个全连接层使用的drop_ratio
                 attn_drop_ratio=0.,
                 drop_path_ratio=0.,
                 act_layer=nn.GELU,
                 norm_layer=nn.LayerNorm,
                 parallel=False):
        super(Block, self).__init__()
        self.parallel = parallel
        self.norm1 = norm_layer(dim)  # layer norm
        self.attn = Attention(dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale,
                              attn_drop_ratio=attn_drop_ratio, proj_drop_ratio=drop_ratio)  # Multihead Attention
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path_ratio) if drop_path_ratio > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)  # MLP第一个全连接层的个数
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop_ratio)
        if parallel:
            self.attn2 = Attention(dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale,
                              attn_drop_ratio=attn_drop_ratio, proj_drop_ratio=drop_ratio)  # Multihead Attention
            self.mlp2 = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop_ratio)

    def forward(self, x):
        if self.parallel:  # 由深变宽
            x = x + self.drop_path(self.attn(self.norm1(x))) + self.drop_path(self.attn2(self.norm1(x)))
            x = x + self.drop_path(self.mlp(self.norm2(x))) + self.drop_path(self.mlp2(self.norm2(x)))
        else:
            x = x + self.drop_path(self.attn(self.norm1(x)))
            x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x


# Non-overlapping
class PatchEmbedding(nn.Module):
    def __init__(self, in_chans=3, embed_dim=768, bias=False):
        super().__init__()
        self.proj = nn.Sequential(
            *[
            nn.Conv2d(in_chans, embed_dim//4, kernel_size=4, stride=4, bias=bias),   # H/4 x W/4
            nn.BatchNorm2d(embed_dim//4),   # BN or LN
            #nn.GELU(),
            nn.ReLU(True),
            nn.Conv2d(embed_dim//4, embed_dim//2, kernel_size=2, stride=2, bias=bias), # H/8 x W/8
            nn.BatchNorm2d(embed_dim//2),
            #nn.GELU(),
            nn.ReLU(True),
            nn.Conv2d(embed_dim//2, embed_dim, kernel_size=2, stride=2, bias=bias),    # H/16 x W/16
            nn.BatchNorm2d(embed_dim),
        ])

        self.gmp = nn.AdaptiveMaxPool2d(1)  # added by me

    def forward(self, x): # (BTN, C, H, W)
        x = self.proj(x)
        x = self.gmp(x)
        return x.squeeze(-1).squeeze(-1)


# Overlapping
class EarlyConv(nn.Module):
    def __init__(self, in_chans=3, embed_dim=768, bias=False):
        super().__init__()
        self.conv_layers = nn.Sequential(*[
            nn.Conv2d(in_chans, embed_dim//4, kernel_size=3, stride=2, padding=1, bias=bias),    
            #nn.Conv2d(in_chans, embed_dim//2, kernel_size=3, stride=2, padding=1, bias=bias),    
            nn.BatchNorm2d(embed_dim//4),   # BN or LN
            #nn.BatchNorm2d(embed_dim//2),   # BN or LN
            nn.ReLU(),
            nn.Conv2d(embed_dim//4, embed_dim//2, kernel_size=3, stride=2, padding=1, bias=bias), 
            nn.BatchNorm2d(embed_dim//2),   # BN or LN
            nn.ReLU(),
            nn.Conv2d(embed_dim//2, embed_dim, kernel_size=3, stride=2, padding=1, bias=bias),
            #nn.BatchNorm2d(embed_dim),   # BN or LN
        ])
        self.gmp = nn.AdaptiveMaxPool2d(1)
        #self.gap = nn.AdaptiveAvgPool2d(1)

    def forward(self, x):
        x = self.conv_layers(x)
        mx = self.gmp(x).squeeze(-1).squeeze(-1)
        #ax = self.gap(x).squeeze(-1).squeeze(-1)
        #out = torch.cat((mx, ax), dim=-1).contiguous()
        return mx


class VisionTransformer(nn.Module):
    def __init__(self, num_classes=0, patch_size=16, in_chans=1, embed_dim=768, depth=6, num_heads=8, mlp_ratio=4.0, qkv_bias=True,
                 qk_scale=None, repr_size=None, drop_ratio=0.1,
                 attn_drop_ratio=0.1, drop_path_ratio=0.1, norm_layer=None,
                 act_layer=None, patch_flatten=False):
        super(VisionTransformer, self).__init__()
        self.num_classes = num_classes
        self.num_features = self.embed_dim = embed_dim  # num_features for consistency with other models
        self.num_tokens = 1
        self.patch_flatten = patch_flatten
        # 把一个函数的某些参数给固定住（也就是设置默认值），返回一个新的函数，调用这个新函数会更简单
        norm_layer = norm_layer or partial(nn.LayerNorm, eps=1e-6)  # 为Norm传入默认参数
        act_layer = act_layer or nn.GELU

        #num_patches = 20 
        #self.patch_embed = nn.Linear(in_chans * patch_size * patch_size, embed_dim) if patch_flatten else PatchEmbedding(in_chans, embed_dim)
        #self.patch_embed = nn.Linear(in_chans * patch_size * patch_size, embed_dim) if patch_flatten else EarlyConv(in_chans, embed_dim)
        #self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))  # 构建可训练参数的0矩阵，用于类别
        #self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + self.num_tokens, embed_dim))  # 位置embedding，和concat后的数据一样
        #self.pos_embed = nn.Parameter(torch.zeros(1, num_patches, embed_dim))  # ignoring CLS but using avg-pooling
        self.dropout = nn.Dropout(p=drop_ratio)
        
        dpr = [x.item() for x in torch.linspace(0, drop_path_ratio, depth)]  # 从0到ratio，有depth个元素的等差序列
        self.blocks = nn.Sequential(*[
            Block(dim=embed_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                  drop_ratio=drop_ratio, attn_drop_ratio=attn_drop_ratio, drop_path_ratio=dpr[i],
                  norm_layer=norm_layer, act_layer=act_layer)
            for i in range(depth)
        ])
        self.norm = norm_layer(embed_dim)

        '''
        self.pre_logits = nn.Sequential(OrderedDict([
            ("fc", nn.Linear(embed_dim, repr_size)),
            ("act", nn.Tanh())
        ]))
        '''
        
        #self.pre_logits = nn.Linear(embed_dim, repr_size)

        # Classifier head(s) 线性分类层
        #self.head = nn.Linear(self.num_features, num_classes) if num_classes > 0 else nn.Identity()
        #nn.init.trunc_normal_(self.pos_embed, std=0.02)
        #nn.init.trunc_normal_(self.cls_token, std=0.02)


    '''    
    def forward_features(self, x, pos_embed=None):   # 输入为Landmark Patches：(B, T, Np, C, H, W)
        """
	  x: (B, T, N, C, H, W)
	  pos_embed: (B, T, N, (N-1)2)
	"""
        B, T, N = x.shape[:3]
        # [BT, Np, CHW] -> [BT, Np, embed_dim]
        x = x.flatten(0, 2).contiguous()  # (BTN, CHW)
        x = self.patch_embed(x)  # (BTN, D)
        x = x.reshape(B*T, N, -1) 
        cls_token = self.cls_token.expand(B*T, -1, -1)
        x = torch.cat((cls_token, x), dim=1).contiguous()
        pos_embed = torch.cat((torch.zeros_like(cls_token), pos_embed), dim=1).contiguous()
        x = self.dropout(x + pos_embed)
        x = self.blocks(x)
        x = self.norm(x)
        #return self.pre_logits(x[:, 0]).reshape(B, T, -1)  # 提取cls_token信息
        return x[:, 0].reshape(B, T, -1)  # 提取cls_token信息
    '''

    def forward_features(self, x, pos_embed=None):   # 输入为Landmark Patches：(B, T, Np, C, H, W)
	# x: (B, T, N, C, H, W)
	# pos_embed: (B*T, N, D)
        '''
        B, T, N = x.shape[:3]
        if self.patch_flatten:
            x = x.reshape(B*T, N, -1)
            # [BT, Np, CHW] -> [BT, Np, embed_dim]
            x = self.patch_embed(x)  # (BT, N, D)
        else:
            x = x.flatten(0, 2).contiguous()  # (BTN, CHW)
            x = self.patch_embed(x)  # (BTN, D)
            x = x.reshape(B*T, N, -1) 
        '''
        #pos_embed = self.pos_embed if pos_embed is None else pos_embed
        #x = self.dropout(x + pos_embed)
        B, T = x.shape[:2]
        x = x.flatten(0, 1).contiguous()  # (BT, N, D)
        x = self.dropout(x)
        x = self.blocks(x)
        x = self.norm(x)
        x = torch.mean(x, dim=1)  # global avg pooling
        #return self.pre_logits(x).reshape(B, T, -1)  # 提取cls_token信息
        return x.reshape(B, T, -1)  # 提取cls_token信息

    def forward(self, x, pos_embed=None):
        x = self.forward_features(x, pos_embed)
        #x = self.head(x)
        return x


