﻿using System;
using System.Linq;

namespace SpreadsheetLLM.Heuristic
{
    internal class CellFeatures
    {
        public const string DefaultContentForFomula = "0.00";
        public static readonly CellFeatures EmptyFeatureVec = new CellFeatures();

        public static void ExtractFeatures(CoreSheet sheet, out CellFeatures[,] features, out string[,] cells, out string[,] formula)
        {
            int height = sheet.Height;
            int width = sheet.Width;

            features = new CellFeatures[height, width];
            cells = new string[height, width];
            formula = new string[height, width];

            for (int i = 0; i < height; i++)
            {
                for (int j = 0; j < width; j++)
                {
                    cells[i, j] = sheet[i, j].Value.ToString().Trim();
                    if (cells[i, j] == "" && sheet[i, j].HasFormula)
                        cells[i, j] = DefaultContentForFomula;
                }
            }

            foreach (var area in sheet.MergedAreas)
            {
                for (int i = Math.Max(area.top, 0); i <= area.bottom; i++)
                {
                    for (int j = Math.Max(area.left, 0); j <= area.right; j++)
                    {
                        if (i >= height || j >= width)
                            continue;

                        // In streaming scenario, a merged cell may not have its content (upper-left) cell existing in given chunk.
                        // Then we have to set its cell value to the first avaiable one, which should be an empty cell.
                        cells[i, j] = cells[Math.Max(area.top, 0), Math.Max(area.left, 0)];
                    }
                }
            }

            for (int i = 0; i < height; i++)
            {
                for (int j = 0; j < width; j++)
                {
                    var format = sheet[i, j].Format;
                    var feature = features[i, j] = new CellFeatures()
                    {
                        HasBottomBorder = format.Border.HasBottom,
                        HasTopBorder = format.Border.HasTop,
                        HasLeftBorder = format.Border.HasLeft,
                        HasRightBorder = format.Border.HasRight,
                        HasFillColor = format.FillColor.Name != "Transparent" && format.FillColor.Name != "White",
                        HasFormula = sheet[i, j].HasFormula,
                        TextLength = cells[i, j].Length,
                    };

                    if (feature.TextLength > 0)
                    {
                        feature.AlphabetRatio = (double)cells[i, j].Count(char.IsLetter) / cells[i, j].Length;
                        feature.NumberRatio = (double)cells[i, j].Count(char.IsDigit) / cells[i, j].Length;
                        feature.SpCharRatio = (double)(cells[i, j].Count(x => x == '*' || x == '/' || x == '：' || x == '\\') + cells[i, j].Skip(1).Count(x => x == '-' || x == '+' || x == '(')) / cells[i, j].Length;
                    }

                    // Disable formula processing for now because the absolute address may not reference to actual cell position
                    formula[i, j] = string.Empty/*sheet.Cells[i, j].FormulaA1*/;
                }
            }
        }

        public bool HasBottomBorder { get; set; }

        public bool HasTopBorder { get; set; }

        public bool HasLeftBorder { get; set; }

        public bool HasRightBorder { get; set; }

        public bool HasFillColor { get; set; }

        public bool HasFormula { get; set; }

        public int TextLength { get; set; }

        public bool MarkText => TextLength > 0;

        public double AlphabetRatio { get; set; }

        public double NumberRatio { get; set; }

        public double SpCharRatio { get; set; }
    }
}