from typing import List, Dict, Any, Optional
import math


class WordInfo(object):
    def __init__(self, form: str) -> None:
        self.form = form
        self.frequency = 0
        self.prob = 0.0

        self.left_storage = {}
        self.right_storage = {}
        self.left_entropy = 0.0
        self.right_entropy = 0.0

        self.entropy = 0.0
        self.aggregation = 2e30 * 1.0
    
    @staticmethod
    def increase_frequency(c: str, storage: Dict[str, List]) -> None:
        if c in storage:
            storage[c][0] += 1
        else:
            storage[c] = [1]

    def compute_entropy(self, storage: Dict[str, List]) -> float:
        tmp_entropy = 0.0

        for c, freqs in storage.items():
            p = freqs[0] / float(self.frequency)
            tmp_entropy -= p * math.log(p)
        
        return tmp_entropy

    def update(self, left: str, right: str) -> None:
        self.frequency += 1
        self.increase_frequency(left, self.left_storage)
        self.increase_frequency(right, self.right_storage)

    def compute_probability_entropy(self, length: int) -> None:
        self.prob = self.frequency / float(length)
        self.left_entropy = self.compute_entropy(self.left_storage)
        self.right_entropy = self.compute_entropy(self.right_storage)

        self.entropy = min(self.left_entropy, self.right_entropy)
    
    def compute_aggregation(self, word_cands: Dict[str, Any]) -> None:
        if len(self.form) == 1:
            self.aggregation = math.sqrt(self.prob)
            return
        
        for i in range(1, len(self.form)):
            self.aggregation = min(self.aggregation, self.prob / (word_cands[self.form[:i]].prob * word_cands[self.form[i:]].prob))

    def __str__(self) -> str:
        return self.form


if __name__ == '__main__':
    w = WordInfo('测试')
    print(w)
