import sys, re
import ipdb

INPUT_STYLE_SET = ['special_type', 'en_type', 'event_type', 'event_type_sent', 
                   'keywords', 'triggerword', 'triggers', 'template']
OUTPUT_STYLE_SET = ['trigger:sentence', 'argument:sentence', 'argument:roletype', 'argument:englishrole']
NO_SPACE_LANGS = {"chinese"}
IN_TYPE = {
    'life': '<--life-->', 
    'movement': '<--movement-->', 
    'transaction': '<--transaction-->', 
    'business': '<--business-->', 
    'conflict': '<--conflict-->', 
    'contact': '<--contact-->', 
    'personnel': '<--personnel-->', 
    'justice': '<--justice-->', 
}
IN_SUBTYPE = {
    'be-born': '<--be-born-->', 
    'marry': '<--marry-->', 
    'divorce': '<--divorce-->', 
    'injure': '<--injure-->', 
    'die': '<--die-->', 
    'transport': '<--transport-->', 
    'transfer-ownership': '<--transfer-ownership-->', 
    'transfer-money': '<--transfer-money-->', 
    'start-organization': '<--start-organization-->', 
    'merge-organization': '<--merge-organization-->', 
    'declare-bankruptcy': '<--declare-bankruptcy-->', 
    'end-organization': '<--end-organization-->', 
    'attack': '<--attack-->', 
    'demonstrate': '<--demonstrate-->', 
    'meet': '<--meet-->', 
    'phone-write': '<--phone-write-->', 
    'start-position': '<--start-position-->', 
    'end-position': '<--end-position-->', 
    'nominate': '<--nominate-->', 
    'elect': '<--elect-->', 
    'arrest-jail': '<--arrest-jail-->', 
    'release-parole': '<--release-parole-->', 
    'trial-hearing': '<--trial-hearing-->', 
    'charge-indict': '<--charge-indict-->', 
    'sue': '<--sue-->', 
    'convict': '<--convict-->', 
    'sentence': '<--sentence-->', 
    'fine': '<--fine-->', 
    'execute': '<--execute-->', 
    'extradite': '<--extradite-->', 
    'acquit': '<--acquit-->', 
    'pardon': '<--pardon-->', 
    'appeal': '<--appeal-->', 
}
IN_SEP = {
    'e_type': '<--e_type-->', 
    's_type': '<--s_type-->', 
    'e_sent': '<--e_sent-->', 
    'triggerword': '<--triggerword-->',
    'template': '<--template-->',
}
ROLE_PH_MAP = {
    'Person': 'somebody',
    'Entity': 'some people or some organization',
    'Defendant': 'somebody',
    'Prosecutor': 'some other',
    'Plaintiff': 'some other',
    'Buyer': 'someone',
    'Artifact': 'something',
    'Seller': 'some seller',
    'Destination': 'somewhere',
    'Origin': 'some place',
    'Vehicle': 'some vehicle',
    'Agent': 'somebody or some organization',
    'Attacker': 'some attacker',
    'Target': 'some facility, someone, or some organization',
    'Victim': 'some victim',
    'Instrument': 'some way',
    'Giver': 'someone',
    'Recipient': 'some other',
    'Org': 'some organization',
    'Place': 'somewhere',
    'Adjudicator': 'some adjudicator',
    'Beneficiary': 'someone or some organization'
}
NO_ROLE = "[None]"
AND = "[and]"
CLSSEP = "[CLS_SEP]"

class eve_template_generator():
    def __init__(self, passage, triggers, roles, input_style, output_style, vocab, lang, instance_base=False):
        """
        generate strctured information for events
        
        args:
            passage(List): a list of tokens
            triggers(List): a list of triggers
            roles(List): a list of Roles
            input_style(List): List of elements; elements belongs to INPUT_STYLE_SET
            input_style(List): List of elements; elements belongs to OUTPUT_STYLE_SET
            instance_base(Bool): if instance_base, we generate only one pair (use for trigger generation), else, we generate trigger_base (use for argument generation)
        """
        self.raw_passage = passage
        self.lang = lang
        self.no_space = lang in NO_SPACE_LANGS
        self.triggers = triggers
        self.roles = roles
        self.events = self.process_events(passage, triggers, roles)
        self.input_style = input_style
        self.output_style = output_style
        self.vocab = vocab
        self.event_templates = []
        if instance_base:
            for e_type in self.vocab['event_type_itos']:
                theclass = getattr(sys.modules[__name__], e_type.replace(':', '_').replace('-', '_'), False)
                if theclass:
                    self.event_templates.append(theclass(self.input_style, self.output_style, passage, e_type, self.lang, self.events))
                else:
                    print(e_type)

        else:
            for event in self.events:
                theclass = getattr(sys.modules[__name__], event['event type'].replace(':', '_').replace('-', '_'), False)
                assert theclass
                self.event_templates.append(theclass(self.input_style, self.output_style, event['tokens'], event['event type'], self.lang, event))
        
        self.data = [x.generate_pair(x.trigger_text) for x in self.event_templates]
        self.data = [x for x in self.data if x]
        
    def get_training_data(self):
        return self.data

    def process_events(self, passage, triggers, roles):
        """
        Given a list of token and event annotation, return a list of structured event

        structured_event:
        {
            'trigger text': str,
            'trigger span': (start, end),
            'event type': EVENT_TYPE(str),
            'arguments':{
                ROLE_TYPE(str):[{
                    'argument text': str,
                    'argument span': (start, end)
                }],
                ROLE_TYPE(str):...,
                ROLE_TYPE(str):....
            }
            'passage': PASSAGE
        }
        """
        
        events = {trigger: [] for trigger in triggers}

        for argument in roles:
            trigger = argument[0]
            events[trigger].append(argument)
        
        event_structures = []
        for trigger, arguments in events.items():
            eve_type = trigger[2]
            eve_text = ''.join(passage[trigger[0]:trigger[1]]) if self.no_space else ' '.join(passage[trigger[0]:trigger[1]])
            eve_span = (trigger[0], trigger[1])
            argus = {}
            for argument in arguments:
                role_type = argument[1][2]
                if role_type not in argus.keys():
                    argus[role_type] = []
                argus[role_type].append({
                    'argument text': ''.join(passage[argument[1][0]:argument[1][1]]) if self.no_space else ' '.join(passage[argument[1][0]:argument[1][1]]),
                    'argument span': (argument[1][0], argument[1][1]),
                })
            event_structures.append({
                'trigger text': eve_text,
                'trigger span': eve_span,
                'event type': eve_type,
                'arguments': argus,
                'passage': ''.join(passage) if self.no_space else ' '.join(passage),
                'tokens': passage
            })
        return event_structures

class event_template():
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        self.input_style = input_style
        self.output_style = output_style
        self.output_template = self.get_output_template()
        self.lang = lang
        self.no_space = lang in NO_SPACE_LANGS
        self.passage = ''.join(passage) if self.no_space else ' '.join(passage)
        self.tokens = passage
        self.event_type = event_type
        if gold_event is not None:
            self.gold_event = gold_event
            if isinstance(gold_event, list):
                # instance base
                
                ######################################################
                # need hangle and if considering triggers            #
                ######################################################
                self.trigger_text = " and ".join([x['trigger text'] for x in gold_event if x['event type']==event_type])
                self.trigger_span = [x['trigger span'] for x in gold_event if x['event type']==event_type]
                self.arguments = [x['arguments'] for x in gold_event if x['event type']==event_type]
            else:
                # trigger base
                self.trigger_text = gold_event['trigger text']
                self.trigger_span = [gold_event['trigger span']]
                self.arguments = [gold_event['arguments']]        
        else:
            self.gold_event = None
        
    @classmethod
    def get_keywords(self):
        pass

    def generate_pair(self, query_trigger):
        """
        Generate model input sentence and output sentence pair
        """
        input_str = self.generate_input_str(query_trigger)
        output_str, gold_sample = self.generate_output_str(query_trigger)
        return (input_str, output_str, self.gold_event, gold_sample, self.event_type, self.tokens)

    def generate_input_str(self, query_trigger):
        return None

    def generate_output_str(self, query_trigger):
        return (None, False)

    def decode(self, prediction):
        pass

    def evaluate(self, predict_output):
        assert self.gold_event is not None
        # categorize prediction
        pred_trigger = []
        pred_argument = []
        for pred in predict_output:
            if pred[1] == self.event_type:
                pred_trigger.append(pred)
            else:
                pred_argument.append(pred)
        # trigger score
        gold_tri_num = len(self.trigger_span)
        pred_tris = []
        for pred in pred_trigger:
            pred_span = self.predstr2span(pred[0])
            if pred_span[0] > -1:
                pred_tris.append((pred_span[0], pred_span[1], pred[1]))
        pred_tri_num = len(pred_tris)
        match_tri = 0
        for pred in pred_tris:
            id_flag = False
            for gold_span in self.trigger_span:
                if gold_span[0] == pred[0] and gold_span[1] == pred[1]:
                    id_flag = True
            match_tri += int(id_flag)

        # argument score
        converted_gold = self.get_converted_gold()
        gold_arg_num = len(converted_gold)
        pred_arg = []
        for pred in pred_argument:
            # find corresponding trigger
            pred_span = None
            if isinstance(self.gold_event, list):
                # end2end case
                try:
                    # we need this ``try'' because we cannot gurantee the model will be bug-free on the matching
                    cor_tri = pred_trigger[pred[2]['cor tri cnt']]
                    cor_tri_span = self.predstr2span(pred[0])[0]
                    if cor_tri_span > -1:
                        pred_span = self.predstr2span(pred[0], cor_tri_span)
                    else:
                        continue
                except Exception as e:
                    print(e)
            else:
                # argument only case
                pred_span = self.predstr2span(pred[0], self.trigger_span[0][0])
            if (pred_span is not None) and (pred_span[0] > -1):
                pred_arg.append((pred_span[0], pred_span[1], pred[1]))
        pred_arg = list(set(pred_arg))
        pred_arg_num = len(pred_arg)
        
        target = converted_gold
        match_id = 0
        match_type = 0
        for pred in pred_arg:
            id_flag = False
            id_type = False
            for gold in target:
                if gold[0]==pred[0] and gold[1]==pred[1]:
                    id_flag = True
                    if gold[2] == pred[2]:
                        id_type = True
                        break
            match_id += int(id_flag)
            match_type += int(id_type)
        return {
            'gold_tri_num': gold_tri_num, 
            'pred_tri_num': pred_tri_num,
            'match_tri_num': match_tri,
            'gold_arg_num': gold_arg_num,
            'pred_arg_num': pred_arg_num,
            'match_arg_id': match_id,
            'match_arg_cls': match_type
        }
    
    def get_converted_gold(self):
        converted_gold = []
        for argu in self.arguments:
            for arg_type, arg_list in argu.items():
                for arg in arg_list:
                    converted_gold.append((arg['argument span'][0], arg['argument span'][1], arg_type))
        return list(set(converted_gold))
    
    def predstr2span(self, pred_str, trigger_idx=None):
        sub_words = [_.strip() for _ in pred_str.strip().lower().split()]
        candidates=[]
        for i in range(len(self.tokens)):
            j = 0
            while j < len(sub_words) and i+j < len(self.tokens):
                if self.tokens[i+j].lower() == sub_words[j]:
                    j += 1
                else:
                    break
            if j == len(sub_words):
                candidates.append((i, i+len(sub_words)))
        if len(candidates) < 1:
            return -1, -1
        else:
            if trigger_idx is not None:
                return sorted(candidates, key=lambda x: abs(trigger_idx-x[0]))[0]
            else:
                return candidates[0]

class Life_Be_Born(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['born', 'birth', 'bore']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was born in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'be born', 'chinese': '出生', 'arabic': 'مولود'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'be born', 'chinese': '出生', 'arabic': 'مولود'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['life'], IN_SUBTYPE['be-born'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to life and someone is given birth to.', 
                               'chinese': '事件与生命有关，有人出生。', 
                               'arabic': 'يرتبط الحدث بالحياة ويولد شخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} was born in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was born in ', 1)[0]
                                    person = person.split(' and ')
                                    place = prediction.split(' was born in ', 1)[1].rsplit('.', 1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Life_Marry(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['marry', 'marriage', 'married']
    
    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody got married in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'marry', 'chinese': '结婚', 'arabic': 'الزواج'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'marry', 'chinese': '结婚', 'arabic': 'الزواج'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['life'], IN_SUBTYPE['marry'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to life and someone is married.', 
                               'chinese': '事件与生活有关，有人结婚。', 
                               'arabic': 'الحدث مرتبط بالحياة وشخص متزوج.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} got married in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                    
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                    pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' got married in ', 1)[0]
                                    person = person.split(' and ')
                                    place = prediction.split(' got married in ', 1)[1].rsplit('.', 1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Life_Divorce(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['divorce', 'divorced', 'Divorce']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody divorced in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'divorce', 'chinese': '离婚', 'arabic': 'الطلاق'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'divorce', 'chinese': '离婚', 'arabic': 'الطلاق'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['life'], IN_SUBTYPE['divorce'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to life and someone was divorced.', 
                               'chinese': '事件与生活有关，有人离婚。', 
                               'arabic': 'الحدث مرتبط بالحياة وطلق أحدهم.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} divorced in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                    
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' divorced in ', 1)[0]
                                    person = person.split(' and ')
                                    place = prediction.split(' divorced in ', 1)[1].rsplit('.', 1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                    pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Life_Injure(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['injure', 'wounded', 'hurt']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody or some organization led to some victim injured by some way in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Agent--> [None] </--Agent--> <--Victim--> [None] </--Victim--> <--Instrument--> [None] </--Instrument--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Agent: [None] [CLS_SEP] Victim: [None] [CLS_SEP] Instrument: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'injure', 'chinese': '受伤', 'arabic': 'مصاب'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'injure', 'chinese': '受伤', 'arabic': 'مصاب'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['life'], IN_SUBTYPE['injure'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to life and someone is injured.', 
                               'chinese': '事件与生命有关，有人受伤。', 
                               'arabic': 'يرتبط الحدث بالحياة ويصاب شخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent'],
                            " and ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else ROLE_PH_MAP['Victim'],
                            " and ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else ROLE_PH_MAP['Instrument'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} led to {} injured by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Agent--> {} </--Agent--> <--Victim--> {} </--Victim--> <--Instrument--> {} </--Instrument--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Agent: {} [CLS_SEP] Victim: {} [CLS_SEP] Instrument: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    agent = prediction.split(' led to ', 1)[0]
                                    agent = agent.split(' and ')
                                    victim = (prediction.split(' led to ', 1)[1]).split(' injured by ', 1)[0]
                                    victim = victim.split(' and ')
                                    instrument = ((prediction.split(' led to ', 1)[1]).split(' injured by ', 1)[1]).split(' in ', )[0]
                                    instrument = instrument.split(' and ')
                                    place = ((prediction.split(' led to ', 1)[1]).split(' injured by ', 1)[1]).split(' in ', 1)[1].rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in victim:
                                        if arg != ROLE_PH_MAP['Victim']:
                                            output.append((arg, 'Victim', {'cor tri cnt': a_cnt}))
                                    for arg in instrument:
                                        if arg != ROLE_PH_MAP['Instrument']:
                                            output.append((arg, 'Instrument', {'cor tri cnt': a_cnt}))                    
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except Exception as e:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Life_Die(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['kill', 'death', 'assassination']
    
    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody or some organization led to some victim died by some way in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Agent--> [None] </--Agent--> <--Victim--> [None] </--Victim--> <--Instrument--> [None] </--Instrument--> <--Place--> [None] </--Place-->') # Person is a mislabeled case.
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Agent: [None] [CLS_SEP] Victim: [None] [CLS_SEP] Instrument: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'die', 'chinese': '死', 'arabic': 'موت'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'life', 'chinese': '生活', 'arabic': 'الحياة'}
                    s_types = {'english': 'die', 'chinese': '死', 'arabic': 'موت'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['life'], IN_SUBTYPE['die'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to life and someone died.', 
                               'chinese': '事件与生命有关，有人死亡。', 
                               'arabic': 'الحدث مرتبط بالحياة ومات شخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent'],
                            " and ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else ROLE_PH_MAP['Victim'],
                            " and ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else ROLE_PH_MAP['Instrument'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} led to {} died by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Agent--> {} </--Agent--> <--Victim--> {} </--Victim--> <--Instrument--> {} </--Instrument--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Victim']]) if "Victim" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Agent: {} [CLS_SEP] Victim: {} [CLS_SEP] Instrument: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    agent = prediction.split(' led to ', 1)[0]
                                    agent = agent.split(' and ')
                                    victim = (prediction.split(' led to ', 1)[1]).split(' died by ', 1)[0]
                                    victim = victim.split(' and ')
                                    instrument = ((prediction.split(' led to ', 1)[1]).split(' died by ', 1)[1]).split(' in ', )[0]
                                    instrument = instrument.split(' and ')
                                    place = ((prediction.split(' led to ', 1)[1]).split(' died by ', 1)[1]).split(' in ', 1)[1].rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in victim:
                                        if arg != ROLE_PH_MAP['Victim']:
                                            output.append((arg, 'Victim', {'cor tri cnt': a_cnt}))
                                    for arg in instrument:
                                        if arg != ROLE_PH_MAP['Instrument']:
                                            output.append((arg, 'Instrument', {'cor tri cnt': a_cnt}))         
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except Exception as e:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Movement_Transport(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['travel', 'go', 'move']
    
    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('something was sent to somewhere from some place by some vehicle. somebody or some organization was responsible for the transport.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Artifact--> [None] </--Artifact--> <--Destination--> [None] </--Destination--> <--Origin--> [None] </--Origin--> <--Vehicle--> [None] </--Vehicle--> <--Agent--> [None] </--Agent--> <--Place--> [None] </--Place-->') # Entity is a mislabeled case.
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Artifact: [None] [CLS_SEP] Destination: [None] [CLS_SEP] Origin: [None] [CLS_SEP] Vehicle: [None] [CLS_SEP] Agent: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'movement', 'chinese': '移动', 'arabic': 'حركة'}
                    s_types = {'english': 'transport', 'chinese': '运输', 'arabic': 'المواصلات'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'movement', 'chinese': '移动', 'arabic': 'حركة'}
                    s_types = {'english': 'transport', 'chinese': '运输', 'arabic': 'المواصلات'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['movement'], IN_SUBTYPE['transport'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to movement. The event occurs when a weapon or vehicle is moved from one place to another.', 
                               'chinese': '事件与运动有关，武器或车辆从一个地方移动到另一个地方。', 
                               'arabic': 'الحدث مرتبط بالحركة. يقع الحدث عندما يتم نقل سلاح أو مركبة من مكان إلى آخر.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else ROLE_PH_MAP['Artifact'],
                            " and ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else ROLE_PH_MAP['Destination'],
                            " and ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else ROLE_PH_MAP['Origin'],
                            " and ".join([ a['argument text'] for a in argu['Vehicle']]) if "Vehicle" in argu.keys() else ROLE_PH_MAP['Vehicle'],
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent']
                        )
                        output_texts.append("{} was sent to {} from {} by {}. {} was responsible for the transport.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Vehicle']]) if "Vehicle" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Artifact--> {} </--Artifact--> <--Destination--> {} </--Destination--> <--Origin--> {} </--Origin--> <--Vehicle--> {} </--Vehicle--> <--Agent--> {} </--Agent--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Vehicle']]) if "Vehicle" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Artifact: {} [CLS_SEP] Destination: {} [CLS_SEP] Origin: {} [CLS_SEP] Vehicle: {} [CLS_SEP] Agent: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    artifact = prediction.split(' was sent to ', 1)[0]
                                    artifact = artifact.split(' and ')
                                    destination = (prediction.split(' was sent to ', 1)[1]).split(' from ', 1)[0]
                                    destination = destination.split(' and ')
                                    origin = ((prediction.split(' was sent to ', 1)[1]).split(' from ', 1)[1]).split(' by ', 1)[0]
                                    origin = origin.split(' and ')
                                    vehicle = (((prediction.split(' was sent to ', 1)[1]).split(' from ', 1)[1]).split(' by ', 1)[1]).split('.', 1)[0]
                                    vehicle = vehicle.split(' and ')
                                    remain = (((prediction.split(' was sent to ', 1)[1]).split(' from ', 1)[1]).split(' by ', 1)[1]).split('.', 1)[1]
                                    if 'was responsible for the transport' in remain:
                                        agent = (remain.split(' was responsible for the transport.')[0]).strip()
                                        agent = agent.split(' and ')
                                    else:
                                        agent = []

                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in artifact:
                                        if arg != ROLE_PH_MAP['Artifact']:
                                            output.append((arg, 'Artifact', {'cor tri cnt': a_cnt}))
                                    for arg in destination:
                                        if arg != ROLE_PH_MAP['Destination']:
                                            output.append((arg, 'Destination', {'cor tri cnt': a_cnt}))          
                                    for arg in origin:
                                        if arg != ROLE_PH_MAP['Origin']:
                                            output.append((arg, 'Origin', {'cor tri cnt': a_cnt}))
                                    for arg in vehicle:
                                        if arg != ROLE_PH_MAP['Vehicle']:
                                            output.append((arg, 'Vehicle', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Transaction_Transfer_Ownership(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['sell', 'buy', 'acquire']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('someone got something from some seller in somewhere.') # Beneficiary was throw away because it's too similar to Buyer
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Buyer--> [None] </--Buyer--> <--Artifact--> [None] </--Artifact--> <--Seller--> [None] </--Seller--> <--Place--> [None] </--Place--> <--Beneficiary--> [None] </--Beneficiary-->') # Org is mislabeled
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Buyer: [None] [CLS_SEP] Artifact: [None] [CLS_SEP] Seller: [None] [CLS_SEP] Place: [None] [CLS_SEP] Beneficiary: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'transaction', 'chinese': '交易', 'arabic': 'عملية تجارية'}
                    s_types = {'english': 'transfer ownership', 'chinese': '转让所有权', 'arabic': 'نقل الملكية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'transaction', 'chinese': '交易', 'arabic': 'عملية تجارية'}
                    s_types = {'english': 'transfer ownership', 'chinese': '转让所有权', 'arabic': 'نقل الملكية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['transaction'], IN_SUBTYPE['transfer-ownership'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to transaction. The event occurs when an item or an organization is sold or given to some other.', 
                               'chinese': '事件与交易有关，物品或组织被出售或赠与他人。', 
                               'arabic': 'الحدث مرتبط بالمعاملة. يحدث هذا الحدث عندما يتم بيع عنصر أو منظمة أو منحها إلى جهة أخرى.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        buyer_list = []
                        if "Beneficiary" in argu.keys():
                            buyer_list.extend([ a['argument text'] for a in argu['Beneficiary']])
                        if "Buyer" in argu.keys():
                            buyer_list.extend([ a['argument text'] for a in argu['Buyer']])
                        filler = (
                            " and ".join(buyer_list) if len(buyer_list)>0 else ROLE_PH_MAP['Buyer'],
                            " and ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else ROLE_PH_MAP['Artifact'],
                            " and ".join([ a['argument text'] for a in argu['Seller']]) if "Seller" in argu.keys() else ROLE_PH_MAP['Seller'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} got {} from {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Buyer']]) if "Buyer" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Seller']]) if "Seller" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Beneficiary']]) if "Beneficiary" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Buyer--> {} </--Buyer--> <--Artifact--> {} </--Artifact--> <--Seller--> {} </--Seller--> <--Place--> {} </--Place--> <--Beneficiary--> {} </--Beneficiary-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Buyer']]) if "Buyer" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Artifact']]) if "Artifact" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Seller']]) if "Seller" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Beneficiary']]) if "Beneficiary" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Buyer: {} [CLS_SEP] Artifact: {} [CLS_SEP] Seller: {} [CLS_SEP] Place: {} [CLS_SEP] Beneficiary: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    buyer = prediction.split(' got ', 1)[0]
                                    buyer = buyer.split(' and ')
                                    remain = prediction.split(' got ', 1)[1]

                                    artifact = remain.split(' from ', 1)[0]
                                    artifact = artifact.split(' and ')
                                    remain = remain.split(' from ', 1)[1]

                                    seller = remain.split(' in ', 1)[0]
                                    seller = seller.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')

                                    for arg in buyer:
                                        if arg != ROLE_PH_MAP['Buyer']:
                                            output.append((arg, 'Buyer', {'cor tri cnt': a_cnt}))
                                    for arg in artifact:
                                        if arg != ROLE_PH_MAP['Artifact']:
                                            output.append((arg, 'Artifact', {'cor tri cnt': a_cnt}))
                                    for arg in seller:
                                        if arg != ROLE_PH_MAP['Seller']:
                                            output.append((arg, 'Seller', {'cor tri cnt': a_cnt}))          
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Transaction_Transfer_Money(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['pay', 'donation', 'loan']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('someone paid some other in somewhere. someone or some organization benefits from the transaction.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Giver--> [None] </--Giver--> <--Recipient--> [None] </--Recipient--> <--Place--> [None] </--Place--> <--Beneficiary--> [None] </--Beneficiary-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Giver: [None] [CLS_SEP] Recipient: [None] [CLS_SEP] Place: [None] [CLS_SEP] Beneficiary: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'transaction', 'chinese': '交易', 'arabic': 'عملية تجارية'}
                    s_types = {'english': 'transfer money', 'chinese': '划款', 'arabic': 'تحويل أموال'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'transaction', 'chinese': '交易', 'arabic': 'عملية تجارية'}
                    s_types = {'english': 'transfer money', 'chinese': '划款', 'arabic': 'تحويل أموال'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['transaction'], IN_SUBTYPE['transfer-money'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to transaction. The event occurs when someone is giving, receiving, borrowing, or lending money.', 
                               'chinese': '事件与交易有关，某人给予、接收、借入或借出资金。', 
                               'arabic': 'الحدث مرتبط بالمعاملة. يحدث الحدث عندما يقوم شخص ما بإعطاء أو تلقي أو اقتراض أو إقراض المال.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Giver']]) if "Giver" in argu.keys() else ROLE_PH_MAP['Giver'],
                            " and ".join([ a['argument text'] for a in argu['Recipient']]) if "Recipient" in argu.keys() else ROLE_PH_MAP['Recipient'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Beneficiary']]) if "Beneficiary" in argu.keys() else ROLE_PH_MAP['Beneficiary']
                        )
                        output_texts.append("{} paid {} in {}. {} benefits from the transaction.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Giver']]) if "Giver" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Recipient']]) if "Recipient" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Beneficiary']]) if "Beneficiary" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Giver--> {} </--Giver--> <--Recipient--> {} </--Recipient--> <--Place--> {} </--Place--> <--Beneficiary--> {} </--Beneficiary-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Giver']]) if "Giver" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Recipient']]) if "Recipient" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Beneficiary']]) if "Beneficiary" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Giver: {} [CLS_SEP] Recipient: {} [CLS_SEP] Place: {} [CLS_SEP] Beneficiary: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    giver = prediction.split(' paid ', 1)[0]
                                    giver = giver.split(' and ')
                                    remain = prediction.split(' paid ', 1)[1]

                                    recipient = remain.split(' in ', 1)[0]
                                    recipient = recipient.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.split('. ',1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. ', 1)[1]

                                    beneficiary = remain.rsplit(' benefits from the transaction.', 1)[0]
                                    beneficiary = beneficiary.split(' and ')

                                    for arg in giver:
                                        if arg != ROLE_PH_MAP['Giver']:
                                            output.append((arg, 'Giver', {'cor tri cnt': a_cnt}))
                                    for arg in recipient:
                                        if arg != ROLE_PH_MAP['Recipient']:
                                            output.append((arg, 'Recipient', {'cor tri cnt': a_cnt}))          
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in beneficiary:
                                        if arg != ROLE_PH_MAP['Beneficiary']:
                                            output.append((arg, 'Beneficiary', {'cor tri cnt': a_cnt}))  
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
        return output

class Business_Start_Org(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['founded', 'create', 'launch']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody or some organization launched some organization in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Agent--> [None] </--Agent--> <--Org--> [None] </--Org--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Agent: [None] [CLS_SEP] Org: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'start organization', 'chinese': '开始组织', 'arabic': 'بداية المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'start organization', 'chinese': '开始组织', 'arabic': 'بداية المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['business'], IN_SUBTYPE['start-organization'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a new organization being created.', 
                               'chinese': '事件与正在创建的新组织有关。', 
                               'arabic': 'يرتبط الحدث بمؤسسة جديدة يتم إنشاؤها.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent'],
                            " and ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else ROLE_PH_MAP['Org'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} launched {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Agent--> {} </--Agent--> <--Org--> {} </--Org--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Agent: {} [CLS_SEP] Org: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                    
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    agent = prediction.split(' launched ', 1)[0]
                                    agent = agent.split(' and ')
                                    remain = prediction.split(' launched ', 1)[1]

                                    org = remain.split(' in ', 1)[0]
                                    org = org.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in org:
                                        if arg != ROLE_PH_MAP['Org']:
                                            output.append((arg, 'Org', {'cor tri cnt': a_cnt}))                  
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Business_Merge_Org(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['merge', 'merging', 'merger']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some organization was merged.') # Place is rarely used, so we ignore.
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Org--> [None] </--Org--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Org: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'merge organization', 'chinese': '合并组织', 'arabic': 'دمج المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'merge organization', 'chinese': '合并组织', 'arabic': 'دمج المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['business'], IN_SUBTYPE['merge-organization'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to two or more organization coming together to form a new organization.', 
                               'chinese': '事件与两个或多个组织联合起来组成一个新组织有关。', 
                               'arabic': 'يرتبط الحدث بمنظمتين أو أكثر يجتمعان معًا لتشكيل منظمة جديدة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else ROLE_PH_MAP['Org']
                        )
                        output_texts.append("{} was merged.".format(filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Org--> {} </--Org--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Org: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    org = prediction.split(' was merged.', 1)[0]
                                    org = org.split(' and ')
                                    for arg in org:
                                        if arg != ROLE_PH_MAP['Org']:
                                            output.append((arg, 'Org', {'cor tri cnt': a_cnt}))                  
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Business_Declare_Bankruptcy(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['bankruptcy', 'bankrupt', 'Bankruptcy']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some organization declared bankruptcy.') # Place is rarely used, so we ignore.
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Org--> [None] </--Org--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Org: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'declare bankruptcy', 'chinese': '宣布破产', 'arabic': 'تعلن الإفلاس'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'declare bankruptcy', 'chinese': '宣布破产', 'arabic': 'تعلن الإفلاس'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['business'], IN_SUBTYPE['declare-bankruptcy'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to some organization declaring bankruptcy.', 
                               'chinese': '事件与一些宣布破产的组织有关。', 
                               'arabic': 'يرتبط الحدث ببعض المنظمات التي أعلنت إفلاسها.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else ROLE_PH_MAP['Org']
                        )
                        output_texts.append("{} declared bankruptcy.".format(filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Org--> {} </--Org--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Org: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    org = prediction.split(' declared bankruptcy.', 1)[0]
                                    org = org.split(' and ')
                                    for arg in org:
                                        if arg != ROLE_PH_MAP['Org']:
                                            output.append((arg, 'Org', {'cor tri cnt': a_cnt}))                  
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
        return output

class Business_End_Org(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['dissolve', 'disbanded', 'close'] 

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some organization dissolved at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Org--> [None] </--Org--> <--Place--> [None] </--Place-->') # Agent is mislabel.
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Org: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'end organization', 'chinese': '结束组织', 'arabic': 'نهاية المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'business', 'chinese': '商业', 'arabic': 'اعمال'}
                    s_types = {'english': 'end organization', 'chinese': '结束组织', 'arabic': 'نهاية المنظمة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['business'], IN_SUBTYPE['end-organization'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to some organization ceasing to exist.', 
                               'chinese': '事件与一些不复存在的组织有关。', 
                               'arabic': 'يرتبط الحدث ببعض المنظمات التي لم تعد موجودة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else ROLE_PH_MAP['Org'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} dissolved at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Org--> {} </--Org--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Org']]) if "Org" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Org: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    org = prediction.split(' dissolved at ', 1)[0]
                                    org = org.split(' and ')
                                    remain = prediction.split(' dissolved at ', 1)[1]

                                    place = remain.rsplit('.', 1)[0]
                                    place = place.split(' and ')
                                    for arg in org:
                                        if arg != ROLE_PH_MAP['Org']:
                                            output.append((arg, 'Org', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))        
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
        return output

class Conflict_Attack(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['war', 'attack', 'terrorism']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some attacker attacked some facility, someone, or some organization by some way in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Attacker--> [None] </--Attacker--> <--Target--> [None] </--Target--> <--Instrument--> [None] </--Instrument--> <--Place--> [None] </--Place-->') # victim is mislabel.
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Attacker: [None] [CLS_SEP] Target: [None] [CLS_SEP] Instrument: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'conflict', 'chinese': '冲突', 'arabic': 'نزاع'}
                    s_types = {'english': 'attack', 'chinese': '攻击', 'arabic': 'هجوم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'conflict', 'chinese': '冲突', 'arabic': 'نزاع'}
                    s_types = {'english': 'attack', 'chinese': '攻击', 'arabic': 'هجوم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['conflict'], IN_SUBTYPE['attack'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to conflict and some violent physical act.', 
                               'chinese': '事件与冲突和一些暴力的身体行为有关。', 
                               'arabic': 'يرتبط الحدث بالصراع وبعض الأعمال الجسدية العنيفة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        attacker_list = []
                        if "Attacker" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Attacker']])
                        if "Agent" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Agent']])  
                        filler = (
                            " and ".join(attacker_list) if len(attacker_list)>0 else ROLE_PH_MAP['Attacker'],
                            " and ".join([ a['argument text'] for a in argu['Target']]) if "Target" in argu.keys() else ROLE_PH_MAP['Target'],
                            " and ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else ROLE_PH_MAP['Instrument'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} attacked {} by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        attacker_list = []
                        if "Attacker" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Attacker']])
                        if "Agent" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Agent']])  
                        filler = (
                            f" {AND} ".join(attacker_list) if len(attacker_list)>0 else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Target']]) if "Target" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Attacker--> {} </--Attacker--> <--Target--> {} </--Target--> <--Instrument--> {} </--Instrument--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        attacker_list = []
                        if "Attacker" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Attacker']])
                        if "Agent" in argu.keys():
                            attacker_list.extend([ a['argument text'] for a in argu['Agent']])  
                        filler = (
                            f" {AND} ".join(attacker_list) if len(attacker_list)>0 else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Target']]) if "Target" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Instrument']]) if "Instrument" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Attacker: {} [CLS_SEP] Target: {} [CLS_SEP] Instrument: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))


        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except Exception as e:
                                #print(e)
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    attack = prediction.split(' attacked ', 1)[0]
                                    attack = attack.split(' and ')
                                    remain = prediction.split(' attacked ', 1)[1]

                                    target = remain.split(' by ', 1)[0]
                                    target = target.split(' and ')
                                    remain = remain.split(' by ', 1)[1]

                                    instrument = remain.split(' in ', 1)[0]
                                    instrument = instrument.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in attack:
                                        if arg != ROLE_PH_MAP['Attacker']:
                                            output.append((arg, 'Attacker', {'cor tri cnt': a_cnt}))
                                    for arg in target:
                                        if arg != ROLE_PH_MAP['Target']:
                                            output.append((arg, 'Target', {'cor tri cnt': a_cnt}))
                                    for arg in instrument:
                                        if arg != ROLE_PH_MAP['Instrument']:
                                            output.append((arg, 'Instrument', {'cor tri cnt': a_cnt}))                    
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Conflict_Demonstrate(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['rally', 'protest', 'demonstrate']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some people or some organization protest at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'conflict', 'chinese': '冲突', 'arabic': 'نزاع'}
                    s_types = {'english': 'demonstrate', 'chinese': '示威', 'arabic': 'احتجاجا على المسيرة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'conflict', 'chinese': '冲突', 'arabic': 'نزاع'}
                    s_types = {'english': 'demonstrate', 'chinese': '示威', 'arabic': 'احتجاجا على المسيرة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['conflict'], IN_SUBTYPE['demonstrate'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a large number of people coming together to protest.', 
                               'chinese': '事件与大量民众聚集抗议有关。', 
                               'arabic': 'يتعلق الحدث بعدد كبير من الناس الذين اجتمعوا للاحتجاج.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} protest at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    entity = prediction.split(' protest at ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = prediction.split(' protest at ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass                                    
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output

class Contact_Meet(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['meeting', 'met', 'summit']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some people or some organization met at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'contact', 'chinese': '联络', 'arabic': 'الإتصال'}
                    s_types = {'english': 'meet', 'chinese': '会晤', 'arabic': 'لقاء'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'contact', 'chinese': '联络', 'arabic': 'الإتصال'}
                    s_types = {'english': 'meet', 'chinese': '会晤', 'arabic': 'لقاء'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['contact'], IN_SUBTYPE['meet'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a group of people meeting and interacting with one another face-to-face.', 
                               'chinese': '事件与一群人面对面的会面和互动有关。', 
                               'arabic': 'يرتبط الحدث بمجموعة من الأشخاص يجتمعون ويتفاعلون مع بعضهم البعض وجهًا لوجه.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} met at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    entity = prediction.split(' met at ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = prediction.split(' met at ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass                                    
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output
    
class Contact_Phone_Write(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['call', 'communicate', 'e-mail']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some people or some organization called or texted messages at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'contact', 'chinese': '联络', 'arabic': 'الإتصال'}
                    s_types = {'english': 'phone write', 'chinese': '通话', 'arabic': 'يتصل'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'contact', 'chinese': '联络', 'arabic': 'الإتصال'}
                    s_types = {'english': 'phone write', 'chinese': '通话', 'arabic': 'يتصل'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['contact'], IN_SUBTYPE['phone-write'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to people phone calling or messaging one another.', 
                               'chinese': '事件与人们互相打电话或发短信有关。', 
                               'arabic': 'يرتبط الحدث بمكالمات هاتفية أو مراسلة الأشخاص لبعضهم البعض.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} called or texted messages at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    entity = prediction.split(' called or texted messages at ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = prediction.split(' called or texted messages at ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass                                   
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    
        return output
            
class Personnel_Start_Position(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['hire', 'appoint', 'join']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody got new job and was hired by some people or some organization in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->') #Agent is mislabel
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'start position', 'chinese': '开始职位', 'arabic': 'يبدأ الموقف'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'start position', 'chinese': '开始职位', 'arabic': 'يبدأ الموقف'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['personnel'], IN_SUBTYPE['start-position'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a person begins working for an organization.', 
                               'chinese': '事件与某人开始为组织工作有关。', 
                               'arabic': 'يرتبط الحدث بشخص يبدأ العمل في مؤسسة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} got new job and was hired by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' got new job and was hired by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' got new job and was hired by ', 1)[1]

                                    entity = remain.split(' in ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Personnel_End_Position(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['former', 'laid off', 'fired']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody stopped working for some people or some organization at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'end position', 'chinese': '结束职位', 'arabic': 'ينتهي الموقف'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'end position', 'chinese': '结束职位', 'arabic': 'ينتهي الموقف'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['personnel'], IN_SUBTYPE['end-position'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a person stops working for an organization.', 
                               'chinese': '事件与某人停止为组织工作有关。', 
                               'arabic': 'يرتبط الحدث بشخص توقف عن العمل في مؤسسة'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} stopped working for {} at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' stopped working for ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' stopped working for ', 1)[1]

                                    entity = remain.split(' at ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = remain.split(' at ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Personnel_Nominate(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['named', 'nomination', 'nominate']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was nominated by somebody or some organization to do a job.') # Place is rarely used, so we ignore.
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Agent--> [None] </--Agent--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Agent: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'nominate', 'chinese': '提名', 'arabic': 'ترشح'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'nominate', 'chinese': '提名', 'arabic': 'ترشح'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['personnel'], IN_SUBTYPE['nominate'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a person being nominated for a position.', 
                               'chinese': '事件与被提名职位的人有关。', 
                               'arabic': 'يرتبط الحدث بشخص يتم ترشيحه لمنصب.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent']
                        )
                        output_texts.append("{} was nominated by {} to do a job.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Agent--> {} </--Agent--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE, 
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Agent: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was nominated by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' was nominated by ', 1)[1]

                                    agent = remain.rsplit(' to do a job.',1)[0]
                                    agent = agent.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Personnel_Elect(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['election', 'elect', 'elected']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was elected a position, and the election was voted by some people or some organization in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'elect', 'chinese': '推选', 'arabic': 'انتخب'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'personnel', 'chinese': '人事', 'arabic': 'شؤون الموظفين'}
                    s_types = {'english': 'elect', 'chinese': '推选', 'arabic': 'انتخب'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['personnel'], IN_SUBTYPE['elect'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a candidate wins an election.', 
                               'chinese': '事件与候选人赢得选举有关。', 
                               'arabic': 'يرتبط الحدث بفوز مرشح في الانتخابات.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} was elected a position, and the election was voted by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was elected a position, and the election was voted by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' was elected a position, and the election was voted by ', 1)[1]

                                    entity = remain.split(' in ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Arrest_Jail(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['arrest', 'jail', 'detained']
    
    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was sent to jailed or arrested by somebody or some organization in somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Agent--> [None] </--Agent--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Agent: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'arrest jail', 'chinese': '逮捕', 'arabic': 'يقبض على'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'arrest jail', 'chinese': '逮捕', 'arabic': 'يقبض على'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['arrest-jail'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a person getting arrested or a person being sent to jail.', 
                               'chinese': '事件与一个人被捕或一个人被送进监狱有关。', 
                               'arabic': 'يتعلق الحدث بإلقاء القبض على شخص أو إرسال شخص إلى السجن.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} was sent to jailed or arrested by {} in {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Agent--> {} </--Agent--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Agent: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was sent to jailed or arrested by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' was sent to jailed or arrested by ', 1)[1]

                                    agent = remain.split(' in ', 1)[0]
                                    agent = agent.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except Exception as e:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1

        return output

class Justice_Release_Parole(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['parole', 'release', 'free']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was released by some people or some organization from somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Entity--> [None] </--Entity--> <--Place--> [None] </--Place-->') # defendant and adjudacator is mislabel
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Entity: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'release parole', 'chinese': '假释', 'arabic': 'الإفراج المشروط'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'release parole', 'chinese': '假释', 'arabic': 'الإفراج المشروط'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['release-parole'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': "The event is related to an end to someone's custody in prison.", 
                               'chinese': '事件与结束某人在监狱中的拘留有关。', 
                               'arabic': 'يتعلق الحدث بإنهاء حبس شخص ما في السجن.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} was released by {} from {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Entity--> {} </--Entity--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Entity: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was released by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' was released by ', 1)[1]

                                    entity = remain.split(' from ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = remain.split(' from ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Trial_Hearing(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['trial', 'hearing', 'proceeding']    
    
    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody, prosecuted by some other, faced a trial in somewhere. The hearing was judged by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Prosecutor--> [None] </--Prosecutor--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Prosecutor: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'trial hearing', 'chinese': '审判 公听会', 'arabic': 'جلسة محاكمة علنية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'trial hearing', 'chinese': '审判 公听会', 'arabic': 'جلسة محاكمة علنية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['trial-hearing'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a trial or hearing for someone.', 
                               'chinese': '事件与某人的审判或听证会有关。', 
                               'arabic': 'يرتبط الحدث محاكمة أو جلسة استماع لشخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else ROLE_PH_MAP['Prosecutor'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{}, prosecuted by {}, faced a trial in {}. The hearing was judged by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Prosecutor--> {} </--Prosecutor--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP]Prosecutor: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(', prosecuted by ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(', prosecuted by ', 1)[1]

                                    prosecutor = remain.split(', faced a trial in ', 1)[0]
                                    prosecutor = prosecutor.split(' and ')
                                    remain = remain.split(', faced a trial in ', 1)[1]

                                    place = remain.split('. The hearing was judged by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. The hearing was judged by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in prosecutor:
                                        if arg != ROLE_PH_MAP['Prosecutor']:
                                            output.append((arg, 'Prosecutor', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except Exception as e:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Charge_Indict(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['indict', 'charged', 'accused']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was charged by some other in somewhere. The adjudication was judged by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Prosecutor--> [None] </--Prosecutor--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Prosecutor: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'charge indict', 'chinese': '控告', 'arabic': 'لائحة الاتهام'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'charge indict', 'chinese': '控告', 'arabic': 'لائحة الاتهام'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['charge-indict'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone or some organization being accused of a crime.', 
                               'chinese': '事件与某人或某个组织被指控犯罪有关。', 
                               'arabic': 'يرتبط الحدث بشخص أو منظمة متهمة بارتكاب جريمة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else ROLE_PH_MAP['Prosecutor'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} was charged by {} in {}. The adjudication was judged by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Prosecutor--> {} </--Prosecutor--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Prosecutor']]) if "Prosecutor" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Prosecutor: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' was charged by ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' was charged by ', 1)[1]

                                    prosecutor = remain.split(' in ', 1)[0]
                                    prosecutor = prosecutor.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.split('. The adjudication was judged by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. The adjudication was judged by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in prosecutor:
                                        if arg != ROLE_PH_MAP['Prosecutor']:
                                            output.append((arg, 'Prosecutor', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
        return output

class Justice_Sue(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['sue', 'lawsuit', 'suit']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was sued by some other in somewhere. The adjudication was judged by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Plaintiff--> [None] </--Plaintiff--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->') # Prosecutor is mislabel. 
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Plaintiff: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'sue', 'chinese': '起诉', 'arabic': 'قاضى'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'sue', 'chinese': '起诉', 'arabic': 'قاضى'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['sue'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to a court proceeding that has been initiated and someone sue the other.', 
                               'chinese': '事件与已启动且有人起诉另一方的法庭程序有关。', 
                               'arabic': 'يرتبط الحدث بإجراءات قضائية بدأت ويقوم أحدهم بمقاضاة الآخر.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else ROLE_PH_MAP['Plaintiff'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} was sued by {} in {}. The adjudication was judged by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Plaintiff--> {} </--Plaintiff--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Plaintiff: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' was sued by ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' was sued by ', 1)[1]

                                    plaintiff = remain.split(' in ', 1)[0]
                                    plaintiff = plaintiff.split(' and ')
                                    remain = remain.split(' in ', 1)[1]

                                    place = remain.split('. The adjudication was judged by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. The adjudication was judged by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in plaintiff:
                                        if arg != ROLE_PH_MAP['Plaintiff']:
                                            output.append((arg, 'Plaintiff', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Convict(event_template):

    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['convicted', 'guilty', 'verdict']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was convicted of a crime in somewhere. The adjudication was judged by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'convict', 'chinese': '定罪', 'arabic': 'قناعة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'convict', 'chinese': '定罪', 'arabic': 'قناعة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['convict'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being found guilty of a crime.', 
                               'chinese': '事件与某人被判有罪有关。', 
                               'arabic': 'يتعلق الحدث بإدانة شخص ما بارتكاب جريمة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} was convicted of a crime in {}. The adjudication was judged by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' was convicted of a crime in ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' was convicted of a crime in ', 1)[1]

                                    place = remain.split('. The adjudication was judged by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. The adjudication was judged by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
        return output

class Justice_Sentence(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['sentenced', 'sentencing', 'sentence']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was sentenced to punishment in somewhere. The adjudication was judged by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'sentence', 'chinese': '判决', 'arabic': 'حكم على'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'sentence', 'chinese': '判决', 'arabic': 'حكم على'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['sentence'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being sentenced to punishment because of a crime.', 
                               'chinese': '事件与某人因犯罪被判刑有关。', 
                               'arabic': 'يرتبط الحدث بشخص حكم عليه بعقوبة بسبب جريمة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} was sentenced to punishment in {}. The adjudication was judged by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' was sentenced to punishment in ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' was sentenced to punishment in ', 1)[1]

                                    place = remain.split('. The adjudication was judged by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split('. The adjudication was judged by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Fine(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['fine', 'fined', 'payouts']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some people or some organization in somewhere was ordered by some adjudicator to pay a fine.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Entity--> [None] </--Entity--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->') # Defendant is mislabel.
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Entity: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'fine', 'chinese': '罚款', 'arabic': 'العقوبة المالية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'fine', 'chinese': '罚款', 'arabic': 'العقوبة المالية'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['fine'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being issued a financial punishment.', 
                               'chinese': '事件与某人受到罚款有关。', 
                               'arabic': 'يتعلق الحدث بعقوبة مالية لشخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else ROLE_PH_MAP['Entity'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} in {} was ordered by {} to pay a fine.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Entity--> {} </--Entity--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Entity']]) if "Entity" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Entity: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    entity = prediction.split(' in ', 1)[0]
                                    entity = entity.split(' and ')
                                    remain = prediction.split(' in ', 1)[1]

                                    place = remain.split(' was ordered by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split(' was ordered by ', 1)[1]

                                    adjudicator = remain.rsplit(' to pay a fine.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in entity:
                                        if arg != ROLE_PH_MAP['Entity']:
                                            output.append((arg, 'Entity', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except Exception as e:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Execute(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['execution', 'executed', 'execute']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was executed by somebody or some organization at somewhere.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Agent--> [None] </--Agent--> <--Place--> [None] </--Place-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Agent: [None] [CLS_SEP] Place: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'execute', 'chinese': '处死', 'arabic': 'أعدم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'execute', 'chinese': '处死', 'arabic': 'أعدم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['execute'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being executed to death.', 
                               'chinese': '事件与某人被处死有关。', 
                               'arabic': 'يتعلق الحدث بإعدام شخص ما حتى الموت.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place']
                        )
                        output_texts.append("{} was executed by {} at {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Agent--> {} </--Agent--> <--Place--> {} </--Place-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Agent: {} [CLS_SEP] Place: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was executed by ', 1)[0]
                                    person = person.split(' and ')
                                    remain = prediction.split(' was executed by ', 1)[1]

                                    agent = remain.split(' at ', 1)[0]
                                    agent = agent.split(' and ')
                                    remain = remain.split(' at ', 1)[1]

                                    place = remain.rsplit('.',1)[0]
                                    place = place.split(' and ')
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output
            
class Justice_Extradite(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['extradition', 'extradited', 'extraditing']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was extradicted to somewhere from some place. somebody or some organization was responsible for the extradition.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Person--> [None] </--Person--> <--Destination--> [None] </--Destination--> <--Origin--> [None] </--Origin--> <--Agent--> [None] </--Agent-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Person: [None] [CLS_SEP] Destination: [None] [CLS_SEP] Origin: [None] [CLS_SEP] Agent: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'extradite', 'chinese': '引渡', 'arabic': 'تسليم مجرم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'extradite', 'chinese': '引渡', 'arabic': 'تسليم مجرم'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['extradite'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to justice. The event occurs when a person was extradited from one place to another place.', 
                               'chinese': '事件与正义有关，一个人从一个地方被引渡到另一个地方。', 
                               'arabic': 'الحدث متعلق بالعدالة. يقع الحدث عندما يتم تسليم شخص من مكان إلى مكان آخر.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else ROLE_PH_MAP['Person'],
                            " and ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else ROLE_PH_MAP['Destination'],
                            " and ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else ROLE_PH_MAP['Origin'],
                            " and ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else ROLE_PH_MAP['Agent']
                        )
                        output_texts.append("{} was extradicted to {} from {}. {} was responsible for the extradition.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Person--> {} </--Person--> <--Destination--> {} </--Destination--> <--Origin--> {} </--Origin--> <--Agent--> {} </--Agent-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Person']]) if "Person" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Destination']]) if "Destination" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Origin']]) if "Origin" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Agent']]) if "Agent" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Person: {} [CLS_SEP] Destination: {} [CLS_SEP] Origin: {} [CLS_SEP] Agent: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    person = prediction.split(' was extradicted to ')[0]
                                    person = person.split(' and ')

                                    destination = (prediction.split(' was extradicted to ')[1]).split(' from ')[0]
                                    destination = destination.split(' and ')

                                    origin = ((prediction.split(' was extradicted to ')[1]).split(' from ')[1]).split('.', 1)[0]
                                    origin = origin.split(' and ')

                                    remain = ((prediction.split(' was extradicted to ')[1]).split(' from ')[1]).split('.', 1)[1]

                                    if 'was responsible for' in remain:
                                        agent = (remain.split(' was responsible for the extradition.')[0]).strip()
                                        agent = agent.split(' and ')
                                    else:
                                        agent = []

                                    for arg in agent:
                                        if arg != ROLE_PH_MAP['Agent']:
                                            output.append((arg, 'Agent', {'cor tri cnt': a_cnt}))
                                    for arg in person:
                                        if arg != ROLE_PH_MAP['Person']:
                                            output.append((arg, 'Person', {'cor tri cnt': a_cnt}))
                                    for arg in destination:
                                        if arg != ROLE_PH_MAP['Destination']:
                                            output.append((arg, 'Destination', {'cor tri cnt': a_cnt}))          
                                    for arg in origin:
                                        if arg != ROLE_PH_MAP['Origin']:
                                            output.append((arg, 'Origin', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Acquit(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['acquitted', 'acquittal', 'acquit']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody was acquitted of the charges at somewhere by some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'acquit', 'chinese': '开释', 'arabic': 'تبرئ'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'acquit', 'chinese': '开释', 'arabic': 'تبرئ'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['acquit'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being acquitted.', 
                               'chinese': '事件与某人被无罪释放有关。', 
                               'arabic': 'يتعلق الحدث ببراءة شخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} was acquitted of the charges at {} by {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' was acquitted of the charges at ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' was acquitted of the charges at ', 1)[1]

                                    place = remain.split(' by ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split(' by ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Pardon(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)

    @classmethod
    def get_keywords(self):
        return ['pardon', 'pardoned', 'remission']

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('somebody received a pardon from some adjudicator.') # Place is a rare role, so we ignore them.
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Defendant--> [None] </--Defendant--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Defendant: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'pardon', 'chinese': '赦免', 'arabic': 'عفو'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'pardon', 'chinese': '赦免', 'arabic': 'عفو'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['pardon'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone being pardoned.', 
                               'chinese': '事件与某人被赦免有关。', 
                               'arabic': 'يتعلق الحدث بالعفو عن شخص ما.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else ROLE_PH_MAP['Defendant'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} received a pardon from {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Defendant--> {} </--Defendant--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Defendant']]) if "Defendant" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Defendant: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))

        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    defendant = prediction.split(' received a pardon from ', 1)[0]
                                    defendant = defendant.split(' and ')
                                    remain = prediction.split(' received a pardon from ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')
                                    for arg in defendant:
                                        if arg != ROLE_PH_MAP['Defendant']:
                                            output.append((arg, 'Defendant', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output

class Justice_Appeal(event_template):
    def __init__(self, input_style, output_style, passage, event_type, lang, gold_event=None):
        super().__init__(input_style, output_style, passage, event_type, lang, gold_event)
    
    @classmethod
    def get_keywords(self):
        return ['appeal', 'appealing', 'appeals']  

    def get_output_template(self):
        output_template = ''
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    output_template += ' \n {}'.format('Event trigger is <Trigger>')
                if o_style == 'argument:sentence':
                    output_template += ' \n {}'.format('some other in somewhere appealed the adjudication from some adjudicator.')
                if o_style == 'argument:roletype':
                    output_template += ' \n {}'.format('<--Plaintiff--> [None] </--Plaintiff--> <--Place--> [None] </--Place--> <--Adjudicator--> [None] </--Adjudicator-->')
                if o_style == 'argument:englishrole':
                    output_template += ' \n {}'.format('Plaintiff: [None] [CLS_SEP] Place: [None] [CLS_SEP] Adjudicator: [None]')
        return ('\n'.join(output_template.split('\n')[1:])).strip()

    def generate_input_str(self, query_trigger):
        input_str = self.passage
        for i_style in INPUT_STYLE_SET:
            if i_style in self.input_style:
                if i_style == 'event_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'appeal', 'chinese': '上诉', 'arabic': 'مناشدة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types[self.lang])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types[self.lang])
                if i_style == 'en_type':
                    e_types = {'english': 'justice', 'chinese': '司法', 'arabic': 'عدالة'}
                    s_types = {'english': 'appeal', 'chinese': '上诉', 'arabic': 'مناشدة'}
                    input_str += ' {} {}'.format(IN_SEP['e_type'], e_types['english'])
                    input_str += ' {} {}'.format(IN_SEP['s_type'], s_types['english'])
                if i_style == 'special_type':
                    input_str += ' {} {}'.format(IN_TYPE['justice'], IN_SUBTYPE['appeal'])
                if i_style == 'event_type_sent':
                    e_sents = {'english': 'The event is related to someone appealing the decision of a court.', 
                               'chinese': '事件与某人提出上诉有关。', 
                               'arabic': 'يتعلق الحدث بشخص يستأنف قرار المحكمة.'}
                    input_str += ' {} {}'.format(IN_SEP['e_sent'], e_sents[self.lang])
                if i_style == 'keywords':
                    input_str += ' \n Similar triggers such as {}'.format(', '.join(self.get_keywords()))
                if i_style == 'triggerword':
                    input_str += ' {} {}'.format(IN_SEP['triggerword'], query_trigger)
                if i_style == 'triggers':
                    input_str += ' \n The event trigger word is {}'.format(query_trigger)
                if i_style == 'template':
                    input_str += ' {} {}'.format(IN_SEP['template'], self.output_template)
        return input_str

    def generate_output_str(self, query_trigger):
        assert self.gold_event is not None
        output_str = ''
        gold_sample = False
        for o_style in OUTPUT_STYLE_SET:
            if o_style in self.output_style:
                if o_style == 'trigger:sentence':
                    if self.trigger_text != '':
                        output_str += ' \n Event trigger is {}'.format(self.trigger_text)
                        gold_sample = True
                    else:
                        output_str += ' \n Event trigger is <Trigger>'
                if o_style == 'argument:sentence':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            " and ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else ROLE_PH_MAP['Plaintiff'],
                            " and ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else ROLE_PH_MAP['Place'],
                            " and ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else ROLE_PH_MAP['Adjudicator']
                        )
                        output_texts.append("{} in {} appealed the adjudication from {}.".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:roletype':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("<--Plaintiff--> {} </--Plaintiff--> <--Place--> {} </--Place--> <--Adjudicator--> {} </--Adjudicator-->".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                if o_style == 'argument:englishrole':
                    output_texts = []
                    for argu in self.arguments:
                        filler = (
                            f" {AND} ".join([ a['argument text'] for a in argu['Plaintiff']]) if "Plaintiff" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Place']]) if "Place" in argu.keys() else NO_ROLE,
                            f" {AND} ".join([ a['argument text'] for a in argu['Adjudicator']]) if "Adjudicator" in argu.keys() else NO_ROLE
                        )
                        output_texts.append("Plaintiff: {} [CLS_SEP] Place: {} [CLS_SEP] Adjudicator: {}".format(*filler))
                        gold_sample = True
                    output_str += ' \n {}'.format(' <sep> '.join(output_texts))
                
        output_str = ('\n'.join(output_str.split('\n')[1:])).strip()
        return (output_str, gold_sample)

    def decode(self, preds):
        output = []
        for cnt, pred in enumerate(preds.split('\n')):
            used_o_cnt = 0
            full_pred = pred.strip()
            for o_style in OUTPUT_STYLE_SET:
                if o_style in self.output_style:
                    if o_style == 'trigger:sentence':
                        if used_o_cnt == cnt:
                            try:
                                triggers = full_pred.split('Event trigger is ', 1)[1]
                                triggers = triggers.split(' and ')
                                for t_cnt, t in enumerate(triggers):
                                    if t != '<Trigger>':
                                        output.append((t, self.event_type, {'tri counter': t_cnt})) # (text, type, kwargs)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:sentence':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    plaintiff = prediction.split(' in ', 1)[0]
                                    plaintiff = plaintiff.split(' and ')
                                    remain = prediction.split(' in ', 1)[1]

                                    place = remain.split(' appealed the adjudication from ', 1)[0]
                                    place = place.split(' and ')
                                    remain = remain.split(' appealed the adjudication from ', 1)[1]

                                    adjudicator = remain.rsplit('.',1)[0]
                                    adjudicator = adjudicator.split(' and ')

                                    for arg in plaintiff:
                                        if arg != ROLE_PH_MAP['Plaintiff']:
                                            output.append((arg, 'Plaintiff', {'cor tri cnt': a_cnt}))
                                    for arg in place:
                                        if arg != ROLE_PH_MAP['Place']:
                                            output.append((arg, 'Place', {'cor tri cnt': a_cnt}))
                                    for arg in adjudicator:
                                        if arg != ROLE_PH_MAP['Adjudicator']:
                                            output.append((arg, 'Adjudicator', {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:roletype':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    
                                    tag_s = re.search('<--[^/>][^>]*-->', prediction)
                                    while tag_s:
                                        prediction = prediction[tag_s.end():]
                                        r_type = tag_s.group()[3:-3]
                                        
                                        if r_type in ROLE_PH_MAP:
                                            tag_e = re.search(f'</--{r_type}-->', prediction)
                                            if tag_e:
                                                arg = prediction[:tag_e.start()].strip()
                                                for a in arg.split(f' {AND} '):
                                                    a = a.strip()
                                                    if a != '' and a != NO_ROLE:
                                                        output.append((a, r_type, {'cor tri cnt': a_cnt}))
                                                prediction = prediction[tag_e.end():]
                                        
                                        tag_s = re.search('<--[^/>][^>]*-->', prediction)
                            except:
                                pass
                        used_o_cnt += 1
                    if o_style == 'argument:englishrole':
                        if used_o_cnt == cnt:
                            try:
                                for a_cnt, prediction in enumerate(full_pred.split(' <sep> ')):
                                    for chunk in prediction.split(f" {CLSSEP} "):
                                        r_type = chunk.split(':', 1)[0].strip()
                                        arg = chunk.split(':', 1)[1].strip()
                                        for a in arg.split(f' {AND} '):
                                            a = a.strip()
                                            if a != '' and a != NO_ROLE:
                                                output.append((a, r_type, {'cor tri cnt': a_cnt}))
                            except:
                                pass
                        used_o_cnt += 1
                        
        return output
